/**********************************************************************************/
/* $Id: Msg.c,v 1.8 2003/06/03 23:05:13 sleeper Exp $                             */
/* 										  */
/* MsgProtocol.c								  */
/*										  */
/* Contains the functions that make up the Message Protocol subsystem.		  */
/* This subsystem is responsable for encoding / decoding messages		  */
/* sending/receiving messages and providing accessors that allow		  */
/* the caller to abort transactions						  */
/*										  */
/* This module reflects the combination of functionality taken			  */
/* from the buffer.c and mp_int.c files in the Windows NDIS code.		  */
/*										  */
/* This file is part of the "ADI USB ADSL Driver for Linux".			  */
/*										  */
/* "ADI USB ADSL Driver for Linux" is free software; you can redistribute it      */
/* and/or modify it under the terms of the GNU General Public License as 	  */
/* published by the Free Software Foundation; either version 2 of the License,    */
/* or (at your option) any later version.					  */
/*										  */
/* "ADI USB ADSL Driver for Linux" is distributed in the hope that it will be     */
/* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of	  */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		  */
/* GNU General Public License for more details.					  */
/*										  */
/* You should have received a copy of the GNU General Public License		  */
/* along with "ADI USB ADSL Driver for Linux"; if not, write to the Free Software */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA	  */
/**********************************************************************************/

#include "Adiutil.h"
#include "Msg.h"
#include "Util.h"
#include "debug.h"


#define SEARCHING   0x0
#define PASSED      0x1
#define FAILED      0x2

static const char symb_addresses[NUM_CMVS][4] =
{
    {'A', 'D', 'P', 'T'},
    {'C', 'N', 'T', 'L'},
    {'C', 'O', 'D', 'E'},
    {'D', 'I', 'A', 'G'},
    {'D', 'O', 'P', 'T'},
    {'F', 'L', 'A', 'G'},
    {'I', 'N', 'F', 'O'},
    {'I', 'N', 'T', 'L'},
    {'O', 'P', 'T', 'N'},
    {'P', 'F', 'C', 'L'},
    {'P', 'F', 'R', 'X'},
    {'P', 'F', 'T', 'X'},
    {'P', 'S', 'D', 'M'},
    {'R', 'A', 'T', 'E'},
    {'R', 'X', 'D', 'A'},
    {'S', 'T', 'A', 'T'},
    {'T', 'E', 'S', 'T'},
    {'T', 'O', 'N', 'E'},
    {'T', 'X', 'D', 'A'},
    {'U', 'O', 'P', 'T'}
};


/**********************************************************************************/
/*										  */
/*  NAME: MsgCheckSymbolicAddress						  */
/*            									  */
/*  ABSTRACT: 									  */
/*      Looks up a symbolic address						  */
/*										  */
/*  RETURNS: 									  */
/*      Success(true) or Failure(false)						  */
/*              								  */
/*  ASSUMPTIONS: 								  */
/*      None.									  */
/*										  */
/**********************************************************************************/
Boolean MsgCheckSymbolicAddress(char *saddr)
{
    Boolean ret_code = FALSE;
    unsigned int search_code = SEARCHING;
    unsigned int cmv = 0;
    unsigned int letter;

    /* search the reference array for a match */
    while ( ( cmv < NUM_CMVS ) && ( search_code == SEARCHING ) )
    {
	/* try to match the CMV to the current reference array CMV */
	letter = 0;
	while ( letter < 4 && ( saddr[letter] == symb_addresses[cmv][letter] ) )
	{
	    letter++;
	}

	switch ( letter )
	{
	case 0:
	    {
		/* no match on first char - check that first char of CMV being */
		/* checked is not before the first char of this entry in */
		/* the alphabet, if it is we can stop looking as entries in the */
		/* reference array are in ascending order */
		if (saddr[0] < symb_addresses[cmv][0])
		{
		    /* give up looking, already too far into alphabet */
		    search_code = FAILED;
		    ret_code = FALSE;
		}
		else
		{
		    /* not too far yet, keep looking */
		}
	    }
	    break;
	case 4:
	    {
		/* found a match */
		search_code = PASSED;
		ret_code = TRUE;
	    }
	    break;
	default:
	    {
		/* partial match, keep looking */
	    }
	    break;
	}
	cmv++;
    }
    
    return (ret_code);
}

/**********************************************************************************/
/* Data tables for CMVs								  */
/**********************************************************************************/
#define PREAMBLE                0x535C
#define RECEIVER_MODEM          0x0000
#define SENDER_HOST             0x0010
#define SUBTYPE_MEM_REQWRITE    0x0100
#define SUBTYPE_MEM_REQREAD     0x0000
#define TYPE_MEMACCESS          0x1000

#define WRITE_TO_MODEM (RECEIVER_MODEM + SENDER_HOST + SUBTYPE_MEM_REQWRITE + TYPE_MEMACCESS)
#define READ_FRM_MODEM (RECEIVER_MODEM + SENDER_HOST + SUBTYPE_MEM_REQREAD  + TYPE_MEMACCESS)

/* for MsgSeq_SoftReset */
static CmvMsg MsgCNTL0_RSET = { PREAMBLE, WRITE_TO_MODEM, 0xCDEF, 'C','N','T','L', 0, 0 };

/* for MsgSeq_EnaFR */
static CmvMsg MsgOPTN0EnaFR = { PREAMBLE, WRITE_TO_MODEM, 0x1111, 'O','P','T','N', 0, 0x82800044 };

/* for MsgSeq_Retrainer */
static CmvMsg MsgCNTL0_RSVD = { PREAMBLE, WRITE_TO_MODEM, 0xEEEE, 'C','N','T','L', 0, 1 };
static CmvMsg MsgCNTL0_INIT = { PREAMBLE, WRITE_TO_MODEM, 0xEEEE, 'C','N','T','L', 0, 2 };
static CmvMsg MsgINFO8a     = { PREAMBLE, READ_FRM_MODEM, 0xEEEE, 'I','N','F','O', 8, 0 };
static CmvMsg MsgINFO10     = { PREAMBLE, READ_FRM_MODEM, 0xEEEE, 'I','N','F','O',10, 0 };
static CmvMsg MsgOPTN0      = { PREAMBLE, WRITE_TO_MODEM, 0x1111, 'O','P','T','N', 0, 0x80020066 };
static CmvMsg MsgOPTN2      = { PREAMBLE, WRITE_TO_MODEM, 0x2222, 'O','P','T','N', 2, 0 };
static CmvMsg MsgOPTN4      = { PREAMBLE, WRITE_TO_MODEM, 0x3333, 'O','P','T','N', 4, 0 };
static CmvMsg MsgOPTN7      = { PREAMBLE, WRITE_TO_MODEM, 0x2222, 'O','P','T','N', 7, 0 };
static CmvMsg MsgOPTN3      = { PREAMBLE, WRITE_TO_MODEM, 0x2222, 'O','P','T','N', 3, 0 };
static CmvMsg MsgOPTN5      = { PREAMBLE, WRITE_TO_MODEM, 0x2222, 'O','P','T','N', 5, 0 };
static CmvMsg MsgOPTN6      = { PREAMBLE, WRITE_TO_MODEM, 0x2222, 'O','P','T','N', 6, 0 };
static CmvMsg MsgOPTN15     = { PREAMBLE, WRITE_TO_MODEM, 0x2222, 'O','P','T','N',15, 0 };
static CmvMsg MsgADPT2      = { PREAMBLE, WRITE_TO_MODEM, 0x1234, 'A','D','P','T', 2, 0 };
static CmvMsg MsgINFO9      = { PREAMBLE, WRITE_TO_MODEM, 0x1234, 'I','N','F','O', 9, 0 };
static CmvMsg MsgMASK8      = { PREAMBLE, WRITE_TO_MODEM, 0x1234, 'M','A','S','K', 8, 0 };
static CmvMsg MsgMASK9      = { PREAMBLE, WRITE_TO_MODEM, 0x1234, 'M','A','S','K', 9, 0 };
static CmvMsg MsgPSDM0      = { PREAMBLE, WRITE_TO_MODEM, 0x1234, 'P','S','D','M', 0, 0 };
static CmvMsg MsgPFCL1      = { PREAMBLE, WRITE_TO_MODEM, 0x89AB, 'P','F','C','L', 1, 0 };

/* for MsgSeq_OpStat */
static CmvMsg MsgSTAT0a     = { PREAMBLE, READ_FRM_MODEM, 0x0123, 'S','T','A','T', 0, 0 };
static CmvMsg MsgDIAG2      = { PREAMBLE, READ_FRM_MODEM, 0x1111, 'D','I','A','G', 2, 0 };
static CmvMsg MsgDIAG22     = { PREAMBLE, READ_FRM_MODEM, 0x2222, 'D','I','A','G',22, 0 };
static CmvMsg MsgDIAG23     = { PREAMBLE, READ_FRM_MODEM, 0x3333, 'D','I','A','G',23, 0 };
static CmvMsg MsgDIAG25     = { PREAMBLE, READ_FRM_MODEM, 0x4444, 'D','I','A','G',25, 0 };
static CmvMsg MsgDIAG51     = { PREAMBLE, READ_FRM_MODEM, 0x5555, 'D','I','A','G',51, 0 };
static CmvMsg MsgDIAG52     = { PREAMBLE, READ_FRM_MODEM, 0x6666, 'D','I','A','G',52, 0 };
static CmvMsg MsgDIAG53     = { PREAMBLE, READ_FRM_MODEM, 0x7777, 'D','I','A','G',53, 0 };
static CmvMsg MsgDIAG54     = { PREAMBLE, READ_FRM_MODEM, 0x8888, 'D','I','A','G',54, 0 };
static CmvMsg MsgRATE0      = { PREAMBLE, READ_FRM_MODEM, 0x9999, 'R','A','T','E', 0, 0 };
static CmvMsg MsgDIAG3      = { PREAMBLE, READ_FRM_MODEM, 0xAAAA, 'D','I','A','G', 3, 0 };
static CmvMsg MsgDIAG47     = { PREAMBLE, READ_FRM_MODEM, 0xBBBB, 'D','I','A','G',47, 0 };
static CmvMsg MsgDIAG49     = { PREAMBLE, READ_FRM_MODEM, 0xCCCC, 'D','I','A','G',49, 0 };
static CmvMsg MsgINFOA      = { PREAMBLE, READ_FRM_MODEM, 0xDDDD, 'I','N','F','O',10, 0 };
static CmvMsg MsgPSDM1      = { PREAMBLE, READ_FRM_MODEM, 0xEEEE, 'P','S','D','M', 1, 0 };
static CmvMsg MsgINFO8b     = { PREAMBLE, READ_FRM_MODEM, 0xEEEE, 'I','N','F','O', 8, 0 };
static CmvMsg MsgINFO14     = { PREAMBLE, READ_FRM_MODEM, 0xEEEE, 'I','N','F','O',14, 0 };

/* for MsgSeq_Stat */
static CmvMsg MsgSTAT0b     = { PREAMBLE, READ_FRM_MODEM, 0x7654, 'S','T','A','T', 0, 0 };

/* for MSgSeq_ModemEna */
/*FLASH_ACC + ENA_MODEMREBOOT_RQ + SENDER_HOST + RECEIVER_MODEM = 0x4110 */
static CmvMsg MsgMODEMENA   = { PREAMBLE, 0x4110, 0xDEAD, 'I','N','F','O', 0, 0 };

static CmvMsg lcl_MsgCNTL0_RSET;
static CmvMsg lcl_MsgOPTN0EnaFR;
static CmvMsg lcl_MsgCNTL0_RSVD;
static CmvMsg lcl_MsgCNTL0_INIT;
static CmvMsg lcl_MsgINFO8a;
static CmvMsg lcl_MsgINFO10;
static CmvMsg lcl_MsgOPTN0;
static CmvMsg lcl_MsgOPTN2;
static CmvMsg lcl_MsgOPTN4;
static CmvMsg lcl_MsgOPTN7;
static CmvMsg lcl_MsgOPTN3;
static CmvMsg lcl_MsgOPTN5;
static CmvMsg lcl_MsgOPTN6;
static CmvMsg lcl_MsgOPTN15;
static CmvMsg lcl_MsgADPT2;
static CmvMsg lcl_MsgINFO9;
static CmvMsg lcl_MsgMASK8;
static CmvMsg lcl_MsgMASK9;
static CmvMsg lcl_MsgPSDM0;
static CmvMsg lcl_MsgPFCL1;
static CmvMsg lcl_MsgSTAT0a;
static CmvMsg lcl_MsgDIAG2;
static CmvMsg lcl_MsgDIAG22;
static CmvMsg lcl_MsgDIAG23;
static CmvMsg lcl_MsgDIAG25;
static CmvMsg lcl_MsgDIAG51;
static CmvMsg lcl_MsgDIAG52;
static CmvMsg lcl_MsgDIAG53;
static CmvMsg lcl_MsgDIAG54;
static CmvMsg lcl_MsgRATE0;
static CmvMsg lcl_MsgDIAG3;
static CmvMsg lcl_MsgDIAG47;
static CmvMsg lcl_MsgDIAG49;
static CmvMsg lcl_MsgINFOA;
static CmvMsg lcl_MsgPSDM1;
static CmvMsg lcl_MsgINFO8b;
static CmvMsg lcl_MsgINFO14;
static CmvMsg lcl_MsgSTAT0b;
static CmvMsg lcl_MsgMODEMENA;


static UInt32 MsgInitDone = FALSE;


#define ADDMSG(Seq,Msg)                                 \
    (Seq).MsgMax.Msgs[(Seq).MsgCount++] = &lcl_##Msg;   \
    build_msg ( &lcl_##Msg, &(Msg) );


#define ADDMSG_IFVAR_NOTZERO(Seq,Msg,Var)       \
    if ( (Var) !=0)                             \
    {                                           \
        (Msg).Data = (Var);                     \
        ADDMSG(Seq,Msg);                        \
    }

#define ADDMSG_FROM_CONFIG(Seq,Msg,Var)         \
   (Msg).Data = (Var);                          \
   ADDMSG (Seq,Msg);

#define ADDMSG_FROM_TEXTFILE(Seq,Msg)                   \
    (Seq).MsgMax.Msgs[(Seq).MsgCount++] = &(Msg);       \
    if ( !pHw->MsgInitialized )                         \
    {                                                   \
        build_msg ( &(Msg), &(Msg) );                   \
    }


#define ___swahw32(x) \
({ \
	UInt32 __x = (x); \
	((UInt32)( \
		(((UInt32)(__x) & (UInt32)0x0000ffffUL) << 16) | \
		(((UInt32)(__x) & (UInt32)0xffff0000UL) >> 16) )); \
})

static void build_msg ( CmvMsg *dest, CmvMsg *src );


/**********************************************************************************/
/*										  */
/*  NAME: MsgDecode								  */
/*            									  */
/*  ABSTRACT: 									  */
/*      Decodes the input message into the output message			  */
/*										  */
/*  RETURNS: 									  */
/*      Success(true) or Failure(false)						  */
/*              								  */
/*  ASSUMPTIONS: 								  */
/*      Must be called after the message protocol subsystem			  */
/*      has been intialised.							  */
/*										  */
/**********************************************************************************/
UInt32 MsgDecode(mp_message *Msg, UInt16 MsgRaw[8])
{
    unsigned int ret_code = MP_DECODE_OK;
    unsigned int mp_decode_stage;
    unsigned short temp;
    
    /* start at beginning of encoding */
    mp_decode_stage = MP_STAGE_PREAMBLE;
    
    /* encode the message one stage at a time */
    while ( mp_decode_stage < END_DECODE )
    {
	switch ( mp_decode_stage )
	{
	case MP_STAGE_PREAMBLE:
	    {
		/* set mp_decode_stage to process next word */
		mp_decode_stage = MP_STAGE_FUNCTION_TYPE;
		
		/* extract the preamble */
		Msg->preamble = cpu_to_le16(MsgRaw[MP_RAW_PREAMBLE]);
		
		/* check the preamble */
		if (Msg->preamble != MSG_PREAMBLE)
		{
		    ret_code |= MP_DECODE_PREAMBLE_ERROR;
		    ret_code |= MP_DECODE_ERROR;
		}
	    }
	    break;
	case MP_STAGE_FUNCTION_TYPE:
	    {
		temp = cpu_to_le16(MsgRaw[MP_RAW_FUNCTION]);
		/* set mp_decode_stage to process next word */
		mp_decode_stage = MP_STAGE_FUNCTION_SUBTYPE;
		
		/* extract the type field */
		Msg->type = ( ( MP_FUNCTION_TYPE_MASK & temp ) >>
		    MP_FUNCTION_TYPE_LSB );
		
		/* check the function subfield */
		switch ( Msg->type )
		{
		case MP_FUNCTION_TYPE_MEMACCESS:
		case MP_FUNCTION_TYPE_FLASHACC:
		case MP_FUNCTION_TYPE_ADSLDIRECTIVE:
		    {
			/* legal function types  */
		    }
		    break;
		case MP_FUNCTION_TYPE_MSGDECERR:
		case MP_FUNCTION_TYPE_MSGACCERR:
		case MP_FUNCTION_TYPE_FLASHACCERR:
		    {
			/* legal function types  */
		    }
		    break;
		default:
		    {
			/* illegal function type - set return code */
			ret_code |= MP_DECODE_TYPE_ERROR;
			ret_code |= MP_DECODE_ERROR;
		    }
		    break;
		}
	    }
	    break;
	case MP_STAGE_FUNCTION_SUBTYPE:
	    {
		temp = cpu_to_le16(MsgRaw[MP_RAW_FUNCTION]);
		/* set mp_decode_stage to process next word */
		mp_decode_stage = MP_STAGE_FUNCTION_SENDERID;
		
		/* extract the sub_type field */
		Msg->sub_type = ( ( MP_FUNCTION_SUBTYPE_MASK & temp ) >> MP_FUNCTION_SUBTYPE_LSB );
		
		/* check that the function type was correct, if it */
		/* wasn't, we can't check the sub - type field */
		if ((ret_code & MP_DECODE_TYPE_ERROR) != 0)
		{
		    /* type field was in error, don't check sub - type */
		}
		else
		{
		    switch ( Msg->type )
		    {
		    case MP_FUNCTION_TYPE_MEMACCESS:
			{
			    /* check the function subfield */
			    switch ( Msg->sub_type )
			    {
			    case SUBTYPE_MEMACCESS_REPLYREAD:
			    case SUBTYPE_MEMACCESS_REPLYWRITE:
				{
				    /* legal function subtypes - and into field */
				}
				break;
			    default:
				{
				    /* illegal function type - set return code */
				    ret_code |= MP_DECODE_SUBTYPE_ERROR;
				    ret_code |= MP_DECODE_ERROR;
				}
				break;
			    }
			}
			break;
		    case MP_FUNCTION_TYPE_FLASHACC:
			{
			    /* check the function subfield */
			    switch ( Msg->sub_type )
			    {
			    case SUBTYPE_FLASHACC_ENKERNELREBOOTREPLY:
			    case SUBTYPE_FLASHACC_ENMODEMREBOOTREPLY:
			    case SUBTYPE_FLASHACC_DECOMPMODEMAPPLREPLY:
			    case SUBTYPE_FLASHACC_STREAMEDDATAWRREPLY:
			    case SUBTYPE_FLASHACC_STREAMEDDATAWRBLOCKREPLY:
			    case SUBTYPE_FLASHACC_READREPLY:
				{
				    /* legal function subtypes - and into field */
				}
				break;
			    default:
				{
				    /* illegal function type - set return code */
				    ret_code |= MP_DECODE_SUBTYPE_ERROR;
				    ret_code |= MP_DECODE_ERROR;
				}
				break;
			    }
			}
			break;
		    case MP_FUNCTION_TYPE_FLASHACCERR:
			{
			    /* check the function subfield */
			    switch ( Msg->sub_type )
			    {
			    case SUBTYPE_FLASHACCERR_DECOMPRESSERR:
			    case SUBTYPE_FLASHACCERR_READCOMPERR:
				{
				    /* legal function subtypes - and into field */
				}
				break;
			    default:
				{
				    /* illegal function type - set return code */
				    ret_code |= MP_DECODE_SUBTYPE_ERROR;
				    ret_code |= MP_DECODE_ERROR;
				}
				break;
			    }
			}
			break;
		    case MP_FUNCTION_TYPE_ADSLDIRECTIVE:
			{
			    /* check the function subfield */
			    switch ( Msg->sub_type )
			    {
			    case SUBTYPE_ADSLDIRECTIVE_KERNELREADY:
			    case SUBTYPE_ADSLDIRECTIVE_MODEMREADY:
			    case SUBTYPE_ADSLDIRECTIVE_MODEMCRCERROR:
				{
				    /* legal function subtypes - and into field */
				}
				break;
			    default:
				{
				    /* illegal function type - set return code */
				    ret_code |= MP_DECODE_SUBTYPE_ERROR;
				    ret_code |= MP_DECODE_ERROR;
				}
				break;
			    }
			}
			break;
		    default:
			{
			    /* type is invalid and passed last test - can't happen */
			    /* got a serious error here */
                            adi_dbg (DBG_MSG,"SW Internal ERROR 'msg->type'\n");
                            
			}
			break;
		    }
		}
	    }
	    break;
	case MP_STAGE_FUNCTION_SENDERID:
	    {
		temp = cpu_to_le16(MsgRaw[MP_RAW_FUNCTION]);
		/* set mp_decode_stage to process next word */
		mp_decode_stage = MP_STAGE_FUNCTION_RECEIVERID;
		
		/* extract the sender_id */
		Msg->sender_id = ( ( MP_FUNCTION_SENDERID_MASK & temp ) >> MP_FUNCTION_SENDERID_LSB );
		
		/* check the function subfield */
		switch ( Msg->sender_id )
		{
		case SUBTYPE_ID_ADSL:
		    {
			/* legal function sender_id - and into field */
		    }
		    break;
		default:
		    {
			/* illegal function type - set return code */
			ret_code |= MP_DECODE_SENDERID_ERROR;
			ret_code |= MP_DECODE_ERROR;
		    }
		    break;
		}
	    }
	    break;
	case MP_STAGE_FUNCTION_RECEIVERID:
	    {
		temp = cpu_to_le16(MsgRaw[MP_RAW_FUNCTION]);
		/* set mp_decode_stage to process next word */
		mp_decode_stage = MP_STAGE_INDEX;
		
		/* extract the sender_id */
		Msg->receiver_id = ( ( MP_FUNCTION_RECEIVER_MASK & temp ) >> MP_FUNCTION_RECEIVER_LSB );
		
		/* check the function subfield */
		switch ( Msg->receiver_id )
		{
		case SUBTYPE_ID_ME:
		    {
			/* legal function receiver_id - and into field */
		    }
		    break;
		default:
		    {
			/* illegal function type - set return code */
			ret_code = MP_DECODE_ERROR;
			/* stop processing the message */
			mp_decode_stage = END_DECODE;
		    }
		    break;
		}
		if (Msg->type == MP_FUNCTION_TYPE_ADSLDIRECTIVE)
		{
		    Msg->index = 0;
		    Msg->MsaOf.fourcc = 0;
		    Msg->offset_addr = 0;
		    Msg->data = 0xDEADBEEF;
		    mp_decode_stage = END_DECODE;
		}
	    }
	    break;
	case MP_STAGE_INDEX:
	    {
		/* set mp_decode_stage to process next word */
		mp_decode_stage = MP_STAGE_SYMBOLIC_ADDR;
		
		/* insert the index */
		Msg->index = cpu_to_le16(MsgRaw[MP_RAW_INDEX]);
	    }
	    break;
	case MP_STAGE_SYMBOLIC_ADDR:
	    {
		/* set mp_decode_stage to process next word */
		mp_decode_stage = MP_STAGE_OFFSET_ADDR;
		
		if (Msg->type != MP_FUNCTION_TYPE_ADSLDIRECTIVE)
		{
		    /* extract the symbolic address */
		    
		    temp = cpu_to_le16(MsgRaw[MP_RAW_SYMB_ADDR_HIGH]);
		    Msg->MsaOf.symb_addr[0] = ( unsigned char )( ( unsigned short )( temp & 0xff00 ) >> 8 );
		    Msg->MsaOf.symb_addr[1] = ( unsigned char )( ( unsigned short )( temp & 0x00ff ) );
		    temp = cpu_to_le16(MsgRaw[MP_RAW_SYMB_ADDR_LOW]);
		    Msg->MsaOf.symb_addr[2] = ( unsigned char )( ( unsigned short )( temp & 0xff00 ) >> 8 );
		    Msg->MsaOf.symb_addr[3] = ( unsigned char )( ( unsigned short )( temp & 0x00ff ) );
		    
		    if (MsgCheckSymbolicAddress( (char *)(&Msg->MsaOf.symb_addr[0]) ) == TRUE)
		    {
			/* symbolic address is valid */
		    }
		    else
		    {
			/* symbolic address invalid - set return code */
			ret_code |= MP_DECODE_SYMB_ADDR;
			ret_code |= MP_DECODE_ERROR;
		    }
		}
		else
		{
		    /* can't check the symbolic address for ADSL */
		    /* directive messages */
		}
	    }
	    break;
	case MP_STAGE_OFFSET_ADDR:
	    {
		/* set mp_decode_stage to process next word */
		mp_decode_stage = MP_STAGE_DATA;
		
		/* not going to check this - just insert regardless */
		/*Msg->offset_addr = cpu_to_le16(MsgRaw[MP_RAW_OFFSET_ADDR]); */
		Msg->offset_addr = MsgRaw[MP_RAW_OFFSET_ADDR];
	    }
	    break;
	case MP_STAGE_DATA:
	    {
		/* set mp_decode_stage to process next word */
		mp_decode_stage = END_DECODE;
		
		/* can't check this - just insert regardless */
		Msg->data = cpu_to_le16(MsgRaw[MP_RAW_DATA_HIGH]) * 0x10000 +
		    cpu_to_le16(MsgRaw[MP_RAW_DATA_LOW]);
	    }
	    break;
	}
    }

    if (ret_code != MP_DECODE_OK) 
    {
        adi_dbg (DBG_MSG,"\tMsgDecode error %x.\n",ret_code);        
    }
    

    /* all done */
    return ret_code;
}

/**********************************************************************************/
/*										  */
/*  NAME: MsgSend								  */
/*            									  */
/*  ABSTRACT: 									  */
/*      Sends the supplied message						  */
/*										  */
/*  RETURNS: 									  */
/*      Success(true) or Failure(false)						  */
/*              								  */
/*  ASSUMPTIONS: 								  */
/*      Must be called after the message protocol subsystem			  */
/*      has been intialised.							  */
/*										  */
/**********************************************************************************/
void MsgSend(Hardware *pHw, UInt16 *Packet)
{
    adi_enters (DBG_MSG);
    
    
    /* if no message pending, its safe to send another one */
    if (pHw->AdiModemSm.OutboundPending == FALSE)
    {
	/* Set the flag because the send is pending*/
	pHw->AdiModemSm.OutboundPending = TRUE;
	
	WriteDspMsg(pHw, kDSP_MP_TX_START, (UInt16 *)Packet);
    }
    else
    {
        adi_dbg (DBG_MSG,"TxPending = TRUE, cannot send new message!\n");
        
    }
    adi_leaves (DBG_MSG);
}

/**********************************************************************************/
/*  NAME: GetConfigValues							  */
/*            									  */
/*  ABSTRACT: 									  */
/*      Prepares the Options, Adapters and Flags for SM.			  */
/*      It will set default values, and possibly read some 			  */
/*      (optional) configuration file to fine tune them.			  */
/*										  */
/*  RETURNS: 									  */
/*      void									  */
/*              								  */
/*  ASSUMPTIONS: 								  */
/*										  */
/*      That <pHw> is valid and points to the globals block			  */
/*										  */
/**********************************************************************************/
static void GetConfigValues(Hardware *pHw, const DriverOptions aOptions)
{
    AdiMSM *pSm = &(pHw->AdiModemSm);

    adi_enters (DBG_MSG);
    

    /* Initialize defaults*/
    pSm->ADPT0 = 0;
    pSm->ADPT1 = 0;
    pSm->ADPT2 = 0;
    pSm->INFO9 = 0;
    pSm->CNTL0 = 0;
    pSm->OPTN0 = 0;
    pSm->OPTN2 = 0;
    pSm->OPTN4 = 0;
    pSm->OPTN7 = 0;
    pSm->OPTN3 = 0;
    pSm->OPTN5 = 0;
    pSm->OPTN6 = 0;
    pSm->PFCL1 = 0;
    pSm->MASK8 = 0;
    pSm->MASK9 = 0;
    pSm->INFO8 = 0;
    pSm->PSDM0 = 0;
    pSm->FLAG0 = 0;

    /* Get the option values from the hardware struct*/
    pSm->OPTN0 = aOptions[ConfigOptn0].Value;
    pSm->OPTN2 = aOptions[ConfigOptn2].Value;
    pSm->OPTN3 = aOptions[ConfigOptn3].Value;
    pSm->OPTN4 = aOptions[ConfigOptn4].Value;
    pSm->OPTN5 = aOptions[ConfigOptn5].Value;
    pSm->OPTN6 = aOptions[ConfigOptn6].Value;
    pSm->OPTN7 = aOptions[ConfigOptn7].Value;
    pSm->OPTN15 = aOptions[ConfigOptn15].Value;
    pSm->LineType = aOptions[ConfigLinetype].Value;
     
    adi_leaves (DBG_MSG);
    
}


/**********************************************************************************/
/*  NAME: MsgInitialize								  */
/*            									  */
/*  ABSTRACT: 									  */
/*      Initializes the Message Protocol Subsystem				  */
/*      Encodes of all the messages that are required for training                */
/*      resetting etc. These encoded messages are stored away in		  */
/*      the global structure, from which they can be sent as desired.		  */
/*										  */
/*  RETURNS: 									  */
/*      Success(true) or Failure(false)						  */
/*              								  */
/*  ASSUMPTIONS: 								  */
/*      Must be called after the various OPTN0.. flags are set -  		  */
/*      this will be done at some stage in the USB init sequence		  */
/*										  */
/*      That <pHw> is valid and points to the globals block			  */
/*										  */
/**********************************************************************************/
void MsgInitialize(Hardware *pHw, const DriverOptions aOptions)
{
    AdiMSM *pSm = &(pHw->AdiModemSm);
    
        
    adi_enters (DBG_MSG);
    

    GetConfigValues(pHw, aOptions);

#if 0
    if ((pSm->LineType == 10) &&
        (pHw->pDriverCMVs[0].Data!=0xF9F9F9F9)) /*Read from CMV text file*/
    {
        int i = 0;
        

        pSm->MsgSeq_Retrainer.MsgCount = 0;

        while (TRUE)
        {
            if (pHw->pDriverCMVs[i].Data == 0xF9F9F9F9)
            {
                break;
            }
            ADDMSG_FROM_TEXTFILE( pSm->MsgSeq_Retrainer, pHw->pDriverCMVs[i]);

            i++;
        }
    } 
    else 
    {
#endif /*  0 */
        
        /*read from configuration file*/
        /* Get our Retrainer message sequence initialized - sequence taken from NDIS code*/
        pSm->MsgSeq_Retrainer.MsgCount = 0;
        ADDMSG(pSm->MsgSeq_Retrainer, MsgCNTL0_RSVD); /*CNTL0 1 - to have enough time
                                                       * sending retrain CMVs*/
        
        ADDMSG_IFVAR_NOTZERO(pSm->MsgSeq_Retrainer, MsgINFO8a,pSm->INFO8);
        ADDMSG(pSm->MsgSeq_Retrainer, MsgINFO10);
        if (pSm->OPTN0 != 0xF9F8F7F6)
        {
            ADDMSG_FROM_CONFIG(pSm->MsgSeq_Retrainer, MsgOPTN0, pSm->OPTN0);
        }
        
        if (pSm->OPTN2 != 0xF9F8F7F6)
        {
            ADDMSG_FROM_CONFIG(pSm->MsgSeq_Retrainer, MsgOPTN2, pSm->OPTN2);
        }
    
        if (pSm->OPTN4 != 0xF9F8F7F6)
        {
            ADDMSG_FROM_CONFIG(pSm->MsgSeq_Retrainer, MsgOPTN4, pSm->OPTN4);
        }
    
        if (pSm->OPTN7 != 0xF9F8F7F6)
        {
            ADDMSG_FROM_CONFIG(pSm->MsgSeq_Retrainer, MsgOPTN7, pSm->OPTN7);
        }
    
        /*The ADI code checks the ISDN var AND the OPTN3 var, but if OPTN3*/
        /*is only set for ISDN - there's no need for the ISDN var         */
        if (pSm->OPTN3 != 0xF9F8F7F6)
        {
            ADDMSG_FROM_CONFIG(pSm->MsgSeq_Retrainer, MsgOPTN3, pSm->OPTN3);
        }
    
        if (pSm->OPTN5 != 0xF9F8F7F6)
        {
            ADDMSG_FROM_CONFIG(pSm->MsgSeq_Retrainer, MsgOPTN5, pSm->OPTN5);
        }
     
        if (pSm->OPTN6 != 0xF9F8F7F6)
        {
            ADDMSG_FROM_CONFIG(pSm->MsgSeq_Retrainer, MsgOPTN6, pSm->OPTN6);
        }
    
        if (pSm->OPTN15 != 0xF9F8F7F6)
        {
            ADDMSG_IFVAR_NOTZERO(pSm->MsgSeq_Retrainer, MsgOPTN15,pSm->OPTN15);
        }
    
        ADDMSG_IFVAR_NOTZERO(pSm->MsgSeq_Retrainer, MsgADPT2, pSm->ADPT2);
        ADDMSG_IFVAR_NOTZERO(pSm->MsgSeq_Retrainer, MsgINFO9, pSm->INFO9);
        ADDMSG_IFVAR_NOTZERO(pSm->MsgSeq_Retrainer, MsgMASK8, pSm->MASK8);
        ADDMSG_IFVAR_NOTZERO(pSm->MsgSeq_Retrainer, MsgMASK9, pSm->MASK9);
        ADDMSG_IFVAR_NOTZERO(pSm->MsgSeq_Retrainer, MsgPSDM0, pSm->PSDM0);
        if (pSm->LineType == 0)  // G.Lite
            ADDMSG_IFVAR_NOTZERO(pSm->MsgSeq_Retrainer, MsgPFCL1, pSm->PFCL1);
    
        /*to tell modem it can go ahead to R_ACT_REQUEST state*/ 
        ADDMSG(pSm->MsgSeq_Retrainer, MsgCNTL0_INIT);
#if 0   
    }
#endif
    
    /*Get our SoftReset message sequence initialized - sequence taken from NDIS code*/
    pSm->MsgSeq_SoftReset.MsgCount = 0;
    ADDMSG( pSm->MsgSeq_SoftReset, MsgCNTL0_RSET);
    
    /*Get our Stat message sequence initialized - sequence taken from NDIS code*/
    pSm->MsgSeq_Stat.MsgCount = 0;
    ADDMSG( pSm->MsgSeq_Stat, MsgSTAT0b);
    
    /*Get our OpStat message sequence initialized - sequence taken from NDIS code*/
    pSm->MsgSeq_OpStat.MsgCount = 0;
    ADDMSG( pSm->MsgSeq_OpStat, MsgSTAT0a);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG2);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG22);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG23);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG25);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG51);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG52);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG53);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG54);
    ADDMSG( pSm->MsgSeq_OpStat, MsgRATE0);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG3);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG47);
    ADDMSG( pSm->MsgSeq_OpStat, MsgDIAG49);
    ADDMSG( pSm->MsgSeq_OpStat, MsgINFOA);
    ADDMSG( pSm->MsgSeq_OpStat, MsgPSDM1);
    ADDMSG( pSm->MsgSeq_OpStat, MsgINFO8b);
    ADDMSG( pSm->MsgSeq_OpStat, MsgINFO14);
    ADDMSG( pSm->MsgSeq_OpStat, MsgSTAT0b);    
    
    /*Get our ModemEna message sequence initialized - CMV taken from NDIS code*/
    pSm->MsgSeq_ModemEna.MsgCount = 0;
    ADDMSG( pSm->MsgSeq_ModemEna, MsgMODEMENA);
    
    /*Set Enable Fast Retrain options - CMV not tested! [IIDos]*/
    pSm->MsgSeq_EnaFR.MsgCount = 0;
    ADDMSG( pSm->MsgSeq_EnaFR, MsgOPTN0EnaFR);
    
    /*Init our message pending flags*/
    pSm->InboundAsyncPending = FALSE;
    pSm->InboundSyncPending  = FALSE;
    pSm->OutboundPending     = FALSE;
    
    MsgInitDone = TRUE;
    

    
    adi_leaves (DBG_MSG);
    
}


static void build_msg ( CmvMsg *dst, CmvMsg *src ) 
{
    dst->Preamble = cpu_to_le16 ( src->Preamble );
    dst->Function = cpu_to_le16 ( src->Function );
    dst->Index = cpu_to_le16 ( src->Index );
    dst->Offset = cpu_to_le16 ( src->Offset );

    dst->SymAddrHiHi = src->SymAddrHiLo;
    dst->SymAddrHiLo = src->SymAddrHiHi;
    dst->SymAddrLoHi = src->SymAddrLoLo;
    dst->SymAddrLoLo = src->SymAddrLoHi;

    dst->Data = ___swahw32 (cpu_to_le32 (src->Data));

}



/************************************************************************************************
$Log: Msg.c,v $
Revision 1.8  2003/06/03 23:05:13  sleeper
Change debug macros

Revision 1.7  2003/04/23 22:26:38  sleeper
PPC support

Revision 1.6  2003/03/31 22:02:32  sleeper
Remove useless comments

Revision 1.5  2003/03/21 00:22:41  sleeper
Msg Initialization modifs from 2.0.1

Revision 1.4  2003/03/20 22:07:55  sleeper
Merge routed mode modfis + info statements added

Revision 1.3  2003/03/11 00:37:25  sleeper
Fix a compile warning

Revision 1.2  2003/03/10 23:21:00  sleeper
First fix for architecture independance

Revision 1.1.1.1  2003/02/10 23:29:49  sleeper
Imported sources

Revision 1.60  2002/05/27 18:03:12  Anoosh Naderi
Clean up the code

Revision 1.5  2002/01/18 18:03:12  chris.edgington
Made GetConfigData read from pDriverOptions if it is there.

Revision 1.4  2002/01/14 21:59:33  chris.edgington
Added GPL header.

Revision 1.3  2002/01/02 22:00:48  chris.edgington
Changed instances of STOLE macro to HSTOLE (Host Short To Little Endian).

Revision 1.2  2001/12/26 22:50:46  chris.edgington
Made x86 (little-endian) compatible by pre-processing CMV messages.

Revision 1.1  2001/12/22 21:38:04  chris.edgington
Initial versions - from MacOS9 project tree.


---------------------------------------------------------------------------
 Log from MacOS9 version
---------------------------------------------------------------------------

Revision 1.47  2001/12/05 22:16:00  chris.edgington
Added Analog Devices copyright notice.

Revision 1.46  2001/11/29 19:00:04  ilko.dossev
Added message composing for EnableFastRetrain lite mode option.

Revision 1.45  2001/11/13 20:34:51  ilko.dossev
Removed a temporary debug trace from <MsgSend>

Revision 1.44  2001/11/08 18:35:51  ilko.dossev
Using Time Services - temporary trace, to be removed later.

Revision 1.43  2001/10/30 15:41:59  chris.edgington
Added setting of LineType member in SM.
Added setting of OPTN3 to support ISDN.
Fixed LineType based option modification code - although it is still disabled.

Revision 1.42  2001/10/30 00:05:21  martyn.deobald
Removed code that was reading config file, replaced with code to get vars from pDriverOptions in Hardware struct

Revision 1.41  2001/10/26 20:07:33  martyn.deobald
Modified code to copy new values from the configGetOptions only if the call succeeded. Modifications as we're not using pointers anymore

Revision 1.40  2001/10/25 19:59:40  chris.edgington
Added all possible OPTN* options to drvOptions.
Fixed pointer issue in options.

Revision 1.39  2001/10/25 19:28:21  chris.edgington
Added support for OPTN15 for Eagle's sake.

Revision 1.38  2001/10/24 21:59:50  martyn.deobald
Modified GetConfigValues to read some of the options values from the configuration file.
NOTE, this file is checked into the Install directory and must be placed in your preferences folder.

Revision 1.37  2001/10/18 21:42:04  ilko.dossev
Writing MAILBOX_FULL not req'd any more w/ latest ADI firmware.

Revision 1.36  2001/10/11 17:08:59  ilko.dossev
DMT_ prefix removed from line rate variabl;es

Revision 1.35  2001/10/07 22:03:25  ilko.dossev
Added easy stat CMV at the end of OpStat CMV chain.
LineType initialization added, LineType set to ZERO.

Revision 1.34  2001/10/07 03:28:38  ilko.dossev
Untabified.

Revision 1.33  2001/10/05 18:31:45  ilko.dossev
ZAP message for bad message decoding placed inside MsgDecode function.

Revision 1.32  2001/10/03 21:38:52  chris.edgington
Removed unnecessary routines.
Rewrote MsgSend.
Put hardcoded CMVs in GetOptions - taken from USB trace of Windows driver.

Revision 1.31  2001/10/03 16:30:45  ilko.dossev
Calls to WriteDSPWord(s) changed; padded w/ <nil> arguments.

Revision 1.30  2001/10/02 16:13:41  ilko.dossev
<MsgSeq_ModemEna> initialized, still not obvious that it will be used.

Revision 1.29  2001/10/01 21:45:58  ilko.dossev
Some extra debug output added.

Revision 1.28  2001/10/01 20:43:04  ilko.dossev
Wrong fix for <MsgCount> reversed.
Initializatoin for OPTNx, ADPTx, INITx and FLAGSx vars moved from <MsgInitialize> to <GetConfigValues>.
For now, the original bit-field assignment is left in place, however it will be changed.
The initial revision of <GetConfigValues> will be w/ hard-coded line parameters for everything, fine-tuning wil be done later.

Revision 1.27  2001/10/01 19:36:58  ilko.dossev
True count of commands in a sequence assigned to <MsgSeq_Zzzt.MsgCount> member.

Revision 1.26  2001/09/29 04:29:39  ilko.dossev
Fixed MsgDecode returning FAIL for all Alert CMVs from modem (e.g. Modem Ready)

Revision 1.25  2001/09/28 19:48:11  chris.edgington
Fixed messages that are supposed to be READ requests.

Revision 1.24  2001/09/28 19:32:02  chris.edgington
Added MsgCheckSymbolicAddress back in.

Revision 1.23  2001/09/28 18:27:57  chris.edgington
Moved message encoding to static arrays - greatly simplifies code. 
Removed MsgEncode and MsgCheckSymbolicAddress.

Revision 1.21  2001/09/28 15:07:48  chris.edgington
Changed includes to just use Adiutil.h like all other files.

Revision 1.20  2001/09/28 13:41:00  chris.edgington
Removed WriteIdma and WriteIdmaMulti.
Changed parms to MsgAbort* functions.
Removed ImmediateError.

Revision 1.19  2001/09/27 22:45:18  ilko.dossev
removed obsolete reference to pre-defined USBMODEM (relic from PC code)

Revision 1.18  2001/09/26 22:31:53  ilko.dossev
Send and Receive prototypes modified to accommodate timeout.
Initialization simplified, precomposed messages moved to AdiMSM structure.
Important functions ZAP-ed (debug output).

Revision 1.17  2001/09/24 20:59:52  ilko.dossev
Fixed interface definitions for some MsgXxx functions, to accept block of UInt16[8]

Revision 1.16  2001/09/24 18:02:12  chris.edgington
Commented out some lines that now are invalid with move of AdiUSBPB members.

Revision 1.15  2001/09/19 21:23:09  martyn.deobald
Removed DANGER warnings from WriteIDMA routines

Revision 1.14  2001/09/19 21:19:29  martyn.deobald
Updated comments in function header

Revision 1.13  2001/09/19 20:56:57  martyn.deobald
Added completion handler to those Msgxx routines that use the WriteIDMAxxx routines
Added pGlobals parameter to the WriteIDMA routines
Create WriteIDMA, and WriteIDMAMulti

Revision 1.12  2001/09/18 18:19:07  martyn.deobald
Removed static pointer to globals and added an AdiUSBPB pointer
parameter to all functions that need it.
This was done to prevent any problems with multiple device instances.

Revision 1.11  2001/09/18 17:26:37  martyn.deobald
Changes to use AdiUSBPB instead of EnetPB

Revision 1.10  2001/09/18 17:17:00  martyn.deobald
Using the return value from WriteIDMA
Implemented Send & Receive

Revision 1.9  2001/09/18 15:49:45  martyn.deobald
Implemented AbortSend, AbortReceive, AbortSwap

Revision 1.8  2001/09/18 14:37:54  martyn.deobald
Changed code that checks return code of MsgEncode to reflect that the function returns a Boolean.
Added return code setting code to MsgEncode

Revision 1.7  2001/09/17 21:58:46  martyn.deobald
Renamed low-level routines, check_symbolic_address & mp_encode_msg called from the GenerateCommands code.
commented out DbgPrints

Revision 1.6  2001/09/17 20:38:28  martyn.deobald
Added internals of MsgGenerateCommands, will require some more modfications

Revision 1.5  2001/09/17 17:05:18  martyn.deobald
Implemented MsgDecode
Added MsgCheckSymbolicAddress function

Revision 1.4  2001/09/14 22:00:13  martyn.deobald
Added encode code verbatim from mp_int.c
Reformatted code to meet requirements
Will probably make modficiations but wanted to get it into cvs.

Revision 1.3  2001/09/14 19:08:57  martyn.deobald
Reformatted braces and comment blocks

Revision 1.2  2001/09/14 17:20:57  martyn.deobald
Fixed typos. Added include file

Revision 1.1  2001/09/14 17:11:27  martyn.deobald
Initial Check In
*************************************************************************************************/
