/*********************************************************************************/
/* $Id: Pipes.c,v 1.6 2003/06/03 22:42:55 sleeper Exp $                          */
/*										 */
/* Copyright (c) 2002, Analog Devices Inc., All Rights Reserved			 */
/*										 */
/* Pipes.c									 */
/*										 */
/* Code for dealing with read and write data pipes				 */
/*										 */
/* This file is part of the "ADI USB ADSL Driver for Linux".			 */
/*										 */
/* "ADI USB ADSL Driver for Linux" is free software; you can redistribute it     */
/* and/or modify it under the terms of the GNU General Public License as 	 */
/* published by the Free Software Foundation; either version 2 of the License,   */
/* or (at your option) any later version.					 */
/*										 */
/* "ADI USB ADSL Driver for Linux" is distributed in the hope that it will be    */
/* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of	 */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 */
/* GNU General Public License for more details.					 */
/*										 */
/* You should have received a copy of the GNU General Public License		 */
/* along with "ADI USB ADSL Driver for Linux"; if not, write to the Free Software*/
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA     */
/*********************************************************************************/

#include "Adiutil.h"
#include "Pipes.h"
#include "Mpoa.h"
#include "Uni.h"
#include "debug.h"

/*********************************************************************************/
/* Local prototypes								 */
/*********************************************************************************/
static void ReadBulkCompletion(struct urb *urb);
static void ReadIsoCompletion(struct urb *urb);
static void WriteCompletion(struct urb *urb);

/**********************************************************************************/
/* StartReadPipe								  */
/*										  */
/* Prepare to receive incoming network data by queueing up multiple BULK IN urbs. */
/* We previously followed the same model as the other USB network drivers, in that*/
/* we only allowed one incoming block of network data at a time. However, this    */
/* resulted in lost data (incoming data would be lost because it wasn't read by   */
/* USB in time). So, we've now switched to the same model that we use in the      */
/* MacOS9 driver, We queue up multiple BULK IN requests, so that (theoretically), */
/* the DMA of an incoming packet could be happening while the CPU is doing the    */
/* reassembly of another packet.						  */
/**********************************************************************************/
void StartReadPipe(Hardware *pHw)
{
    int result;
    int i;

    /***********************************************************/
    /*Midified on Jan 21,2003 by Anoosh to make re-sync problem*/
    /*which caused to loose up-stream working                  */
    if (pHw->AdiModemSm.CurrentAdiState & STATE_OPERATIONAL )
    {
        adi_enters (DBG_SAR);
        

#ifdef USEBULK
        /*Start the BULK read pipe URBs*/
        for (i=0; i<INCOMING_Q_SIZE; i++)
        {
            memset(&pHw->urbRead[i], 0, sizeof(pHw->urbRead[i]));
            usb_fill_bulk_urb(&pHw->urbRead[i], pHw->usbdev, pHw->pipeBulkDataIn, 
                              pHw->pIncomingData[i], INCOMING_DATA_SIZE, ReadBulkCompletion, pHw);
            pHw->urbRead[i].transfer_flags |= USB_QUEUE_BULK;
            if ((result = usb_submit_urb(&pHw->urbRead[i]))) 
            {
                adi_dbg (DBG_READ,"Error %d on read URB submit.\n", result);
            }
            
        }
#else
        /*****************************************************************************/
        /* Start the ISOCH read URBs ... first we go through the ring and initialize */
        /* all the data. Since we setup an URB ring, we only have to actually submit */
        /* the first URB in the ring, the rest will be submitted through the URB     */
        /* linking mechanism (via the ->next field)					 */
        /*****************************************************************************/
        for (i=0; i<INCOMING_Q_SIZE; i++)
        {
            int j;
            struct urb *pUrb;

            pUrb = pHw->pUrbReadIso[i];
            memset(pUrb, 0, sizeof(*pUrb));
            spin_lock_init(&pUrb->lock);
            /*Init the URB fields per the spec*/
            pUrb->next                   = pHw->pUrbReadIso[(i+1) % INCOMING_Q_SIZE];
            pUrb->dev                    = pHw->usbdev;
            pUrb->pipe                   = pHw->pipeIsoDataIn;
            pUrb->context                = pHw;
            pUrb->complete               = ReadIsoCompletion;
            pUrb->transfer_buffer        = pHw->pIncomingData[i];
            pUrb->transfer_buffer_length = pHw->IsoPipeSize * pHw->IsoFramesPerUrb;
            pUrb->number_of_packets      = pHw->IsoFramesPerUrb;
            pUrb->transfer_flags         = USB_ISO_ASAP;
            /*
              CC - FIXME : Voir ce que ces flags signifient avant de les activer :
              pUrb->transfer_flags         &= ~USB_DISABLE_SPD;	// Curieux: cela suppose que ISO_ASAP contiendrait DISABLE_SPD ?
              pUrb->status	             = 0;			// Curieux: c'est un paramtre de retour...
              pUrb->actual_length          = 0;			// Curieux: idem.
              pUrb->start_frame            = 0;			// Curieux: la doc USB dit que ce paramtre est ignor avec ISO_ASP...
            */
            /*Setup the frame descriptors that will receive the data*/
            for (j=0; j<pUrb->number_of_packets; ++j)
            {
                pUrb->iso_frame_desc[j].offset = j * pHw->IsoPipeSize;
                pUrb->iso_frame_desc[j].length = pHw->IsoPipeSize;
            }
        }

        for (i=0; i<INCOMING_Q_SIZE; i++)
        {
            /*Ok, data is initialized, fire off the ISO URB ring*/
            if ((result = usb_submit_urb(pHw->pUrbReadIso[i])))
            {
                adi_dbg (DBG_READ,"Error %d on ISO read URB submit.\n", result);
                
            }
            
        }
#endif
        adi_leaves (DBG_READ);
    }
    
}

/**********************************************************************************/
/* ReadBulkCompletion								  */
/*										  */
/* Called when incoming BULK data is ready for our consumption. We check a few    */
/* conditions to make sure all is well with the state of things, then we	  */
/* let the ATM code reassemble the cells into an ethernet packet. The last	  */
/* thing we'll do is queue up another BULK IN urb, to get the next packet.	  */
/**********************************************************************************/
static void ReadBulkCompletion(struct urb *urb)
{
    Hardware *pHw = urb->context;
    struct net_device *ether;
    int result;

    adi_enters (DBG_READ);
    

    /*This should never happen - but in a kernel driver, better safe than sorry!*/
    if (!pHw)
    {
        adi_dbg (DBG_READ,"NULL pHw from urb->context!\n");
        
	goto Read_exit;
    }

    /* If we're not open for business, ignore this callback*/
    if (!pHw->IsOpen)
    {
        adi_dbg (DBG_READ,"Read callback, but IsOpen = FALSE!\n");
	goto Read_exit;
    }

    ether = pHw->pLinuxNet;
    if (!netif_device_present(ether))
    {
        adi_dbg (DBG_READ,"Somebody has killed our network interface.\n");
        
	goto Read_exit;
    }

#ifdef SYNC_READS
    /***************************************************************************/
    /* We only allow one read to happen at a time. Since we only queue up one  */
    /* BULK IN urb, I don't know how we could get called again. But, we do need*/
    /* to make sure that we're not being reentered, just in case.	       */
    /***************************************************************************/
    if (pHw->IsReading)
    {
	pHw->LinuxStats.rx_errors++;
        adi_dbg (DBG_READ,"Read callback, but IsReading = TRUE!\n");
        
	goto Read_exit;
    }

    pHw->IsReading = TRUE;
#endif

    if (urb->status != USB_ST_NOERROR)
    {
	/* These errors seem to be what we get when the device has been unplugged, so*/
	/* lets stop a bunch of these from happening by unlinking the urb now        */
	if ((urb->status != -EILSEQ) &&
	    (urb->status != -ETIMEDOUT))
	{
            adi_dbg (DBG_READ,"Error status = %d in ReadBulkCompletion - requeuing\n", urb->status);
            
	    goto ReadResubmit;        
	}
	else
	{
            adi_dbg (DBG_READ,"Error status = %d in ReadBulkCompletion - not requeuing\n", urb->status);
            
	    goto Read_exit;        
	}
    }

    /*If this was a zero-length read, we're done*/
    if (!urb->actual_length)
	goto ReadResubmit;

    adi_dbg (DBG_READ,"Read data length is %#x bytes.\n", urb->actual_length);
    

    /*Ok, we're ready to deal with the incoming data*/
    result = UniProcessInboundData(pHw, urb->transfer_buffer, urb->actual_length);
    if (result) 
    {
        adi_dbg (DBG_READ,"Error %d from UniProcessInboundData\n", result);        
    }
    

ReadResubmit:
    /***********************************************************/
    /*Midified on Jan 21,2003 by Anoosh to make re-sync problem*/
    /*which caused to loose up-stream working                  */
    if (pHw->AdiModemSm.CurrentAdiState & STATE_OPERATIONAL )
    {
        memset(urb, 0, sizeof(*urb));
        usb_fill_bulk_urb(urb, pHw->usbdev, pHw->pipeBulkDataIn, 
                          urb->transfer_buffer, INCOMING_DATA_SIZE, ReadBulkCompletion, pHw);
        if ((result = usb_submit_urb(urb))) 
        {
            adi_dbg (DBG_READ,"Error %d on read URB submit.\n", result);
            
        }
        
    }
    
#ifdef SYNC_READS
    pHw->IsReading = FALSE;
#endif

/*We will only jump to here on conditions where we don't want to queue another read*/
Read_exit:
    adi_leaves (DBG_READ);
    
    return;
}

/**********************************************************************************/
/* ReadIsoCompletion								  */
/*										  */
/* Called when incoming ISO data is ready for our consumption. We check a few     */
/* conditions to make sure all is well with the state of things, then we	  */
/* let the ATM code reassemble the cells into an ethernet packet. We don't	  */
/* have to re-submit the URB because it will get automatically done by the	  */
/* URB linking logic in the usb driver						  */
/**********************************************************************************/
static void ReadIsoCompletion(struct urb *urb)
{
    Hardware *pHw = urb->context;
    struct net_device *ether;
    int result, i;
    UInt8 *pBuf;

    adi_enters (DBG_READ);
    

    /*This should never happen - but in a kernel driver, better safe than sorry!*/
    if (!pHw)
    {
        adi_dbg (DBG_READ,"NULL pHw from urb->context!\n");
        
	goto ReadIso_exit;
    }

    /*If we're not open for business, ignore this callback*/
    if (!pHw->IsOpen)
    {
        adi_dbg (DBG_READ,"Read callback, but IsOpen = FALSE!\n");
        
	goto ReadIso_exit;
    }

    ether = pHw->pLinuxNet;
    if (!netif_device_present(ether))
    {
        adi_dbg (DBG_READ,"Somebody has killed our network interface.\n");
        
	goto ReadIso_exit;
    }

    if (urb->status == -ENOENT)
    {
        adi_dbg (DBG_READ,"Error status = -ENOENT in ReadIsoCompletion\n");
        
	goto ReadIso_exit;
    }

    /*If this was a zero-length read, we're done*/
    if (!urb->actual_length)
	goto ReadIso_exit;


    /* Ok, we're ready to deal with the incoming data. We must go through each packet
    even if the URB's status code is not USB_ST_NOERROR, since a partial transfer
    may have occured: */
    pBuf = urb->transfer_buffer;
    for (i=0; i<urb->number_of_packets; i++)
    {
	/* Check the packet completion code: */
        if (!urb->iso_frame_desc[i].status)
        {
	    /* OK, process the incoming data: */
	    result = UniProcessInboundData(pHw, pBuf+urb->iso_frame_desc[i].offset,
		urb->iso_frame_desc[i].actual_length);
	    if (result)
            {
                adi_dbg (DBG_READ,"Error %d from UniProcessInboundData\n", result);
                
            }
            
	}
    }

ReadIso_exit:
    /* chinda_keodouangsy@usr.com
	As some time, we received USB_ST_CRC (-EILSEQ) as status even if the urb
	is correct. We need to reset the urb before the next use.
	CC - FIXME : Voir si cette chose peut toujours se produire avec un noyau rcent.
	De plus, je ne vois pas pourquoi il faudrait resetter l'URB, USB est suppos
        le faire comme un grand, non ?
    */

     urb->status = 0;
    for (i=0; i<urb->number_of_packets; i++)
    {
/*        
	urb->iso_frame_desc[i].offset = i * pHw->IsoPipeSize;
	urb->iso_frame_desc[i].length = pHw->IsoPipeSize;
*/        
	urb->iso_frame_desc[i].status = 0;
/*        
	urb->iso_frame_desc[i].actual_length = 0;
*/        
    }
/*    
    urb->actual_length = 0;
    urb->dev = pHw->usbdev;	// Curieux: pourquoi l'URB oublierait-il son
    device ?
*/
    
    adi_leaves (DBG_READ);
    
    return;
}


/**********************************************************************************/
/* StopReadPipe									  */
/*										  */
/* Simple, really, but maybe we'll eventually have to do more.			  */
/**********************************************************************************/
void StopReadPipe(Hardware *pHw)
{
    int i;

    adi_enters (DBG_READ);
    
    for (i=0; i<INCOMING_Q_SIZE; i++)
    {
#ifdef USEBULK
	pHw->urbRead[i].transfer_flags &= ~USB_ASYNC_UNLINK;
        usb_unlink_urb(&pHw->urbRead[i]);
#else
	pHw->pUrbReadIso[i]->transfer_flags &= ~USB_ASYNC_UNLINK;
        usb_unlink_urb(pHw->pUrbReadIso[i]);
#endif
    }

    adi_leaves (DBG_READ);
}

/**********************************************************************************/
/* ReadSendItUp									  */
/*										  */
/* This is called by the Uni code once it has determined that a valid		  */
/* ethernet packet has been reassembled and is ready for processing		  */
/* by higher level network code. First we'll do things that a normal		  */
/* ethernet card would do in hardware (packet filtering, etc.), then		  */
/* we'll follow the standard procedure for sending data back up the stack.	  */
/* Comment by Anoosh on Apri30,2002, In MAC driver in ReadWaitingData function    */
/* in read.c we check if we are using PPPoA then use RealSize for the size of     */
/* data that we send up and for ethernet we pad packets smaller than 64 bytes     */
/* and add space for Ethernet FCS. But here in linux it looks that we just use    */
/* RealSize for the size of data without adding anything therefore we just        */
/* ignore adding anything here to match with MAC driver. Look into read.c in      */
/* MAC driver line 168                                                            */
/**********************************************************************************/
void ReadSendItUp(Hardware *pHw, ETHERNET_PACKET_BUFFER *pPacket)
{
    adi_enters (DBG_READ);
    
    
    MpoaProcessReceivedPdu(pHw, pPacket);
    if (pPacket->DropThisPacket)
    {
	pHw->LinuxStats.rx_errors++;
    }
    else
    {
	UInt32 PacketSize = 0;
	struct sk_buff *skb = NULL;
	switch (pHw->MpoaMode)
	{
	case MPOA_MODE_BRIDGED_ETH_LLC:
	case MPOA_MODE_BRIDGED_ETH_VC :
	    {
		PacketSize = pPacket->PacketSize - pPacket->EncapSkipSize;
		skb = dev_alloc_skb(PacketSize);
		if (skb != NULL)
		    eth_copy_and_sum(skb, pHw->pReadyData+pPacket->EncapSkipSize, PacketSize, 0);
	    }
	    break;
	case MPOA_MODE_ROUTED_IP_LLC  :
	case MPOA_MODE_ROUTED_IP_VC   :
	    {
		PacketSize = pPacket->PacketSize - pPacket->EncapSkipSize + 14;
		skb = dev_alloc_skb(PacketSize);
		if (skb != NULL)
		{
		    memcpy(skb->data, pHw->pLinuxNet->dev_addr, ETH_LENGTH_OF_ADDRESS);
		    memcpy(skb->data + 6, pHw->pLinuxNet->dev_addr, ETH_LENGTH_OF_ADDRESS);
		    skb->data[11] ^=0x1;		/* Try a fake address */
		    skb->data[12] = (UInt8) 0x08;	/* ETH_P_IP */
		    skb->data[13] = (UInt8) 0x00;
		    memcpy(skb->data + 14, pHw->pReadyData+pPacket->EncapSkipSize,PacketSize-14);
		}
	    }
	    break;
	case MPOA_MODE_PPPOA_LLC      :
	case MPOA_MODE_PPPOA_VC       :
	    {
		PacketSize = pPacket->PacketSize - pPacket->EncapSkipSize + 16;
		skb = dev_alloc_skb(PacketSize);
		if (skb != NULL)
		{
		    static const UInt8 PPP_ETH_ADDR_DEST[ETH_LENGTH_OF_ADDRESS] = {0x01, 0x02, 0x03, 0x04, 0x05, 0x06};
		    static const UInt8 PPP_ETH_ADDR_SRC[ETH_LENGTH_OF_ADDRESS] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
		    /* The following are fake source and destination address*/
		    /* to make PPPoA work:                                  */
		    memcpy(skb->data, PPP_ETH_ADDR_DEST, ETH_LENGTH_OF_ADDRESS);
		    memcpy(skb->data+ETH_LENGTH_OF_ADDRESS, PPP_ETH_ADDR_SRC, ETH_LENGTH_OF_ADDRESS);
		    /* Set the Ethernet protocol field (0x0080 is for ICMP  */
		    /* and 0x0003 is for every packet, for more details     */
		    /* look at if_ether.h in /usr/src/linux/include/linux   */
		    /* directory.                                           */
		    skb->data[12] = (UInt8) 0x00;	/*every packet*/
		    skb->data[13] = (UInt8) 0x03;	/*every packet*/
		    /* Set the packet size field: */
		    skb->data[14] = (UInt8) ((PacketSize - 16) >> 8);
		    skb->data[15] = (UInt8) (PacketSize - 16);
		    /* Copy the data: */
		    memcpy(skb->data + 16, pHw->pReadyData+pPacket->EncapSkipSize,PacketSize-16);
		}
	    }
	    break;
        }

        /*Ok, we've got the green light to send the packet on up*/
        if (skb != NULL)
	{
	    /* Link the socket buffer to this device: */
	    skb->dev = pHw->pLinuxNet;
	    /* Update the tail and len field of the socket data buffer: */
	    skb_put(skb, PacketSize);
	    /* Set up packet type: */
	    skb->protocol = eth_type_trans(skb, pHw->pLinuxNet);
	    /* Update statistics: */
	    pHw->pLinuxNet->last_rx = jiffies;
	    ++pHw->LinuxStats.rx_packets;
	    pHw->LinuxStats.rx_bytes += PacketSize;
	    /* Send the packet to upper layer: */
	    netif_rx(skb);
	}
	else
        {
            adi_dbg (DBG_READ,"Error allocating skb!\n");            
	    ++pHw->LinuxStats.rx_dropped;
        }

        adi_leaves (DBG_READ);
        
    }
    return;
}


/**********************************************************************************/
/* WriteAtmData									  */
/*										  */
/* Another simple routine, just sends the ATM cells on their merry way.		  */
/**********************************************************************************/
int WriteAtmData(Hardware *pHw)
{
    int result;

    /***********************************************************/
    /*Midified on Jan 21,2003 by Anoosh to make re-sync problem*/
    /*which caused to loose up-stream working                  */
    if (pHw->AdiModemSm.CurrentAdiState & STATE_OPERATIONAL )
    {
        adi_dbg (DBG_READ,"*** Writing %#x bytes to USB\n", pHw->SegmentationBuffer.TotalBytes);

        /* Prepare the URB: */
        memset(&pHw->urbWrite, 0, sizeof(pHw->urbWrite));
        usb_fill_bulk_urb(&pHw->urbWrite, pHw->usbdev, pHw->pipeBulkDataOut,
                          pHw->pOutgoingData, pHw->SegmentationBuffer.TotalBytes, 
                          WriteCompletion, pHw);
        /* Marks the writing URB as pending */
        pHw->IsWriting = TRUE;
        pHw->urbWrite.transfer_flags |= USB_QUEUE_BULK;
        
        /* Send it: */
        result = usb_submit_urb(&pHw->urbWrite);
    }
    else
    {
        /*I want to make sure that we return error if modem is*/
        /*not in OPERATIONAL mode because otherwise we screw  */
        /*USB pipes again. We do not want to send anything to */
        /*USB if modem is not in OPERATIONAL mode             */ 
        result = 1;
    }
    
    return result;
}


/**********************************************************************************/
/* WriteCompletion								  */
/*										  */
/* Called when the usb write of outgoing ATM cells is complete. 		  */
/* Since we do not do any internal queueing of outgoing data, the		  */
/* main purpose for this routine is to wake up the kernel's outgoing		  */
/* network queue so we get another packet to send out (if anybody has one).	  */
/**********************************************************************************/
static void WriteCompletion(struct urb *urb)
{
    Hardware *pHw = urb->context;
    struct net_device *ether;

    adi_enters (DBG_READ);
    

    if ((pHw == NULL) || (!pHw->IsOpen))
    {
       adi_dbg (DBG_READ,"NULL urb->context or IsOpen=FALSE in WriteCompletion!\n");
       
       goto WriteCompletion_exit;
    }

    ether = pHw->pLinuxNet;

    if (!netif_device_present(ether))
    {
       adi_dbg (DBG_READ,"Network interface no longer present!\n");
       
       goto WriteCompletion_exit;
    }

    /* Update transmission statistics: */
    if (0 == urb->status)
    {
	++pHw->LinuxStats.tx_packets;
	pHw->LinuxStats.tx_bytes += pHw->OutgoingPacketSize;
    }
    else
    {
	if (-ECONNRESET == urb->status)
	{
	    adi_warn ("transmit URB %p cancelled\n", urb);
        
	    ++pHw->LinuxStats.tx_aborted_errors;
           /*
              Clear status in order to be able to re-use this URB : otherwise
              host (UHCI/OHCI), will see status as -ECONNRESET when doing the
              next usb_submit_urb, and will refuse to send it, returning
              -EINVAL.
            */
            urb->status = 0;
            urb->transfer_flags &= ~USB_ASYNC_UNLINK;
	}
	else
	{
	    adi_warn ("transmit error with URB status %d\n", urb->status);
        
	    ++pHw->LinuxStats.tx_carrier_errors;
	}
	++pHw->LinuxStats.tx_errors;
	pHw->OutgoingPacketSize = 0;
    }

    /* Ask for another outgoing socket buffer: */
    pHw->IsWriting = FALSE;
    netif_wake_queue(ether);

WriteCompletion_exit:
    adi_leaves (DBG_READ);
    
}

/*****************************************************************************************
$Log: Pipes.c,v $
Revision 1.6  2003/06/03 22:42:55  sleeper
Changed ZAPS to adi_dbg/err macros

Revision 1.5  2003/03/21 00:22:41  sleeper
Msg Initialization modifs from 2.0.1

Revision 1.4  2003/03/05 22:25:31  sleeper
Add queuing of Bulk msg on write

Revision 1.3  2003/02/14 23:30:56  sleeper
Merge CC5 diffs

Revision 1.2  2003/02/10 23:41:02  sleeper
Fix a non-cleared flag

Revision 2.00  2002/05/24 17:56:44  Anoosh Naderi
Clean up the code.
Added support for PPPoA (LLC & VC).

Revision 1.5  2002/03/12 17:56:44  chris.edgington
Added ISO support for incoming data.

Revision 1.4  2002/02/11 21:13:55  chris.edgington
Added support for a variable number of read urbs in StartReadPipe and StopReadPipe.
Ifdefed out code for not allowing multiple incoming buffers to be processed simultaneously.

Revision 1.3  2002/01/14 21:59:30  chris.edgington
Added GPL header.

Revision 1.2  2002/01/08 16:08:37  chris.edgington
Added some necessary comments.

Revision 1.1  2002/01/04 20:42:17  chris.edgington
First version.
******************************************************************************************/
