/**********************************************************************************/
/* $Id: Util.c,v 1.7 2003/06/03 22:52:10 sleeper Exp $           	          */
/*										  */
/* Copyright (c) 2002, Analog Devices Inc., All Rights Reserved			  */
/*										  */
/* Util.c									  */
/*										  */
/* Miscellaneous utility routines used in the driver				  */
/*										  */
/* This file is part of the "ADI USB ADSL Driver for Linux".			  */
/*										  */
/* "ADI USB ADSL Driver for Linux" is free software; you can redistribute it      */
/* and/or modify it under the terms of the GNU General Public License as          */
/* published by the Free Software Foundation; either version 2 of the License,    */
/* or (at your option) any later version.					  */
/*										  */
/* "ADI USB ADSL Driver for Linux" is distributed in the hope that it will be     */
/* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of	  */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		  */
/* GNU General Public License for more details.					  */
/*										  */
/* You should have received a copy of the GNU General Public License		  */
/* along with "ADI USB ADSL Driver for Linux"; if not, write to the Free Software */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA      */
/**********************************************************************************/

#include "Adiutil.h"
#include "Util.h"
#include "Konst.h"
#include "debug.h"
#include "Firmware8051.h"

/**********************************************************************************/
/* Local utilities								  */
/**********************************************************************************/

/**********************************************************************************/
/* htoi										  */
/*										  */
/* Converts a hex char to an integer (my own wacky implementation).		  */
/* I don't know why this isn't part of a standard library or something,		  */
/* it seems like this is a quite-frequently-rewritten function.			  */
/**********************************************************************************/
static UInt8 htoi(UInt8 h)
{
    switch (toupper(h))
    {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
	return h-'0';
	break;
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
	return h-'A'+10;
	break;
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':
	return h-'a'+10;
	break;
    }
    
    /*If we get here ... the character was invalid*/
    return -1;
}


/**********************************************************************************/
/* EZUSBWrite									  */
/**********************************************************************************/
static int EZUSBWrite(Hardware *pHw, UInt16 EZUSBAddress, UInt16 count, UInt8 writeBuffer[])
{
    int ret = -ENOMEM;
    UInt8 *xfer_buf = GET_KBUFFER(count);
    if (NULL != xfer_buf)
    {
	memcpy(xfer_buf, writeBuffer, count);
	ret = usb_control_msg(pHw->usbdev,
			      usb_sndctrlpipe( pHw->usbdev, 0 ),
			      kEZUSB_LOAD_INTERNAL,
			      USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			      EZUSBAddress, 0, xfer_buf, count, 300);
	FREE_KBUFFER(xfer_buf);
    }
    return ret;
}


/**********************************************************************************/
/* Local prototypes								  */
/**********************************************************************************/

static void queue_ctrl_urb(Hardware *pHw, struct urb *urb);
static void ctrl_urb_completion(struct urb *urb);
static queued_urb_t *get_queued_urb(struct list_head *q);
static void rm_queued_urb(struct urb *urb);

/**********************************************************************************/
/* Helper implementations							  */
/**********************************************************************************/

/**********************************************************************************/
/* GetMacAddress								  */
/*										  */
/* Queries the ethernet (MAC) address from the USB firmware, storing it in	  */
/* pHw->MAC in numeric form (as opposed to ascii).				  */
/**********************************************************************************/
void GetMacAddress(Hardware *pHw)
{
    int maclen, i;
    UInt8 macbuf[ETH_LENGTH_OF_ADDRESS*2+1];
    
    /****************************************************************************/
    /* Even though this device advertises itself as being an ATM class          */
    /* device, it doesn't have ATM functional descriptors - it has Ethernet ones*/
    /* We already know what the string index (iMACAddress) is from the		*/
    /* Ethernet descriptor, so all we have to do is query the string		*/
    /****************************************************************************/
    maclen = usb_string(pHw->usbdev, kMAC_ADDRESS_INDEX, macbuf, ETH_LENGTH_OF_ADDRESS*2+1);
    
    /*Make sure we got a string of the right length*/
    if (maclen != ETH_LENGTH_OF_ADDRESS*2)
    {
        adi_dbg(DBG_UTILS,"GetMacAddress: invalid length %x.\n", maclen);
	return;
    }
    
    /*Convert the string into a numeric format    */
    /*(each string byte is a nibble in the result)*/
    for (i=0; i<ETH_LENGTH_OF_ADDRESS; i++)
    {
	pHw->MAC[i] = (htoi(macbuf[i*2])<<4) + (htoi(macbuf[i*2+1]));
    }
}

/*
 * loadUsbFirmware : called by adi_probe when it detects a pre-firmware modem.
 */
int loadUsbFirmware ( Hardware *pHw, UInt32 pid ) 
{
    int ret= USB_ST_NOERROR;
    UInt8 writeVal;
    UInt32 i = 0;
    INTEL_HEX_RECORD *pFirm;
    
    adi_enters (DBG_UTILS);

    writeVal = 1;

    /* Send reset */
    ret = EZUSBWrite ( pHw, k8051_USBCS, 1, &writeVal );
    if ( ret < 0 ) 
    {
        adi_err ("EZUSBWrite reset failure with error %d\n",ret);
        goto byebye;
    }

    if ( IS_EAGLE_I (pid) ) 
    {
        pFirm = &Firmware8051_EagleI[0];
    }
    else
    {
        pFirm = &Firmware8051_EagleII[0];
    }

    i = 0;

    while (pFirm->Type == 0 ) 
    {
        ret = EZUSBWrite (pHw,
                          pFirm->Address,
                          pFirm->Length,
                          pFirm->Data );
        if ( ret < 0 ) 
        {
            adi_err ("EZUSBWrite of download data failed (%d)\n",ret);
            goto byebye;
        }
        pFirm ++;
    }

    /* De-assert reset */
    writeVal = 0;
    ret = EZUSBWrite(pHw, k8051_USBCS, 1, &writeVal);
    if ( ret < 0 ) 
    {
        adi_err ("EZUSBWrite de-reset failure (%d)\n",ret);
        goto byebye;
    }

    
  byebye:  
    adi_leaves (DBG_UTILS);
    return (ret);
}


#if 0
/**********************************************************************************/
/* loadUsbFirmware								  */
/*										  */
/* Called by ioctl from user space to upload the ADI-specific firmware            */
/* to the EZUSB chip.                                                             */
/**********************************************************************************/
int loadUsbFirmware(Hardware *pHw, FIRMWARE_RECORD *aRecords, UInt32 BufSize)
{
    /*Assert reset*/
    UInt8 writeVal = 1;
    int ret = EZUSBWrite(pHw, k8051_USBCS, 1, &writeVal);
    if (ret >= 0)
    {
	int ret2;
	/*Download code*/
	UInt32 i=0;
	while (1)
	{
	    ret = EZUSBWrite(pHw, ntohs(aRecords[i].Address),
			     (aRecords[i].Length < MAX_FIRMWARE_RECORD_LENGTH ?
			     aRecords[i].Length : MAX_FIRMWARE_RECORD_LENGTH),
			     aRecords[i].Data);
	    /*Check for write error*/
	    if (ret < 0)
	    {
		adi_err ("Error while writing firmware to device!\n");
		break;
	    }
	    /*Check for end of firmware*/
	    if (aRecords[i].Type != 0)
		break;
	    /*Check for end of buffer*/
	    if (i*sizeof(FIRMWARE_RECORD) >= BufSize)
	    {
		adi_err ("Incomplete firmware received, some parts already downloaded to device!\n");
		ret = -1;
		break;
	    }
	    /*Go to next record*/
	    ++i;
	}
	/*De-assert reset*/
	writeVal = 0;
	ret2 = EZUSBWrite(pHw, k8051_USBCS, 1, &writeVal);
	if (ret2 < 0)
	{
	    adi_err ("EZUSBWrite de-reset failure\n");
	    ret = ret2;
	}
    }
    else
    {
	adi_err ("EZUSBWrite reset failure with error %x\n", ret);
    }
    return ret;
}
#endif /* 0 */



/**********************************************************************************/
/* WriteDspMsg									  */
/**********************************************************************************/
void WriteDspMsg(Hardware *pHw, UInt16 Addr, UInt16 *pData)
{
    /*get space for data and request struct*/
    struct urb *urb;
    devrequest *pdr;
    UInt8 *xfer_buf = GET_KBUFFER((CMV_DATA_WORDS*2)+sizeof(devrequest));

    if (!xfer_buf)
    {
        adi_err("kmalloc in WriteDspMsg for %#X bytes failed\n",
                CMV_DATA_WORDS*2+sizeof(devrequest));
	goto WriteDsp_exit;
    }

    memcpy(xfer_buf, pData, CMV_DATA_WORDS*2);
    pdr = (devrequest *)(xfer_buf+CMV_DATA_WORDS*2);

    /*Get an URB and prepare it for submission*/
    urb = usb_alloc_urb(0);

    FILL_USB_CTRL_REQUEST(pdr,
	    USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
	    kCDC_REQ_SEND_ENCAPSULATED_COMMAND,
	    kADI_CMD_SET_BLOCK, Addr | 0x4000, CMV_DATA_WORDS*2);

    usb_fill_control_urb(urb, pHw->usbdev, usb_sndctrlpipe(pHw->usbdev,0), (UInt8 *)pdr,
                     xfer_buf, CMV_DATA_WORDS*2, ctrl_urb_completion, NULL);
    
    queue_ctrl_urb(pHw, urb);

WriteDsp_exit:
    return ;
}

/**********************************************************************************/
/* AdiCustomWrite								  */
/**********************************************************************************/
int AdiCustomWrite(Hardware *pHw, UInt32 Cmd, UInt32 Index, UInt32 Count, UInt8 *pData)
{
    /*get space for data and request struct*/
    struct urb *urb;
	devrequest *pdr;
    UInt8 *xfer_buf = GET_KBUFFER(Count+sizeof(devrequest));
    int ret = USB_ST_NOERROR;

    if (!xfer_buf)
    {
        adi_err ("kmalloc in AdiCustomWrite for %#X bytes failed\n", Count+sizeof(devrequest));
        
	ret = -ENOMEM;
	goto custom_exit;
    }

    memcpy(xfer_buf, pData, Count);
    pdr = (devrequest *)(xfer_buf+CMV_DATA_WORDS*2);

    /*Get an URB and prepare it for submission*/
    urb = usb_alloc_urb(0);

    FILL_USB_CTRL_REQUEST(pdr,
	    USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
	    kCDC_REQ_SEND_ENCAPSULATED_COMMAND,
	    Cmd, Index, Count);

    usb_fill_control_urb(urb, pHw->usbdev, usb_sndctrlpipe(pHw->usbdev,0), (UInt8 *)pdr,
                     xfer_buf, Count, ctrl_urb_completion, NULL);
    
    queue_ctrl_urb(pHw, urb);
custom_exit:
    return ret;
} 

/*
 * Allocates the number (count) of queued_urb_t and
 * put them in the free queue
 *
 * Return:
 *    0     :   OK
 * -ENOMEM  : Not enough memory. No data has been allocated
 */
int alloc_queued_urb_ctrl ( Hardware *pHw, unsigned int count )
{
    unsigned int i;
    queued_urb_t *pu;
    int retcode = 0;
    
    /* Initialize the free control urb queue */
    INIT_LIST_HEAD( &pHw->ctrl_urb_free_q );
    
    for (i=0; i< count; i++ ) 
    {
        pu = (queued_urb_t *) GET_VBUFFER (sizeof(queued_urb_t));
        if (pu == NULL ) 
        {
            retcode = -ENOMEM;
            adi_err ( "Not enough memory to init. free urb q.\n");
            goto byebye;
        }

        memset ( pu, 0, sizeof(queued_urb_t) );
        
        /* And add it to the list */
        list_add ( &pu->list, &pHw->ctrl_urb_free_q );
    }

  byebye:
    if ( retcode != 0 ) 
    {
        /* We have to free previously allocated queued_urb_t */
        free_queued_urb_ctrl ( &pHw->ctrl_urb_free_q );
    }
    else
    {
        adi_dbg(DBG_UTILS,"%d queued_urb_t allocated.\n",count);
    }
    
    return (retcode);
}

/*
 * Free all previously allocated queued_urb_t
 *
 */

void free_queued_urb_ctrl ( struct list_head *q ) 
{
    queued_urb_t *pu;
    
    while ( !list_empty (q) ) 
    {
        pu = list_entry (q->next, queued_urb_t, list );
        list_del (&pu->list);
	FREE_VBUFFER(pu);
    }    
    
}

/*
 * Return the next queued_urb_t from the list, and delete it
 */
 
static queued_urb_t *get_queued_urb(struct list_head *q)
{
    queued_urb_t *pu;
    

    /*Check first for an empty list*/
    if (list_empty(q))
    {
	pu = NULL;
    }
    else
    {
        
        pu = list_entry (q->next, queued_urb_t, list );
        list_del(q->next);
    }
    
    return (pu);
}


/*
 * Put the given urb in the in-progress control queue,
 * and eventually send it to the modem
 *
 */
static void queue_ctrl_urb(Hardware *pHw, struct urb *urb)
{
    int ret;
    unsigned int list_was_empty;
    queued_urb_t *pq;
    unsigned long flags;
    
    
    adi_enters (DBG_UTILS);
    
    adi_dbg(DBG_UTILS,"queue_ctrl_urb (%p)\n", urb);

    /* Lock spinlock */
    spin_lock_irqsave (&pHw->ctrl_q_lock, flags);    
    
    /* Look at the status of the list */
    list_was_empty = list_empty ( &pHw->ctrl_urb_ipg_q );

    /* Get a free queued_urb_t */
    pq = get_queued_urb ( &pHw->ctrl_urb_free_q );

    if ( pq == NULL ) 
    {
        /* Too bad : no more free queued_urb_t */
        adi_err ("Can't get free queued_urb_t !\n");
        goto byebye;
    }
    
    pq->urb = urb;
    pq->dev = pHw;
    urb->context = pq;          /* usefull in the complete handler ! */
    
    /* Enqueue Control Urb */
    list_add_tail ( &pq->list, &pHw->ctrl_urb_ipg_q);

    /*
     * Now, if the list was empty before we introduce our urb, we can safely
     * send it to the modem.
     */ 
    if ( list_was_empty ) 
    {
        adi_dbg(DBG_UTILS,"Submit urb %p immediately\n", urb);
        ret = usb_submit_urb (urb);
        if ( ret != 0 ) 
        {
            adi_err ("Failed to send ctrl urb (%p) with err=%d\n",
                    urb,ret);
            adi_err ("retry in 100ms\n");

            /* Schedule retry send in 200 ms */
            pHw->ctrl_urb_failed = urb;
            pHw->ctrl_urb_retry.expires = jiffies + MSEC_TO_JIFFIES (100);
            add_timer ( &pHw->ctrl_urb_retry );
            
        }
        else
        {
            pHw->ctrl_urb_failed = NULL;
        
            /*
             * For safety we also set a timer : if something weird happens, like
             * lost urb acknowledgement, we will be able to submit other URBs
             */
            pHw->CtrlUrbQueueTimer.expires = jiffies + MSEC_TO_JIFFIES(2000);
            add_timer ( &pHw->CtrlUrbQueueTimer );
        }        

    }

  byebye:
    spin_unlock_irqrestore (&pHw->ctrl_q_lock, flags);
    
    adi_leaves (DBG_UTILS);
    
    return;
}



/*
 * Completion handler for Control Urbs
 * Remove the head of the ipgt list, and eventually submit a new
 * control urb to the modem
 *
 */
static void ctrl_urb_completion(struct urb *urb)
{
    queued_urb_t *pq;
    Hardware *pHw;
    unsigned long flags;
    
    adi_enters (DBG_UTILS);
    

    pq = urb->context;
    pHw = (Hardware *)(pq->dev);
    
    spin_lock_irqsave (&pHw->ctrl_q_lock, flags);
    rm_queued_urb (urb );
    spin_unlock_irqrestore (&pHw->ctrl_q_lock,flags);
    
    adi_leaves (DBG_UTILS);
}

/*
 * This watcher is used only to rescue the modem, from an unacknowledged
 * Control Urb.
 *
 */
void ctrl_urb_q_watcher ( unsigned long ptr ) 
{    
    Hardware *pHw = (Hardware *)ptr;
    queued_urb_t *qu;
    unsigned long flags;
    
    
    adi_enters (DBG_UTILS);
    

    spin_lock_irqsave (&pHw->ctrl_q_lock, flags);

    qu = list_entry ( pHw->ctrl_urb_ipg_q.next, queued_urb_t, list);
    
    /* We need to delete the head of the list,
       and submit the next one if there's one.
       ???? Should we also CANCEL the pending URB ????
       To do this the simplest approach is to call ctrl_urb_completion
    */
    
    if  ( (list_empty (  &pHw->ctrl_urb_ipg_q ) ) ||
          ( qu == NULL ) )
    {
        adi_err ("Ctrl Urb Watcher expires but not entry in ipg queue\n");
        return;
    }
    
    adi_err ("Ctrl Urb Watcher expired. About to remove URB %p\n",
            qu->urb );
    
    rm_queued_urb ( qu->urb );


    spin_unlock_irqrestore (&pHw->ctrl_q_lock,flags);
    
    adi_leaves (DBG_UTILS);
    
}


/*
 * Remove the first urb of the ipg queue, and if the queue is not empty, send
 * the first URB.
 * Also deals with the Control Urb Queue watcher timer ( stop it or modify it)
 *
 * This function is called by ctrl_urb_completion and ctrl_urb_q_watcher
 *
 */
static void rm_queued_urb(struct urb *urb)
{
    queued_urb_t *qu,*qhead;
    Hardware *pHw;
    
    /* Check if we're able to find queued_urb_t related to this urb */
    qu = urb->context;

    if ( qu == NULL ) 
    {
        adi_err ("URB (%p) has not context.\n",urb);
        goto free_urb_and_quit;
    }

    /* Get back hardware  */
    pHw = (Hardware *) qu->dev;
    if ( !pHw ) 
    {
        adi_err ("URB %p not associated to Hdw\n",urb);
        goto free_urb_and_quit;
    }
    
    if ( list_empty ( &pHw->ctrl_urb_ipg_q ) ) 
    {
        adi_err ("URB %p not present in list !\n",urb);
        goto free_urb_and_quit;
    }
    
    /* Check that URB is at the head of the list */
    qhead = list_entry ( pHw->ctrl_urb_ipg_q.next, queued_urb_t, list);
    
    if ( qhead->urb != urb ) 
    {
        adi_err ("Head of IPG queue(%p) is not equal to URB %p\n",
                qhead->urb,urb);
    }
    /*
     * Remove this queued_urb_t entry and put it in the
     *  free queued_urb_t queue
     */
    list_del ( &qu->list );
    list_add_tail ( &qu->list, &pHw->ctrl_urb_free_q);
    

    /* Check if there's another pending URB */
    if ( list_empty ( &pHw->ctrl_urb_ipg_q ) ) 
    {
        /* The only pending URB has been acknowledged, and
           nothing is pending : we can stop the running timer
        */
        adi_dbg(DBG_UTILS,"rm_queued_urb: Stopping timer\n");
        del_timer ( &pHw->CtrlUrbQueueTimer );
    }
    else
    {
        int ret;
        
        /* Send the 1st entry of the list */
        qhead = list_entry ( pHw->ctrl_urb_ipg_q.next, queued_urb_t, list);

        adi_dbg(DBG_UTILS,"rm_queued_urb: Submitting urb %p\n",qhead->urb);
        

        ret = usb_submit_urb ( qhead->urb );
        if ( ret != 0 ) 
        {
            adi_err ("Failed to send pending ctrl urb (%p) with err=%d\n",
                    qhead->urb,ret);
            adi_err ("Retrying in 100 ms\n");
            
            /* Schedule retry send in 200 ms */
            pHw->ctrl_urb_failed = urb;
            pHw->ctrl_urb_retry.expires = jiffies + MSEC_TO_JIFFIES (100);
            add_timer ( &pHw->ctrl_urb_retry );
        }
        else
        {
            /* Set the timer to expire in 2s : just to be sure nothing is lost */
            mod_timer ( &pHw->CtrlUrbQueueTimer, jiffies + MSEC_TO_JIFFIES(2000) );
        }
    }
    
  free_urb_and_quit:
    /* Now free transfer buffer and URB */
    FREE_KBUFFER(urb->transfer_buffer);
    usb_free_urb(urb);                 
}

/**
 * Retry to send urb
 */
void ctrl_urb_retry_send ( unsigned long ptr ) 
{    
    Hardware *pHw = (Hardware *)ptr;
    unsigned long flags;
    queued_urb_t *qu;
    int ret;
    
        
    adi_enters (DBG_UTILS);
    adi_dbg(DBG_UTILS,"ctrl_urb_retry_send: WAKE UP !!\n");
    

    spin_lock_irqsave (&pHw->ctrl_q_lock, flags);

    if ( pHw->ctrl_urb_failed == NULL ) 
    {
        adi_err ("ctrl_urb_retry_send expires but failed urb is NULL !\n");
        return;
    }
        
    
    qu = list_entry ( pHw->ctrl_urb_ipg_q.next, queued_urb_t, list);    
    
    if  ( (list_empty (  &pHw->ctrl_urb_ipg_q ) ) ||
          ( qu == NULL ) )
    {
        adi_err ("ctrl_urb_retry_send expires but not entry in ipg queue\n");
        return;
    }

    if ( qu->urb != pHw->ctrl_urb_failed ) 
    {
        adi_warn ("faulty urb not at beginning of list !\n");
    }
    
    
    ret = usb_submit_urb (pHw->ctrl_urb_failed);
    if ( ret != 0 ) 
    {
        adi_err ("ADI: Failed to send faulty ctrl urb (%p) with err=%d\n",
                        pHw->ctrl_urb_failed ,ret);
        adi_err ("ADI: retry in 100ms\n");

        /* Schedule retry send in 100 ms */
        pHw->ctrl_urb_retry.expires = jiffies + MSEC_TO_JIFFIES (100);
        add_timer ( &pHw->ctrl_urb_retry );
    }
    else
    {
        pHw->ctrl_urb_failed = NULL;

        if ( !timer_pending ( &pHw->CtrlUrbQueueTimer ) ) 
        {
            /*
             * For safety we also set a timer : if something weird happens, like
             * lost urb acknowledgement, we will be able to submit other URBs
             */
            pHw->CtrlUrbQueueTimer.expires = jiffies + MSEC_TO_JIFFIES(2000);
            add_timer ( &pHw->CtrlUrbQueueTimer );
        }        
    }

    spin_unlock_irqrestore (&pHw->ctrl_q_lock,flags);
    
    adi_leaves (DBG_UTILS);
    
    
}

    

/**
 * Unlink 1st control urb if it's pending
 */
void unlink_ipg_ctrl_urb ( Hardware *pHw ) 
{
    queued_urb_t *qhead;
    unsigned long flags;
    
    /* Take lock */
    spin_lock_irqsave (&pHw->ctrl_q_lock, flags);    

    if ( !list_empty ( &pHw->ctrl_urb_ipg_q ) ) {
			
        /* We have to unlink it */
        qhead = list_entry ( pHw->ctrl_urb_ipg_q.next, queued_urb_t, list);
        
        qhead->urb->transfer_flags &= ~USB_ASYNC_UNLINK;
        usb_unlink_urb(qhead->urb);
    }
    
    spin_unlock_irqrestore (&pHw->ctrl_q_lock, flags);
}



/*********************************************************************************
$Log: Util.c,v $
Revision 1.7  2003/06/03 22:52:10  sleeper
Changed ZAPS to adi_dbg/err macros

Revision 1.6  2003/04/23 22:29:01  sleeper
Fix a bug in Ctrl URB when submission failed

Revision 1.5  2003/03/11 00:36:50  sleeper
Fix a compil bug

Revision 1.4  2003/03/10 23:19:39  sleeper
Change Control URB submission

Revision 1.3  2003/03/05 22:25:07  sleeper
Fix a typo

Revision 1.2  2003/02/10 23:40:41  sleeper
Fix memory leak

Revision 2.00  2002/05/24 21:59:30  Anoosh Naderi
Clean upthe code.

Revision 1.9  2002/01/14 21:59:30  chris.edgington
Added GPL header.

Revision 1.8  2002/01/08 20:06:12  chris.edgington
Removed UnicodeHexStringToBytes - usb does Unicode conversion for us.
Wrote htoi to assist in converting MAC string to numeric format.
Finished GetMacAddress.

Revision 1.7  2002/01/08 16:23:40  chris.edgington
Added #ifdef around GetMacAddress code until it is working.

Revision 1.6  2002/01/08 16:10:10  chris.edgington
Added some necessary comments.
Removed AdiCustomWriteIm - no longer needed.
Added GetMacAddress and UnicodeHexStringToBytes.

Revision 1.5  2002/01/02 22:02:49  chris.edgington
Added PutBufferInQ function.

Revision 1.4  2001/12/28 19:31:33  chris.edgington
Fixed AllocQueuedBuffers and DeAllocQueuedBuffers to work on Linux.
Changed all routines that send Control urbs to use QueueControlUrb.
Added QueueControlUrb, CtrlUrbCompletion, and CtrlUrbQueueWatcher.

Revision 1.3  2001/12/26 22:46:51  chris.edgington
Changed WriteDspMsg to use URB - fixes kernel crash when in interrupt.
Moved AdiCustomWrite to here from Boot.c and changed it to use a dynamic URB.

Revision 1.2  2001/12/22 21:40:29  chris.edgington
Added WriteDspMsg.

Revision 1.1  2001/12/22 19:54:33  chris.edgington
First version - moved routines from AdiUsbAdslDriver.c to here.
***********************************************************************************/
