/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Defines procedures for ATR/Hist. bytes/Protocols handling
 *
 * Author :         Frank Thater (FTH)
 *
 * Last modified:   01/24/2000
 *
 *****************************************************************************/

#ifdef DEBUG 
#include <stdio.h>      /* Includes */
#endif

#include <string.h>
#include "eco5000.h"
#include "defines.h"
#include "ctapi.h"
#include "sercom.h"
#include "atr.h"
#include "ecotools.h"


int FTable[]  = { 372, 372, 558, 744, 1116, 1488, 1860, -1, -1, 512, 768, 1024, 1536, 2048, -1, -1};
int DTable[]  = { -1, 1, 2, 4, 8, 16, 32, -1, 12, 20, -1, -1, -1, -1, -1, -1};
int DxTable[] = {  0, 1, 2, 3, 4,  3,  3,  0,  8,  3,  0,  0,  0,  0,  0,  0};


#define MATCH(x,y) ((x >= (y - y / 20)) && (x <= (y + y / 20)))

int DetermineBaudrate(int F, int D)

{
    int br;

    br = 14318000 * D / (F * 4);

    if (MATCH(br, 9600))
        br = 9600;
    else if (MATCH(br, 19200))
        br = 19200;
    else if (MATCH(br, 38400))
        br = 38400;
    else if (MATCH(br, 57600))
        br = 57600;
    else if (MATCH(br, 115200))
        br = 115200;
    else
        br = -1;

    return br;
}



/*
 * Perform PPS Sequence
 *
 */

int PerformPPS(struct eco5000_t *ctx, unsigned char PPS0,
               unsigned char PPS1, unsigned char PPS2, unsigned char PPS3)

{
    unsigned char buff[6], *p, lrc, temp;
    int len, rc;

    lrc = 0;
    p = buff;

    *p++ = 0xFF;
    *p++ = PPS0;

    lrc = 0xFF ^ PPS0;
    
    if (PPS0 & 0x10) {
        *p++ = PPS1;
        lrc ^= PPS1;
    }

    if (PPS0 & 0x20) {
        *p++ = PPS2;
        lrc ^= PPS2;
    }

    if (PPS0 & 0x40) {
        *p++ = PPS3;
        lrc ^= PPS3;
    }

    *p++ = lrc;
    len = p - buff;

    iccWrite(ctx->fh, ctx->Indirect, buff, len);
    iccRead(ctx->fh, ctx->Indirect, buff, len);

    rc = iccRead(ctx->fh, ctx->Indirect, &temp, 1);

    if (rc < 0)
        return rc;

    if ((rc != 1) || (temp != 0xFF))
        return -1;

    lrc = 0xFF;
    
    if (iccRead(ctx->fh, ctx->Indirect, &PPS0, 1) != 1)
        return -1;

    lrc ^= PPS0;

    if (PPS0 & 0x10) {
        if (iccRead(ctx->fh, ctx->Indirect, &PPS1, 1) != 1)
            return -1;
        lrc ^= PPS1;
    }

    if (PPS0 & 0x20) {
        if (iccRead(ctx->fh, ctx->Indirect, &PPS2, 1) != 1)
            return -1;
        lrc ^= PPS2;
    }

    if (PPS0 & 0x40) {
        if (iccRead(ctx->fh, ctx->Indirect, &PPS3, 1) != 1)
            return -1;
        lrc ^= PPS3;
    }

    if (iccRead(ctx->fh, ctx->Indirect, &temp, 1) != 1)
        return -1;

    if (temp != lrc)
        return -1;

    ctx->Protocol = PPS0 & 0x0F;

    if (PPS0 & 0x10) {
        ecoChangeBaudrate(ctx, DetermineBaudrate(FTable[PPS1 >> 4], DTable[PPS1 & 0x0F]));
    }

    return 0;
}



int GetATR(struct eco5000_t *ctx)

{  /* 
      GetATR for sync/async cards - fill structure vars (ctx->HCC, ctx->cATR, ...)
   
      determines whether sync. or async. card present !!           
   */   
    int response;
    int prot;

    memset(ctx->ATR, 0, sizeof(ctx->ATR));              /* Clear ATR structure */
    memset(ctx->HCC, 0, sizeof(ctx->HCC));              
   
    ctx->Protocol = 0;
    ctx->NumOfHB = 0;
    ctx->LenOfATR = 0;
   
    if ((response = ASync_GetATR(ctx)) == 1) {
        if((response = Sync_GetATR(ctx) < 0))
            return ERR_CT;;
        ctx->Type = SYNC_ICC;
    } else if(response < 0) {
        return response;
    } else {
        ctx->Type = ASYNC_ICC;

        if (!ctx->Specific) {
            if ((ctx->AvailProt & 0x02) && (ctx->Protocol == 0))
                prot = 1;
            else
                prot = ctx->Protocol;

#if 0
            if (DetermineBaudrate(FTable[ctx->FI], DTable[ctx->DI]) != -1) {
                PerformPPS(ctx, 0x10 | prot, (ctx->FI << 4) | ctx->DI, 0, 0);
            } else
#endif
            if (prot != ctx->Protocol)
                PerformPPS(ctx, prot, 0, 0, 0);
        }
    }

    return OK;

};



/*
 * Deactivate contacts in the specified order
 *
 */

int Async_PowerOff(struct eco5000_t *ctx)

{
    int rc;
    
    if((rc = ecoCommand(ctx, CLR_RESET, 0, NULL, 0, NULL)) < 0)
        return rc;

    if((rc = ecoCommand(ctx, CLOCK_OFF, 0, NULL, 0, NULL)) < 0)
        return rc;
   
    if((rc = ecoCommand(ctx, POWER_OFF, 0, NULL, 0, NULL)) < 0)
        return rc;

    return 0;
}



/* 
 * GetATR for async cards - fill structure vars (ctx->HCC, ctx->ATR, ...)
 */

int ASync_GetATR(struct eco5000_t *ctx)

{
    int ATRCnt,rc;
    int protsection;
    enum { cold, warm, done, error} ResetState;
    int F,Fi;
    unsigned char i;
    unsigned char temp;
    unsigned char help;

    if ((rc = ecoCommand(ctx, POWER_ON, 0, NULL, 0, NULL)) < 0) /* Try async. reset              */
        return rc;

    if ((rc = ecoCommand(ctx, CLOCK_ON, 0, NULL, 0, NULL)) < 0) 
        return rc;

    ResetState = cold;
        
    while (ResetState != done) {       /* We come back here for warm   */
                                        /* reset if required            */
        if ((rc = ecoCommand(ctx, SET_RESET_RX_ON, 0, NULL, 0, NULL)) < 0)  
            return 1;

        /* Expect answer within 40000 clock cycles, which translates to */
        /* approx. 10ms for 3.57 Mhz clock rate. We take 100ms just to  */
        /* be sure everything works O.K.                                */

        rs232Mode(ctx->fh, -1, 0, -1, -1, 100);

        if (rs232Read(ctx->fh, &temp, 1) == 0)
            temp = 0;

        ATRCnt = 0;

        switch(temp) {              /* Get/Save initial char in ATR [0] */
            case 0x3b:
                ctx->ATR[ATRCnt++] = temp;
                ctx->Indirect = FALSE;
                break;
            case 0x03:
                ctx->ATR[ATRCnt++] = 0x3F;
                ctx->Indirect = TRUE;
                break;
            default:
                if (ResetState == warm) {
                    if ((rc = Async_PowerOff(ctx)) < 0)
                        return rc;

                    return 1;
                }

                ResetState = warm;

                if ((rc = ecoCommand(ctx, CLR_RESET, 0, NULL, 0, NULL)) < 0)
                    return rc;

                continue;
        }

        protsection = 0;                /* Which protocol section are we in */

        F = Fi = 372;
        ctx->FI = 1;
        ctx->DI = 1;
        ctx->D = DTable[ctx->DI];       /* Set D and DI to default value    */
        ctx->AvailProt = 0;             /* Bitmap of supported protocols    */
        ctx->Protocol = 0;              /* Assume T=0 as default            */
        ctx->Specific = FALSE;          /* Assume negotible mode            */

        ctx->T0_WI = 10;                /* T=0: work waiting time from TC(2)*/
        ctx->T1_IFSC = 32;              /* T=1: information field size TA(i)*/
        ctx->T1_CWI = 13;               /* T=1: Char waiting time indx TB(i)*/
        ctx->T1_BWI = 4;                /* T=1: Block waiting time inx TB(i)*/

        ctx->disableRX_ON = FALSE;      /* Assume old firmware and old T=0 handling as standard  */
        
        temp = 0;
        if (iccRead(ctx->fh, ctx->Indirect, &temp, 1) != 1)
            ResetState = (ResetState == warm ? error : warm);

        ctx->NumOfHB = temp & 0x0F;     /* Number of hist. Bytes            */
        ctx->ATR[ATRCnt++] = temp;      /* Save T0                          */

        i = 1;
        do {
            help = (temp >> 4);
            if (help & 1) {             /* Get TAx                          */
                if (iccRead(ctx->fh, ctx->Indirect, &temp, 1) <= 0)
                    ResetState = (ResetState == warm ? error : warm);

                ctx->ATR[ATRCnt++] = temp;

                if (i == 1) {           /* TA(1) present ?                  */
                    ctx->FI = temp >> 4;
                    Fi = FTable[temp >> 4];     /* Store FI and Di          */
                    ctx->DI = temp & 0xF;
                }

                /* If the card sends TA(2), then we are in specific mode    */
                
                if (i == 2) {           /* TA(2) present ?                  */
                    ctx->Specific = TRUE;

                    if (!(temp & 0x10)) {
                        F = Fi;
                        ctx->D = DTable[ctx->DI];
                    }

                    /* If the card requires a F other than 372 or a not     */
                    /* supported baud rate, but allows a warm reset,        */
                    /* then we try that, if we have not already done so     */

                    if ((F != 372) || (ctx->D == -1)) {
                        if (!(temp & 0x80) && (ResetState == cold))
                            ResetState = warm;
                        else            /* Card not supported               */
                            ResetState = error;
                    } else {
                        ctx->Protocol = temp & 0x0F;
                        ctx->AvailProt = ctx->Protocol << 1;
                    }
                }

                if ((i > 2) && (protsection == 1))
                    ctx->T1_IFSC = temp;
            }

            if (help & 2) {             /* Get TBx                          */
                if (iccRead(ctx->fh, ctx->Indirect, &temp, 1) <= 0)
                    ResetState = (ResetState == warm ? error : warm);

                ctx->ATR[ATRCnt++] = temp;

                if ((i > 2) && (protsection == 1)) {
                    ctx->T1_CWI = temp & 0x0F;
                    ctx->T1_BWI = temp >> 4;
                }
            }
            
            if (help & 4) {             /* Get TCx                          */
                if (iccRead(ctx->fh, ctx->Indirect, &temp, 1) <= 0)
                    ResetState = (ResetState == warm ? error : warm);

                ctx->ATR[ATRCnt++] = temp;

                if (i == 2)             /* Store WI for T=0                 */
                    ctx->T0_WI = temp;
            }
            
            if (help & 8) {             /* Get TDx                          */
                if (iccRead(ctx->fh, ctx->Indirect, &temp, 1) <= 0)
                    ResetState = (ResetState == warm ? error : warm);

                ctx->ATR[ATRCnt++] = temp;

                protsection = temp & 0x0F;
                ctx->AvailProt = 1 << protsection;

                if (i == 1)
                    ctx->Protocol = protsection;

            } else
                temp = 0;
            i++;
        } while (temp);

        for (i = 0; i < ctx->NumOfHB; i++) {
            if (iccRead(ctx->fh, ctx->Indirect, &temp, 1) <= 0)
                ResetState = (ResetState == warm ? error : warm);

            ctx->ATR[ATRCnt++] = temp;
            ctx->HCC[i] = temp;     
        }
        
        if (!ctx->AvailProt)            /* We expect card to have at least T=0 */
            ctx->AvailProt |= 1;

        if (ctx->AvailProt != 1) {                      /* Get check character */
            if (iccRead(ctx->fh, ctx->Indirect, &temp, 1) <= 0)
                ResetState = (ResetState == warm ? error : warm);

            ctx->ATR[ATRCnt++] = temp;
        }

        ctx->LenOfATR = ATRCnt; /* - 2;  to delete SW1/SW2 */

        if (ResetState == cold)
            ResetState = done;

        if (ResetState == error) {
            if ((rc = Async_PowerOff(ctx)) < 0)
                return rc;
            return 1;
        }
    }


    if (ctx->Indirect) {
        if ((rc = ecoCommand(ctx, SET_INDIRECT_CONVENTION, 0, NULL, 0, NULL)) < 0)
            return rc;
    }  

#ifdef DEBUG
    printf("\nHB: %u\n", ctx->NumOfHB);
    printf("\nATRSize: %u\n", ctx->LenOfATR);
    printf("Protocol: T=%u  ", ctx->Protocol);
    printf("\nAsync. Card present!!\n");     
    for(i=0;i < ctx->LenOfATR; i++)
      printf("%02x", ctx->ATR[i]);
#endif

    return 0;
}



/* 
 *  GetATR for sync cards - fill structure vars (ctx->HCC, ctx->ATR, ...)
 */
 
int Sync_GetATR(struct eco5000_t *ctx)

{  
    unsigned char i;
    int status = 0;     
        
    if ((status = ecoCommand(ctx, POWER_ON, 0, NULL, 0, NULL)) < 0)     
        return status;
        
    if ((status = ecoCommand(ctx, SYNC_ATR, 0, NULL, 4, ctx->ATR)) < 0) {                
        return status;
    }

#ifdef DEBUG
    printf("\nSync. Card present!!\n");
#endif

    ctx->LenOfATR = 4;
    ctx->NumOfHB = 2;
        
    for(i=0;i<2;i++)
        ctx->HCC[i] = ctx->ATR[i+2];
        
    /* Determine what type of memory card is inserted */
    if(ctx->ATR[0] == 0xff)
        ctx->Protocol = MC_SDAP;
    else
        ctx->Protocol = ctx->ATR[0] >> 4;
        
#ifdef DEBUG
    printf("Protocol Type = %02x", ctx->Protocol);
#endif    
    return 0;
}



