#ifndef _ECS_MAILLAGE_H_
#define _ECS_MAILLAGE_H_

/*============================================================================
 *  Prototypes des fonctions de base
 *   associees a la structure `ecs_maillage_t' decrivant un maillage
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilite
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Post-Traitement"
 *----------------------------------------------------------------------------*/

#include "ecs_post_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_entmail_publ.h"
#include "ecs_famille_publ.h"
#include "ecs_connect_publ.h"
#include "ecs_param_perio_glob.h"
#include "ecs_param_rc_glob.h"
#include "ecs_select_fac_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_maillage_publ.h"


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *    Fonction qui cree une structure definissant un maillage
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_maillage__cree
(                                      /* <-- Pointeur structure maillage     */
 ecs_dim_t         dim_e        ,      /* --> Dimension spatiale              */
 ecs_entmail_t  ** vect_entmail ,      /* --> Tableau des entites de maillage */
 ecs_connect_t  ** vect_connect        /* --> Tableau des connectivites       */
) ;


/*----------------------------------------------------------------------------
 *  Fonction liberant une structure `ecs_maillage_t' donnee en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_maillage__detruit
(
 ecs_maillage_t * this_maillage
) ;


/*----------------------------------------------------------------------------
 *  Fonction imprimant le contenu d'une structure `ecs_maillage_t' donnee
 *   dans le fichier de nom donne
 *   precede par le `titre'
 *----------------------------------------------------------------------------*/

void ecs_maillage__imprime
(
 const ecs_maillage_t  *const this_maillage    ,
 const char            *const nom_fichier_dump ,
 const ecs_int_t              nbr_imp          ,
 const char            *const titre
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le type d'ntit de plus grande dimension
 *   contenue dans une structure `ecs_maillage_t'
 *----------------------------------------------------------------------------*/


ECS_ENTMAIL_E ecs_maillage__ret_entmail_max
(
 const ecs_maillage_t * const this_maillage
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets d'une structure `ecs_maillage_t'
 *----------------------------------------------------------------------------*/

float ecs_maillage__ret_taille
(
 const ecs_maillage_t *const this_maillage
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui cree une structure maillage en connectivite nodale
 *   a partir du vecteur des entites de maillage
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_maillage__cree_nodal
(
 ecs_dim_t              dim_e        ,
 ecs_entmail_t * *const vect_entmail
) ;


/*----------------------------------------------------------------------------
 *  Fusion des sommets confondus et suppression des lments dgnrs
 *----------------------------------------------------------------------------*/

void ecs_maillage__nettoie_descend
(
 ecs_maillage_t  *this_maillage
) ;


/*----------------------------------------------------------------------------
 *  Suppression des sommets ne participant pas  la connectivit
 *   et fusion des lments surfaciques confondus ventuels
 *----------------------------------------------------------------------------*/

void ecs_maillage__nettoie_nodal
(
 ecs_maillage_t  *this_maillage
) ;


/*----------------------------------------------------------------------------
 *  Correction si ncessaire de l'orientation des lments en
 *   connectivit nodale.
 *----------------------------------------------------------------------------*/

void ecs_maillage__orient_nodal
(
       ecs_maillage_t  *const this_maillage ,  /*  -> Dfinition du maillage  */
       ecs_tab_int_t   *const liste_cel_err ,  /* <-  Liste cellules avec
                                                *     erreur (optionnelle)    */
       ecs_tab_int_t   *const liste_cel_cor ,  /* <-  Liste cellules cor-
                                                *     riges (optionnelle)    */
 const ecs_bool_t             correc_orient    /*  -> Correction ou non       */
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui assigne la tete de la liste chainee des familles donnee
 *   a la structure de maillage donnee
 *----------------------------------------------------------------------------*/

void ecs_maillage__definit_famille
(
 ecs_maillage_t   *const maillage     ,
 ecs_famille_t  * *const vect_famille
) ;


/*----------------------------------------------------------------------------
 *  Fonction realisant, pour tout le maillage,
 *   la transformation des references a des etiquettes d'entite de maillage
 *   en references par rapport aux indices de l'entite de maillage
 *----------------------------------------------------------------------------*/

void ecs_maillage__cree_indice
(
 ecs_maillage_t  *this_maillage
) ;


/*----------------------------------------------------------------------------
 *  Fonction realisant, a partir d'une connectivite de maillage donnee,
 *   la connectivite descendante du maillage
 *----------------------------------------------------------------------------*/

void ecs_maillage__connect_descend
(
 ecs_maillage_t * maillage    /* <-> Maillage pour laconnectivite descendante */
) ;


/*----------------------------------------------------------------------------
 *  Fonction realisant le tri des elements suivant leur type geometrique
 *  La fonction affiche le nombre d'elements par type geometrique
 *----------------------------------------------------------------------------*/

void ecs_maillage__trie_typ_geo
(
 ecs_maillage_t * maillage     /* <-> Maillage contenant les elements a trier */
) ;


/*----------------------------------------------------------------------------
 *  Fonction ralisant le dcoupage des faces polygonales en triangles
 *
 *  Le maillage doit tre en connectivit nodale (faces -> sommets)
 *----------------------------------------------------------------------------*/

void ecs_maillage__dec_poly_tria
(
 ecs_maillage_t  *const maillage
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui dfinit un nouveau maillage
 *   par extraction d'une partie du maillage donn
 *
 *  La partie extraite correspond aux lments :
 *  -       ayant        une des couleurs donnes dans la liste des couleurs
 *  - et/ou appartenant  un des groupes  donns  dans la liste des groupes
 *
 *  On peut aussi choisir d'inverser la slection
 *
 *  Les lments  extraire doivent tre tous de mme dimension :
 *  cellules ou faces ou artes ou sommets
 *
 *  On construit automatiquement une filiation.
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_maillage__extrait
(
       ecs_maillage_t       *const maillage         ,
       ECS_ENTMAIL_E               entmail_sel_e    ,
 const ecs_tab_int_t        *const liste_filtre     ,
 const ecs_tab_int_t               liste_couleur    ,
 const ecs_tab_char_t              liste_groupe     ,
 const ecs_bool_t                  inv_selection    ,
 const ecs_bool_t                  herite_attributs
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui dfinit un nouveau maillage nodal
 *   par extraction d'une partie du maillage nodal donn
 *
 *  La partie extraite correspond aux lments :
 *  -       ayant        une des couleurs donnes dans la liste des couleurs
 *  - et/ou appartenant a un des groupes  donns  dans la liste des groupes
 *
 *  On peut aussi choisir d'inverser la selection
 *
 *  Les elements  extraire doivent etre tous de mme dimension :
 *  cellules ou faces ou artes ou sommets
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_maillage__extrait_nodal
(
       ecs_maillage_t       *const maillage         ,
       ECS_ENTMAIL_E               entmail_sel_e    ,
 const ecs_tab_int_t        *const liste_filtre     ,
 const ecs_tab_int_t               liste_couleur    ,
 const ecs_tab_char_t              liste_groupe     ,
 const ecs_bool_t                  inv_selection
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui slectionne une liste d'entits portant des familles
 *   non vides dans un maillage donn
 *
 *  La partie slectionne correspond aux lments :
 *  -       ayant        une des couleurs donnes dans la liste des couleurs
 *  - et/ou appartenant  un des groupes  donns  dans la liste des groupes
 *
 *  La slection peut tre inverse (en restant dans les entits de
 *  famille non vides)
 *----------------------------------------------------------------------------*/

ecs_tab_int_t  ecs_maillage__selectionne_fam
(
       ecs_maillage_t       *const maillage      ,
       ECS_ENTMAIL_E               entmail_sel_e ,
 const ecs_tab_int_t        *const liste_filtre  ,
 const ecs_tab_int_t               liste_couleur ,
 const ecs_tab_char_t              liste_groupe  ,
 const ecs_bool_t                  inv_selection
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui concatene dans un maillage recepteur donne,
 *   un maillage a concatener donne
 *----------------------------------------------------------------------------*/

void ecs_maillage__concatene
(
 ecs_maillage_t *const maillage_recept ,
 ecs_maillage_t *const maillage_concat
) ;


/*----------------------------------------------------------------------------
 *  Cration et initialisation d'un maillage extrait correspondant a un cas de
 *  coupe de faces : on extrait un maillage correspondant  la zone
 *  slectione par liste, puis on dcoupe ventuellement les faces de plus
 *  de 4 artes de ce maillage en triangles ; finalement, on l'ordonne
 *  selon le numro de type de face.
 *
 *  On remplit un tableau de pointeurs sur les structures maillage associes
 *   chaque niveau de dcoupage de la coupe.
 *----------------------------------------------------------------------------*/

void ecs_maillage__cree_coupe
(
 const char              *const   nom_coupe                            ,
       ecs_maillage_t    *const   maillage                             ,
 const ecs_tab_int_t              liste_fac                            ,
       ecs_maillage_t    *        maillage_coupe[ECS_POST_FAC_MAX_FIN] ,
 const ecs_int_t                  nbr_dump                             ,
 const char              *const   nom_fic_dump                         ,
 const ecs_bool_t                 herite_attributs                     ,
       ecs_post_type_t            type_post                            ,
       ecs_post_t        *const   cas_post
) ;


/*----------------------------------------------------------------------------
 *  Destruction des structures correspondant  une coupe
 *----------------------------------------------------------------------------*/

void ecs_maillage__detruit_coupe
(
       ecs_maillage_t    *const maillage                               ,
       ecs_maillage_t    *      maillage_extrait[ECS_POST_FAC_MAX_FIN]
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui transforme la connectivit descendante de la partie
 *  surfacique d'un maillage en connectivit nodale
 *
 *  La connectivit "aretes -> sommets" est supprime, ainsi que
 *  l'entit de maillage "artes".
 *----------------------------------------------------------------------------*/

void ecs_maillage__connect_fac_som
(
 ecs_maillage_t  *const maillage
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui cre la liste des faces intrieures  un maillage.
 *  On ne compte pas ici les faces priodiques parmi les faces intrieures,
 *  cette fonction tant destine  filtrer le post-triatment, et les
 *  faces priodiques tant dj affichables par ailleurs.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_maillage__liste_fac_int
(
 ecs_maillage_t  *const maillage
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit la liste des cellules attaches  une liste
 *  de faces fournie en argument.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_maillage__liste_cel_fac
(
       ecs_maillage_t  *const maillage  ,
 const ecs_tab_int_t          liste_fac
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui calcule les coordonnes min et max du domaine
 *----------------------------------------------------------------------------*/

void ecs_maillage__calc_coo_ext
(
 ecs_maillage_t *const maillage
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui modifie les coordonnes du maillage
 *----------------------------------------------------------------------------*/

void ecs_maillage__transf_coo
(
 ecs_maillage_t  *const maillage,
 const double           matrice[3][4]
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui recolle les faces non conformes
 *
 *  Les listes des faces nouvelles ou modifies sont construites (et alloues)
 *  ici ; les structures liste_fac_new et liste_fac_mod correspondantes sont
 *  donc vides en entre ; idem pour liste_fac_err qui indiquera les indices
 *  des faces pour lesquelles le dcoupage en sous-faces a chou
 *
 *  On renvoie 1 si l'on a appell effectivement le recollement, 0 sinon
 *  (i.e. si la slection de faces  recoller est vide).
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_maillage__recolle
(
       ecs_maillage_t    *const maillage      ,
 const ecs_select_fac_t         select_fac_rc ,
       ecs_tab_int_t     *const liste_fac_new ,
       ecs_tab_int_t     *const liste_fac_mod ,
       ecs_tab_int_t     *const liste_fac_err ,
 const ecs_param_rc_t           param_rc
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui recolle des faces non conformes dont la visibilit
 *  est connue (par exemple une filiation) ;
 *
 *  Les listes des faces nouvelles ou modifies sont construites (et alloues)
 *  ici ; les structures liste_fac_new et liste_fac_mod correspondantes sont
 *  donc vides en entre ; idem pour liste_fac_err qui indiquera les indices
 *  des faces pour lesquelles le dcoupage en sous-faces a chou
 *
 *  On renvoie 1 si l'on a effectivement appell le recollement, 0 sinon
 *  (i.e. si le champ de type "visibilit" est introuvable).
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_maillage__recolle_vis
(
 ecs_maillage_t    *const maillage      ,
 ecs_tab_int_t     *const liste_fac_new ,
 ecs_tab_int_t     *const liste_fac_mod ,
 ecs_tab_int_t     *const liste_fac_err
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit les familles
 *----------------------------------------------------------------------------*/

void ecs_maillage__cree_famille
(
 ecs_maillage_t *const maillage  ,
 ECS_ENTMAIL_E         ent_deb_e ,
 ECS_ENTMAIL_E         ent_fin_e
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui detruit les familles
 *----------------------------------------------------------------------------*/

void ecs_maillage__detruit_famille
(
 ecs_maillage_t *const maillage
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit les attributs "groupe" et "couleur"
 *   a partir des familles
 *----------------------------------------------------------------------------*/

void ecs_maillage__cree_attribut
(
 ecs_maillage_t *const maillage  ,
 ECS_ENTMAIL_E         ent_deb_e ,
 ECS_ENTMAIL_E         ent_fin_e
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui supprime les attributs "groupe" et "couleur"
 *----------------------------------------------------------------------------*/

void ecs_maillage__supprime_attributs
(
 ecs_maillage_t * this_maillage
) ;


/*----------------------------------------------------------------------------
 *  Vrification d'un maillage et calcul de critres de qualit
 *----------------------------------------------------------------------------*/

ecs_bool_t ecs_maillage__verif
(
       ecs_maillage_t      * const maillage          ,
 const ecs_int_t                   nbr_dump          ,
 const char                * const nom_fic_dump      ,
       ecs_post_t          * const cas_post
) ;


#endif /* _ECS_MAILLAGE_H_ */
