/*============================================================================
 *  Dfinitions des fonctions de base
 *   associes  la structure `ecs_connect_t' dcrivant une connectivit
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/

#include <assert.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' systeme ou BFT
 *----------------------------------------------------------------------------*/

#include <bft_file.h>
#include <bft_mem.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_fic.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_entmail_pcp.h"
#include "ecs_entmail.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associ au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_connect.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' prives   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_connect_priv.h"


/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui cre une structure `ecs_connect_t'
 *----------------------------------------------------------------------------*/

ecs_connect_t * ecs_connect__cree
(                                    /* <-- Rfrence sur la connect. cre   */
 ecs_entmail_t     * entmail       , /* --> Pointeur sur lments du maillage */
 ecs_entmail_t     * entmail_inf   , /* --> Pointeur sur lments `fils'      */
 ECS_CONNECT_NAT_E   connect_nat_e   /* --> Type de connect. entre entits    */
)
{

  ecs_connect_t *this_connect ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(entmail != NULL && entmail_inf != NULL) ;


  BFT_MALLOC(this_connect, 1, ecs_connect_t);

  this_connect->entmail       = entmail       ;
  this_connect->entmail_inf   = entmail_inf   ;

  this_connect->connect_nat_e = connect_nat_e ;


  return this_connect ;


}


/*----------------------------------------------------------------------------
 *  Fonction librant une structure `ecs_connect_t' donne en argument
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_connect_t * ecs_connect__detruit
(
 ecs_connect_t *this_connect
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_connect != NULL) ;


  /* Libration de la structure de type `ecs_connect_t' */
  /*================================================*/


  BFT_FREE(this_connect) ;


  /* Retourne `NULL' */

  return this_connect ;


}


/*----------------------------------------------------------------------------
 *  Fonction imprimant le contenu d'une structure `ecs_connect_t' donne
 *   sur le flux dcrit par la structure `bft_file_t'
 *----------------------------------------------------------------------------*/

void ecs_connect__imprime
(
 const ecs_connect_t *const this_connect ,      /* --> Structure  imprimer   */
       ecs_int_t            imp_col      ,
       bft_file_t    *const fic_imp             /* --> Descripteur du fichier */
)
{


#define ECS_FCT_IMP_CONNECT_ENTMAIL      "entmail"
#define ECS_FCT_IMP_CONNECT_ENTMAIL_INF  "entmail_inf"
#define ECS_FCT_IMP_CONNECT_CONNECT_NAT  "connect_nat_e"


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_connect != NULL) ;


  imp_col++ ;


  /* Impression des pointeurs contenus dans la structure `ecs_connect_t' */

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_CONNECT_ENTMAIL,
                       ECS_TYPE_void, this_connect->entmail) ;

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_CONNECT_ENTMAIL_INF,
                       ECS_TYPE_void, this_connect->entmail_inf) ;

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_CONNECT_CONNECT_NAT,
                       ECS_TYPE_ecs_int_t, &this_connect->connect_nat_e) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets d'une structure `ecs_connect_t'
 *----------------------------------------------------------------------------*/

float ecs_connect__ret_taille
(
 const ecs_connect_t *const this_connect
)
{

  float         taille ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_connect != NULL) ;


  taille = sizeof(*this_connect) ;


  return taille ;


}


/*----------------------------------------------------------------------------
 *  Fonction ralisant, pour une connectivit donne,
 *   une transformation entre ses 2 entits de maillage membres
 *----------------------------------------------------------------------------*/

void ecs_connect__cree_indice
(
 ecs_connect_t     *this_connect /* <-> Connectivit concerne par la transf. */
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (this_connect->connect_nat_e == ECS_CONNECT_NAT_LABEL) {

    ecs_entmail_pcp__cree_indice(this_connect->entmail    ,
                                 this_connect->entmail_inf ) ;

    this_connect->connect_nat_e = ECS_CONNECT_NAT_INDEX ;

  }

}


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'une connectivit donne
 *   en une "sous-connectivit"
 *
 *  Elle renvoie la nouvelle connectivit associant
 *   l'entit de maillage des lments      avec
 *   l'entit de maillage des sous-lments
 *
 *  La connectivit donne met en jeu des lments dfinis
 *   par des lments d'une entit de maillage lmentaire
 *  La "sous-connectivit" correspondante met en jeu les sous-lments dfinis
 *   par des lments de  l'entit de maillage lmentaire
 *
 *  Exemple :
 *  -------
 *
 * connectivit donne                : Cellule->Sommet (ECS_CONNECT_CEL_SOM)
 *                           lments : cellules
 *     Entit de maillage lmentaire : Sommet
 *
 * "sous-connectivit" correspondante : Face->Sommet    (ECS_CONNECT_FAC_SOM)
 *                      Sous-lments : faces
 *----------------------------------------------------------------------------*/

ecs_connect_t * ecs_connect__cree_sous_connect
(                                 /* <-- Nouvelle connectivit associant      */
                                  /*      les lments  leurs sous-lments  */
 ecs_connect_t   * this_connect , /* --> Connectivit donne  transformer    */
 ecs_connect_t  ** sous_connect , /* <-> Sous-connectivit                    */
 ecs_entmail_t  ** ent_sous_elt , /* <-- Entit de maillage des sous-lments */
 ECS_CONNECT_E     connect_typ    /* --> Type de connectivit de dpart       */
)
{

  ecs_int_t     ient ;

  ecs_connect_t * connect_elt_sselt ;

  ecs_entmail_t * entmail_sselt_dec ;
  ecs_entmail_t * liste_entmail_connect[ECS_ENTMAIL_FIN] ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_connect != NULL) ;


  liste_entmail_connect[0] = this_connect->entmail ;
  for (ient = 1 ; ient < ECS_ENTMAIL_FIN ; ient++) {
    liste_entmail_connect[ient] = NULL ;
  }


  /* Dcomposition des lments en leurs sous-lments */
  /*---------------------------------------------------*/

  assert(   connect_typ == ECS_CONNECT_CEL_SOM
         || connect_typ == ECS_CONNECT_FAC_SOM) ;

  if (connect_typ == ECS_CONNECT_CEL_SOM)
    entmail_sselt_dec = ecs_entmail_pcp__decompose_cel(this_connect->entmail) ;

  else if (connect_typ == ECS_CONNECT_FAC_SOM)
    entmail_sselt_dec = ecs_entmail_pcp__decompose_fac(this_connect->entmail) ;

  else
    entmail_sselt_dec = NULL ;


  /* Concatnation, dans la mme entit de maillage, des sous-lments cres, */
  /*  avec les sous-lments de mme entit dj existants (s'ils existent !) */
  /*--------------------------------------------------------------------------*/

  if (*sous_connect != NULL) {

    ecs_entmail_pcp__concatene(*ent_sous_elt,
                               entmail_sselt_dec,
                               liste_entmail_connect) ;

    entmail_sselt_dec = ecs_entmail__detruit(entmail_sselt_dec) ;

  }
  else {

    *ent_sous_elt = entmail_sselt_dec ;

    *sous_connect = ecs_connect__cree(*ent_sous_elt,
                                      this_connect->entmail_inf,
                                      ECS_CONNECT_NAT_INDEX) ;

  }


  (*sous_connect)->entmail = *ent_sous_elt ;


  /* Fusion des sous-lments ayant la mme dfinition topologique */
  /*---------------------------------------------------------------*/

  ecs_entmail_pcp__fusionne((*sous_connect)->entmail,
                            liste_entmail_connect) ;


  /* Cration de la nouvelle connectivit associant */
  /*  l'entit de maillage des lments avec        */
  /*  l'entit de maillage des sous-lments        */
  /*------------------------------------------------*/

  connect_elt_sselt = ecs_connect__cree(this_connect->entmail   ,
                                        (*sous_connect)->entmail,
                                        ECS_CONNECT_NAT_INDEX        ) ;


  /* Renvoi de la nouvelle connectivit */
  /*------------------------------------*/

  return connect_elt_sselt ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui cre la liste des connectivits
 *   dans une reprsentation du maillage en connectivit nodale
 *    partir du vecteur des entits de maillage
 *         et de la nature de la connectivit (par tiquette ou par indice)
 *----------------------------------------------------------------------------*/

ecs_connect_t * * ecs_connect__cree_nodal
(
 ecs_entmail_t     * *const vect_entmail  ,
 ECS_CONNECT_NAT_E          connect_nat_e
)
{

  ecs_int_t        iconnect       ; /* Indice de boucle sur les connectivits */

  ecs_connect_t  * * vect_connect ; /* Tableau des connectivits              */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  BFT_MALLOC(vect_connect, ECS_CONNECT_FIN, ecs_connect_t *) ;


  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++)
    vect_connect[iconnect] = NULL ;

  if (vect_entmail[ECS_ENTMAIL_CEL] != NULL) {

    vect_connect[ECS_CONNECT_CEL_SOM]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_CEL],
                          vect_entmail[ECS_ENTMAIL_SOM],
                          connect_nat_e) ;
  }

  if (vect_entmail[ECS_ENTMAIL_FAC] != NULL) {

    vect_connect[ECS_CONNECT_FAC_SOM]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_FAC],
                          vect_entmail[ECS_ENTMAIL_SOM],
                          connect_nat_e) ;

  }

  if (vect_entmail[ECS_ENTMAIL_ARE] != NULL) {

    vect_connect[ECS_CONNECT_ARE_SOM]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_ARE],
                          vect_entmail[ECS_ENTMAIL_SOM],
                          connect_nat_e) ;
  }


  return vect_connect ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui cre la liste des connectivits
 *   dans une reprsentation du maillage en connectivit descendante
 *    partir du vecteur des entits de maillage
 *----------------------------------------------------------------------------*/

ecs_connect_t * * ecs_connect__cree_descendant
(
 ecs_entmail_t * *const vect_entmail
)
{

  ecs_int_t        iconnect       ; /* Indice de boucle sur les connectivits */

  ecs_connect_t  * * vect_connect ; /* Tableau des connectivits              */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  BFT_MALLOC(vect_connect, ECS_CONNECT_FIN, ecs_connect_t *) ;


  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++)
    vect_connect[iconnect] = NULL ;

  if (vect_entmail[ECS_ENTMAIL_CEL] != NULL) {

    vect_connect[ECS_CONNECT_CEL_FAC]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_CEL],
                          vect_entmail[ECS_ENTMAIL_FAC],
                          ECS_CONNECT_NAT_INDEX) ;
  }

  if (vect_entmail[ECS_ENTMAIL_FAC] != NULL) {

    vect_connect[ECS_CONNECT_FAC_ARE]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_FAC],
                          vect_entmail[ECS_ENTMAIL_ARE],
                          ECS_CONNECT_NAT_INDEX) ;

  }

  if (vect_entmail[ECS_ENTMAIL_ARE] != NULL) {

    vect_connect[ECS_CONNECT_ARE_SOM]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_ARE],
                          vect_entmail[ECS_ENTMAIL_SOM],
                          ECS_CONNECT_NAT_INDEX) ;
  }


  return vect_connect ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le type de la connectivit :
 *  - connectivit nodale      : ECS_CONNECT_TYP_NODALE
 *  - connectivit descendante : ECS_CONNECT_TYP_DESCENDANTE
 *   associ au tableau des connectivits
 *
 *  Si le maillage ne comporte que des artes,
 *   la fonction ne peut pas dterminer le type de la connectivit
 *   et retourne ECS_CONNECT_TYP_NUL
 *----------------------------------------------------------------------------*/

ECS_CONNECT_TYP_E ecs_connect__ret_connect_typ
(
 ecs_connect_t *const liste_connect[] ,
 ecs_entmail_t *const liste_entmail[]
)
{

  ECS_CONNECT_TYP_E    connect_typ_e ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(liste_entmail[ECS_ENTMAIL_SOM] != NULL) ;

  if (liste_entmail[ECS_ENTMAIL_ARE] != NULL)
    assert(liste_connect[ECS_CONNECT_ARE_SOM] != NULL) ;


  if (liste_entmail[ECS_ENTMAIL_FAC] != NULL) {

    if      (liste_connect[ECS_CONNECT_FAC_ARE] != NULL &&
             liste_connect[ECS_CONNECT_FAC_SOM] == NULL   )
      connect_typ_e = ECS_CONNECT_TYP_DESCENDANTE ;
    else if (liste_connect[ECS_CONNECT_FAC_SOM] != NULL &&
             liste_connect[ECS_CONNECT_FAC_ARE] == NULL   )
      connect_typ_e = ECS_CONNECT_TYP_NODALE ;
    else
      assert((liste_connect[ECS_CONNECT_FAC_ARE] != NULL &&
              liste_connect[ECS_CONNECT_FAC_SOM] == NULL   ) ||
             (liste_connect[ECS_CONNECT_FAC_SOM] != NULL &&
              liste_connect[ECS_CONNECT_FAC_ARE] == NULL   )   ) ;

  }
  else {

    connect_typ_e = ECS_CONNECT_TYP_NUL ;

  }


  if (liste_entmail[ECS_ENTMAIL_CEL] != NULL) {

    if (connect_typ_e == ECS_CONNECT_TYP_NUL) {

      if      (liste_connect[ECS_CONNECT_CEL_FAC] != NULL &&
               liste_connect[ECS_CONNECT_CEL_SOM] == NULL   )
        connect_typ_e = ECS_CONNECT_TYP_DESCENDANTE ;
      else if (liste_connect[ECS_CONNECT_CEL_SOM] != NULL &&
               liste_connect[ECS_CONNECT_CEL_FAC] == NULL   )
        connect_typ_e = ECS_CONNECT_TYP_NODALE ;
      else
        assert((liste_connect[ECS_CONNECT_CEL_FAC] != NULL &&
                liste_connect[ECS_CONNECT_CEL_SOM] == NULL   ) ||
               (liste_connect[ECS_CONNECT_CEL_SOM] != NULL &&
                liste_connect[ECS_CONNECT_CEL_FAC] == NULL   )   ) ;


    }
    else {

      if      (connect_typ_e == ECS_CONNECT_TYP_DESCENDANTE) {

        assert(liste_connect[ECS_CONNECT_CEL_FAC] != NULL &&
               liste_connect[ECS_CONNECT_CEL_SOM] == NULL   ) ;

      }
      else if (connect_typ_e == ECS_CONNECT_TYP_NODALE) {

        assert(liste_connect[ECS_CONNECT_CEL_SOM] != NULL &&
               liste_connect[ECS_CONNECT_CEL_FAC] == NULL   ) ;

      }

    }

  }


  return connect_typ_e ;


}

