/*============================================================================
 *  Dfinitions des fonctions de base
 *   associes  la structure `ecs_descr_t' dcrivant un descripteur de champ
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <string.h>  /* memcmp() */


/*----------------------------------------------------------------------------
 *  Fichiers `include' systme ou BFT
 *----------------------------------------------------------------------------*/

#include <bft_file.h>
#include <bft_mem.h>
#include <bft_printf.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_fic.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associ au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_descr.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privs   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_descr_priv.h"


/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *    Fonction de cration d'une structure de descripteur de champ
 *----------------------------------------------------------------------------*/

ecs_descr_t *ecs_descr__cree
(
       ECS_DESCR_TYP_E         typ_e      , /* --> Type (groupe, couleur, ...)*/
 const ecs_int_t               ide        , /* --> Identificateur descripteur */
 const char             *const nom          /* --> Nom du descripteur         */
)
{
  /* Variables locales */

  ecs_descr_t * descr_loc ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Allocation et initialisation de la structure */
  /*==============================================*/

  /* Allocation de la structure globale du descripteur de champ */
  /*------------------------------------------------------------*/

  BFT_MALLOC(descr_loc, 1, ecs_descr_t) ;

  /* Initialisation du numro du descripteur */
  /*-----------------------------------------*/

  descr_loc->num = 1 ;


  /* Affectation du type du descripteur */
  /*------------------------------------*/

  descr_loc->typ_e = typ_e ;


  /* Affectation de l'identificateur du descripteur */
  /*------------------------------------------------*/

  descr_loc->ide = ide   ;


  /* Affectation du nom du descripteur */
  /*-----------------------------------*/

  if (nom != NULL) {
    BFT_MALLOC(descr_loc->nom, strlen(nom) + 1, char) ;
    strcpy(descr_loc->nom, nom) ;
  }
  else {
    descr_loc->nom = NULL ;
  }


  /* Initialisation par dfaut du lien sur le descripteur suivant */
  /*--------------------------------------------------------------*/

  descr_loc->l_descr_sui = NULL ;


  /* On renvoie un pointeur sur la structure */
  /*=========================================*/

  return descr_loc ;


}


/*----------------------------------------------------------------------------
 *  Fonction librant la structure `ecs_descr_t' donne en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_descr_t *ecs_descr__detruit
(
 ecs_descr_t * this_descr
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_descr != NULL) ;


  /* Libration du contenu de la structure */
  /*=======================================*/

  /* Libration du nom du descripteur */
  /*----------------------------------*/

  if (this_descr->nom != NULL)
    BFT_FREE(this_descr->nom) ;


  /* Libration de la structure */
  /*============================*/

  BFT_FREE(this_descr) ;


  return this_descr ;


}


/*----------------------------------------------------------------------------
 *  Fonction imprimant le contenu d'une structure `ecs_descr_t' donne
 *   sur le flux dcrit par la structure `bft_file_t'
 *----------------------------------------------------------------------------*/

void ecs_descr__imprime
(
 const ecs_descr_t *const this_descr ,          /* --> Structure a imprimer   */
       ecs_int_t          imp_col    ,
       bft_file_t  *const fic_imp               /* --> Descripteur du fichier */
)
{

#define ECS_FCT_IMP_DESCR_NUM         "numero"
#define ECS_FCT_IMP_DESCR_TYP         "type"
#define ECS_FCT_IMP_DESCR_IDE         "ide"
#define ECS_FCT_IMP_DESCR_NOM         "nom"
#define ECS_FCT_IMP_DESCR_L_DESCR     "l_descr_sui"


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_descr != NULL) ;


  imp_col++ ;


  /* criture du contenu de la structure `ecs_descr_t' */
  /*===============================================*/

  /* Numro du descripteur */
  /*-----------------------*/

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_DESCR_NUM,
                       ECS_TYPE_ecs_int_t, &this_descr->num) ;

  /* Type du descripteur */
  /*---------------------*/

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_DESCR_TYP,
                       ECS_TYPE_ecs_int_t, &this_descr->typ_e) ;


  /* Identificateur du descripteur */
  /*-------------------------------*/

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_DESCR_IDE,
                       ECS_TYPE_ecs_int_t, &this_descr->ide) ;


  /* Nom du descripteur */
  /*--------------------*/

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_DESCR_NOM,
                       ECS_TYPE_char, this_descr->nom) ;


  /* Lien sur un ventuel descripteur */
  /*----------------------------------*/

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_DESCR_L_DESCR,
                       ECS_TYPE_void, this_descr->l_descr_sui) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets d'une structure `ecs_descr_t'
 *----------------------------------------------------------------------------*/

float ecs_descr__ret_taille
(
  const ecs_descr_t *const this_descr
)
{

  float   taille ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_descr != NULL) ;


  taille = (float)sizeof(*this_descr) ;

  taille += (float)sizeof(*(this_descr->nom)) ;


  return taille ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui alloue une structure `ecs_descr_t' et qui remplit
 *   son contenu en copiant le contenu de la structure donne en argument
 *   sauf le numro de descripteur qui n'est pas recopie
 *----------------------------------------------------------------------------*/

ecs_descr_t * ecs_descr__copie
(                                 /* <-- Renvoie la structure copie          */
 ecs_descr_t * this_descr         /* --> Structure a copier                   */
)
{

  ecs_descr_t   * descr_loc ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_descr != NULL) ;


  descr_loc = ecs_descr__cree(this_descr->typ_e,
                              this_descr->ide,
                              this_descr->nom) ;

  return descr_loc ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui compare 2 descripteurs
 *
 *  La fonction renvoie :
 *  - `ECS_TRUE'  si les deux descripteurs ont les mmes identificateur et nom
 *  - `ECS_FALSE' sinon
 *----------------------------------------------------------------------------*/

ecs_bool_t ecs_descr__compare
(
 const ecs_descr_t *const descr_1 ,        /* --> 1er  descripteur a comparer */
 const ecs_descr_t *const descr_2          /* --> 2eme descripteur a comparer */
)
{

  ecs_bool_t bool_retour ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(descr_1 != NULL && descr_2 != NULL) ;


  if (descr_1->typ_e == descr_2->typ_e && descr_1->ide == descr_2->ide)
    bool_retour = ECS_TRUE ;
  else
    bool_retour = ECS_FALSE ;


  if (bool_retour == ECS_TRUE) {

    if (descr_1->nom != NULL || descr_2->nom != NULL) {

      if((descr_1->nom == NULL && descr_2->nom != NULL) ||
         (descr_1->nom != NULL && descr_2->nom == NULL) ||
         strcmp(descr_1->nom,
                descr_2->nom ) != 0 ) {

        bool_retour = ECS_FALSE ;

      }

    }

  }


  return bool_retour ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui compare 2 descripteurs
 *
 *  La fonction renvoie :
 *  - `ECS_TRUE' si les types des deux identificateurs sont identiques et
 *           -       si les noms            des 2 descripteurs sont a `NULL'
 *                et si les identificateurs des 2 descripteurs sont identiques
 *           - ou    si les noms            des 2 descripteurs sont identiques
 *                   (et tous deux differents de `NULL')
 *  - `ECS_FALSE' sinon
 *----------------------------------------------------------------------------*/

ecs_bool_t ecs_descr__compare_selection
(
 const ecs_descr_t *const descr_1 ,        /* --> 1er  descripteur a comparer */
 const ecs_descr_t *const descr_2          /* --> 2eme descripteur a comparer */
)
{

  assert(descr_1 != NULL && descr_2 != NULL) ;

  if (descr_1->typ_e != descr_2->typ_e)
    return ECS_FALSE ;


  if (descr_1->nom == NULL && descr_2->nom == NULL) {


    /* Les noms des 2 descripteurs sont a `NULL' : */
    /* on compare leurs identificateurs            */

    if (descr_1->ide != descr_2->ide)

      return ECS_FALSE ;

  }
  else {


    /* Au moins un nom des deux descripteurs n'est pas a `NULL' : */
    /*    on compare les deux noms                                */


    if((descr_1->nom == NULL && descr_2->nom != NULL) ||
       (descr_1->nom != NULL && descr_2->nom == NULL) ||
       strcmp(descr_1->nom,
              descr_2->nom ) != 0 )

      return ECS_FALSE ;

  }


  return ECS_TRUE ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche le contenu d'un descripteur
 *
 *  Sont affichs :
 *   - le type du descripteur ("Couleur" ou "Groupe")
 *   - l'identificateur dans le cas d'une couleur
 *   - la chane de caractres dans le cas d'un groupe
 *----------------------------------------------------------------------------*/

void ecs_descr__affiche
(
 const ecs_descr_t  *const descr ,
 const int                 decal
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  switch (descr->typ_e) {

  case ECS_DESCR_TYP_COULEUR:

    if (descr->nom == NULL)
      bft_printf("  %*s%s %" ECS_FORMAT_ecs_int_t "\n",
                 decal, "", _("Color"),
                 descr->ide) ;
    else
      bft_printf("  %*s%s %" ECS_FORMAT_ecs_int_t " (%s)\n",
                 decal, "", _("Color"), descr->ide, descr->nom) ;

    break ;

  case ECS_DESCR_TYP_GROUPE:

    assert(descr->nom != NULL) ;

    bft_printf("  %*s%s \"%s\"\n",
               decal, "", _("Group"),
               descr->nom) ;

    break ;

  default:

    assert(descr->typ_e == ECS_DESCR_TYP_COULEUR ||
           descr->typ_e == ECS_DESCR_TYP_GROUPE     ) ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie le type du descripteur :
 *  - soit `ECS_DESCR_TYP_COULEUR' pour un descripteur de type "couleur"
 *  - soit `ECS_DESCR_TYP_GROUPE'  pour un descripteur de type "groupe"
 *----------------------------------------------------------------------------*/

ECS_DESCR_TYP_E ecs_descr__ret_typ
(
 const ecs_descr_t *const descr
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(descr != NULL) ;


  return descr->typ_e ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie l'identificateur du descripteur donn en argument
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_descr__ret_ide
(
 const ecs_descr_t *const descr
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(descr != NULL) ;


  return descr->ide ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie le nom du descripteur donn en argument
 *----------------------------------------------------------------------------*/

const char * ecs_descr__ret_nom
(
 const ecs_descr_t *const descr
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(descr != NULL) ;


  return descr->nom ;


}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/
