/*============================================================================
 *  Dfinitions des fonctions de base
 *   associes  la structure `ecs_entmail_t' dcrivant une entit de maillage
 *   quelconque (une entit principale ou une sous-entit)
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <string.h> /* strlen() */

#include <bft_file.h>
#include <bft_mem.h>
#include <bft_printf.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_chaine_glob.h"
#include "ecs_def.h"
#include "ecs_fic.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_champ.h"
#include "ecs_champ_chaine.h"
#include "ecs_champ_att.h"
#include "ecs_champ_def.h"
#include "ecs_champ_vec.h"
#include "ecs_champ_vec_int.h"
#include "ecs_champ_vec_real.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associ au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_entmail.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privs   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_entmail_priv.h"



/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/



/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Fonction allouant et initialisant une structure `ecs_entmail_t'
 *----------------------------------------------------------------------------*/

ecs_entmail_t * ecs_entmail__cree
(
 void
)
{

  ecs_entmail_t  * this_entmail  ;

  ecs_int_t      ichamp        ;        /* Indice de boucle sur les champs    */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  BFT_MALLOC(this_entmail, 1, ecs_entmail_t);


  /* Initialisation des pointeurs d'acces aux champs */


  for (ichamp = ECS_CHAMP_DEB ; ichamp < ECS_CHAMP_FIN ; ichamp++) {

    this_entmail->champ[ichamp] = NULL ;

  }


  return this_entmail ;


}


/*----------------------------------------------------------------------------
 *  Fonction librant une structure `ecs_entmail_t'donne en argument
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_entmail_t * ecs_entmail__detruit
(
 ecs_entmail_t * this_entmail
)
{


  ecs_int_t       ichamp                 ; /* Indice de boucle sur les champs */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_entmail != NULL) ;



  /* Libration du contenu de la structure `ecs_entmail_t' */
  /*===================================================*/


  /* Libration des listes de champs */
  /*---------------------------------*/

  for (ichamp = ECS_CHAMP_DEB ; ichamp < ECS_CHAMP_FIN ; ichamp++) {

    if (this_entmail->champ[ichamp] != NULL)
      ecs_champ_chaine__detruit(&this_entmail->champ[ichamp]) ;

  }


  /* Libration de la structure de type `ecs_entmail_t' */
  /*================================================*/


  BFT_FREE(this_entmail);



  /* Retoune `NULL' */

  return this_entmail ;


}


/*----------------------------------------------------------------------------
 *  Fonction imprimant le contenu d'une structure `ecs_entmail_t' donne
 *   sur le flux dcrit par la structure `bft_file_t'
 *----------------------------------------------------------------------------*/

void ecs_entmail__imprime
(
 const ecs_entmail_t  *const this_entmail ,     /* --> Structure a imprimer   */
       ecs_int_t             imp_col      ,
 const ecs_int_t             nbr_imp      ,
       bft_file_t     *const fic_imp            /* --> Descripteur du fichier */
)
{

  const char *nom_champ_c[ECS_CHAMP_FIN] = {
    "ECS_CHAMP_DEF" ,
    "ECS_CHAMP_ATT"
  } ;


  ecs_int_t       ichamp                 ; /* Indice de boucle sur les champs */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_entmail != NULL) ;


  imp_col++ ;


  /* Impression du nom de l'entite */
  /*-------------------------------*/

  for (ichamp = ECS_CHAMP_DEB ; ichamp < ECS_CHAMP_FIN ; ichamp++) {

    ecs_fic__imprime_ptr(fic_imp, imp_col, nom_champ_c[ichamp],
                         (void *)this_entmail->champ[ichamp]) ;

    if (this_entmail->champ[ichamp] != NULL) {

      /* Impression du contenu d'une chaine de champs */

      ecs_champ_chaine__imprime(this_entmail->champ[ichamp],
                                imp_col + 1,
                                nbr_imp,
                                fic_imp);

    }

  }

}


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le nombre d'lments d'une structure `ecs_entmail_t'
 *----------------------------------------------------------------------------*/


size_t ecs_entmail__ret_nbr_ele
(
 const ecs_entmail_t * const this_entmail
)
{

  assert(this_entmail != NULL) ;

  if (this_entmail->champ == NULL)
    return 0 ;

  else if (this_entmail->champ[ECS_CHAMP_DEF] == NULL)
    return 0 ;

  else
    return (ecs_champ__ret_pos_nbr(this_entmail->champ[ECS_CHAMP_DEF]) - 1) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets d'une structure `ecs_entmail_t'
 *----------------------------------------------------------------------------*/

float ecs_entmail__ret_taille
(
 const ecs_entmail_t *const this_entmail
)
{

  float         taille ;
  float         taille_def ;


  ecs_int_t       ichamp ;               /* Indice de boucle sur les champs   */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_entmail != NULL) ;


  taille = sizeof(*this_entmail) ;


  for (ichamp = ECS_CHAMP_DEB ; ichamp < ECS_CHAMP_FIN ; ichamp++) {

    if (this_entmail->champ[ichamp] != NULL) {

      taille_def
        = ecs_champ_chaine__ret_taille(this_entmail->champ[ichamp]);

      taille += taille_def ;

    }

  }


  return taille ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui ajoute un champ donn dans une entit de maillage donne
 *----------------------------------------------------------------------------*/

void ecs_entmail__ajoute_champ
(
 ecs_entmail_t * this_entmail ,     /* <-> Structure  remplir                */
 ecs_champ_t   * champ        ,     /* --> Rfrence du champ  ajouter       */
 ECS_CHAMP_E     champ_e            /* --> Identificateur du champ            */
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_entmail != NULL && champ != NULL) ;

  assert(champ_e == ECS_CHAMP_DEF || champ_e == ECS_CHAMP_ATT) ;


  /* Placement du nouveau champ dans le vecteur des champs */
  /*-------------------------------------------------------*/

  ecs_champ_chaine__ajoute(&this_entmail->champ[champ_e],
                           champ) ;

}


/*----------------------------------------------------------------------------
 *  Fonction supprimant un champ d'une entit de maillage.
 *----------------------------------------------------------------------------*/

void ecs_entmail__supprime_champ
(
       ecs_entmail_t   *const  this_entmail , /* <-> Entit de maillage       */
 const char            *const  nom_champ    , /* --> Nom du champ             */
       ECS_CHAMP_E             champ_e        /* --> Identificateur du champ  */
)
{

  ecs_champ_t         * champ_suppr ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (this_entmail == NULL)
    return ;


  assert(nom_champ != NULL) ;

  assert(champ_e == ECS_CHAMP_DEF || champ_e == ECS_CHAMP_ATT) ;


  /* On regarde si un champ du type indiqu            */
  /*  a le meme descripteur de champ que `champ_suppr' */

  if (this_entmail->champ[(ecs_int_t)champ_e] != NULL)

    champ_suppr
      = ecs_champ_chaine__trouve_nom(this_entmail->champ[(ecs_int_t)champ_e],
                                     nom_champ) ;
  else

    champ_suppr = NULL ;


  if (champ_suppr != NULL) {

    /* Un champ de meme descripteur existe bien dans l'entite de maillage */

    /* On le detruit et on supprime sa reference  */
    /* dans la chaine des champs de type grandeur */

    ecs_champ_chaine__supprime(&champ_suppr,
                               &this_entmail->champ[(ecs_int_t)champ_e]);

    champ_suppr = ecs_champ__detruit(champ_suppr) ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui retourne la rfrence du champ correspondant
 *    l'identificateur de champ donn d'une entit de maillage donne
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_entmail__ret_champ
(
 ecs_entmail_t * this_entmail ,    /* --> Entite de maillage du champ         */
 ECS_CHAMP_E     champ_e           /* --> Identificateur du champ a retourner */
)
{

  assert(this_entmail != NULL) ;


  return this_entmail->champ[champ_e] ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche le nombre d'lments par entit
 *----------------------------------------------------------------------------*/

void ecs_entmail__aff_nbr_par_ent
(
 const size_t  nbr_elt_ent[] ,
 int           lng_imp
)
{

  int max_lng_var_nbr ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  max_lng_var_nbr = ECS_MAX((int)strlen(_("Number of vertices")),
                            (int)strlen(_("Number of edges")) ) ;
  max_lng_var_nbr = ECS_MAX(max_lng_var_nbr,
                            (int)strlen(_("Number of faces")) ) ;
  max_lng_var_nbr = ECS_MAX(max_lng_var_nbr,
                            (int)strlen(_("Number of cells")) ) ;

  max_lng_var_nbr = ECS_MAX(max_lng_var_nbr,
                            lng_imp) ;


  if (nbr_elt_ent[ECS_ENTMAIL_CEL] != 0) {

    bft_printf("    ") ;
    ecs_print_padded_str(_("Number of cells"), max_lng_var_nbr) ;
    bft_printf(" : %*lu\n", ECS_LNG_AFF_ENT,
               (unsigned long)(nbr_elt_ent[ECS_ENTMAIL_CEL])) ;

  }

  if (nbr_elt_ent[ECS_ENTMAIL_FAC] != 0) {

    bft_printf("    ") ;
    ecs_print_padded_str(_("Number of faces"), max_lng_var_nbr) ;
    bft_printf(" : %*lu\n", ECS_LNG_AFF_ENT,
               (unsigned long)(nbr_elt_ent[ECS_ENTMAIL_FAC])) ;

  }

  if (nbr_elt_ent[ECS_ENTMAIL_ARE] != 0) {

    bft_printf("    ") ;
    ecs_print_padded_str(_("Number of edges"), max_lng_var_nbr) ;
    bft_printf(" : %*lu\n", ECS_LNG_AFF_ENT,
               (unsigned long)(nbr_elt_ent[ECS_ENTMAIL_ARE])) ;

  }

  if (nbr_elt_ent[ECS_ENTMAIL_SOM] != 0) {

    bft_printf("    ") ;
    ecs_print_padded_str(_("Number of vertices"), max_lng_var_nbr) ;
    bft_printf(" : %*lu\n", ECS_LNG_AFF_ENT,
               (unsigned long)(nbr_elt_ent[ECS_ENTMAIL_SOM])) ;

  }

}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/
