/*============================================================================
 *  Dfinitions des fonctions
 *   associes  la structure `ecs_vec_int_t' dcrivant un vecteur index
 *   entier et propres aux vecteurs indexs
 *   lis aux champs auxiliaires de type "attribut"
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>

#include <bft_mem.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_chrono.h"
#include "ecs_def.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_vec_int.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associe au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_vec_int_att.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' prives   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_vec_int_priv.h"


/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui assemble :
 *  - une table de positions donne dans une table de positions rceptrice
 *  - la table des valeurs correspondante dans la table des valeurs rceptrice
 *
 *  Les 2 tables de positions doivent avoir le mme nombre de positions
 *  (et les tables de valeurs doivent tre de mme type !)
 *
 *  Seule la table rceptrice est modifie,
 *   la table  assembler reste inchange
 * ----------------------------------------------------------------------------
 *
 *  Exemple : (pour des tables de valeurs entires)
 *  =======
 *
 *  Soit la table  assembler :
 *
 *                         .---.---..---..---.---.---.
 *     assemb->val_tab     | 5 | 3 || 4 || 5 | 2 | 6 |
 *                         `---'---'`---'`---'---'---'
 *                           0   1    2    3   4   5
 *
 *
 *                         .---.---.---.---.---.
 *     assemb->pos_tab     | 1 | 3 | 4 | 4 | 7 |
 *                         `---'---'---'---'---'
 *                           0   1   2   3   4
 *
 *
 *  dans la table rceptrice :
 *
 *                         .---..---..---.---..---.
 *     recept->val_tab     | 4 || 5 || 6 | 6 || 1 |
 *                         `---'`---'`---'---'`---'
 *                           0    1    2   3    4
 *
 *
 *                         .---.---.---.---.---.
 *     recept->pos_tab     | 1 | 2 | 3 | 5 | 6 |
 *                         `---'---'---'---'---'
 *                           0   1   2   3   4
 *
 *
 *  La table rceptrice devient :
 *
 *                         .---.---.---..---.---..---.---..---.---.---.---.
 *     recept->val_tab     | 4 | 5 | 3 || 5 | 4 || 6 | 6 || 1 | 5 | 2 | 6 |
 *                         `---'---'---'`---'---'`---'---'`---'---'---'---'
 *                           0   1   2    3   4    5   6    7   8   9   10
 *
 *
 *                         .---.---.---.---.---.
 *     recept->pos_tab     | 1 | 4 | 6 | 8 | 12|
 *                         `---'---'---'---'---'
 *                           0   1   2   3   4
 *
 *----------------------------------------------------------------------------*/

void ecs_vec_int_att__assemble
(
 ecs_vec_int_t *const vec_int_recept,
 ecs_vec_int_t *const vec_int_assemb
)
{

  size_t    ipos ;
  size_t    ival ;
  size_t    cpt_val ;

  ecs_size_t * recept_pos_tab_new ;
  ecs_int_t  * recept_val_tab_new ;

  size_t      recept_val_nbr ;
  size_t      assemb_val_nbr ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(   vec_int_recept != NULL
         && vec_int_assemb != NULL) ;

  assert(vec_int_recept->pos_nbr == vec_int_assemb->pos_nbr) ;


  /* Allocation des nouveaux tableaux de la structure receptrice */
  /*-------------------------------------------------------------*/

  recept_val_nbr
    = vec_int_recept->pos_tab[vec_int_recept->pos_nbr - 1] - 1 ;
  assemb_val_nbr
    = vec_int_assemb->pos_tab[vec_int_assemb->pos_nbr - 1] - 1 ;


  BFT_MALLOC(recept_pos_tab_new,
             vec_int_recept->pos_nbr,
             ecs_size_t);

  BFT_MALLOC(recept_val_tab_new,
             recept_val_nbr + assemb_val_nbr,
             ecs_int_t);


  /* Assemblage des tables de positions et de valeurs */
  /*--------------------------------------------------*/

  cpt_val = 0 ;
  recept_pos_tab_new[0] = 1 ;

  for (ipos = 0 ; ipos < vec_int_recept->pos_nbr - 1 ; ipos++) {

    recept_pos_tab_new[ipos + 1]
      =   recept_pos_tab_new[ipos]
        + vec_int_recept->pos_tab[ipos + 1] - vec_int_recept->pos_tab[ipos]
        + vec_int_assemb->pos_tab[ipos + 1] - vec_int_assemb->pos_tab[ipos] ;


    for (ival = (vec_int_recept->pos_tab[ipos    ] - 1) ;
         ival < (vec_int_recept->pos_tab[ipos + 1] - 1) ;
         ival++) {

      recept_val_tab_new[cpt_val++] = vec_int_recept->val_tab[ival] ;

    }


    for (ival = (vec_int_assemb->pos_tab[ipos    ] - 1) ;
         ival < (vec_int_assemb->pos_tab[ipos + 1] - 1) ;
         ival++) {

      recept_val_tab_new[cpt_val++] = vec_int_assemb->val_tab[ival] ;

    }

  }


  BFT_FREE(vec_int_recept->pos_tab) ;
  BFT_FREE(vec_int_recept->val_tab) ;

  vec_int_recept->pos_tab = recept_pos_tab_new ;
  vec_int_recept->val_tab = recept_val_tab_new ;

}


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'un vecteur indexe
 *   en fusionnant les proprits de ses lments
 *   qui sont identiquement transformes par le vecteur de transformation donn
 *----------------------------------------------------------------------------*/

void ecs_vec_int_att__fusionne
(
       ecs_vec_int_t *const this_vec_int,
       size_t               nbr_elt_new,
 const ecs_tab_int_t        vect_transf
)
{

  ecs_size_t  * pos_tab_transf ;
  ecs_int_t  ** val_tab_transf ;
  ecs_int_t   * val_tab_unidim ;

  size_t       cpt_val_transf ;
  size_t       nbr_elt_ref ;
  size_t       nbr_val_transf ;
  ecs_int_t    num_elt_transf ;
  size_t       pos_inf ;
  size_t       pos_sup ;
  ecs_int_t    val_ref ;

  size_t       ielt_ref ;
  size_t       ielt_transf ;
  size_t       ipos_ref ;
  size_t       ipos_transf ;

  size_t       elt_nbr_val_max ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  nbr_elt_ref = this_vec_int->pos_nbr - 1 ;

  elt_nbr_val_max = 0 ;

  /* Dtermination du nombre maximal de proprits pour un lment */

  for (ielt_ref = 0 ; ielt_ref < nbr_elt_ref ; ielt_ref++) {

    elt_nbr_val_max = ECS_MAX(elt_nbr_val_max,
                              this_vec_int->pos_tab[ielt_ref + 1] -
                              this_vec_int->pos_tab[ielt_ref]) ;

  }

  /* On suppose qu'une fusion ne concerne qu'au maximum 2 lments       */
  /* Le nombre maximal de proprits pour un lment fusionne est donc : */

  elt_nbr_val_max *= 2 ;


  BFT_MALLOC(pos_tab_transf, nbr_elt_new, ecs_size_t) ;
  BFT_MALLOC(val_tab_transf, nbr_elt_new, ecs_int_t *) ;
  BFT_MALLOC(val_tab_unidim, nbr_elt_new * elt_nbr_val_max, ecs_int_t) ;

  for (ielt_transf = 0 ; ielt_transf < nbr_elt_new ; ielt_transf++) {
    pos_tab_transf[ielt_transf]= 0 ;
    val_tab_transf[ielt_transf]= &val_tab_unidim[ielt_transf * elt_nbr_val_max];
  }


  nbr_val_transf = 0 ;


  for (ielt_ref = 0 ; ielt_ref < nbr_elt_ref ; ielt_ref++) {

    num_elt_transf  = vect_transf.val[ielt_ref] ;


    pos_inf = this_vec_int->pos_tab[ielt_ref]     - 1 ;
    pos_sup = this_vec_int->pos_tab[ielt_ref + 1] - 1 ;


    for (ipos_ref = pos_inf ; ipos_ref < pos_sup ; ipos_ref++) {

      val_ref = this_vec_int->val_tab[ipos_ref] ;


      ipos_transf = 0 ;
      while (ipos_transf < pos_tab_transf[num_elt_transf]           &&
             val_tab_transf[num_elt_transf][ipos_transf] != val_ref    )
        ipos_transf++ ;

      if (ipos_transf == pos_tab_transf[num_elt_transf]) {

        /* C'est une nouvelle valeur : on la stocke */

        val_tab_transf[num_elt_transf][ipos_transf] = val_ref ;
        pos_tab_transf[num_elt_transf]++ ;
        nbr_val_transf++ ;

      }
      /* else : rien  faire (la valeur a dj t stocke) */


    } /* Fin : boucle sur les valeurs de l'lment de rfrence */


  } /* Fin : boucle sur les lments de rfrence */


  this_vec_int->pos_nbr = nbr_elt_new + 1 ;

  BFT_REALLOC(this_vec_int->pos_tab, nbr_elt_new + 1, ecs_size_t) ;

  BFT_REALLOC(this_vec_int->val_tab, nbr_val_transf, ecs_int_t) ;

  this_vec_int->pos_tab[0] = 1 ;
  cpt_val_transf = 0 ;


  for (ielt_transf = 0 ; ielt_transf < nbr_elt_new ; ielt_transf++) {

    this_vec_int->pos_tab[ielt_transf + 1]
      = this_vec_int->pos_tab[ielt_transf] + pos_tab_transf[ielt_transf] ;

    for (ipos_transf = 0 ;
         ipos_transf < pos_tab_transf[ielt_transf] ;
         ipos_transf++) {

      this_vec_int->val_tab[cpt_val_transf++]
        = val_tab_transf[ielt_transf][ipos_transf] ;

    }

  }


  BFT_FREE(pos_tab_transf) ;
  BFT_FREE(val_tab_transf) ;
  BFT_FREE(val_tab_unidim) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la liste des numros de famille des lments
 *
 *  Pour les lments de famille 0 ou n'ayant pas de famille, on leur
 *   attribue le numro de famille par dfaut
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_vec_int_att__fam_elt
(
 ecs_vec_int_t  *const this_vec_int_fam,
 ecs_tab_int_t  *const tab_nbr_elt_fam
)
{

  size_t      nbr_elt ;
  size_t      ielt ;
  size_t      ifam ;
  int         num_fam ;
  size_t      pos_inf ;
  size_t      pos_sup ;

  ecs_tab_int_t tab_fam_elt ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_vec_int_fam != NULL) ;


  nbr_elt = this_vec_int_fam->pos_nbr - 1 ;

  BFT_MALLOC(tab_fam_elt.val, nbr_elt, ecs_int_t) ;
  tab_fam_elt.nbr = nbr_elt ;

  for (ifam = 0 ; ifam < tab_nbr_elt_fam->nbr ; ifam++)
    tab_nbr_elt_fam->val[ifam] = 0 ;

  for (ielt = 0 ; ielt < nbr_elt ; ielt++) {

    pos_inf = this_vec_int_fam->pos_tab[ielt    ] - 1 ;
    pos_sup = this_vec_int_fam->pos_tab[ielt + 1] - 1 ;

    if (pos_sup > pos_inf) {

      assert(pos_sup == pos_inf + 1) ;

      num_fam = this_vec_int_fam->val_tab[pos_inf] ;

      if (num_fam != 0)

        tab_nbr_elt_fam->val[num_fam - 1]++ ;

      else {

        num_fam = 0 ;

      }

      tab_fam_elt.val[ielt] = num_fam ;

    }
    else {

      /* pos_inf == pos_sup                    */
      /* L'lment n'a pas de valeur           */

      /* On lui attribue une valeur par dfaut */

      tab_fam_elt.val[ielt] = 0 ;

    }
    /* else : rien  faire */


  } /* Fin : boucle sur les lments */


  return tab_fam_elt ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la liste des numros de famille
 *   des lments filtrs par la liste optionnelle donne en argument
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_vec_int_att__liste_elt_fam
(
       ecs_vec_int_t   *const this_vec_int_fam,
 const ecs_tab_int_t   *const liste_filtre,
 const ecs_tab_bool_t  *const indic_famille
)
{
  size_t      ielt ;
  size_t      cpt_elt ;
  size_t      ind_elt ;
  size_t      nbr_elt ;
  size_t      ipos ;
  int         num_fam ;
  size_t      pos_inf ;
  size_t      pos_sup ;

  ecs_tab_int_t tab_liste_elt_fam ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_vec_int_fam != NULL) ;


  if (liste_filtre != NULL)
    nbr_elt = liste_filtre->nbr ;
  else
    nbr_elt = this_vec_int_fam->pos_nbr - 1 ;


  /* Comptage */

  cpt_elt = 0 ;

  for (ielt = 0 ; ielt < nbr_elt ; ielt++) {

    if (liste_filtre != NULL)
      ind_elt = liste_filtre->val[ielt] ;
    else
      ind_elt = ielt ;

    pos_inf = this_vec_int_fam->pos_tab[ind_elt    ] - 1 ;
    pos_sup = this_vec_int_fam->pos_tab[ind_elt + 1] - 1 ;

    if (pos_inf != pos_sup) {

      for (ipos = pos_inf ; ipos < pos_sup ; ipos++) {

        num_fam = this_vec_int_fam->val_tab[ipos] ;

        if (indic_famille->val[num_fam] == ECS_TRUE) {
          cpt_elt++ ;
          break ;
        }

      }

    }

  }


  /* Construction de la liste */

  tab_liste_elt_fam.nbr = cpt_elt ;
  BFT_MALLOC(tab_liste_elt_fam.val, tab_liste_elt_fam.nbr, ecs_int_t) ;


  cpt_elt = 0 ;

  for (ielt = 0 ; ielt < nbr_elt ; ielt++) {

    if (liste_filtre != NULL)
      ind_elt = liste_filtre->val[ielt] ;
    else
      ind_elt = ielt ;

    pos_inf = this_vec_int_fam->pos_tab[ind_elt    ] - 1 ;
    pos_sup = this_vec_int_fam->pos_tab[ind_elt + 1] - 1 ;

    if (pos_inf != pos_sup) {

      for (ipos = pos_inf ; ipos < pos_sup ; ipos++) {

        num_fam = this_vec_int_fam->val_tab[ipos] ;

        if (indic_famille->val[num_fam] == ECS_TRUE) {
          tab_liste_elt_fam.val[cpt_elt++] = ind_elt ;
          break ;
        }

      }

    }

  }

  assert(tab_liste_elt_fam.nbr == cpt_elt) ;


  /* On renvoie la liste */

  return tab_liste_elt_fam ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui comptabilise, pour chaque valeur d'attribut,
 *   les nombre d'lments ayant la mme valeur d'attribut
 *----------------------------------------------------------------------------*/

ecs_int_t * ecs_vec_int_att__cpt_elt_val
(
 const ecs_vec_int_t  *const vec_int_att,
       size_t                nbr_val_att
)
{

  size_t      ipos ;
  size_t      ival ;
  size_t      ival_att ;
  size_t      pos_val ;
  size_t      nbr_val ;

  ecs_int_t * nbr_elt_par_att ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  BFT_MALLOC(nbr_elt_par_att, nbr_val_att, ecs_int_t) ;

  for (ival_att = 0 ; ival_att < nbr_val_att ; ival_att++)
    nbr_elt_par_att[ival_att] = 0 ;


  if (vec_int_att != NULL) {

    for (ipos = 0 ; ipos < vec_int_att->pos_nbr - 1 ; ipos++) {

      pos_val = vec_int_att->pos_tab[ipos    ] - 1 ;
      nbr_val = vec_int_att->pos_tab[ipos + 1] - 1 - pos_val ;

      for (ival = 0 ; ival < nbr_val ; ival++)
        if (vec_int_att->val_tab[pos_val + ival] > 0)
          nbr_elt_par_att[vec_int_att->val_tab[pos_val + ival] - 1]++ ;

    }

  }


  return nbr_elt_par_att ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui cre une liste d'lments ayant pour valeur de l'attribut
 *   la valeur donne en argument
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_vec_int_att__cree_liste
(
 ecs_vec_int_t  *const vec_int_att,
 ecs_int_t             valeur_att
)
{

  size_t       ielt ;
  size_t       ival ;
  size_t       nbr_val ;

  ecs_tab_int_t  liste_elt = {0, NULL} ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(vec_int_att != NULL) ;


  nbr_val = ecs_vec_int__ret_val_nbr(vec_int_att) ;

  assert(nbr_val != 0) ;

  BFT_MALLOC(liste_elt.val, nbr_val, ecs_int_t) ;
  liste_elt.nbr = 0 ;


  for (ielt = 0 ; ielt < vec_int_att->pos_nbr - 1 ; ielt++) {

    for (ival = vec_int_att->pos_tab[ielt] ;
         ival < vec_int_att->pos_tab[ielt + 1] ;
         ival++) {

      if (vec_int_att->val_tab[ival - 1] == valeur_att)

        liste_elt.val[liste_elt.nbr++] = ielt + 1 ;

    }

  }


  if (liste_elt.nbr != 0)
    BFT_REALLOC(liste_elt.val, liste_elt.nbr, ecs_int_t) ;
  else
    BFT_FREE(liste_elt.val) ;


  return liste_elt ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui remplace les rfrences  des lments
 *  en des rfrences  d'autres lments lis aux premiers
 *  par un vecteur index contenant des valeurs positives mais pouvant
 *  contenir la valeur 0
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__ret_remplace_ref_att
(
       ecs_vec_int_t *const vec_rep,
 const ecs_vec_int_t *const vec_def
)
{

  ecs_vec_int_t * vec_rep_new ;

  size_t        cpt_val_rep ;
  size_t        ielt_rep ;
  size_t        ipos_def ;
  size_t        ipos_rep ;
  size_t        nbr_elt_rep ;
  size_t        nbr_pos_def ;
  size_t        nbr_val_rep_new ;
  ecs_int_t     num_val_def ;
  size_t        pos_deb_def ;
  size_t        pos_deb_rep ;
  size_t        pos_fin_def ;
  size_t        pos_fin_rep ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  nbr_elt_rep = vec_rep->pos_nbr - 1 ;


  /* Passage pour dimensionnement */


  cpt_val_rep = 0 ;
  nbr_val_rep_new = 0 ;


  for (ielt_rep = 0 ; ielt_rep < nbr_elt_rep ; ielt_rep++) {


    pos_deb_rep = vec_rep->pos_tab[ielt_rep]     - 1 ;
    pos_fin_rep = vec_rep->pos_tab[ielt_rep + 1] - 1 ;


    for (ipos_rep = pos_deb_rep ; ipos_rep < pos_fin_rep ; ipos_rep++) {

      num_val_def = vec_rep->val_tab[ipos_rep] ;

      if (num_val_def > 0) {

        pos_deb_def = vec_def->pos_tab[num_val_def - 1] - 1 ;
        pos_fin_def = vec_def->pos_tab[num_val_def    ] - 1 ;

        nbr_pos_def = pos_fin_def - pos_deb_def ;

      }
      else if (num_val_def == 0) {

        nbr_pos_def = 0 ;

      }
      else {

        assert(num_val_def >= 0) ;

      }

      nbr_val_rep_new = ECS_MAX(nbr_val_rep_new, cpt_val_rep + nbr_pos_def) ;


      cpt_val_rep += nbr_pos_def ;

    }


  }


  /* Traitement effectif */

  vec_rep_new = ecs_vec_int__alloue(nbr_elt_rep + 1,
                                    nbr_val_rep_new) ;

  vec_rep_new->pos_tab[0] = 1 ;


  cpt_val_rep = 0 ;


  for (ielt_rep = 0 ; ielt_rep < nbr_elt_rep ; ielt_rep++) {


    pos_deb_rep = vec_rep->pos_tab[ielt_rep]     - 1 ;
    pos_fin_rep = vec_rep->pos_tab[ielt_rep + 1] - 1 ;


    for (ipos_rep = pos_deb_rep ; ipos_rep < pos_fin_rep ; ipos_rep++) {

      num_val_def = vec_rep->val_tab[ipos_rep] ;

      if (num_val_def > 0) {

        pos_deb_def = vec_def->pos_tab[num_val_def - 1] - 1 ;
        pos_fin_def = vec_def->pos_tab[num_val_def    ] - 1 ;

        nbr_pos_def = pos_fin_def - pos_deb_def ;

        for (ipos_def = 0 ; ipos_def < nbr_pos_def ; ipos_def++) {

          vec_rep_new->val_tab[cpt_val_rep + ipos_def]
            = vec_def->val_tab[pos_deb_def + ipos_def] ;

        }

      }
      else if (num_val_def == 0)

        nbr_pos_def = 0 ;

      else {

        assert(num_val_def >= 0) ;

      }

      cpt_val_rep += nbr_pos_def ;

    }

    vec_rep_new->pos_tab[ielt_rep + 1] = cpt_val_rep + 1 ;

  }


  return vec_rep_new ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renumrote  partir de `1' les rfrences
 *   des numros de descripteurs
 *   dans le tableau des valeurs d'un attribut donn
 *
 *  La fonction renvoie un tableau de renumrotation des descripteurs
 *   d'un attribut de rfrence donn
 *   Si le numro de descripteur n'a pas t renumrote, on lui attribue
 *    la valeur `num_descr_defaut' dans le tableau de renumrotation
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_vec_int_att__renum_descr
(
       ecs_vec_int_t  *const vec_att,
 const ecs_int_t             num_descr_defaut,
       size_t                nbr_descr_old
)
{

  size_t          cpt_descr ;
  size_t          idescr_old ;
  size_t          ival ;
  size_t          nbr_val ;
  ecs_int_t       num_descr_old ;
  ecs_int_t     * renum_descr_old_new ;

  ecs_tab_int_t   vect_transf_num_descr_new_old ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(vec_att != NULL) ;


  BFT_MALLOC(renum_descr_old_new, nbr_descr_old, ecs_int_t) ;

  BFT_MALLOC(vect_transf_num_descr_new_old.val, nbr_descr_old, ecs_int_t) ;
  vect_transf_num_descr_new_old.nbr = nbr_descr_old ;


  for (idescr_old = 0 ; idescr_old < nbr_descr_old ; idescr_old++) {

    renum_descr_old_new[idescr_old] = num_descr_defaut ;

  }


  nbr_val = ecs_vec_int__ret_val_nbr(vec_att) ;

  cpt_descr = 0 ;

  for (ival = 0 ; ival < nbr_val ; ival++) {

    num_descr_old = vec_att->val_tab[ival] - 1 ;

    if (renum_descr_old_new[num_descr_old] == num_descr_defaut) {

      renum_descr_old_new[num_descr_old] = cpt_descr ;
      vec_att->val_tab[ival] = cpt_descr + 1 ;
      vect_transf_num_descr_new_old.val[cpt_descr++] = num_descr_old ;

    }
    else {

      vec_att->val_tab[ival] = renum_descr_old_new[num_descr_old] + 1 ;
    }

  }

  BFT_FREE(renum_descr_old_new) ;

  BFT_REALLOC(vect_transf_num_descr_new_old.val, cpt_descr, ecs_int_t) ;
  vect_transf_num_descr_new_old.nbr = cpt_descr ;


  return vect_transf_num_descr_new_old ;


}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/
