/*============================================================================
 *  Dfinitions des fonctions de base
 *   associes  la structure `ecs_vec_real_t' dcrivant un vecteur index rel
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>

#include <bft_mem.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_vec_real_tri.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associ au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_vec_real.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privs   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_vec_real_priv.h"



/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction allouant et initialisant une structure `ecs_vec_real_t'
 *----------------------------------------------------------------------------*/

ecs_vec_real_t * ecs_vec_real__cree
(                      /* <-- Pointeur sur une structure `ecs_vec_real_t'     */
 void
)
{

  ecs_vec_real_t *this_vec_real ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  BFT_MALLOC(this_vec_real, 1, ecs_vec_real_t) ;


  /* Initialisation de la structure de type `ecs_vec_real_t' */

  this_vec_real->pos_nbr = 0 ;

  this_vec_real->pos_pas = 0 ;

  this_vec_real->val_tab = NULL ;


  return this_vec_real ;


}


/*----------------------------------------------------------------------------
 *  Fonction librant une structure `ecs_vec_real_t' donne en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_vec_real_t * ecs_vec_real__detruit
(
 ecs_vec_real_t * this_vec_real
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_vec_real != NULL) ;


  /* Libration du contenu de la structure `ecs_vec_real_t' */
  /*===================================================*/


  /* Libration du tableau des valeurs */
  /*-----------------------------------*/

  if (this_vec_real->val_tab != NULL)

    BFT_FREE(this_vec_real->val_tab) ;



  /* Libration de la structure `ecs_vec_real_t' */
  /*======================================*/

  BFT_FREE(this_vec_real) ;


  return this_vec_real ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui alloue une structure `ecs_vec_real_t' ainsi que ses membres
 *----------------------------------------------------------------------------*/

ecs_vec_real_t * ecs_vec_real__alloue
(                    /* <-- Renvoie la reference a la structure creee         */
 size_t     pos_nbr, /* --> Nombre de positions pour le tableau des positions */
 size_t     val_nbr  /* --> Nombre de valeurs   pour le tableau des valeurs   */
)
{

  ecs_vec_real_t *this_vec_real ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Allocation de la structure elle-meme */

  this_vec_real = ecs_vec_real__cree() ;


  /* Allocation de ses membres */


  if (pos_nbr != 0) {

    this_vec_real->pos_nbr = pos_nbr ;

    if (pos_nbr > 1)
      this_vec_real->pos_pas = val_nbr / (pos_nbr - 1) ;
    else
      this_vec_real->pos_pas = 0 ;

  }
  else {

    /* this_vec_real->pos_tab == NULL grace a ecs_vec_real__cree() */

    this_vec_real->pos_nbr = val_nbr + 1 ;
    this_vec_real->pos_pas = 1 ;

  }


  BFT_MALLOC(this_vec_real->val_tab, val_nbr, ecs_real_t   );


  return this_vec_real ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui redimensionne les membres d'une structure `ecs_vec_real_t'
 *----------------------------------------------------------------------------*/

void ecs_vec_real__redimensionne
(
 ecs_vec_real_t  *const this_vec_real,
 size_t                 pos_nbr,  /* --> Nouvelle dim. du tableau des pos.    */
 size_t                 val_nbr   /* --> Nouvelle dim. du tableau des val. x  */
)
{
  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_vec_real != NULL) ;


  /* Rallocation des membres */

  if (pos_nbr != 0) {

    this_vec_real->pos_nbr = pos_nbr ;

    if (pos_nbr > 1)
      this_vec_real->pos_pas = val_nbr / (pos_nbr - 1) ;
    else
      this_vec_real->pos_pas = 0 ;

  }
  else {

    this_vec_real->pos_nbr = val_nbr + 1 ;
    this_vec_real->pos_pas = 1 ;

  }

  BFT_REALLOC(this_vec_real->val_tab, val_nbr, ecs_real_t);

}


/*----------------------------------------------------------------------------
 *  Fonction librant une structure `ecs_vec_real_t'
 *----------------------------------------------------------------------------*/

ecs_vec_real_t * ecs_vec_real__libere
(                               /* <-- Renvoie un pointeur NULL               */
 ecs_vec_real_t  *this_vec_real /* <-> Structure `ecs_vec_real_t' a liberer   */
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_vec_real != NULL) ;


  /* Libration de la structure `ecs_vec_real_t' */
  /*=========================================*/

  BFT_FREE(this_vec_real) ;


  return this_vec_real ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui cre un vecteur `ecs_vec_real_t'
 *   et qui l'initialise avec les rfrences donnes
 *----------------------------------------------------------------------------*/

ecs_vec_real_t * ecs_vec_real__initialise
(
 const ecs_int_t          pos_nbr,
 const ecs_int_t          pos_pas,
       ecs_real_t  *const val_tab
)
{

  ecs_vec_real_t *this_vec_real ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(pos_nbr != 0) ;


  /* Cration de la structure de type `ecs_vec_real_t' */

  this_vec_real = ecs_vec_real__cree() ;


  /* Initialisation de la structure de type `ecs_vec_real_t' */

  this_vec_real->pos_nbr = pos_nbr ;

  this_vec_real->pos_pas = pos_pas ;

  this_vec_real->val_tab = val_tab ;


  return this_vec_real ;


}


/*----------------------------------------------------------------------------
 *  Fonction retournant le nombre de positions
 *----------------------------------------------------------------------------*/

size_t ecs_vec_real__ret_pos_nbr
(
 const ecs_vec_real_t *const this_vec_real
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_vec_real != NULL) ;


  return this_vec_real->pos_nbr ;


}


/*----------------------------------------------------------------------------
 *  Fonction retournant le nombre de valeurs associees au tableau des positions
 *----------------------------------------------------------------------------*/

size_t ecs_vec_real__ret_val_nbr
(
 const ecs_vec_real_t *const this_vec_real
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_vec_real != NULL) ;


  if (this_vec_real->pos_pas == 0)

    return this_vec_real->pos_nbr ;

  else

    return (this_vec_real->pos_nbr - 1) *  this_vec_real->pos_pas ;


}

/*----------------------------------------------------------------------------
 *  Fonction retournant le pas des positions
 *----------------------------------------------------------------------------*/

size_t ecs_vec_real__ret_pos_pas
(
 const ecs_vec_real_t *const this_vec_real
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_vec_real != NULL) ;


  return this_vec_real->pos_pas ;


}


/*----------------------------------------------------------------------------
 *  Fonction retournant la reference du tableau des valeurs
 *----------------------------------------------------------------------------*/

ecs_real_t    * ecs_vec_real__ret_val_tab
(
 const ecs_vec_real_t *const this_vec_real
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_vec_real != NULL) ;


  return this_vec_real->val_tab ;


}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/

