#ifndef _ECS_MAILLAGE_H_
#define _ECS_MAILLAGE_H_

/*============================================================================
 *  Prototypes des fonctions de base
 *   associees a la structure `ecs_maillage_t' decrivant un maillage
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2009 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilite
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Post-Traitement"
 *----------------------------------------------------------------------------*/

#include "ecs_post.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_entmail.h"
#include "ecs_famille.h"
#include "ecs_param_perio_glob.h"
#include "ecs_param_rc_glob.h"
#include "ecs_select_fac_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/


/*============================================================================
 *                         Déclaration de la structure
 *============================================================================*/

typedef struct _ecs_maillage_t ecs_maillage_t;


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *    Fonction qui cree une structure definissant un maillage
 *----------------------------------------------------------------------------*/

ecs_maillage_t *
ecs_maillage__cree(ecs_entmail_t  **vect_entmail);


/*----------------------------------------------------------------------------
 *  Fonction liberant une structure `ecs_maillage_t' donnee en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_maillage_t *
ecs_maillage__detruit(ecs_maillage_t  *this_maillage);


/*----------------------------------------------------------------------------
 *  Fonction imprimant le contenu d'une structure `ecs_maillage_t' donnee
 *   dans le fichier de nom donne
 *   precede par le `titre'
 *----------------------------------------------------------------------------*/

void
ecs_maillage__imprime(const ecs_maillage_t  *this_maillage,
                      const char            *nom_fichier_dump,
                      const ecs_int_t        nbr_imp,
                      const char            *titre);


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le type d'éntité de plus grande dimension
 *   contenue dans une structure `ecs_maillage_t'
 *----------------------------------------------------------------------------*/


ECS_ENTMAIL_E
ecs_maillage__ret_entmail_max(const ecs_maillage_t  *this_maillage);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets d'une structure `ecs_maillage_t'
 *----------------------------------------------------------------------------*/

float
ecs_maillage__ret_taille(const ecs_maillage_t  *this_maillage);


/*----------------------------------------------------------------------------
 *  Fonction qui cree une structure maillage en connectivite nodale
 *   a partir du vecteur des entites de maillage
 *----------------------------------------------------------------------------*/

ecs_maillage_t *
ecs_maillage__cree_nodal(ecs_entmail_t  **vect_entmail);


/*----------------------------------------------------------------------------
 *  Fusion des sommets confondus et suppression des éléments dégénérés
 *----------------------------------------------------------------------------*/

void
ecs_maillage__nettoie_descend(ecs_maillage_t  *this_maillage);


/*----------------------------------------------------------------------------
 *  Suppression des sommets ne participant pas à la connectivité
 *   et fusion des éléments surfaciques confondus éventuels
 *----------------------------------------------------------------------------*/

void
ecs_maillage__nettoie_nodal(ecs_maillage_t  *this_maillage);


/*----------------------------------------------------------------------------
 *  Correction si nécessaire de l'orientation des éléments en
 *   connectivité nodale.
 *
 *  Les listes de cellules avec erreur ou corrigées sont optionnelles.
 *----------------------------------------------------------------------------*/

void
ecs_maillage__orient_nodal(ecs_maillage_t    *this_maillage,
                           ecs_tab_int_t     *liste_cel_err,
                           ecs_tab_int_t     *liste_cel_cor,
                           bool               correc_orient);


/*----------------------------------------------------------------------------
 *  Fonction qui assigne la tete de la liste chainee des familles donnee
 *   a la structure de maillage donnee
 *----------------------------------------------------------------------------*/

void
ecs_maillage__definit_famille(ecs_maillage_t   *maillage,
                              ecs_famille_t   **vect_famille);


/*----------------------------------------------------------------------------
 *  Fonction realisant, a partir d'une connectivite de maillage donnee,
 *   la connectivite descendante du maillage
 *----------------------------------------------------------------------------*/

void
ecs_maillage__connect_descend(ecs_maillage_t * maillage);


/*----------------------------------------------------------------------------
 *  Fonction realisant le tri des elements suivant leur type geometrique
 *  La fonction affiche le nombre d'elements par type geometrique
 *----------------------------------------------------------------------------*/

void
ecs_maillage__trie_typ_geo(ecs_maillage_t  *maillage);


/*----------------------------------------------------------------------------
 *  Fonction qui définit un nouveau maillage
 *   par extraction d'une partie du maillage donné
 *
 *  Les éléments à extraire doivent être tous de même dimension :
 *  cellules ou faces ou arêtes ou sommets
 *
 *  On construit automatiquement une filiation.
 *----------------------------------------------------------------------------*/

ecs_maillage_t *
ecs_maillage__extrait(ecs_maillage_t       *maillage,
                      ECS_ENTMAIL_E         entmail_sel_e,
                      const ecs_tab_int_t  *liste_filtre,
                      const bool            herite_attributs);


/*----------------------------------------------------------------------------
 *  Fonction qui sélectionne une liste d'entités portant des familles
 *   non vides dans un maillage donné
 *
 *  La partie sélectionnée correspond aux éléments :
 *  -       ayant        une des couleurs données dans la liste des couleurs
 *  - et/ou appartenant à un des groupes  donnés  dans la liste des groupes
 *
 *  La sélection peut être inversée (en restant dans les entités de
 *  famille non vides)
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_maillage__selectionne_fam(ecs_maillage_t        *maillage,
                              ECS_ENTMAIL_E          entmail_sel_e,
                              const ecs_tab_int_t   *liste_filtre,
                              const ecs_tab_int_t    liste_couleur,
                              const ecs_tab_char_t   liste_groupe,
                              const bool             inv_selection);


/*----------------------------------------------------------------------------
 *  Fonction qui concatene dans un maillage recepteur donne,
 *   un maillage a concatener donne
 *----------------------------------------------------------------------------*/

void
ecs_maillage__concatene_nodal(ecs_maillage_t  *maillage_recept,
                              ecs_maillage_t  *maillage_concat);


/*----------------------------------------------------------------------------
 *  Création et initialisation d'un maillage extrait correspondant a un cas de
 *  coupe de faces : on extrait un maillage correspondant à la zone
 *  sélectionée par liste et on l'ordonne selon le numéro de type de face.
 *----------------------------------------------------------------------------*/

ecs_maillage_t *
ecs_maillage__cree_coupe(const char           *nom_coupe,
                         ecs_maillage_t       *maillage,
                         const ecs_tab_int_t   liste_fac,
                         const ecs_int_t       nbr_dump,
                         const char           *nom_fic_dump,
                         const bool            herite_attributs,
                         ecs_post_type_t       type_post,
                         ecs_post_t           *cas_post);


/*----------------------------------------------------------------------------
 *  Fonction qui crée la liste des faces intérieures à un maillage.
 *  On ne compte pas ici les faces périodiques parmi les faces intérieures,
 *  cette fonction étant destinée à filtrer le post-triatment, et les
 *  faces périodiques étant déjà affichables par ailleurs.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_maillage__liste_fac_int(ecs_maillage_t  *maillage);


/*----------------------------------------------------------------------------
 *  Fonction qui construit la liste des cellules attachées à une liste
 *  de faces fournie en argument.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_maillage__liste_cel_fac(ecs_maillage_t       *maillage,
                            const ecs_tab_int_t   liste_fac);


/*----------------------------------------------------------------------------
 *  Fonction qui calcule les coordonnées min et max du domaine
 *----------------------------------------------------------------------------*/

void
ecs_maillage__calc_coo_ext(ecs_maillage_t  *maillage);


/*----------------------------------------------------------------------------
 *  Fonction qui modifie les coordonnées du maillage
 *----------------------------------------------------------------------------*/

void
ecs_maillage__transf_coo(ecs_maillage_t  *maillage,
                         const double     matrice[3][4]);


/*----------------------------------------------------------------------------
 *  Fonction qui recolle les faces non conformes
 *
 *  Les listes des faces nouvelles ou modifiées sont construites (et allouées)
 *  ici ; les structures liste_fac_new et liste_fac_mod correspondantes sont
 *  donc vides en entrée ; idem pour liste_fac_err qui indiquera les indices
 *  des faces pour lesquelles le découpage en sous-faces a échoué
 *
 *  On renvoie 1 si l'on a appellé effectivement le recollement, 0 sinon
 *  (i.e. si la sélection de faces à recoller est vide).
 *----------------------------------------------------------------------------*/

int
ecs_maillage__recolle(ecs_maillage_t          *maillage,
                      const ecs_select_fac_t   select_fac_rc,
                      ecs_tab_int_t           *liste_fac_new,
                      ecs_tab_int_t           *liste_fac_mod,
                      ecs_tab_int_t           *liste_fac_err,
                      const ecs_param_rc_t     param_rc);


/*----------------------------------------------------------------------------
 *  Fonction qui recolle des faces non conformes dont la visibilité
 *  est connue (par exemple une filiation) ;
 *
 *  Les listes des faces nouvelles ou modifiées sont construites (et allouées)
 *  ici ; les structures liste_fac_new et liste_fac_mod correspondantes sont
 *  donc vides en entrée ; idem pour liste_fac_err qui indiquera les indices
 *  des faces pour lesquelles le découpage en sous-faces a échoué
 *
 *  On renvoie 1 si l'on a effectivement appellé le recollement, 0 sinon
 *  (i.e. si le champ de type "visibilité" est introuvable).
 *----------------------------------------------------------------------------*/

int
ecs_maillage__recolle_vis(ecs_maillage_t  *maillage,
                          ecs_tab_int_t   *liste_fac_new,
                          ecs_tab_int_t   *liste_fac_mod,
                          ecs_tab_int_t   *liste_fac_err);


/*----------------------------------------------------------------------------
 *  Fonction qui construit les familles
 *----------------------------------------------------------------------------*/

void
ecs_maillage__cree_famille(ecs_maillage_t  *maillage,
                           ECS_ENTMAIL_E    ent_deb_e,
                           ECS_ENTMAIL_E    ent_fin_e);


/*----------------------------------------------------------------------------
 *  Fonction qui detruit les familles
 *----------------------------------------------------------------------------*/

void
ecs_maillage__detruit_famille(ecs_maillage_t  *maillage);


/*----------------------------------------------------------------------------
 *  Fonction qui construit les attributs "groupe" et "couleur"
 *   a partir des familles
 *----------------------------------------------------------------------------*/

void
ecs_maillage__cree_attribut(ecs_maillage_t  *maillage,
                            ECS_ENTMAIL_E    ent_deb_e,
                            ECS_ENTMAIL_E    ent_fin_e);


/*----------------------------------------------------------------------------
 *  Fonction qui supprime les attributs "groupe" et "couleur"
 *----------------------------------------------------------------------------*/

void
ecs_maillage__supprime_attributs(ecs_maillage_t  *this_maillage);


/*----------------------------------------------------------------------------
 *  Vérification d'un maillage et calcul de critères de qualité
 *----------------------------------------------------------------------------*/

bool
ecs_maillage__verif(ecs_maillage_t   *maillage,
                    const ecs_int_t   nbr_dump,
                    const char       *nom_fic_dump,
                    ecs_post_t       *cas_post);


#endif /* _ECS_MAILLAGE_H_ */
