/*============================================================================
 *  Définitions des fonctions de base
 *   réalisant les sorties au format MED
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2011 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/

#include "ecs_config.h"

#if defined(HAVE_MED)


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <ctype.h>
#include <string.h>

#include <bft_error.h>
#include <bft_mem.h>
#include <bft_printf.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_fic.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associe au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_post_med.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privés   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_med_priv.h"


/*============================================================================
 *                       Prototypes de fonctions privées
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction initialisant une structure `ecs_med_t` utilisée en écriture.
 *----------------------------------------------------------------------------*/

ecs_med_t *
ecs_post_med__cree_cas(const char  *nom_cas,
                       bool         no_poly)
{

  ecs_int_t    ind ;
  ecs_int_t    lng_nom_fic ;

  ecs_med_t  * cas_med ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Creation de la structure `ecs_med_t` et du fichier MED */
  /*--------------------------------------------------------*/

  BFT_MALLOC(cas_med, 1, ecs_med_t) ;

  BFT_MALLOC(cas_med->nom_cas, strlen(nom_cas) + 1, char) ;
  strcpy(cas_med->nom_cas, nom_cas) ;

  lng_nom_fic
    = strlen(nom_cas) + strlen(".med") + 1 ;

  BFT_MALLOC(cas_med->nom_fic, lng_nom_fic, char) ;
  strcpy(cas_med->nom_fic, nom_cas) ;
  strcat(cas_med->nom_fic, ".med") ;

  for (ind = 0 ; ind < (ecs_int_t)strlen(nom_cas) ; ind++) {
    if (cas_med->nom_fic[ind] == ' ')
      cas_med->nom_fic[ind] = '_' ;
  }

#if ECS_MED_VERSION == 2
  cas_med->fid = MEDouvrir(cas_med->nom_fic, MED_CREATION);
#else
  cas_med->fid = MEDfileOpen(cas_med->nom_fic, MED_ACC_CREAT);
#endif

  if (cas_med->fid < 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error opening file \"%s\"."),
              cas_med->nom_fic);


  cas_med->nbr_var = 0 ;
  cas_med->nom_var = NULL ;

  cas_med->nbr_maillages = 0 ;
  cas_med->tab_maillages = NULL ;

  cas_med->no_poly = no_poly;


  /* Info sur la création du fichier MED */
  /*-------------------------------------*/

  bft_printf("  %s %s\n", _("Creating file:"), cas_med->nom_fic) ;


  return cas_med ;

}


/*----------------------------------------------------------------------------
 *  Fonction détruisant une structure `ecs_med_t` utilisée en écriture.
 *----------------------------------------------------------------------------*/

ecs_med_t *
ecs_post_med__detruit_cas(ecs_med_t  *cas_med)
{

  ecs_int_t ind ;

  ecs_med_maillage_t  * maillage_med ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (cas_med != NULL) {

    /* Destruction du cas */

#if ECS_MED_VERSION == 2
    if (MEDfermer(cas_med->fid) != 0)
      bft_error(__FILE__, __LINE__, 0,
                _("MED: error closing file \"%s\"."),
                cas_med->nom_fic);
#else
    if (MEDfileClose(cas_med->fid) != 0)
      bft_error(__FILE__, __LINE__, 0,
                _("MED: error closing file \"%s\"."),
                cas_med->nom_fic);
#endif

    BFT_FREE(cas_med->nom_cas) ;
    BFT_FREE(cas_med->nom_fic) ;

    for (ind = 0 ; ind < cas_med->nbr_var ; ind++)
      BFT_FREE(cas_med->nom_var[ind]) ;

    BFT_FREE(cas_med->nom_var) ;
    cas_med->nbr_var = 0 ;

    for (ind = 0 ; ind < cas_med->nbr_maillages ; ind++) {

      maillage_med = cas_med->tab_maillages[ind] ;

      BFT_FREE(maillage_med->nom_maillage) ;
      BFT_FREE(maillage_med->nbr_ele_typ) ;
      BFT_FREE(maillage_med->med_typ) ;

      BFT_FREE(maillage_med) ;

    }

    BFT_FREE(cas_med->tab_maillages) ;
    cas_med->nbr_maillages = 0 ;

    BFT_FREE(cas_med) ;

  }

  return cas_med ;

}


/*----------------------------------------------------------------------------
 *  Fonction définissant un maillage pour une structure `ecs_med_t`.
 *----------------------------------------------------------------------------*/

void
ecs_post_med__ajoute_maillage(const char       *nom_maillage,
                              const ecs_int_t   dim_m,
                              const ecs_int_t   dim_e,
                              ecs_med_t        *cas_med)
{
  ecs_int_t             ind ;
  ecs_int_t             lng_nom_maillage ;

  ecs_med_maillage_t  * maillage_med ;

  char  desc_maillage_med[MED_COMMENT_SIZE + 1] = "";

#if ECS_MED_VERSION == 3
  char  dtunit[MED_LNAME_SIZE + 1] = "s";
  char  axisname[MED_SNAME_SIZE*3 + 1];
  char  axisunit[MED_SNAME_SIZE*3 + 1];
#endif

  med_err      ret_med = 0 ;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  if (cas_med == NULL)
    return ;

#if ECS_MED_VERSION == 3
  for (ind = 0; ind < MED_SNAME_SIZE*3; ind++) {
    axisname[ind] = ' ';
    axisunit[ind] = ' ';
  }
  axisname[0] = 'x';
  axisname[MED_SNAME_SIZE] = 'y';
  axisname[MED_SNAME_SIZE*2] = 'z';
  axisname[MED_SNAME_SIZE*3] = '\0';
  for (ind = 0; ind < 3; ind++)
    axisunit[ind * MED_SNAME_SIZE] = 'm';
  axisunit[MED_SNAME_SIZE*3] = '\0';
#endif

  /* Vérification que le maillage n'a pas déjà été défini */

  for (ind = 0 ; ind < cas_med->nbr_maillages ; ind++) {

    maillage_med = cas_med->tab_maillages[ind] ;

    if (strcmp(nom_maillage, maillage_med->nom_maillage) == 0)

      bft_error(__FILE__, __LINE__, 0,
                _("A mesh named: %s\n"
                  "is already defined in MED case: %s\n"),
                nom_maillage, cas_med->nom_cas) ;

  }

  BFT_MALLOC(maillage_med, 1, ecs_med_maillage_t) ;

  lng_nom_maillage = strlen(nom_maillage) ;
  BFT_MALLOC(maillage_med->nom_maillage, lng_nom_maillage + 1, char) ;
  strcpy(maillage_med->nom_maillage, nom_maillage) ;

  strncpy(maillage_med->nom_maillage_med, nom_maillage, MED_NAME_SIZE) ;

  for (ind = 0 ; ind < lng_nom_maillage ; ind++) {
    if (maillage_med->nom_maillage_med[ind] == ' ')
      maillage_med->nom_maillage_med[ind] = '_' ;
    else
      maillage_med->nom_maillage_med[ind]
        = tolower(maillage_med->nom_maillage_med[ind]) ;
  }
  for (ind = lng_nom_maillage ; ind < MED_NAME_SIZE ; ind++)
    maillage_med->nom_maillage_med[ind] = '\0' ;
  maillage_med->nom_maillage_med[MED_NAME_SIZE] = '\0' ;

  /* BUG: En théorie, on devrait utiliser
     maillage_med->dim_entite = dim_entite ;
     mais L'API MED 2.x est incohérente */

  maillage_med->dim_entite = 3 ;
  maillage_med->nbr_typ_ele = 0 ;
  maillage_med->nbr_ele_typ = NULL ;
  maillage_med->med_typ = NULL ;

  cas_med->nbr_maillages += 1 ;
  BFT_REALLOC(cas_med->tab_maillages,
              cas_med->nbr_maillages,
              ecs_med_maillage_t *) ;
  cas_med->tab_maillages[cas_med->nbr_maillages - 1] = maillage_med ;

  /* Initialisation du maillage */

#if ECS_MED_VERSION == 2

  ret_med = MEDmaaCr(cas_med->fid,
                     maillage_med->nom_maillage_med,
                     (med_int)(maillage_med->dim_entite),
                     MED_NON_STRUCTURE,
                     desc_maillage_med) ;

#else

  ret_med = MEDmeshCr(cas_med->fid,
                      maillage_med->nom_maillage_med,
                      (med_int)3,
                      (med_int)(maillage_med->dim_entite),
                      MED_UNSTRUCTURED_MESH,
                      desc_maillage_med,
                      dtunit,
                      MED_SORT_DTIT,
                      MED_CARTESIAN,
                      axisname,
                      axisunit);

#endif

  if (ret_med != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error writing file \"%s\".\n"
                "Name of mesh to create: \"%s\"\n"),
              cas_med->nom_fic, maillage_med->nom_maillage_med, (int)dim_e);

#if ECS_MED_VERSION == 2

  ret_med = MEDdimEspaceCr(cas_med->fid,
                           maillage_med->nom_maillage_med,
                           (med_int)dim_e) ;

#endif

  if (ret_med != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error writing file \"%s\".\n"
                "Mesh name: \"%s\"\n"
                "mesh dimension: \"%d\", space dimension: \"%d\""),
              cas_med->nom_fic, maillage_med->nom_maillage_med,
              (int)dim_m, (int)dim_e) ;
}


/*----------------------------------------------------------------------------
 *  Fonction vérifiant si un champ figure dans la liste des champs du cas
 *----------------------------------------------------------------------------*/

bool
ecs_post_med__test_champ_liste(const char  *nom_champ,
                               ecs_med_t   *cas_med)
{

  ecs_int_t  ind ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  for (ind = 0 ;
       ind < cas_med->nbr_var
         && strncmp(nom_champ, cas_med->nom_var[ind], MED_NAME_SIZE) != 0 ;
       ind++) ;

  if (ind < cas_med->nbr_var)
    return true ;
  else
    return false ;

}


/*----------------------------------------------------------------------------
 *  Fonction ajoutant un champ à la liste des champs du cas
 *----------------------------------------------------------------------------*/

void
ecs_post_med__ajoute_champ_liste(const char  *nom_champ,
                                 ecs_med_t   *cas_med)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (ecs_post_med__test_champ_liste(nom_champ, cas_med) == true)
    bft_error(__FILE__, __LINE__, 0,
              _("Field \"%s\" has already been defined in file \"%s\"\n"),
              nom_champ, cas_med->nom_fic) ;

  cas_med->nbr_var += 1 ;
  BFT_REALLOC(cas_med->nom_var, cas_med->nbr_var, char *) ;
  BFT_MALLOC(cas_med->nom_var[cas_med->nbr_var - 1], MED_NAME_SIZE + 1, char) ;
  strncpy(cas_med->nom_var[cas_med->nbr_var - 1], nom_champ, MED_NAME_SIZE) ;
  cas_med->nom_var[cas_med->nbr_var - 1][MED_NAME_SIZE] = '\0';

}


/*============================================================================
 *                              Fonctions privées
 *============================================================================*/


#endif /* HAVE_MED */
