/* This file is part of EdiTeX, an editor of mathematical
 * expressions based on TeX syntax.
 * 
 * Copyright (C) 2002-2003 Luca Padovani <lpadovan@cs.unibo.it>,
 *                    2003 Paolo Marinelli <pmarinel@cs.unibo.it>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, please visit the project's home page
 * http://helm.cs.unibo.it/editex/
 * or send an email to <lpadovan@cs.unibo.it>
 */

#include <assert.h>

#include <caml/memory.h>
#include <caml/custom.h>
#include <caml/callback.h>

#include "mlgdomevalue.h"

#include "c_mathml_editor.h"

typedef struct
{
  Editor* c_editor;
  value   callback;
} ml_Editor;

ml_Editor*
Editor_val(value v)
{
  ml_Editor* editor = *((ml_Editor**) Data_custom_val(v));
  assert(editor != NULL);
  return editor;
}

static void
ml_mathml_editor_finalize(value v)
{
  ml_Editor* editor = Editor_val(v);
  assert(editor);

  remove_global_root(&editor->callback);
  c_mathml_editor_destroy(editor->c_editor);
  free(editor);
}

static void
ml_mathml_editor_log_callback(int level, const char* msg, void* user_data)
{
  ml_Editor* ml_editor = (ml_Editor*) user_data;
  assert(ml_editor);
  callback2(ml_editor->callback, Val_int(level), copy_string(msg));
}

value
ml_mathml_editor_get_default_dictionary_path(value unit)
{
  CAMLparam1(unit);
  CAMLreturn(copy_string(c_mathml_editor_get_default_dictionary_path()));
}

value
ml_mathml_editor_get_default_mathml_stylesheet_path(value unit)
{
  CAMLparam1(unit);
  CAMLreturn(copy_string(c_mathml_editor_get_default_mathml_stylesheet_path()));
}

value
ml_mathml_editor_get_default_tex_stylesheet_path(value unit)
{
  CAMLparam1(unit);
  CAMLreturn(copy_string(c_mathml_editor_get_default_tex_stylesheet_path()));
}

value
ml_mathml_editor_new(value alt,
		     value dictionary_uri,
		     value tml_mml_uri,
		     value tml_tex_uri,
		     value log_message_cb)
{
  static struct custom_operations ops =
  {
    "HELM/MathML Editor",
    ml_mathml_editor_finalize,
    custom_compare_default,
    custom_hash_default,
    custom_serialize_default,
    custom_deserialize_default
  };
  
  value v = alloc_custom(&ops, sizeof(ml_Editor*), 0, 1);
  ml_Editor** ml_editor_ref = (ml_Editor**) Data_custom_val(v);
  ml_Editor* ml_editor = *ml_editor_ref = malloc(sizeof(ml_Editor));
  ml_editor->c_editor = c_mathml_editor_new(Bool_val(alt),
		  			    String_val(dictionary_uri),
					    String_val(tml_mml_uri),
					    String_val(tml_tex_uri),
					    ml_mathml_editor_log_callback,
					    (void*) ml_editor);
  ml_editor->callback = log_message_cb;
  register_global_root(&ml_editor->callback);

  return v;
}

value
ml_mathml_editor_freeze(value v)
{
  CAMLparam1(v);
  ml_Editor* editor = Editor_val(v);
  CAMLreturn(Val_bool(c_mathml_editor_freeze(editor->c_editor)));
}

value
ml_mathml_editor_thaw(value v)
{
  CAMLparam1(v);
  ml_Editor* editor = Editor_val(v);
  CAMLreturn(Val_bool(c_mathml_editor_thaw(editor->c_editor)));
}

value
ml_mathml_editor_push(value v, value ch)
{
  CAMLparam2(v, ch);
  ml_Editor* editor = Editor_val(v);
  c_mathml_editor_push(editor->c_editor, Int_val(ch));
  CAMLreturn(Val_unit);
}

value
ml_mathml_editor_drop(value v, value alt)
{
  CAMLparam2(v, alt);
  ml_Editor* editor = Editor_val(v);
  c_mathml_editor_drop(editor->c_editor, Bool_val(alt));
  CAMLreturn(Val_unit);
}

value
ml_mathml_editor_cursor_hide(value v)
{
  CAMLparam1(v);
  ml_Editor* editor = Editor_val(v);
  CAMLreturn(Val_bool(c_mathml_editor_cursor_hide(editor->c_editor)));
}

value
ml_mathml_editor_cursor_show(value v)
{
  CAMLparam1(v);
  ml_Editor* editor = Editor_val(v);
  CAMLreturn(Val_bool(c_mathml_editor_cursor_show(editor->c_editor)));
}

value
ml_mathml_editor_get_tex(value v)
{
  CAMLparam1(v);
  ml_Editor* editor = Editor_val(v);
  char* res = c_mathml_editor_get_tex(editor->c_editor);
  CAMLlocal1(ml_res);
  ml_res = copy_string(res);
  free(res);
  CAMLreturn(ml_res);
}

value
ml_mathml_editor_reset(value v, value s)
{
  CAMLparam1(v);
  ml_Editor* editor = Editor_val(v);
  c_mathml_editor_reset(editor->c_editor);
  CAMLreturn(Val_unit);
}

value
ml_mathml_editor_get_tml(value v)
{
  CAMLparam1(v);
  ml_Editor* editor = Editor_val(v);
  GdomeDocument* doc = c_mathml_editor_get_tml(editor->c_editor);
  CAMLreturn(Val_Document(doc));
}

value
ml_mathml_editor_get_mml(value v)
{
  CAMLparam1(v);
  ml_Editor* editor = Editor_val(v);
  GdomeDocument* doc = c_mathml_editor_get_mml(editor->c_editor);
  CAMLreturn(Val_Document(doc));
}

