/* Copyright (C) 2001 2002 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <unistd.h>

#include <iostream>
#include <fstream>
#include <vector>
#include <algorithm>
#include <iterator>
#include <cstdlib>

#include <gtkmm/main.h>
#include <gtkmm/style.h>
#include <gdkmm/pixbuf.h>
#include <gtkmm/image.h>
#include <gtkmm/stock.h>

#include "settings.h"
#include "settings_icons.h"
#include "dialogs.h"
#include "window_icon.h"

#ifdef HAVE_GETTEXT
#include <libintl.h>
#endif

IdentityTable::IdentityTable(const int standard_size): Gtk::Table(4, 3, false),
						       name_label(gettext("Name: ")),
						       number_label(gettext("Number: ")) {

  name_label.set_justify(Gtk::JUSTIFY_RIGHT);
  number_label.set_justify(Gtk::JUSTIFY_RIGHT);
  name_entry.set_size_request(standard_size * 7, standard_size);
  number_entry.set_size_request(standard_size * 7, standard_size);

  Glib::RefPtr<Gdk::Pixbuf> pixbuf_r(Gdk::Pixbuf::create_from_xpm_data(help_xpm));
  Gtk::Image* image_p = manage(new Gtk::Image(pixbuf_r));
  name_help_button.add(*image_p);
  image_p = manage(new Gtk::Image(pixbuf_r));
  number_help_button.add(*image_p);

  attach(filler1, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);
  attach(name_label, 0, 1, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(name_entry, 1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(name_help_button, 2, 3, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(number_label, 0, 1, 2, 3, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(number_entry, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(number_help_button, 2, 3, 2, 3, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(filler2, 1, 2, 3, 4, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);

  name_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &IdentityTable::show_help),
						       IdentityMessages::name));
  number_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &IdentityTable::show_help),
							 IdentityMessages::number));

  tooltips.set_tip(name_help_button, help_messages.get_message(IdentityMessages::name));
  tooltips.set_tip(number_help_button, help_messages.get_message(IdentityMessages::number));

  set_border_width(standard_size/3);
}

void IdentityTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void IdentityTable::clear(void) {
  name_entry.set_text("");
  number_entry.set_text("");
}

ModemTable::ModemTable(const int standard_size): Gtk::Table(8, 3, false),
				    device_label(gettext("Serial Device: ")),
				    lock_label(gettext("Lock File: ")),
				    capabilities_label(gettext("Capabilities: ")),
                                    rings_label(gettext("Rings (1-9): ")),
				    class_label(gettext("Modem Class: ")),
                                    dialmode_label(gettext("Dial Mode: ")),
				    auto_button(gettext("Auto")),
				    class1_button(gettext("Class 1")),
                                    class2_button(gettext("Class 2")),
				    class20_button(gettext("Class 2.0")),
				    tone_button(gettext("Tone")),
				    pulse_button(gettext("Pulse")),
                                    class_box(false, 2), dialmode_box(false, 2) {

  device_label.set_justify(Gtk::JUSTIFY_RIGHT);
  lock_label.set_justify(Gtk::JUSTIFY_RIGHT);
  class_label.set_justify(Gtk::JUSTIFY_RIGHT);
  dialmode_label.set_justify(Gtk::JUSTIFY_RIGHT);
  capabilities_label.set_justify(Gtk::JUSTIFY_RIGHT);
  rings_label.set_justify(Gtk::JUSTIFY_RIGHT);
  device_entry.set_size_request(standard_size * 7, standard_size);
  lock_entry.set_size_request(standard_size * 7, standard_size);
  capabilities_entry.set_size_request(standard_size * 7, standard_size);
  rings_entry.set_size_request(standard_size * 7, standard_size);

  Gtk::RadioButton::Group class_group(auto_button.get_group());
  class1_button.set_group(class_group);
  class2_button.set_group(class_group);
  class20_button.set_group(class_group);
  auto_button.set_active(true);

  class_box.pack_start(auto_button, false, false, 0);
  class_box.pack_start(class1_button, false, false, 0);
  class_box.pack_start(class2_button, false, false, 0);
  class_box.pack_start(class20_button, false, false, 0);
  class_box.pack_start(class_box_filler, true, false, 0);

  class_frame.add(class_box);

  Gtk::RadioButton::Group dialmode_group(tone_button.get_group());
  pulse_button.set_group(dialmode_group);
  tone_button.set_active(true);

  dialmode_box.pack_start(tone_button, false, false, 0);
  dialmode_box.pack_start(pulse_button, false, false, 0);
  dialmode_box.pack_start(dialmode_box_filler, true, false, 0);

  dialmode_frame.add(dialmode_box);

  Glib::RefPtr<Gdk::Pixbuf> pixbuf_r(Gdk::Pixbuf::create_from_xpm_data(help_xpm));
  Gtk::Image* image_p = manage(new Gtk::Image(pixbuf_r));
  device_help_button.add(*image_p);
  image_p = manage(new Gtk::Image(pixbuf_r));
  lock_help_button.add(*image_p);
  image_p = manage(new Gtk::Image(pixbuf_r));
  class_help_button.add(*image_p);
  image_p = manage(new Gtk::Image(pixbuf_r));
  dialmode_help_button.add(*image_p);
  image_p = manage(new Gtk::Image(pixbuf_r));
  capabilities_help_button.add(*image_p);
  image_p = manage(new Gtk::Image(pixbuf_r));
  rings_help_button.add(*image_p);

  attach(filler1, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);
  attach(device_label, 0, 1, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(device_entry, 1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(device_help_button, 2, 3, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/5);
  attach(lock_label, 0, 1, 2, 3, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(lock_entry, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(lock_help_button, 2, 3, 2, 3, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/5);
  attach(capabilities_label, 0, 1, 3, 4, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(capabilities_entry, 1, 2, 3, 4, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(capabilities_help_button, 2, 3, 3, 4, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/5);
  attach(rings_label, 0, 1, 4, 5, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(rings_entry, 1, 2, 4, 5, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(rings_help_button, 2, 3, 4, 5, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/5);
  attach(class_label, 0, 1, 5, 6, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(class_frame, 1, 2, 5, 6, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(class_help_button, 2, 3, 5, 6, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/5);
  attach(dialmode_label, 0, 1, 6, 7, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(dialmode_frame, 1, 2, 6, 7, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/5, standard_size/5);
  attach(dialmode_help_button, 2, 3, 6, 7, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/5);
  attach(filler2, 1, 2, 7, 8, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);

  device_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &ModemTable::show_help),
							 ModemMessages::device));
  lock_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &ModemTable::show_help),
						       ModemMessages::lock));
  class_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &ModemTable::show_help),
							ModemMessages::modem_class));
  dialmode_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &ModemTable::show_help),
							   ModemMessages::dialmode));
  capabilities_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &ModemTable::show_help),
							       ModemMessages::capabilities));
  rings_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &ModemTable::show_help),
							ModemMessages::rings));

  tooltips.set_tip(device_help_button, help_messages.get_message(ModemMessages::device));
  tooltips.set_tip(lock_help_button, help_messages.get_message(ModemMessages::lock));
  tooltips.set_tip(class_help_button, help_messages.get_message(ModemMessages::modem_class));
  tooltips.set_tip(dialmode_help_button, help_messages.get_message(ModemMessages::dialmode));
  tooltips.set_tip(capabilities_help_button, help_messages.get_message(ModemMessages::capabilities));
  tooltips.set_tip(rings_help_button, help_messages.get_message(ModemMessages::rings));

  set_border_width(standard_size/3);
}

Glib::ustring ModemTable::get_class(void) const {

  Glib::ustring return_val; // an empty string will indicate an "Auto" setting
  if (class1_button.get_active()) return_val = "1";
  else if (class2_button.get_active()) return_val = "2";
  else if (class20_button.get_active()) return_val = "2.0";
  return return_val;
}

void ModemTable::set_class(const Glib::ustring& class_string) {

  if (!class_string.compare("1")) class1_button.set_active(true);
  else if (!class_string.compare("2")) class2_button.set_active(true);
  else if (!class_string.compare("2.0")) class20_button.set_active(true);
  else auto_button.set_active(true); // "Auto" setting
}

Glib::ustring ModemTable::get_dialmode(void) const {
  
  Glib::ustring return_val("tone");
  if (pulse_button.get_active()) return_val = "pulse";
  return return_val;
}

void ModemTable::set_dialmode(const Glib::ustring& mode_string) {

  if (!mode_string.compare("tone")) tone_button.set_active(true);
  else if (!mode_string.compare("pulse")) pulse_button.set_active(true);
}

void ModemTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));

}

void ModemTable::clear(void) {
  device_entry.set_text("");
  lock_entry.set_text("");
  capabilities_entry.set_text("");
  rings_entry.set_text("");

  auto_button.set_active(true);
  tone_button.set_active(true);
}

ParmsTable::ParmsTable(const int standard_size): Gtk::Table(5, 3, false),
				    init_label(gettext("Initialisation Params:")),
				    reset_label(gettext("Reset Params: ")),
				    parms_label(gettext("Other Params: ")) {

  init_label.set_justify(Gtk::JUSTIFY_RIGHT);
  reset_label.set_justify(Gtk::JUSTIFY_RIGHT);
  parms_label.set_justify(Gtk::JUSTIFY_RIGHT);
  init_entry.set_size_request(standard_size * 7, standard_size);
  reset_entry.set_size_request(standard_size * 7, standard_size);
  parms_entry.set_size_request(standard_size * 7, standard_size);

  Glib::RefPtr<Gdk::Pixbuf> pixbuf_r(Gdk::Pixbuf::create_from_xpm_data(help_xpm));
  Gtk::Image* image_p = manage(new Gtk::Image(pixbuf_r));
  init_help_button.add(*image_p);
  image_p = manage(new Gtk::Image(pixbuf_r));
  reset_help_button.add(*image_p);
  image_p = manage(new Gtk::Image(pixbuf_r));
  parms_help_button.add(*image_p);

  attach(filler1, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);
  attach(init_label, 0, 1, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/2);
  attach(init_entry, 1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size/2);
  attach(init_help_button, 2, 3, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/2);
  attach(reset_label, 0, 1, 2, 3, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/2);
  attach(reset_entry, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size/2);
  attach(reset_help_button, 2, 3, 2, 3, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/2);
  attach(parms_label, 0, 1, 3, 4, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/2);
  attach(parms_entry, 1, 2, 3, 4, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size/2);
  attach(parms_help_button, 2, 3, 3, 4, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size/2);
  attach(filler2, 1, 2, 4, 5, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);

  init_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &ParmsTable::show_help),
					      ParmsMessages::init));
  reset_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &ParmsTable::show_help),
					       ParmsMessages::reset));
  parms_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &ParmsTable::show_help),
					       ParmsMessages::extra_parms));

  tooltips.set_tip(init_help_button, help_messages.get_message(ParmsMessages::init));
  tooltips.set_tip(reset_help_button, help_messages.get_message(ParmsMessages::reset));
  tooltips.set_tip(parms_help_button, help_messages.get_message(ParmsMessages::extra_parms));

  set_border_width(standard_size/3);
}

void ParmsTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void ParmsTable::clear(void) {
  init_entry.set_text("");
  reset_entry.set_text("");
  parms_entry.set_text("");
}

PrintTable::PrintTable(const int standard_size): Gtk::Table(4, 3, false),
			    command_label(gettext("Print Program: ")),
			    shrink_label(gettext("Print Shrink (50-100):")) {

  command_label.set_justify(Gtk::JUSTIFY_RIGHT);
  shrink_label.set_justify(Gtk::JUSTIFY_RIGHT);
  command_entry.set_size_request(standard_size * 7, standard_size);
  shrink_entry.set_size_request(standard_size * 7, standard_size);

  Glib::RefPtr<Gdk::Pixbuf> pixbuf_r(Gdk::Pixbuf::create_from_xpm_data(help_xpm));
  Gtk::Image* image_p = manage(new Gtk::Image(pixbuf_r));
  command_help_button.add(*image_p);
  image_p = manage(new Gtk::Image(pixbuf_r));
  shrink_help_button.add(*image_p);

  attach(filler1, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);
  attach(command_label, 0, 1, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(command_entry, 1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(command_help_button, 2, 3, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(shrink_label, 0, 1, 2, 3, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(shrink_entry, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(shrink_help_button, 2, 3, 2, 3, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(filler2, 1, 2, 3, 4, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);

  command_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &PrintTable::show_help),
						 PrintMessages::command));
  shrink_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &PrintTable::show_help),
						PrintMessages::shrink));
  tooltips.set_tip(command_help_button, help_messages.get_message(PrintMessages::command));
  tooltips.set_tip(shrink_help_button, help_messages.get_message(PrintMessages::shrink));

  set_border_width(standard_size/3);
}

void PrintTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void PrintTable::clear(void) {
  command_entry.set_text("");
  shrink_entry.set_text("");
}

ViewTable::ViewTable(const int standard_size): Gtk::Table(3, 3, false),
				     ps_view_command_label(gettext("Postscript Viewer\nProgram:")) {

  ps_view_command_label.set_justify(Gtk::JUSTIFY_LEFT);
  ps_view_command_entry.set_size_request(standard_size * 7, standard_size);

  Gtk::Image* image_p = manage(new Gtk::Image(Gdk::Pixbuf::create_from_xpm_data(help_xpm)));
  ps_view_command_help_button.add(*image_p);

  attach(filler1, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);
  attach(ps_view_command_label, 0, 1, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(ps_view_command_entry, 1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(ps_view_command_help_button, 2, 3, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(filler2, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);

  ps_view_command_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &ViewTable::show_help),
							 ViewMessages::ps_view_command));

  tooltips.set_tip(ps_view_command_help_button, help_messages.get_message(ViewMessages::ps_view_command));

  set_border_width(standard_size/3);
}

void ViewTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void ViewTable::clear(void) {
  ps_view_command_entry.set_text("");
}

LoggingTable::LoggingTable(const int standard_size): Gtk::Table(3, 3, false),
						     logfile_label(gettext("Log File: ")) {

  logfile_label.set_justify(Gtk::JUSTIFY_RIGHT);
  logfile_entry.set_size_request(standard_size * 7, standard_size);

  Gtk::Image* image_p = manage(new Gtk::Image(Gdk::Pixbuf::create_from_xpm_data(help_xpm)));
  logfile_help_button.add(*image_p);

  attach(filler1, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);
  attach(logfile_label, 0, 1, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(logfile_entry, 1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(logfile_help_button, 2, 3, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(filler2, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);

  logfile_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &LoggingTable::show_help),
						 LoggingMessages::logfile));

  tooltips.set_tip(logfile_help_button, help_messages.get_message(LoggingMessages::logfile));

  set_border_width(standard_size/3);
}

void LoggingTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void LoggingTable::clear(void) {
  logfile_entry.set_text("");
}

PageTable::PageTable(const int standard_size): Gtk::Table(4, 3, false),
				    page_label(gettext("Page Size: ")),
				    res_label(gettext("Sent Fax Resolution: ")),
				    a4_button("A4"), letter_button("Letter"),
				    legal_button("Legal"), standard_button(gettext("Standard")),
				    fine_button(gettext("Fine")), page_box(false, 2),
                                    res_box(false, 2) {

  page_label.set_justify(Gtk::JUSTIFY_RIGHT);
  res_label.set_justify(Gtk::JUSTIFY_RIGHT);

  Gtk::RadioButton::Group size_group(a4_button.get_group());
  letter_button.set_group(size_group);
  legal_button.set_group(size_group);
  a4_button.set_active(true);

  page_box.pack_start(a4_button, false, false, 10);
  page_box.pack_start(letter_button, false, false, 10);
  page_box.pack_start(legal_button, false, false, 10);
  page_box.pack_start(page_box_filler, true, false, 0);

  page_frame.add(page_box);

  Gtk::RadioButton::Group res_group(standard_button.get_group());
  fine_button.set_group(res_group);
  fine_button.set_active(true);

  res_box.pack_start(standard_button, false, false, 10);
  res_box.pack_start(fine_button, false, false, 10);
  res_box.pack_start(res_box_filler, true, false, 0);

  res_frame.add(res_box);

  Glib::RefPtr<Gdk::Pixbuf> pixbuf_r(Gdk::Pixbuf::create_from_xpm_data(help_xpm));
  Gtk::Image* image_p = manage(new Gtk::Image(pixbuf_r));
  page_help_button.add(*image_p);
  image_p = manage(new Gtk::Image(pixbuf_r));
  res_help_button.add(*image_p);

  attach(filler1, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);
  attach(page_label, 0, 1, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(page_frame, 1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(page_help_button, 2, 3, 1, 2, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(res_label, 0, 1, 2, 3, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(res_frame, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(res_help_button, 2, 3, 2, 3, Gtk::SHRINK,
         Gtk::SHRINK, standard_size/3, standard_size);
  attach(filler2, 1, 2, 3, 4, Gtk::EXPAND | Gtk::FILL,
	 Gtk::EXPAND | Gtk::FILL, 0, 0);

  page_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &PageTable::show_help),
						 PageMessages::page));
  res_help_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &PageTable::show_help),
						PageMessages::res));
  tooltips.set_tip(page_help_button, help_messages.get_message(PageMessages::page));
  tooltips.set_tip(res_help_button, help_messages.get_message(PageMessages::res));

  set_border_width(standard_size/3);
}

Glib::ustring PageTable::get_page(void) const {
  
  Glib::ustring return_val("a4");
  if (letter_button.get_active()) return_val = "letter";
  else if (legal_button.get_active()) return_val = "legal";
  return return_val;
}

void PageTable::set_page(const Glib::ustring& page_string) {

  if (!page_string.compare("a4")) a4_button.set_active(true);
  else if (!page_string.compare("letter")) letter_button.set_active(true);
  else if (!page_string.compare("legal")) legal_button.set_active(true);
}

Glib::ustring PageTable::get_res(void) const {
  
  Glib::ustring return_val("fine");
  if (standard_button.get_active()) return_val = "standard";
  return return_val;
}

void PageTable::set_res(const Glib::ustring& res_string) {

  if (!res_string.compare("fine")) fine_button.set_active(true);
  else if (!res_string.compare("standard")) standard_button.set_active(true);
}

void PageTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void PageTable::clear(void) {
  a4_button.set_active(true);
  fine_button.set_active(true);
}

SettingsDialog::SettingsDialog(const int size, Gtk::Window& window, bool skip_old_settings):
                                            // skip_old_settings has a default value of false
                                            Gtk::Window(Gtk::WINDOW_TOPLEVEL),
					    standard_size(size), in_run_loop(false),
			                    is_home_config(false), ok_button(Gtk::Stock::OK),
					    cancel_button(Gtk::Stock::CANCEL),
					    button_box(Gtk::BUTTONBOX_END, standard_size/2),
                                            window_table(2, 2, false),
					    parent(window), identity_table(standard_size),
					    modem_table(standard_size), parms_table(standard_size),
					    print_table(standard_size), view_table(standard_size),
					    logging_table(standard_size), page_table(standard_size) {

  // we set skip_old_settings as true in MainWindow::MainWindow(), where no config file has
  // been found, to prevent that fact being reported twice
  if (!skip_old_settings) read_config();

  if (!is_home_config) { // if skip_old_settings is true, then is_home_config is always false

    label.set_line_wrap(true);
    Glib::ustring label_text(gettext("Note: pressing the OK button will save the "
				     "settings in file"));
    label_text += " ~/." RC_FILE;
    label.set_text(label_text);
  }

  else {
    Gtk::Button* reset_button_p = manage(new Gtk::Button(gettext("Reset")));
    reset_button_p->signal_clicked().connect(SigC::slot(*this, &SettingsDialog::get_reset_settings_prompt));
    reset_button_p->set_flags(Gtk::CAN_DEFAULT);
    button_box.add(*reset_button_p);
  }
    
  button_box.add(cancel_button);
  button_box.add(ok_button);

  window_table.attach(notebook, 0, 2, 0, 1, Gtk::FILL | Gtk::EXPAND,
		      Gtk::FILL | Gtk::EXPAND, standard_size/2, standard_size/4);
  window_table.attach(label, 0, 1, 1, 2, Gtk::EXPAND,
		      Gtk::SHRINK, standard_size/2, standard_size/4);
  window_table.attach(button_box, 1, 2, 1, 2, Gtk::FILL | Gtk::EXPAND,
			Gtk::SHRINK, standard_size/2, standard_size/4);

  notebook.set_tab_pos(Gtk::POS_TOP);

  notebook.set_scrollable(true);
  // set up the notebook pages
  {
    using namespace Gtk::Notebook_Helpers;
    PageList& page_list = notebook.pages();
    page_list.push_back(TabElem(identity_table, gettext("Identity")));
    page_list.push_back(TabElem(modem_table, gettext("Modem")));
    page_list.push_back(TabElem(parms_table, gettext("Params")));
    page_list.push_back(TabElem(print_table, gettext("Print")));
    page_list.push_back(TabElem(view_table, gettext("View")));
    page_list.push_back(TabElem(logging_table, gettext("Logging")));
    page_list.push_back(TabElem(page_table, gettext("Page")));
  }

  ok_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &SettingsDialog::selected), true));
  cancel_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &SettingsDialog::selected), false));
  ok_button.set_flags(Gtk::CAN_DEFAULT);
  cancel_button.set_flags(Gtk::CAN_DEFAULT);

  identity_table.show_help_sig.connect(SigC::slot(*this, &SettingsDialog::show_help));
  modem_table.show_help_sig.connect(SigC::slot(*this, &SettingsDialog::show_help));
  parms_table.show_help_sig.connect(SigC::slot(*this, &SettingsDialog::show_help));
  print_table.show_help_sig.connect(SigC::slot(*this, &SettingsDialog::show_help));
  view_table.show_help_sig.connect(SigC::slot(*this, &SettingsDialog::show_help));
  logging_table.show_help_sig.connect(SigC::slot(*this, &SettingsDialog::show_help));
  page_table.show_help_sig.connect(SigC::slot(*this, &SettingsDialog::show_help));

  add(window_table);
  
  set_title(gettext("efax-gtk: settings"));
  set_transient_for(parent);
  set_type_hint(Gdk::WINDOW_TYPE_HINT_DIALOG);
  parent.set_sensitive(false);
  set_modal(true);

  notebook.set_size_request(standard_size * 19, standard_size * 12);
  set_border_width(standard_size/2);
  cancel_button.grab_focus();
  set_position(Gtk::WIN_POS_CENTER);

  set_icon(Gdk::Pixbuf::create_from_xpm_data(window_icon_xpm));

  show_all();
}

void SettingsDialog::selected(bool accept) {
  bool finish = false;

  if (accept) {
    if (write_config()) {
      parent.set_sensitive(true); // do this before we emit accepted()
      hide_all();
      finish = true;
      accepted(configure_prog(true));
    }
  }
  else {
    parent.set_sensitive(true);
    hide_all();
    finish = true;
  }

  if (finish) {
    if (in_run_loop) Gtk::Main::quit();
    // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
    else delete this;
  }
}

bool SettingsDialog::on_delete_event(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

void SettingsDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

bool SettingsDialog::write_config(void) {
  // returns false if any of the entered parameters is invalid, otherwise it returns true

  bool return_val = true;
  ifstream filein;

  if (!rcfile.empty()) {
    filein.open(rcfile.c_str(), ios::in);
    if (!filein) {
      Glib::ustring message("Can't open file ");
      message += rcfile;
      message += "\n\n";
      write_error(message.c_str());
    }
  }

  vector<std::string> file_list;
  std::string file_read;
  
  Glib::ustring temp;

  Glib::ustring name_line("NAME: ");
  temp = identity_table.get_name();
  strip(temp);
  name_line += temp;
  
  Glib::ustring number_line("NUMBER: ");
  temp =  identity_table.get_number();
  strip(temp);
  number_line += temp;
  
  Glib::ustring device_line;
  temp = modem_table.get_device();
  strip(temp);
  if (temp.empty()) device_line = "#DEVICE: modem";
  else {
    device_line = "DEVICE: ";
    device_line += temp;
  }
  
  Glib::ustring lock_line;
  temp = modem_table.get_lock();
  strip(temp);
  if (temp.empty()) lock_line = "#LOCK: /var/lock";
  else {
    lock_line = "LOCK: ";
    lock_line += temp;
  }
  
  Glib::ustring class_line;
  temp = modem_table.get_class();
  strip(temp);
  if (temp.empty()) class_line = "#CLASS: 2";
  else {
    if (temp.compare("1") && temp.compare("2") && temp.compare("2.0")) {
      class_line = "#CLASS: 2";
      return_val = false;
      InfoDialog dialog("Invalid modem class specified", "Config Error",
			standard_size, Gtk::MESSAGE_ERROR, *this);
      dialog.run();
    }
    else {    
      class_line = "CLASS: ";
      class_line += temp;
    }
  }
  
  Glib::ustring page_line;
  temp = page_table.get_page();
  strip(temp);
  if (temp.empty()) page_line = "#PAGE: a4";
  else {
    if (temp.compare("a4") && temp.compare("letter") && temp.compare("legal")) {
      page_line = "#PAGE: a4";
      return_val = false;
      InfoDialog dialog("Invalid page size specified", "Config Error",
			standard_size, Gtk::MESSAGE_ERROR, *this);
      dialog.run();
    }
    else {
      page_line = "PAGE: ";
      page_line += temp;
    }
  }
  
  Glib::ustring res_line;
  temp = page_table.get_res();
  strip(temp);
  if (temp.empty()) res_line = "#RES: fine";
  else {
    if (temp.compare("fine") && temp.compare("standard")) {
      res_line = "#RES: fine";
      return_val = false;
      InfoDialog dialog("Invalid sent fax resolution specified", "Config Error",
			standard_size, Gtk::MESSAGE_ERROR, *this);
      dialog.run();
    }
    else {
      res_line = "RES: ";
      res_line += temp;
    }
  }
  
  Glib::ustring rings_line;
  temp = modem_table.get_rings();
  strip(temp);
  if (temp.empty()) rings_line = "#RINGS: 1";
  else {
    if (temp.size() > 1 || temp[0] < '1' ||temp[0] > '9') {
      rings_line = "#RINGS: 1";
      return_val = false;
      InfoDialog dialog("Invalid rings number specified", "Config Error",
			standard_size, Gtk::MESSAGE_ERROR, *this);
      dialog.run();
    }
    else {
      rings_line = "RINGS: ";
      rings_line += temp;
    }
  }
  
  Glib::ustring dialmode_line;
  temp = modem_table.get_dialmode();
  strip(temp);
  if (temp.empty()) dialmode_line = "#DIALMODE: tone";
  else {
    if (temp.compare("tone") && temp.compare("pulse")) {
      dialmode_line = "#DIALMODE: tone";
      return_val = false;
      InfoDialog dialog("Invalid dial mode specified", "Config Error",
			standard_size, Gtk::MESSAGE_ERROR, *this);
      dialog.run();
    }
    else {
      dialmode_line = "DIALMODE: ";
      dialmode_line += temp;
    }
  }
  
  Glib::ustring init_line;
  temp = parms_table.get_init();
  strip(temp);
  if (temp.empty()) init_line = "#INIT: Z &FE&D2S7=120 &C0 M1L0";
  else {
    init_line = "INIT: ";
    init_line += temp;
  }
  
  Glib::ustring reset_line;
  temp = parms_table.get_reset();
  strip(temp);
  if (temp.empty()) reset_line = "#RESET: Z";
  else {
    reset_line = "RESET: ";
    reset_line += temp;
  }
  
  Glib::ustring capabilities_line;
  temp = modem_table.get_capabilities();
  strip(temp);
  if (temp.empty()) capabilities_line = "#CAPABILITIES: 1,5,0,2,0,0,0,0";
  else {
    capabilities_line = "CAPABILITIES: ";
    capabilities_line += temp;
  }
  
  Glib::ustring extra_parms_line;
  temp = parms_table.get_parms();
  strip(temp);
  if (temp.empty()) extra_parms_line = "#PARMS: ";
  else {
    extra_parms_line = "PARMS: ";
    extra_parms_line += temp;
  }
  
  Glib::ustring print_cmd_line;
  temp = print_table.get_command();
  strip(temp);
  if (temp.empty()) print_cmd_line = "#PRINT_CMD: lpr";
  else {
    print_cmd_line = "PRINT_CMD: ";
    print_cmd_line += temp;
  }
  
  Glib::ustring print_shrink_line;
  temp = print_table.get_shrink();
  strip(temp);
  if (temp.empty()) print_shrink_line = "#PRINT_SHRINK: 100";
  else {
    if (atoi(temp.c_str()) < 50 || atoi(temp.c_str()) > 100) {
      print_shrink_line = "#PRINT_SHRINK: 100";
      return_val = false;
      InfoDialog dialog("Invalid print shrink parameter specified", "Config Error",
			standard_size, Gtk::MESSAGE_ERROR, *this);
      dialog.run();
    }
    else {
      print_shrink_line = "PRINT_SHRINK: ";
      print_shrink_line += temp;
    }
  }
  
  Glib::ustring ps_view_cmd_line;
  temp = view_table.get_ps_view_command();
  strip(temp);
  if (temp.empty()) ps_view_cmd_line = "#PS_VIEWER: gv";
  else {
    ps_view_cmd_line = "PS_VIEWER: ";
    ps_view_cmd_line += temp;
  }
  
  Glib::ustring logfile_line;
  temp = logging_table.get_logfile();
  strip(temp);
  if (temp.empty()) logfile_line = "#LOG_FILE: logfile";
  else {
    logfile_line = "LOG_FILE: ";
    logfile_line += temp;
  }
  
  if (return_val) { // no errors -- write out the configuration file
    
    bool found_name = false;
    bool found_number = false;
    bool found_device = false;
    bool found_lock = false;
    bool found_class = false;
    bool found_page = false;
    bool found_res = false;
    bool found_rings = false;
    bool found_dialmode = false;
    bool found_init = false;
    bool found_reset = false;
    bool found_capabilities = false;
    bool found_parms = false;
    bool found_print_cmd = false;
    bool found_print_shrink = false;
    bool found_ps_view_cmd = false;
    bool found_log_file = false;
    
    const char terminating_line[] = "## end of " RC_FILE " ##";
    
    while (getline(filein, file_read)) {
      
      // look for "NAME:"
      if (find_prog_parm("NAME:", file_read) || find_prog_parm("#NAME:", file_read)) {
	if (!found_name) {
	  found_name = true;
	  file_list.push_back(Glib::locale_from_utf8(name_line));
	}
      }
      
      // look for "NUMBER:"
      else if (find_prog_parm("NUMBER:", file_read) || find_prog_parm("#NUMBER:", file_read)) {
	if (!found_number) {
	  found_number = true;
	  file_list.push_back(number_line);
	}
      }
      
      // look for "DEVICE:"
      else if (find_prog_parm("DEVICE:", file_read) || find_prog_parm("#DEVICE:", file_read)) {
	if (!found_device) {
	  found_device = true;
	  file_list.push_back(device_line);
	}
      }
      
      // look for "LOCK:"
      else if (find_prog_parm("LOCK:", file_read)  || find_prog_parm("#LOCK:", file_read)) {
	if (!found_lock) {
	  found_lock = true;
	  file_list.push_back(Glib::filename_from_utf8(lock_line));
	}
      }
      
      // look for "CLASS:"
      else if (find_prog_parm("CLASS:", file_read) || find_prog_parm("#CLASS:", file_read))  {
	if (!found_class) {
	  found_class = true;
	  file_list.push_back(class_line);
	}
      }
      
      // look for "PAGE:"
      else if (find_prog_parm("PAGE:", file_read) || find_prog_parm("#PAGE:", file_read))  {
	if (!found_page) {
	  found_page = true;
	  file_list.push_back(page_line);
	}
	}
      
      // look for "RES:"
      else if (find_prog_parm("RES:", file_read) || find_prog_parm("#RES:", file_read))  {
	if (!found_res) {
	  found_res = true;
	  file_list.push_back(res_line);
	}
      }
      
      // look for "RINGS:"
      else if (find_prog_parm("RINGS:", file_read) || find_prog_parm("#RINGS:", file_read)) {
	if (!found_rings) {
	  found_rings = true;
	  file_list.push_back(rings_line);
	}
      }
      
      // look for "DIALMODE:"
      else if (find_prog_parm("DIALMODE:", file_read) || find_prog_parm("#DIALMODE:", file_read)) {
	if (!found_dialmode) {
	  found_dialmode = true;
	  file_list.push_back(dialmode_line);
	}
      }
      
      // look for "INIT:"
      else if (find_prog_parm("INIT:", file_read) || find_prog_parm("#INIT:", file_read)) {
	if (!found_init) {
	  found_init = true;
	  file_list.push_back(init_line);
	}
      }
      
      // look for "RESET:"
      else if (find_prog_parm("RESET:", file_read) || find_prog_parm("#RESET:", file_read)) {
	if (!found_reset) {
	  found_reset = true;
	  file_list.push_back(reset_line);
	}
      }
      
      // look for "CAPABILITIES:"
      else if (find_prog_parm("CAPABILITIES:", file_read) || find_prog_parm("#CAPABILITIES:", file_read)) {
	if (!found_capabilities) {
	  found_capabilities = true;
	  file_list.push_back(capabilities_line);
	}
      }
      
      // look for "PARMS:"
      else if (find_prog_parm("PARMS:", file_read) || find_prog_parm("#PARMS:", file_read)) {
	if (!found_parms) {
	  found_parms = true;
	  file_list.push_back(extra_parms_line);
	}
      }
      
      // look for "PRINT_CMD:"
      else if (find_prog_parm("PRINT_CMD:", file_read) || find_prog_parm("#PRINT_CMD:", file_read)) {
	if (!found_print_cmd) {
	  found_print_cmd = true;
	  file_list.push_back(print_cmd_line);
	}
      }
      
      // look for "PRINT_SHRINK:"
      else if (find_prog_parm("PRINT_SHRINK:", file_read) || find_prog_parm("#PRINT_SHRINK:", file_read)) {
	if (!found_print_shrink) {
	  found_print_shrink = true;
	  file_list.push_back(print_shrink_line);
	}
      }
      
      // look for "PS_VIEWER:"
      else if (find_prog_parm("PS_VIEWER:", file_read) || find_prog_parm("#PS_VIEWER:", file_read)) {
	if (!found_ps_view_cmd) {
	  found_ps_view_cmd = true;
	  file_list.push_back(ps_view_cmd_line);
	}
      }
      
      // look for "LOG_FILE:"
      else if (find_prog_parm("LOG_FILE:", file_read) || find_prog_parm("#LOG_FILE:", file_read)) {
	if (!found_log_file) {
	  found_log_file = true;
	  file_list.push_back(Glib::filename_from_utf8(logfile_line));
	}
      }
      
      // add any residual lines to the list, except the terminating line
      else if (!find_prog_parm(terminating_line, file_read)) file_list.push_back(file_read);
    }
    
    if (found_name == false) file_list.push_back(Glib::locale_from_utf8(name_line));
    if (found_number == false) file_list.push_back(number_line);
    if (found_device == false) file_list.push_back(device_line);
    if (found_lock == false) file_list.push_back(Glib::filename_from_utf8(lock_line));
    if (found_class == false) file_list.push_back(class_line);
    if (found_page == false) file_list.push_back(page_line);
    if (found_res == false) file_list.push_back(res_line);
    if (found_rings == false) file_list.push_back(rings_line);
    if (found_dialmode = false) file_list.push_back(dialmode_line);
    if (found_init == false) file_list.push_back(init_line);
    if (found_reset == false) file_list.push_back(reset_line);
    if (found_capabilities == false) file_list.push_back(capabilities_line);
    if (found_parms == false) file_list.push_back(extra_parms_line);
    if (found_print_cmd == false) file_list.push_back(print_cmd_line);
    if (found_print_shrink == false) file_list.push_back(print_shrink_line);
    if (found_ps_view_cmd == false) file_list.push_back(ps_view_cmd_line);
    if (found_log_file == false) file_list.push_back(Glib::filename_from_utf8(logfile_line));
    
    // add the terminating line
    file_list.push_back(terminating_line);
    
    // now write out the new config file
    
    rcfile = prog_config.homedir + "/.";
    rcfile += RC_FILE;
    
    ofstream fileout(rcfile.c_str(), ios::out);
    if (!fileout) {
      Glib::ustring message("Can't open file");
      message += rcfile;
      message += '\n';
      write_error(message.c_str());
      return_val = false;
    }
    else {
      copy(file_list.begin(), file_list.end(),
	   ostream_iterator<std::string>(fileout, "\n"));
    }
  }
  return return_val;
}

void SettingsDialog::read_config(bool search_localfile) {
// search_localfile has a default value of true
// get rc file
  if (!get_rcfile_path(search_localfile)) {
    Glib::ustring message;
    if (search_localfile) {
      message = "Can't find or open file /etc/" RC_FILE ",\n"
	        "/usr/local/etc/" RC_FILE " or ";
      message += prog_config.homedir + "/." RC_FILE "\n";
    }
    else {
      message = "Can't find or open file /etc/" RC_FILE "\n"
	        "or /usr/local/etc/" RC_FILE;
      message += prog_config.homedir + "/." RC_FILE "\n";
    }
    write_error(message.c_str());
  }

  else {
// now extract settings from file

    ifstream filein(rcfile.c_str(), ios::in);
    if (!filein) {
      Glib::ustring message("Can't open file ");
      message += rcfile;
      write_error(message.c_str());
    }

    else {
      std::string file_read;
      Glib::ustring result;
      while (getline(filein, file_read)) {

	if (!file_read.empty() && file_read[0] != '#') { // valid line to check
	  // now check for other comment markers
	  string::size_type pos = file_read.find_first_of('#');
	  if (pos != string::npos) file_read.resize(pos); // truncate
	
	  // look for "NAME:"
	  if (get_prog_parm("NAME:", file_read, result)) {
	    identity_table.set_name(result);
	  }
	
	  // look for "NUMBER:"
	  else if (get_prog_parm("NUMBER:", file_read, result)) {
	    identity_table.set_number(result);
	  }
	
	  // look for "DEVICE:"
	  else if (get_prog_parm("DEVICE:", file_read, result)) {
	    modem_table.set_device(result);
	  }
	
	  // look for "LOCK:"
	  else if (get_prog_parm("LOCK:", file_read, result, Glib::filename_to_utf8)) {
	    modem_table.set_lock(result);
	  }

	  // look for "CLASS:"
	  else if (get_prog_parm("CLASS:", file_read, result)) {
	    modem_table.set_class(result);
	  }

	  // look for "PAGE:"
	  else if (get_prog_parm("PAGE:", file_read, result)) {
	    page_table.set_page(result);
	  }

	  // look for "RES:"
	  else if (get_prog_parm("RES:", file_read, result)) {
	    page_table.set_res(result);
	  }
	
	  // look for "RINGS:"
	  else if (get_prog_parm("RINGS:", file_read, result)) {
	    modem_table.set_rings(result);
	  }
	
	  // look for "DIALMODE:"
	  else if (get_prog_parm("DIALMODE:", file_read, result)) {
	    modem_table.set_dialmode(result);
	  }
	
	  // look for "INIT:"
	  else if (get_prog_parm("INIT:", file_read, result)) {
	    parms_table.set_init(result);
	  }
	
	  // look for "RESET:"
	  else if (get_prog_parm("RESET:", file_read, result)) {
	    parms_table.set_reset(result);
	  }

	  // look for "CAPABILITIES:"
	  else if (get_prog_parm("CAPABILITIES:", file_read, result)) {
	    modem_table.set_capabilities(result);
	  }

	  // look for "PARMS:"
	  else if (get_prog_parm("PARMS:", file_read, result)) {
	    parms_table.set_parms(result);
	  }

	  // look for "PRINT_CMD:"
	  else if (get_prog_parm("PRINT_CMD:", file_read, result)) {
	    print_table.set_command(result);
	  }

	  // look for "PRINT_SHRINK:"
	  else if (get_prog_parm("PRINT_SHRINK:", file_read, result)) {
	    print_table.set_shrink(result);
	  }

	  // look for "PS_VIEWER:"
	  else if (get_prog_parm("PS_VIEWER:", file_read, result)) {
	    view_table.set_ps_view_command(result);
	  }

	  // look for "LOG_FILE:"
	  else if (get_prog_parm("LOG_FILE:", file_read, result, Glib::filename_to_utf8)) {
	    logging_table.set_logfile(result);
	  }
	}
      }
    }
  }
}

void SettingsDialog::get_reset_settings_prompt(void) {
  Glib::ustring message(gettext("Enter settings from"));
  message += " /etc/" RC_FILE " ";
  message += gettext("or");
  message += "\n/usr/local/etc/" RC_FILE;
  message += gettext("?");
  PromptDialog* dialog_p = new PromptDialog(message, "Reset settings", standard_size, *this);
  if (!dialog_p) {
    cerr << "Memory allocation error in SettingsDialog::get_reset_settings_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->accepted.connect(SigC::slot(*this, &SettingsDialog::get_reset_settings));
  // there is no memory leak -- the memory will be deleted when PromptDialog closes
}

void SettingsDialog::get_reset_settings(void) {

  // clear all the existing settings
  identity_table.clear();
  modem_table.clear();
  parms_table.clear();
  print_table.clear();
  view_table.clear();
  logging_table.clear();
  page_table.clear();

  read_config(false); // read settings without searching for config file in home directory
}

bool SettingsDialog::get_prog_parm(const char* name, std::string& line, Glib::ustring& result,
				   Glib::ustring(*convert_func)(const std::string&)) {
// This function looks for a setting named `name' in the string `line'
// and returns the values stated after it in string `result'.  It returns
// `true' if the setting was found.  If there are trailing spaces or tabs,
// string `line' will be modified.  string `result' is only modified if
// the `name' setting is found.  Anything extracted from `line' will be
// converted (when placed into `result') to UTF-8 as maintained by
// Glib::ustring, using the function assigned to function pointer
// convert_func (you would normally use Glib::locale_to_utf8() or
// Glib::filename_to_utf8(), and there is a default inline method
// using Glib::locale_to_utf8()

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) {
    // erase any trailing space or tab
    while (line.find_last_of(" \t") == line.size() - 1) line.resize(line.size() - 1);
    if (line.size() > length) {
      // ignore any preceding space or tab from the setting value given
      string::size_type pos = line.find_first_not_of(" \t", length); // pos now is set to beginning of setting value
      if (pos != string::npos) result.assign(convert_func(line.substr(pos)));
    }
    return true;
  }
  return false;
}

bool SettingsDialog::find_prog_parm(const char* name, const std::string& line) {
// this function looks for a setting named `name' in the string `line'
// it returns `true' if the setting was found or false otherwise

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) return true;
  return false;
}

bool SettingsDialog::get_rcfile_path(bool search_localfile) {
// search_localfile has a default value of true

  bool found_rcfile = false;

  if (search_localfile) {
    rcfile = prog_config.homedir + "/.";
    rcfile += RC_FILE;

    if (!access(rcfile.c_str(), F_OK)) {
      found_rcfile = true;
      is_home_config = true;
    }
  }

  if (!found_rcfile) {

    rcfile = "/usr/local/etc/";
    rcfile += RC_FILE;
    if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;
  }

  if (!found_rcfile) {

    rcfile = "/etc/";
    rcfile += RC_FILE;
    if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;
  }
  if (!found_rcfile) rcfile = "";
  return found_rcfile;
}

void SettingsDialog::show_help(const Glib::ustring& message, const Glib::ustring& caption) {
  SettingsHelpDialog* dialog_p = new SettingsHelpDialog(standard_size,
							message, caption, *this);
  if (!dialog_p) {
    cerr << "Memory allocation error in IdentityDialog::show_help()" << endl;
    exit(MEM_ERROR);
  }
  // there is no memory leak
  // the SettingsHelpDialog object will delete its own memory when it is closed
}

void SettingsDialog::strip(Glib::ustring& setting) {

  // erase any trailing space or tab
  while (!setting.empty() && setting.find_last_of(" \t") == setting.size() - 1) {
    setting.resize(setting.size() - 1);
  }
  // erase any leading space or tab
  while (!setting.empty() && (setting[0] == ' ' || setting[0] == '\t')) {
    setting.erase(0, 1);
  }
}
