/* Copyright (C) 2005 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYING distributed with the source files.

*/

#ifndef GOBJ_HANDLE_H
#define GOBJ_HANDLE_H

/*
   This is a class which manages the reference count of g_objects.  It
   does not maintain its own reference count, but interfaces with that
   kept by the g_object system.  To that extent it is analogous to an
   intrusive smart pointer or handle.

   It can also be used to reference GTK+ widgets derived from
   GtkObject not embedded in a container (or from which ownership is
   to be claimed from a container), but if the object concerned has
   never been embedded in a container, g_object_ref() and
   gtk_object_sink() must be called on it (in that order) before the
   GobjHandle object managing it goes out of scope.

   The class has operator*() and operator->() dereferencing operators,
   so has normal smart pointer functionality, but as it is intended
   for use with the normal C g_object/pango/GTK+ interfaces, ordinary
   use would involve passing the handle to a function taking a pointer
   to the contained object by means of the operatorT*() type
   conversion operator (which returns the underlying pointer), or by
   explicitly calling the get() method to obtain the underlying
   pointer.
*/

#include <glib-object.h>

template <class T> class GobjHandle {

  T* obj_p;

  void unreference(void) {
    if (obj_p) g_object_unref(G_OBJECT(obj_p));
  }

  void reference(void) {
    if (obj_p) g_object_ref(G_OBJECT(obj_p));
  }

public:
  // constructor of first GobjHandle holding the referenced object
  explicit GobjHandle(T* ptr = 0) {
    obj_p = ptr;
  }

  // copy constructor
  GobjHandle(const GobjHandle& gobj_ptr) throw() {
    obj_p = gobj_ptr.obj_p;
    reference();
  }

  // copy assignment
  GobjHandle& operator=(const GobjHandle& gobj_ptr) throw() {

    // check whether we are already referencing this object -
    // if so make this a null op.  This will also deal with
    // self-assignment
    if (obj_p != gobj_ptr.obj_p) {

      // first unreference any object referenced by this handle
      unreference();

      // now inherit the GObject from the assigning handle
      // and reference it
      obj_p = gobj_ptr.obj_p;
      reference();
    }
    return *this;
  }

  T* get(void) const throw() {return obj_p;}
  T& operator*(void) const throw() {return *obj_p;}
  T* operator->(void) const throw() {return obj_p;}
  operator T*(void) const throw() {return obj_p;}

  // destructor
  ~GobjHandle(void) throw() {unreference();}
};

#endif
