/* Copyright (C) 2005 and 2009 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the src/utils sub-directory);
   if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/

#include <c++-gtk-utils/lib_defs.h>

#include <c++-gtk-utils/convert.h>
#include <c++-gtk-utils/shared_handle.h>
#include <c++-gtk-utils/gerror_handle.h>

namespace Cgu {

namespace Utf8 {

std::wstring uniwide_from_utf8(const std::string& input) {
  GError* error_p = 0;
  glong written = 0;
#if SIZEOF_WCHAR_T == 2
  ScopedHandle<gunichar2*, GFree> result_h(g_utf8_to_utf16(input.data(), input.size(), 0,
							   &written, &error_p)));
#else
  ScopedHandle<gunichar*, GFree> result_h(g_utf8_to_ucs4(input.data(), input.size(), 0,
							 &written, &error_p));
#endif
  if (error_p) {
    GerrorScopedHandle handle_h(error_p);
    throw ConversionError(error_p);
  }
  return std::wstring(reinterpret_cast<wchar_t*>(result_h.get()), written);
}

std::string uniwide_to_utf8(const std::wstring& input) {
  GError* error_p = 0;
  glong written = 0;
#if SIZEOF_WCHAR_T == 2
  GcharScopedHandle result_h(g_utf16_to_utf8(reinterpret_cast<const gunichar2*>(input.data()),
					     input.size(), 0,
					     &written, &error_p));
#else
  GcharScopedHandle result_h(g_ucs4_to_utf8(reinterpret_cast<const gunichar*>(input.data()),
					    input.size(), 0,
					    &written, &error_p));
#endif
  if (error_p) {
    GerrorScopedHandle handle_h(error_p);
    throw ConversionError(error_p);
  }
  return std::string(result_h.get(), written);
}

std::wstring wide_from_utf8(const std::string& input) {
  GError* error_p = 0;
  gsize written = 0;
  GcharScopedHandle result_h(g_convert(input.data(), input.size(),
				       "WCHAR_T", "UTF-8",
				        0, &written, &error_p));
  if (error_p) {
    GerrorScopedHandle handle_h(error_p);
    throw ConversionError(error_p);
  }
  return std::wstring(reinterpret_cast<wchar_t*>(result_h.get()), written/sizeof(wchar_t));
}

std::string wide_to_utf8(const std::wstring& input) {
  GError* error_p = 0;
  gsize written = 0;
  GcharScopedHandle result_h(g_convert(reinterpret_cast<const gchar*>(input.data()),
				       input.size() * sizeof(wchar_t),
				       "UTF-8", "WCHAR_T",
				       0, &written, &error_p));
  if (error_p) {
    GerrorScopedHandle handle_h(error_p);
    throw ConversionError(error_p);
  }
  return std::string(result_h.get(), written);
}

std::string filename_from_utf8(const std::string& input) {
  GError* error_p = 0;
  gsize written = 0;
  GcharScopedHandle result_h(g_filename_from_utf8(input.data(), input.size(), 0,
						  &written, &error_p));
    
  if (error_p) {
    GerrorScopedHandle handle_h(error_p);
    throw ConversionError(error_p);
  }
  return std::string(result_h.get(), written);
}

std::string filename_to_utf8(const std::string& input) {
  GError* error_p = 0;
  gsize written = 0;
  GcharScopedHandle result_h(g_filename_to_utf8(input.data(), input.size(), 0,
						&written, &error_p));
    
  if (error_p) {
    GerrorScopedHandle handle_h(error_p);
    throw ConversionError(error_p);
  }
  return std::string(result_h.get(), written);
}

std::string locale_from_utf8(const std::string& input) {
  GError* error_p = 0;
  gsize written = 0;
  GcharScopedHandle result_h(g_locale_from_utf8(input.data(), input.size(), 0,
						&written, &error_p));
    
  if (error_p) {
    GerrorScopedHandle handle_h(error_p);
    throw ConversionError(error_p);
  }
  return std::string(result_h.get(), written);
}

std::string locale_to_utf8(const std::string& input) {
  GError* error_p = 0;
  gsize written = 0;
  GcharScopedHandle result_h(g_locale_to_utf8(input.data(), input.size(), 0,
					      &written, &error_p));
    
  if (error_p) {
    GerrorScopedHandle handle_h(error_p);
    throw ConversionError(error_p);
  }
  return std::string(result_h.get(), written);
}

} // namespace Utf8

} // namespace Cgu
