/*
    Eiciel - GNOME editor of ACL file permissions.
    Copyright (C) 2004-2005 Roger Ferrer Ibez

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "eiciel_xattr_window.hpp"

EicielXAttrWindow::EicielXAttrWindow(EicielXAttrControler* contr)
	: controlador(contr), 
	afegirAtribut(Gtk::Stock::ADD),
	eliminarAtribut(Gtk::Stock::REMOVE)
{
	refLlistaXAttr = Gtk::ListStore::create(modelLlistaXAttr);

	controlador->finestra = this;

	set_border_width(4);
	vistaLlistaXAttr.set_size_request(100, 100);

	refLlistaXAttr->set_sort_column(modelLlistaXAttr.nomAtribut, Gtk::SORT_ASCENDING);

	vistaLlistaXAttr.set_reallocate_redraws();
	vistaLlistaXAttr.set_model(refLlistaXAttr);

	// Ara s'afegeixen les columnes i es vinculen amb el model
    vistaLlistaXAttr.append_column(_("Name"), modelLlistaXAttr.nomAtribut);

	// Aquesta columna requereix opcions especials
	Gtk::TreeViewColumn* col = vistaLlistaXAttr.get_column(0);
	Gtk::CellRenderer* cellRenderer = col->get_first_cell_renderer();
	Gtk::CellRendererText* cellRendererText = dynamic_cast<Gtk::CellRendererText*>(cellRenderer);
	cellRendererText->property_editable() = true;

	cellRendererText->signal_edited().connect(
			sigc::mem_fun(*this, &EicielXAttrWindow::nomAtributEditat)
			);

    vistaLlistaXAttr.append_column_editable(_("Value"), modelLlistaXAttr.valorAtribut);

	col = vistaLlistaXAttr.get_column(1);
	cellRenderer = col->get_first_cell_renderer();
	cellRendererText = dynamic_cast<Gtk::CellRendererText*>(cellRenderer);
	cellRendererText->property_editable() = true;

	cellRendererText->signal_edited().connect(
			sigc::mem_fun(*this, &EicielXAttrWindow::valorAtributEditat)
			);

	contenidorVistaLlistaXAttr.add(vistaLlistaXAttr);
	contenidorVistaLlistaXAttr.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

	pack_start(contenidorVistaLlistaXAttr, Gtk::PACK_EXPAND_WIDGET, 2);

	botoneraInferior.pack_start(afegirAtribut, Gtk::PACK_SHRINK, 2);
	botoneraInferior.pack_start(eliminarAtribut, Gtk::PACK_SHRINK, 2);

	pack_start(botoneraInferior, Gtk::PACK_SHRINK, 2);

	// Atributs de prova
	// Gtk::TreeModel::iterator iter = refLlistaXAttr->append();
	// Gtk::TreeModel::Row row;
	// row = *iter;

	// row[modelLlistaXAttr.nomAtribut] = "rev";
	// row[modelLlistaXAttr.valorAtribut] = "245";

	// iter = refLlistaXAttr->append();
	// row = *iter;

	// row[modelLlistaXAttr.nomAtribut] = "Confidencial";
	// row[modelLlistaXAttr.valorAtribut] = "No";

	// Vinculaci dels senyals
	Glib::RefPtr<Gtk::TreeSelection> referenciaSeleccioLlistaXAttr = vistaLlistaXAttr.get_selection();

	// Canvi de sellecci XAttr
	referenciaSeleccioLlistaXAttr->signal_changed().connect(
			sigc::mem_fun(*this, &EicielXAttrWindow::canviDeSeleccioXAttr)
			);

	// Boto eliminar
	eliminarAtribut.signal_clicked().connect(
			sigc::mem_fun(*this, &EicielXAttrWindow::eliminarAtributSeleccionat)
			);
	
	// Boto editar
	afegirAtribut.signal_clicked().connect(
			sigc::mem_fun(*this, &EicielXAttrWindow::afegirAtributSeleccionat)
			);

	activacio(false);
}

void EicielXAttrWindow::activacio(bool b)
{
	this->set_sensitive(b);
}

EicielXAttrWindow::~EicielXAttrWindow()
{
	delete controlador;
}

void EicielXAttrWindow::canviDeSeleccioXAttr()
{
	Glib::RefPtr<Gtk::TreeSelection> referenciaSeleccioLlista = vistaLlistaXAttr.get_selection();
	Gtk::TreeModel::iterator iter = referenciaSeleccioLlista->get_selected();
	if (!iter)
	{
		noHiHaSeleccioXAttr();
	}
	else
	{
		hiHaSeleccioXAttr();
	}
}

void EicielXAttrWindow::noHiHaSeleccioXAttr()
{
	eliminarAtribut.set_sensitive(false);
}

void EicielXAttrWindow::hiHaSeleccioXAttr()
{
	if (!this->nomesLectura)
	{
		eliminarAtribut.set_sensitive(true);
	}
}

void EicielXAttrWindow::nomAtributEditat(const Glib::ustring& path, const Glib::ustring& value)
{
	Gtk::TreeModel::iterator iter = refLlistaXAttr->get_iter(path);

	if (iter)
	{
		Gtk::TreeModel::Row row(*iter);

		// No pot estar buit
		if (value.empty())
		{
			return;
		}

		// No admetem repetits
		Gtk::TreeModel::Children children = refLlistaXAttr->children();

		for(Gtk::TreeModel::Children::iterator iter = children.begin(); 
				iter != children.end(); ++iter)
		{
			Gtk::TreeModel::Row irow(*iter);

			if (irow[modelLlistaXAttr.nomAtribut] == value)
			{
				return;
			}
		}

		try
		{
			controlador->modificarNomAtribut(row[modelLlistaXAttr.nomAtribut], value);
			row[modelLlistaXAttr.nomAtribut] = value;
		}
		catch (GestorXAttrException e)
		{
			Glib::ustring s = _("Could not rename attribute name: ") + e.getMessage();
			Gtk::MessageDialog renombrarXAttr(s, false,
					Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK);
			renombrarXAttr.run();
		}
	}
}

void EicielXAttrWindow::valorAtributEditat(const Glib::ustring& path, const Glib::ustring& value)
{
	Gtk::TreeModel::iterator iter = refLlistaXAttr->get_iter(path);

	if (iter)
	{
		Gtk::TreeModel::Row row(*iter);
		try
		{
			controlador->actualitzarValorAtribut(row[modelLlistaXAttr.nomAtribut], value);
			row[modelLlistaXAttr.valorAtribut] = value;
		}
		catch (GestorXAttrException e)
		{
			Glib::ustring s = _("Could not change attribute value: ") + e.getMessage();
			Gtk::MessageDialog editarXAttr(s, false,
					Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK);
			editarXAttr.run();
		}
	}
}

void EicielXAttrWindow::eliminarAtributSeleccionat()
{
	Glib::RefPtr<Gtk::TreeSelection> referenciaSeleccioLlista = vistaLlistaXAttr.get_selection();
	Gtk::TreeModel::iterator iter = referenciaSeleccioLlista->get_selected();

	if (iter)
	{
		Gtk::TreeModel::Row row(*iter);

		try
		{
			controlador->eliminarAtribut(row[modelLlistaXAttr.nomAtribut]);
			refLlistaXAttr->erase(iter);
		}
		catch (GestorXAttrException e)
		{
			Glib::ustring s = _("Could not remove attribute: ") + e.getMessage();
			Gtk::MessageDialog eliminarXAttr(s, false,
					Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK);
			eliminarXAttr.run();
		}
	}
}

void EicielXAttrWindow::afegirAtributSeleccionat()
{
	// Calculem el nom del nou atribut
	bool repetit;
	int numCops = 0;

	Glib::ustring nomNou;

	do {
		if (numCops == 0)
		{
			nomNou = _("New attribute");
		}
		else
		{
			char* numCopsStr = new char[20];

			snprintf(numCopsStr, 20, " (%d)", numCops);
			numCopsStr[19] = '\0';

			nomNou = _("New attribute");
			nomNou += numCopsStr;

			delete[] numCopsStr;
		}
		numCops++;

		Gtk::TreeModel::Children children = refLlistaXAttr->children();

		repetit = false;
		for(Gtk::TreeModel::Children::iterator iter = children.begin(); 
				iter != children.end(); ++iter)
		{
			Gtk::TreeModel::Row irow(*iter);

			if (irow[modelLlistaXAttr.nomAtribut] == nomNou)
			{
				repetit = true;
				break;
			}
		}
	}
	while (repetit);

	
	Gtk::TreeModel::iterator iter = refLlistaXAttr->append();
	Gtk::TreeModel::Row row;
	row = *iter;

	row[modelLlistaXAttr.nomAtribut] = nomNou;
	row[modelLlistaXAttr.valorAtribut] = _("New value");

	try
	{
		controlador->afegirAtribut(row[modelLlistaXAttr.nomAtribut],
				row[modelLlistaXAttr.valorAtribut]);

		Gtk::TreePath path = refLlistaXAttr->get_path(iter);
		Gtk::TreeViewColumn* col = vistaLlistaXAttr.get_column(0);

		vistaLlistaXAttr.set_cursor(path, *col, true);
	}
	catch (GestorXAttrException e)
	{
		refLlistaXAttr->erase(iter);
		Glib::ustring s = _("Could not add attribute: ") + e.getMessage();
		Gtk::MessageDialog afegirXAttr(s, false,
				Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK);
		afegirXAttr.run();
	}
}

void EicielXAttrWindow::ompleAtributs(GestorXAttr::atributs_t llista)
{
	GestorXAttr::atributs_t::iterator it;
	refLlistaXAttr->clear();

	Gtk::TreeModel::iterator iter;
	Gtk::TreeModel::Row row;
	
	for (it = llista.begin(); it != llista.end(); it++)
	{
		iter = refLlistaXAttr->append();
		row = *iter;
		
		row[modelLlistaXAttr.nomAtribut] = it->first;
		row[modelLlistaXAttr.valorAtribut] = it->second;
	}
}

void EicielXAttrWindow::establirNomesLectura(bool b)
{
	nomesLectura = b;

	afegirAtribut.set_sensitive(!b);
	eliminarAtribut.set_sensitive(!b);

	Gtk::TreeViewColumn* col = vistaLlistaXAttr.get_column(0);
	Gtk::CellRenderer* cellRenderer = col->get_first_cell_renderer();
	Gtk::CellRendererText* cellRendererText = dynamic_cast<Gtk::CellRendererText*>(cellRenderer);
	cellRendererText->property_editable() = !b;

	col = vistaLlistaXAttr.get_column(1);
	cellRenderer = col->get_first_cell_renderer();
	cellRendererText = dynamic_cast<Gtk::CellRendererText*>(cellRenderer);
	cellRendererText->property_editable() = !b;
}
