// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2008 Benoit Jacob <jacob@math.jussieu.fr>
// Copyright (C) 2008 Gael Guennebaud <g.gael@free.fr>
//
// Eigen is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 3 of the License, or (at your option) any later version.
//
// Alternatively, you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License or the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License and a copy of the GNU General Public License along with
// Eigen. If not, see <http://www.gnu.org/licenses/>.

#include "main.h"
#include <Eigen/LU>

template<typename MatrixType> void determinant(const MatrixType& m)
{
  /* this test covers the following files:
     Determinant.h
  */
  int size = m.rows();

  MatrixType m1(size, size), m2(size, size);
  m1.setRandom();
  m2.setRandom();
  typedef typename MatrixType::Scalar Scalar;
  Scalar x = ei_random<Scalar>();
  VERIFY(ei_isApprox(MatrixType::Identity(size, size).determinant(), Scalar(1)));
  VERIFY(ei_isApprox((m1*m2).determinant(), m1.determinant() * m2.determinant()));
  if(size==1) return;
  int i = ei_random<int>(0, size-1);
  int j;
  do {
    j = ei_random<int>(0, size-1);
  } while(j==i);
  m2 = m1;
  m2.row(i).swap(m2.row(j));
  VERIFY(ei_isApprox(m2.determinant(), -m1.determinant()));
  m2 = m1;
  m2.col(i).swap(m2.col(j));
  VERIFY(ei_isApprox(m2.determinant(), -m1.determinant()));
  VERIFY(ei_isApprox(m2.determinant(), m2.transpose().determinant()));
  VERIFY(ei_isApprox(ei_conj(m2.determinant()), m2.adjoint().determinant()));
  m2 = m1;
  m2.row(i) += x*m2.row(j);
  VERIFY(ei_isApprox(m2.determinant(), m1.determinant()));
  m2 = m1;
  m2.row(i) *= x;
  VERIFY(ei_isApprox(m2.determinant(), m1.determinant() * x));
}

void test_determinant()
{
  for(int i = 0; i < g_repeat; i++) {
    CALL_SUBTEST( determinant(Matrix<float, 1, 1>()) );
    CALL_SUBTEST( determinant(Matrix<double, 2, 2>()) );
    CALL_SUBTEST( determinant(Matrix<double, 3, 3>()) );
    CALL_SUBTEST( determinant(Matrix<double, 4, 4>()) );
    CALL_SUBTEST( determinant(Matrix<std::complex<double>, 10, 10>()) );
    CALL_SUBTEST( determinant(MatrixXd(20, 20)) );
  }
}
