%%%----------------------------------------------------------------------
%%% File    : mod_shared_roster_ldap.erl
%%% Authors : Alexey Shchepin <alexey@sevcom.net>
%%%           Realloc <realloc@realloc.spb.ru>
%%%           Marcin Owsiany <marcin@owsiany.pl>
%%% Purpose : LDAP shared roster management
%%% Created :  5 Mar 2005 by Alexey Shchepin <alexey@sevcom.net>
%%% Version : 0.5.3
%%%----------------------------------------------------------------------

%%%----------------------------------------------------------------------
%%% Contribution page: http://www.ejabberd.im/mod_shared_roster_ldap
%%% Project page:  https://alioth.debian.org/projects/ejabberd-msrl/
%%% Documentation: https://alioth.debian.org/docman/?group_id=100433
%%%----------------------------------------------------------------------

-module(mod_shared_roster_ldap).
-author('alexey@sevcom.net').

-behaviour(gen_server).
-behaviour(gen_mod).

%% gen_server callbacks
-export([
	 init/1,
	 handle_info/2,
	 handle_call/3,
	 handle_cast/2,
	 terminate/2,
	 code_change/3
	]).

-export([
	 start/2,
	 start_link/2,
	 stop/1,
	 get_user_roster/2,
	 get_subscription_lists/3,
	 get_jid_info/4,
	 process_item/2,
	 in_subscription/6,
	 out_subscription/4
	]).

% Technically internal functions that need to be accessible to
% mod_shared_roster_ldap_helpers:get_user_to_groups_map
-export([
         get_user_displayed_groups/1,
         get_group_name/2,
         get_group_users/2
        ]).

%% Test entry point
-export([main/0]).

%% This function is only referenced from this module, but the references have
%% to be fully qualified, so compiler thinks it's unused. Export to silence the
%% compiler warning.
-export([true_fun/2]).

-include("ejabberd.hrl").
-include("eldap/eldap.hrl").
-include("jlib.hrl").
-include("mod_roster.hrl").
-include("mod_shared_roster_ldap.hrl").

-define(LDAP_REQUEST_TIMEOUT, 10000).
-define(assert_state_field_equals(Expected, Parsed, Field),
        assert_equals(Expected#state.Field, Parsed#state.Field, atom_to_list(Field))).
-define(assert_roster_field_equals(Expected, Parsed, Field),
        assert_equals(Expected#roster.Field, Parsed#roster.Field, atom_to_list(Field))).

-record(gen_server_program, {sname, instr}).
-record(gen_server_instr, {req, resp}).

%% Unused callbacks.
handle_cast(_Request, State) ->
    {noreply, State}.
code_change(_OldVsn, State, _Extra) ->
    {ok, State}.
handle_info(_Info, State) ->
    {noreply, State}.
%% -----

start(Host, Opts) ->
    Proc = gen_mod:get_module_proc(Host, ?MODULE),
    ChildSpec = {
      Proc, {?MODULE, start_link, [Host, Opts]},
      permanent, 1000, worker, [?MODULE]
     },
    supervisor:start_child(ejabberd_sup, ChildSpec).

stop(Host) ->
    Proc = gen_mod:get_module_proc(Host, ?MODULE),
    gen_server:call(Proc, stop),
    supervisor:terminate_child(ejabberd_sup, Proc),
    supervisor:delete_child(ejabberd_sup, Proc).

start_link(Host, Opts) ->
    Proc = gen_mod:get_module_proc(Host, ?MODULE),
    gen_server:start_link({local, Proc}, ?MODULE, [Host, Opts], []).

terminate(_Reason, State) ->
    Host = State#state.host,
    ejabberd_hooks:delete(roster_get, Host,
			  ?MODULE, get_user_roster, 70),
    ejabberd_hooks:delete(roster_in_subscription, Host,
        		  ?MODULE, in_subscription, 30),
    ejabberd_hooks:delete(roster_out_subscription, Host,
        		  ?MODULE, out_subscription, 30),
    ejabberd_hooks:delete(roster_get_subscription_lists, Host,
        		  ?MODULE, get_subscription_lists, 70),
    ejabberd_hooks:delete(roster_get_jid_info, Host,
        		  ?MODULE, get_jid_info, 70),
    ejabberd_hooks:delete(roster_process_item, Host,
			  ?MODULE, process_item, 50).

init([Host, Opts]) ->
    State = parse_options(Host, Opts),
    ejabberd_hooks:add(roster_get, Host,
		       ?MODULE, get_user_roster, 70),
    ejabberd_hooks:add(roster_in_subscription, Host,
        	       ?MODULE, in_subscription, 30),
    ejabberd_hooks:add(roster_out_subscription, Host,
        	       ?MODULE, out_subscription, 30),
    ejabberd_hooks:add(roster_get_subscription_lists, Host,
		       ?MODULE, get_subscription_lists, 70),
    ejabberd_hooks:add(roster_get_jid_info, Host,
        	       ?MODULE, get_jid_info, 70),
    ejabberd_hooks:add(roster_process_item, Host,
        	       ?MODULE, process_item, 50),

    % ejabberd before 2.1.0 does not support the 6th argument to
    % eldap:start_link
    case application:get_key(ejabberd, vsn) of
	{ok, VSN} when VSN >= "2.1.0" ->
	    eldap:start_link(State#state.eldap_id,
			     State#state.servers,
			     State#state.port,
			     State#state.dn,
			     State#state.password,
			     State#state.tls_options);
	_ ->
	    eldap:start_link(State#state.eldap_id,
			     State#state.servers,
			     State#state.port,
			     State#state.dn,
			     State#state.password)
    end,
    {ok, State}.

get_user_roster(Items, US) ->
    {U, S} = US,
    SRUsers = mod_shared_roster_ldap_helpers:get_user_to_groups_map(US, true),
    %% If partially subscribed users are also in shared roster, show them as
    %% totally subscribed:
    {NewItems1, SRUsersRest} =
	lists:mapfoldl(
	  fun(Item, SRUsers1) ->
		  {_, _, {U1, S1, _}} = Item#roster.usj,
		  US1 = {U1, S1},
		  case dict:find(US1, SRUsers1) of
		      {ok, _GroupNames} ->
			  {Item#roster{subscription = both, ask = none},
			   dict:erase(US1, SRUsers1)};
		      error ->
			  {Item, SRUsers1}
		  end
	  end, SRUsers, Items),

    %% Export items in roster format:
    SRItems = [#roster{usj = {U, S, {U1, S1, ""}},
		       us = US,
		       jid = {U1, S1, ""},
		       name = get_user_name(U1,S1),
		       subscription = both,
		       ask = none,
		       groups = GroupNames} ||
		  {{U1, S1}, GroupNames} <- dict:to_list(SRUsersRest)],
    SRItems ++ NewItems1.

%% This function in use to rewrite the roster entries when moving or renaming
%% them in the user contact list.
process_item(RosterItem, _Host) ->
    USFrom = RosterItem#roster.us,
    {User,Server,_Resource} = RosterItem#roster.jid,
    USTo = {User,Server},
    Map = mod_shared_roster_ldap_helpers:get_user_to_groups_map(USFrom, false),
    case dict:find(USTo, Map) of
        error ->
            RosterItem;
        {ok, []} ->
            RosterItem;
        {ok, GroupNames} when RosterItem#roster.subscription == remove ->
            %% Roster item cannot be removed: We simply reset the original groups:
            RosterItem#roster{subscription = both, ask = none, groups=GroupNames};
        _ ->
            RosterItem#roster{subscription = both, ask = none}
    end.

get_subscription_lists({F, T}, User, Server) ->
    LUser = jlib:nodeprep(User),
    LServer = jlib:nameprep(Server),
    US = {LUser, LServer},
    DisplayedGroups = get_user_displayed_groups(US),
    SRUsers =
	lists:usort(
	  lists:flatmap(
	    fun(Group) ->
		    get_group_users(LServer, Group)
	    end, DisplayedGroups)),
    SRJIDs = [{U1, S1, ""} || {U1, S1} <- SRUsers],
    {lists:usort(SRJIDs ++ F), lists:usort(SRJIDs ++ T)}.

get_jid_info({Subscription, Groups}, User, Server, JID) ->
    LUser = jlib:nodeprep(User),
    LServer = jlib:nameprep(Server),
    US = {LUser, LServer},
    {U1, S1, _} = jlib:jid_tolower(JID),
    US1 = {U1, S1},
    SRUsers = mod_shared_roster_ldap_helpers:get_user_to_groups_map(US, false),
    case dict:find(US1, SRUsers) of
	{ok, GroupNames} ->
	    NewGroups = if
			    Groups == [] -> GroupNames;
			    true -> Groups
			end,
	    {both, NewGroups};
	error ->
	    {Subscription, Groups}
    end.

in_subscription(Acc, User, Server, JID, Type, _Reason) ->
    process_subscription(in, User, Server, JID, Type, Acc).

out_subscription(User, Server, JID, Type) ->
    process_subscription(out, User, Server, JID, Type, false).

process_subscription(Direction, User, Server, JID, _Type, Acc) ->
    LUser = jlib:nodeprep(User),
    LServer = jlib:nameprep(Server),
    US = {LUser, LServer},
    {U1, S1, _} = jlib:jid_tolower(jlib:jid_remove_resource(JID)),
    US1 = {U1, S1},
    DisplayedGroups = get_user_displayed_groups(US),
    SRUsers =
	lists:usort(
	  lists:flatmap(
	    fun(Group) ->
		    get_group_users(LServer, Group)
	    end, DisplayedGroups)),
    case lists:member(US1, SRUsers) of
	true ->
	    case Direction of
		in ->
		    {stop, false};
		out ->
		    stop
	    end;
	false ->
	    Acc
    end.

get_group_users(Host, Group) ->
    make_request(Host, {get_group_users, Group}, []).

get_group_name(Host, Group) ->
    make_request(Host, {get_group_name, Group}, Group).

get_user_displayed_groups({User, Host}) ->
    make_request(Host, {get_user_displayed_groups, User}, []).

get_user_name(User, Host) ->
    make_request(Host, {get_user_name, User}, []).


%%%-----------------------
%%% Internal functions.
%%%-----------------------
handle_call({get_user_displayed_groups, _User}, _From, State) ->
    GroupAttr = State#state.group_attr,
    Entries = mod_shared_roster_ldap_helpers:eldap_search(State, [State#state.rfilter], [GroupAttr]),
    Reply = lists:flatmap(
        fun(#eldap_entry{attributes = Attrs}) ->
            case Attrs of
                [{GroupAttr, ValuesList}] ->
                    ValuesList;
                _ ->
                    []
            end
        end,
        Entries
    ),
    {reply, lists:usort(Reply), State};

handle_call({get_group_name, Group}, _From, State) ->
    % Make sure the groups cache is fresh.
    NewState = with_fresh_cache(group, State, now_seconds()),
    % Look up the description of the given group
    Reply = case dict:find(Group, NewState#state.cached_groups) of
        {ok, #group_info{desc = GroupName}} when GroupName =/= undefined ->
            GroupName;
        _ ->
            Group
    end,
    {reply, Reply, NewState};

handle_call({get_group_users, Group}, _From, State) ->
    % Make sure the groups cache is fresh.
    NewState = with_fresh_cache(group, State, now_seconds()),
    % Look up the members of the given group
    Reply = case dict:find(Group, NewState#state.cached_groups) of
        {ok, #group_info{members = Members}} when Members =/= undefined ->
            Members;
        _ ->
            []
    end,
    {reply, Reply, NewState};

handle_call({get_user_name, User}, _From, State) ->
    % Make sure the username cache is fresh.
    NewState = with_fresh_cache(user, State, now_seconds()),
    % Look up the first description of the given user
    Reply = case dict:find(User, NewState#state.cached_users) of
        {ok, [UserName | _]} ->
            UserName;
        _ ->
            User
    end,
    {reply, Reply, NewState};

handle_call(stop, _From, State) ->
    {stop, normal, ok, State};

handle_call(_Request, _From, State) ->
    {reply, bad_request, State}.

%%%-----------------------
%%% Auxiliary functions.
%%%-----------------------
parse_options(Host, Opts) ->
    Eldap_ID = atom_to_list(gen_mod:get_module_proc(Host, ?MODULE)),
    LDAPServers = case gen_mod:get_opt(ldap_servers, Opts, undefined) of
		      undefined ->
			  ejabberd_config:get_local_option({ldap_servers, Host});
		      S -> S
		  end,
    LDAPEncrypt = case gen_mod:get_opt(ldap_encrypt, Opts, undefined) of
		      undefined ->
			  ejabberd_config:get_local_option({ldap_encrypt, Host});
		      E -> E
		  end,
    LDAPTLSVerify = case gen_mod:get_opt(ldap_tls_verify, Opts, undefined) of
			undefined ->
				ejabberd_config:get_local_option({ldap_tls_verify, Host});
			Verify -> Verify
		    end,
    LDAPPort = case gen_mod:get_opt(ldap_port, Opts, undefined) of
		   undefined ->
		       case ejabberd_config:get_local_option({ldap_port, Host}) of
			   undefined -> case LDAPEncrypt of
                               tls -> ?LDAPS_PORT;
                               starttls -> ?LDAP_PORT;
                               _ -> ?LDAP_PORT
                           end;
			   P -> P
		       end;
		   P -> P
	       end,
    LDAPBase = case gen_mod:get_opt(ldap_base, Opts, undefined) of
		   undefined ->
		       ejabberd_config:get_local_option({ldap_base, Host});
		   B -> B
	       end,
    GroupAttr = case gen_mod:get_opt(ldap_groupattr, Opts, undefined) of
		    undefined -> "cn";
		    GA -> GA
		end,
    GroupDesc = case gen_mod:get_opt(ldap_groupdesc, Opts, undefined) of
		    undefined -> GroupAttr;
		    GD -> GD
		end,
    UserDesc = case gen_mod:get_opt(ldap_userdesc, Opts, undefined) of
		   undefined -> "cn";
		   UD -> UD
	       end,
    UserUID = case gen_mod:get_opt(ldap_useruid, Opts, undefined) of
		   undefined -> "cn";
		   UU -> UU
	       end,
    UIDAttr = case gen_mod:get_opt(ldap_memberattr, Opts, undefined) of
		  undefined -> "memberUid";
		  UA -> UA
	      end,
    UIDAttrFormat = case gen_mod:get_opt(ldap_memberattr_format, Opts, undefined) of
			undefined -> "%u";
			UAF -> UAF
		    end,
    UIDAttrFormatRe = case gen_mod:get_opt(ldap_memberattr_format_re, Opts, undefined) of
			undefined -> "";
			UAFre -> case catch re:compile(UAFre) of
                            {ok, MP} ->
                                MP;
                            _ ->
                                ?ERROR_MSG("Invalid ldap_memberattr_format_re '~s' "++
                                           "or no RE support in this erlang version. "++
                                           "ldap_memberattr_format '~s' will be used "++
                                           "instead.", [UAFre, UIDAttrFormat]),
                                ""
                        end
		      end,
    AuthCheck = case gen_mod:get_opt(ldap_auth_check, Opts, undefined) of
			undefined -> on;
			AC -> AC
		end,
    RootDN = case gen_mod:get_opt(ldap_rootdn, Opts, undefined) of
		 undefined ->
		     case ejabberd_config:get_local_option({ldap_rootdn, Host}) of
			 undefined -> "";
			 RDN -> RDN
		     end;
		 RDN -> RDN
	     end,
    Password = case gen_mod:get_opt(ldap_password, Opts, undefined) of
		   undefined ->
		       case ejabberd_config:get_local_option({ldap_password, Host}) of
			   undefined -> "";
			   Pass -> Pass
		       end;
		   Pass -> Pass
	       end,
    UserValidity = case gen_mod:get_opt(ldap_user_cache_validity, Opts, undefined) of
		   undefined ->
		       case ejabberd_config:get_local_option({ldap_user_cache_validity, Host}) of
			   undefined -> 5*60;
			   USeconds -> USeconds
		       end;
		   USeconds -> USeconds
	       end,
    GroupValidity = case gen_mod:get_opt(ldap_group_cache_validity, Opts, undefined) of
		   undefined ->
		       case ejabberd_config:get_local_option({ldap_group_cache_validity, Host}) of
			   undefined -> 5*60;
			   GSeconds -> GSeconds
		       end;
		   GSeconds -> GSeconds
	       end,
    ConfigFilter = case gen_mod:get_opt(ldap_filter, Opts, undefined) of
		       undefined ->
			   ejabberd_config:get_local_option({ldap_filter, Host});
		       F ->
			   F
		   end,

    ConfigUserFilter = case gen_mod:get_opt(ldap_ufilter, Opts, undefined) of
			   undefined ->
                               ejabberd_config:get_local_option({ldap_ufilter, Host});
			   UF -> UF
		       end,

    ConfigGroupFilter = case gen_mod:get_opt(ldap_gfilter, Opts, undefined) of
			    undefined ->
			        ejabberd_config:get_local_option({ldap_gfilter, Host});
                            GF -> GF
                        end,

    RosterFilter = case gen_mod:get_opt(ldap_rfilter, Opts, undefined) of
		       undefined ->
			   ejabberd_config:get_local_option({ldap_rfilter, Host});
		       RF ->
			   RF
		   end,

    SubFilter = "(&("++UIDAttr++"="++UIDAttrFormat++")("++GroupAttr++"=%g))",
    UserSubFilter = case ConfigUserFilter of
                        undefined -> eldap_filter:do_sub(SubFilter, [{"%g", "*"}]);
                        "" -> eldap_filter:do_sub(SubFilter, [{"%g", "*"}]);
                        UString -> UString
                    end,
    GroupSubFilter = case ConfigGroupFilter of
			 undefined -> eldap_filter:do_sub(SubFilter, [{"%u", "*"}]);
		         "" -> eldap_filter:do_sub(SubFilter, [{"%u", "*"}]);
		         GString -> GString
                     end,
    Filter = case ConfigFilter of
		 undefined -> SubFilter;
		 "" -> SubFilter;
		 _ -> "(&" ++ SubFilter ++ ConfigFilter ++ ")"
	     end,
    UserFilter = case ConfigFilter of
		     undefined -> UserSubFilter;
		     "" -> UserSubFilter;
		     _ -> "(&" ++ UserSubFilter ++ ConfigFilter ++ ")"
		 end,
    GroupFilter = case ConfigFilter of
		      undefined -> GroupSubFilter;
		      "" -> GroupSubFilter;
		      _ -> "(&" ++ GroupSubFilter ++ ConfigFilter ++ ")"
		  end,
    #state{
		    host = Host,
		    eldap_id = Eldap_ID,
		    servers = LDAPServers,
		    port = LDAPPort,
		    tls_options = [{encrypt, LDAPEncrypt},
				   {tls_verify, LDAPTLSVerify}],
		    dn = RootDN,
		    base = LDAPBase,
		    password = Password,
		    uid = UIDAttr,
		    group_attr = GroupAttr,
		    group_desc = GroupDesc,
		    user_desc = UserDesc,
		    user_uid = UserUID,
		    uid_format = UIDAttrFormat,
		    uid_format_re = UIDAttrFormatRe,
		    filter = Filter,
		    ufilter = UserFilter,
		    rfilter = RosterFilter,
		    gfilter = GroupFilter,
		    auth_check = AuthCheck,
		    user_cache_validity = UserValidity,
		    group_cache_validity = GroupValidity
		   }.

%% Getting User ID part by regex pattern
%%
get_user_part_re(String, Pattern) ->
    case catch re:run(String, Pattern) of
	{match, Captured} ->
		{First, Len} = lists:nth(2,Captured),
		Result = string:sub_string(String, First+1, First+Len),
		{ok,Result};
	_ -> {error, badmatch}
    end.

now_seconds() ->
    {Msec, Sec, _} = mod_shared_roster_ldap_helpers:now(),
    Msec * 1000000 + Sec.

make_request(Host, Request, Fallback) ->
    Proc = gen_mod:get_module_proc(Host, ?MODULE),
    case catch gen_server:call(Proc, Request, ?LDAP_REQUEST_TIMEOUT) of
	{'EXIT', Reason} ->
            error_logger:error_msg("~p crashed: ~p~n", [Proc, Reason]),
	    Fallback;
	Result ->
	    Result
    end.


% If necessary, retrieve entries and cache them in State.
with_fresh_cache(CacheType, State, Now) ->
    {FreshnessFunction, Filter, FPattern, SearchArgs} = case CacheType of
        user ->
            {fun mod_shared_roster_ldap_helpers:users_cache_fresh/2,
             State#state.ufilter, "%u",
             [State#state.user_desc, State#state.user_uid]};
        group ->
            {fun mod_shared_roster_ldap_helpers:groups_cache_fresh/2,
             State#state.gfilter, "%g",
             [State#state.group_attr, State#state.group_desc, State#state.uid]}
    end,
    case FreshnessFunction(State, Now) of
        fresh -> State;
        stale ->
	    % TODO: since eldap_search never reports errors, a potentially
	    % broken result may be cached and propagated to clients for up to
	    % State#state.*_cache_validity seconds.
            Retrieved = mod_shared_roster_ldap_helpers:eldap_search(State,
                [eldap_filter:do_sub(Filter, [{FPattern, "*"}])],
                SearchArgs),
            % Convert the list to a dictionary for lookup efficiency.
            case CacheType of
                user ->
                    RetrievedDict = mod_shared_roster_ldap_helpers:user_entries_to_dict(
                        State#state.user_uid, State#state.user_desc, Retrieved
                    ),
                    State#state{cached_users = RetrievedDict, cached_users_timestamp = Now};
                group ->
                    Extractor = case State#state.uid_format_re of
                        "" -> fun(UID) -> catch eldap_utils:get_user_part(UID, State#state.uid_format) end;
                        _  -> fun(UID) -> catch get_user_part_re(UID, State#state.uid_format_re) end
                    end,
                    Checker = case State#state.auth_check of
                        on -> fun ejabberd_auth:is_user_exists/2;
                        off -> fun ?MODULE:true_fun/2
                    end,
                    RetrievedDict = mod_shared_roster_ldap_helpers:group_entries_to_dict(
			State#state.group_attr, State#state.group_desc, State#state.uid,
                        State#state.host, Extractor, Checker, Retrieved
                    ),
                    State#state{cached_groups = RetrievedDict, cached_groups_timestamp = Now}
            end
    end.

true_fun(_, _) ->
    true.

%%%-----------------------
%%% Unit tests.
%%%
%%% They do not use any standard erlang unit testing library, for the simple
%%% reason that none was available in the erlang release I used when writing them.
%%%
%%% The tests require gen_server_mock and mock modules. They are not otherwise
%%% required for using the module with ejabberd.
%%%
%%% Run with:
%%% erlc -W mod_shared_roster_ldap.erl mod_shared_roster_ldap_helpers.erl && \
%%% erl -noshell -s mod_shared_roster_ldap main -s init stop -kernel error_logger silent
%%%
%%% (Omit the part after "stop" if you want to see mock module messages.)
%%%-----------------------

% Run all tests, catch exceptions and pause to give time for messages to appear
% on standard output.
main() ->
    io:format("Starting tests.~n"),
    try run_tests() of
        _ -> io:format("Tests executed successfully.~n")
    catch
        Type:Error ->
          io:format("Tests failed: at ~p~n", [erlang:get_stacktrace()]),
          receive after 2000 -> true end,
          exit({Type, Error})
    end.

% Run all tests
run_tests() ->
    prepare_stringprep(),
    % Config parsing tests
    % - module defaults are taken if top-level settings are undefined.
    run_confparse_test(fun parse_no_options_test/0),
    % - default LDAP port selection logic,
    run_confparse_test(fun parse_port_tests/0),
    % - module config overrides top-level config values, whether the latter are defined or not.
    run_confparse_test(fun parse_all_standard_options_test/0, [{1, undefined}]),
    % - however the gfilter option needs to stay undefined on top level.
    run_confparse_test(fun parse_all_standard_options_test/0, [
        {1, "something defined"},
        {[{ldap_ufilter, "foobar"}], undefined},
        {[{ldap_gfilter, "foobar"}], undefined}
    ]),
    % - g/ufilter taken from options when defined
    run_confparse_test(fun parse_gfilter_options_test/0),
    run_confparse_test(fun parse_ufilter_options_test/0),
    % Tests for get_user_roster()
    empty_roster_get_user_roster_test(),
    nonempty_roster_get_user_roster_test(),
    % Tests for get_jid_info()
    empty_roster_and_no_old_groups_get_jid_info_test(),
    empty_roster_with_old_groups_get_jid_info_test(),
    nonmatching_roster_with_old_groups_get_jid_info_test(),
    matching_roster_with_old_groups_get_jid_info_test(),
    matching_roster_without_old_groups_get_jid_info_test(),
    % Tests for with_fresh_cache
    uncached_with_fresh_username_cache_test(),
    fresh_with_fresh_username_cache_test(),
    stale_with_fresh_username_cache_test(),
    uncached_with_fresh_groups_cache_test(),
    fresh_with_fresh_groups_cache_test(),
    stale_with_fresh_groups_cache_test(),
    % Tests for get_user_name
    ok_get_user_name_test(),
    nomatch_get_user_name_test(),
    failing_get_user_name_test(),
    % Cache handling tests
    user_cache_expiry_test(),
    group_cache_expiry_test(),
    % Tests for get_user_to_groups_map
    no_groups_get_user_to_groups_map_test(),
    empty_group_get_user_to_groups_map_test(),
    two_nonempty_groups_get_user_to_groups_map_test(),
    % Tests for query functions
    % - get_group_name()
    searchfail_get_group_name_test(),
    bad_attr_get_group_name_test(),
    empty_get_group_name_test(),
    ok_get_group_name_test(),
    % - get_group_users()
    ok_get_group_users_test(),
    searchfail_get_group_users_test(),
    bad_attr_get_group_users_test(),
    empty_get_group_users_test(),
    % Test for query helper
    ok_eldap_search_test(),
    empty_eldap_search_test(),
    search_fail_eldap_search_test(),
    parse_fail_eldap_search_test(),
    % Tests for mod_shared_roster_ldap_helpers:user_entries_to_dict
    ok_user_entries_to_dict_test(),
    % Tests for mod_shared_roster_ldap_helpers:group_entries_to_dict
    ok_group_entries_to_dict_test(),
    % Inserted 'ok' to make function non-tail-recursive, to make it easier to
    % see where we get a failure.
    ok.

%
% Tests for mod_shared_roster_ldap_helpers:group_entries_to_dict
%
ok_group_entries_to_dict_test() ->
    % empty list
    assert_equals(mod_shared_roster_ldap_helpers:group_entries_to_dict("groupid_attr", "groupdesc_attr", "groupmember_attr", "h", fun(UID) -> {ok, UID} end, fun(_UID, _Host) -> true end, []), dict:new()),
    Entries = [
        #eldap_entry{attributes = [
	    {"groupid_attr", ["g1"]},
	    {"groupdesc_attr", ["Group 1"]},
            {"groupmember_attr", ["bla=u1", "bla=u2", "bla=u3 long UID to trigger jlib:nodeprep to return atom `error` follows"
            "0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789"
            "0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789"
            "0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789"
            "0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789"
            "0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789"
            "0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789"
            "0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789"
            "0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789"
            "0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789"
            "0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789"
            ]}
	]},
	#eldap_entry{attributes = [
	    {"groupid_attr", ["g2"]},
	    {"groupdesc_attr", ["Group 2"]},
            {"groupmember_attr", ["bla=u1", "bla=u2"]}
	]},
	#eldap_entry{attributes = [
            {"groupmember_attr", ["bla=u4", "bla=u3", "bla=u2", "bla=uX"]},
	    {"groupdesc_attr", ["Group 2.5"]},
	    {"groupid_attr", ["g2"]}
	]}
    ],
    % First value only from each entry
    D1 = dict:store("g1", #group_info{desc="Group 1", members=[{"u1","h"}, {"u2","h"}]}, dict:new()),
    D2 = dict:store("g2", #group_info{desc="Group 2", members=[{"u1","h"}, {"u2","h"}, {"u3","h"}, {"u4","h"}]}, D1),
    assert_dicts_equal(
        mod_shared_roster_ldap_helpers:group_entries_to_dict("groupid_attr", "groupdesc_attr", "groupmember_attr", "h", fun(UID) -> eldap_utils:get_user_part(UID, "bla=%u") end, fun(UID,_Host) -> case UID of "ux" -> false; _ -> true end end, Entries),
        D2
    ),
    ok.

%
% Tests for mod_shared_roster_ldap_helpers:user_entries_to_dict
%
ok_user_entries_to_dict_test() ->
    % empty list
    assert_equals(mod_shared_roster_ldap_helpers:user_entries_to_dict(whatever, something, []), dict:new()),
    Entries = [
        #eldap_entry{attributes = [
	    {"useruid_attr", ["blah"]},
	    {"userdesc_attr", ["NotCzeslaw", "NotCzeslaw2"]}
	]},
	#eldap_entry{attributes = [
	    {"useruid_attr", ["aUser"]},
	    {"userdesc_attr", ["Czeslaw"]}
	]},
	#eldap_entry{attributes = [
	    {"useruid_attr", ["auser"]},
	    {"userdesc_attr", ["Czeslaw Duplicate"]}
	]}
    ],
    % First value only from each entry
    D1 = dict:append_list("blah", ["NotCzeslaw"], dict:new()),
    D2 = dict:append_list("auser", ["Czeslaw", "Czeslaw Duplicate"], D1),
    assert_dicts_equal(
        mod_shared_roster_ldap_helpers:user_entries_to_dict("useruid_attr", "userdesc_attr", Entries),
        D2
    ),
    ok.

%
% Tests for with_fresh_cache
%

uncached_with_fresh_username_cache_test() ->
    run_with_fresh_cache_test(user, uncached, stale, on),
    ok.

fresh_with_fresh_username_cache_test() ->
    run_with_fresh_cache_test(user, cached, fresh, on),
    ok.

stale_with_fresh_username_cache_test() ->
    run_with_fresh_cache_test(user, cached, stale, on),
    ok.

uncached_with_fresh_groups_cache_test() ->
    run_with_fresh_cache_test(group, uncached, stale, on),
    run_with_fresh_cache_test(group, uncached, stale, off),
    ok.

fresh_with_fresh_groups_cache_test() ->
    run_with_fresh_cache_test(group, cached, fresh, on),
    run_with_fresh_cache_test(group, cached, fresh, off),
    ok.

stale_with_fresh_groups_cache_test() ->
    run_with_fresh_cache_test(group, cached, stale, on),
    run_with_fresh_cache_test(group, cached, stale, off),
    ok.


%
% Tests for get_user_name
%

% There are matching results.
ok_get_user_name_test() ->
    UsersDict = dict:append("auser", "Czeslaw", dict:new()),
    ExpectedReply = "Czeslaw",
    run_get_user_name_test(UsersDict, ExpectedReply),
    ok.

% There are results, but none matches.
nomatch_get_user_name_test() ->
    UsersDict0 = dict:append("anotheruser", "Czeslaw", dict:new()),
    UsersDict = dict:append("anotheruser", "or even something else", UsersDict0),
    ExpectedReply = "auser",
    run_get_user_name_test(UsersDict, ExpectedReply),
    ok.

% No results.
failing_get_user_name_test() ->
    UsersDict = dict:new(),
    ExpectedReply = "auser",
    run_get_user_name_test(UsersDict, ExpectedReply),
    ok.

% Whether user cache expiry works properly
user_cache_expiry_test() ->
    Now = 100,
    S = #state{user_cache_validity = 60},
    assert_equals(
        mod_shared_roster_ldap_helpers:users_cache_fresh(S, Now),
        stale, "Default state record has a stale cache, just in case."),
    assert_equals(
        mod_shared_roster_ldap_helpers:users_cache_fresh(
            S#state{cached_users_timestamp = Now}, Now),
        stale, "State record with recent timestamp but no cache is stale."),
    assert_equals(
        mod_shared_roster_ldap_helpers:users_cache_fresh(
            S#state{cached_users_timestamp = Now - 61, cached_users = dict:new()}, Now),
        stale, "State record with cache older than 60 seconds is stale."),
    assert_equals(
        mod_shared_roster_ldap_helpers:users_cache_fresh(
            S#state{cached_users_timestamp = Now - 1, cached_users = dict:new()}, Now),
        fresh, "State record with cache younger than 60 seconds is fresh."),
    assert_equals(
        mod_shared_roster_ldap_helpers:users_cache_fresh(
            S#state{cached_users_timestamp = Now, cached_users = dict:new()}, Now),
        fresh, "State record with cache younger than 60 seconds is fresh."),
    ok.

% Whether group cache expiry works properly
group_cache_expiry_test() ->
    Now = 100,
    S = #state{group_cache_validity = 60},
    assert_equals(
        mod_shared_roster_ldap_helpers:groups_cache_fresh(S, Now),
        stale, "Default state record has a stale cache, just in case."),
    assert_equals(
        mod_shared_roster_ldap_helpers:groups_cache_fresh(
            S#state{cached_groups_timestamp = Now}, Now),
        stale, "State record with recent timestamp but no cache is stale."),
    assert_equals(
        mod_shared_roster_ldap_helpers:groups_cache_fresh(
            S#state{cached_groups_timestamp = Now - 61, cached_groups = dict:new()}, Now),
        stale, "State record with cache older than 60 seconds is stale."),
    assert_equals(
        mod_shared_roster_ldap_helpers:groups_cache_fresh(
            S#state{cached_groups_timestamp = Now - 1, cached_groups = dict:new()}, Now),
        fresh, "State record with cache younger than 60 seconds is fresh."),
    assert_equals(
        mod_shared_roster_ldap_helpers:groups_cache_fresh(
            S#state{cached_groups_timestamp = Now, cached_groups = dict:new()}, Now),
        fresh, "State record with cache younger than 60 seconds is fresh."),
    ok.

%
% Tests for get_user_to_groups_map
%

% When there are no groups, roster should be empty.
no_groups_get_user_to_groups_map_test() ->
    Server = "foobar",
    assert_equals(
      dict:new(),
      test_gen_server_action(
        mod_shared_roster_ldap_helpers, get_user_to_groups_map, [{"czesio", Server}, true],
        Server, [
        #gen_server_instr{req={get_user_displayed_groups, "czesio"}, resp=[]}
      ])
    ),
    ok.

% When all groups are empty, roster should be empty.
empty_group_get_user_to_groups_map_test() ->
    Server = "foobar",
    assert_equals(
      dict:new(),
      test_gen_server_action(
        mod_shared_roster_ldap_helpers, get_user_to_groups_map, [{"czesio", Server}, true],
        Server, [
          #gen_server_instr{req={get_user_displayed_groups, "czesio"}, resp=["empty_group"]},
          #gen_server_instr{req={get_group_name, "empty_group"}, resp="An Empty Group"},
          #gen_server_instr{req={get_group_users,"empty_group"}, resp=[]}
      ])
    ),
    ok.

% A more complex example with two groups, overlapping members and a member from
% another domain. Tested twice - once when skipping and not.
two_nonempty_groups_get_user_to_groups_map_test() ->
    Server = "foobar",
    lists:foreach(fun(Skip) ->
      Dict = test_gen_server_action(
        mod_shared_roster_ldap_helpers, get_user_to_groups_map, [{"czesio", Server}, Skip],
        Server, [
          #gen_server_instr{req={get_user_displayed_groups, "czesio"}, resp=["g1", "g2"]},
          #gen_server_instr{req={get_group_name, "g1"}, resp="Group One"},
          #gen_server_instr{req={get_group_users,"g1"}, resp=[
              {"czesio", Server},
              {"u1", Server},
              {"u2", Server}
          ]},
          #gen_server_instr{req={get_group_name, "g2"}, resp="Group Two"},
          #gen_server_instr{req={get_group_users,"g2"}, resp=[
              {"u2", Server},
              {"u3", "another-domain"}
          ]}
        ]
      ),
      UserCount = dict:fold(fun(_,_,I) -> I+1 end, 0, Dict),
      case Skip of
          false ->
              assert_equals(4, UserCount, "User count"),
              assert_equals({ok, ["Group One"]}, dict:find({"czesio", Server}, Dict));
          true ->
              assert_equals(3, UserCount, "User count"),
              assert_equals(error, dict:find({"czesio", Server}, Dict))
      end,
      assert_equals({ok, ["Group One"]}, dict:find({"u1", Server}, Dict)),
      assert_equals({ok, ["Group One", "Group Two"]}, dict:find({"u2", Server}, Dict)),
      assert_equals({ok, ["Group Two"]}, dict:find({"u3", "another-domain"}, Dict))
    end, [true, false]),
    ok.

%
% Tests for query functions
%

% Test successful eldap_search
ok_eldap_search_test() ->
    Reply = #eldap_search_result{entries = #eldap_entry{
        object_name = "blah",
        attributes = [{"attr1", ["val1", "val2"]}]}
    },
    run_eldap_search_test(
        {okParse, Reply},
        Reply#eldap_search_result.entries
    ),
    ok.

% Test empty eldap_search
empty_eldap_search_test() ->
    Reply = #eldap_search_result{entries = []},
    run_eldap_search_test(
        {okParse, Reply},
        []
    ),
    ok.

% Test search failing eldap_search
search_fail_eldap_search_test() ->
    Reply = error,
    run_eldap_search_test(
        {okParse, Reply},
        []
    ),
    ok.

% Test parse failing eldap_search
parse_fail_eldap_search_test() ->
    run_eldap_search_test(
        {failParse},
        []
    ),
    ok.

%
% Tests for get_group_name/2
%

% There are no results.
searchfail_get_group_name_test() ->
    GroupsDict = dict:new(),
    ExpectedReply = "agroup",
    run_get_group_name_test(GroupsDict, ExpectedReply),
    ok.

% There are results, but none matches.
bad_attr_get_group_name_test() ->
    GroupsDict = dict:from_list([{"anothergroup", #group_info{desc = "Some group"}},
                                 {"yetanothergroup", #group_info{desc = "Other group"}}]),
    ExpectedReply = "agroup",
    run_get_group_name_test(GroupsDict, ExpectedReply),
    ok.

% There are results, and one matches, but has no data.
empty_get_group_name_test() ->
    GroupsDict = dict:from_list([{"anothergroup", #group_info{desc = "Some group"}},
                                 {"agroup", #group_info{}}]),
    ExpectedReply = "agroup",
    run_get_group_name_test(GroupsDict, ExpectedReply),
    ok.

% There are results, and one matches.
ok_get_group_name_test() ->
    GroupsDict = dict:from_list([{"anothergroup", #group_info{desc = "Some group"}},
                                 {"agroup", #group_info{desc = "Group desc"}}]),
    ExpectedReply = "Group desc",
    run_get_group_name_test(GroupsDict, ExpectedReply),
    ok.

%
% Tests for get_group_users/2
%

% There are no results.
searchfail_get_group_users_test() ->
    GroupsDict = dict:new(),
    ExpectedReply = [],
    run_get_group_users_test(GroupsDict, ExpectedReply),
    ok.

% There are results, but none matches.
bad_attr_get_group_users_test() ->
    GroupsDict = dict:from_list([
        {"anothergroup", #group_info{members = [{"u1","h1"}, {"u2","h2"}]}},
        {"yetanothergroup", #group_info{members = [{"u3","h3"}, {"u4","h4"}]}}
    ]),
    ExpectedReply = [],
    run_get_group_users_test(GroupsDict, ExpectedReply),
    ok.

% There are results, and one matches, but has not data.
empty_get_group_users_test() ->
    GroupsDict = dict:from_list([
        {"anothergroup", #group_info{members = [{"u1","h1"}, {"u2","h2"}]}},
        {"agroup", #group_info{}}
    ]),
    ExpectedReply = [],
    run_get_group_users_test(GroupsDict, ExpectedReply),
    ok.

% There are results, and one matches.
ok_get_group_users_test() ->
    GroupsDict = dict:from_list([
        {"anothergroup", #group_info{members = [{"u1","h1"}, {"u2","h2"}]}},
        {"agroup", #group_info{members = [{"u3","h3"}, {"u4","h4"}]}}
    ]),
    ExpectedReply = [{"u3","h3"}, {"u4","h4"}],
    run_get_group_users_test(GroupsDict, ExpectedReply),
    ok.

%
% Tests for get_jid_info()
%

% If roster is empty, subscription is unmodified and groups are left empty if
% they were empty.
empty_roster_and_no_old_groups_get_jid_info_test() ->
    Server = "foobar",
    assert_equals(
        {old_subscription, []},
        run_get_jid_info_test(
          [{old_subscription, []}, "czesio", Server, {"contact", Server, "resource"}],
          {"czesio", Server}, dict:new()
        )
    ),
    ok.

% If roster is empty, subscription is unmodified and groups are left as they were untouched.
empty_roster_with_old_groups_get_jid_info_test() ->
    Server = "foobar",
    assert_equals(
        {old_subscription, ["Old G1", "Old G2"]},
        run_get_jid_info_test(
          [{old_subscription, ["Old G1", "Old G2"]}, "czesio", Server, {"contact", Server, "resource"}],
          {"czesio", Server}, dict:new()
        )
    ),
    ok.

% If roster is not empty, but JID is not in it, subscription is unmodified and
% groups are left as they were untouched.
nonmatching_roster_with_old_groups_get_jid_info_test() ->
    Server = "foobar",
    assert_equals(
        {old_subscription, ["Old G1", "Old G2"]},
        run_get_jid_info_test(
          [{old_subscription, ["Old G1", "Old G2"]}, "czesio", Server, {"contact", Server, "resource"}],
          {"czesio", Server}, dict:from_list([
              {{"contact_another", Server}, ["New G1"]}
           ])
        )
    ),
    ok.

% If roster is not empty, and JID is in it, subscription is set to both.
% If groups were passed in, they are left as they were untouched.
matching_roster_with_old_groups_get_jid_info_test() ->
    Server = "foobar",
    assert_equals(
        {both, ["Old G1", "Old G2"]},
        run_get_jid_info_test(
          [{old_subscription, ["Old G1", "Old G2"]}, "czesio", Server, {"contact", Server, "resource"}],
          {"czesio", Server}, dict:from_list([
              {{"contact", Server}, ["New G1"]}
          ])
        )
    ),
    ok.

% If roster is not empty, and JID is in it, subscription is set to both.
% If groups were not passed in, they are set to those in roster.
matching_roster_without_old_groups_get_jid_info_test() ->
    Server = "foobar",
    assert_equals(
        {both, ["New G1", "New G2"]},
        run_get_jid_info_test(
          [{old_subscription, []}, "czesio", Server, {"contact", Server, "resource"}],
          {"czesio", Server}, dict:from_list([
              {{"contact", Server}, ["New G1", "New G2"]}
          ])
        )
    ),
    ok.

%
% Tests for get_user_roster()
%

% When the user to groups map is empty, roster should be empty.
empty_roster_get_user_roster_test() ->
    Server = "foobar",
    M = mock:new(),
    mock:strict(M, mod_shared_roster_ldap_helpers, get_user_to_groups_map,
        [{"czesio", Server}, true], {return, dict:new()}
    ),
    mock:replay(M),
    assert_equals([], get_user_roster([], {"czesio", Server})),
    mock:verify(M),
    ok.

% A more complex example with two groups, overlapping members and a member from
% another domain.
nonempty_roster_get_user_roster_test() ->
    Server = "foobar",
    Xuser1 = make_roster_item({"czesio", Server}, {"u1", Server}, "OldU1"),
    Xuser2 = make_roster_item({"czesio", Server}, {"ux", Server}, "NotShared"),
    Roster = [
        Xuser1#roster{subscription = from, ask = to},
        Xuser2#roster{subscription = from, ask = to}
    ],
    M = mock:new(),
    Map = dict:from_list([
        {{"u1", Server}, ["Group One"]},
        {{"u2", Server}, ["Group One", "Group Two"]},
        {{"u3", "another-domain"}, ["Group Two"]}
    ]),
    mock:strict(M, mod_shared_roster_ldap_helpers, get_user_to_groups_map,
        [{"czesio", Server}, true], {return, Map}
    ),
    mock:replay(M),
    [E1, E2, E3, E4] = test_gen_server_action(?MODULE, get_user_roster,
      [Roster, {"czesio", Server}],
      [
        #gen_server_program{sname=Server, instr=[
          #gen_server_instr{req={get_user_name, "u2"}, resp="U2"}
        ]},
        #gen_server_program{sname="another-domain", instr=[
          #gen_server_instr{req={get_user_name, "u3"}, resp="U3"}
        ]}
      ]
    ),
    mock:verify(M),
    % contact in another domain
    assert_rosters_equal(E1, E1#roster{
        jid = {"u3", "another-domain", ""}, name = "U3",
        groups = ["Group Two"],
        ask = none, subscription = both
    }),
    % contact in 2 groups
    assert_rosters_equal(E2, E2#roster{
        jid = {"u2", Server, ""}, name = "U2",
        groups = ["Group One", "Group Two"],
        ask = none, subscription = both
    }),
    % contact on shared and private roster, ignored shared group membership,
    % but expanded "subscription" and deleted "ask"
    assert_rosters_equal(E3, E3#roster{
        jid = {"u1", Server, ""}, name = "OldU1",
        groups = [],
        ask = none, subscription = both
    }),
    % contact just on private roster, untouched
    assert_rosters_equal(E4, E4#roster{
        jid = {"ux", Server, ""}, name = "NotShared",
        groups = [],
        ask = to, subscription = from
    }),
    ok.

%
% Config parsing tests.
%

% Test parsing of all pre-0.3.0 options, and some newer ones.
parse_all_standard_options_test() ->
    Parsed = parse_options("foobar", [
      {ldap_servers, ["ldap.server"]},
      {ldap_port, 1234},
      {ldap_base, "the=base"},
      {ldap_groupattr, "somegroupAttr"},
      {ldap_groupdesc, "othergroupAttr"},
      {ldap_userdesc, "displayName"},
      {ldap_useruid, "cn"},
      {ldap_memberattr, "someMemberAttr"},
      {ldap_memberattr_format, "%u,memberFormat"},
      {ldap_rootdn, "the=root"},
      {ldap_password, "thepass"},
      {ldap_filter, "(x=y)"},
      {ldap_rfilter, "(objectClass=posixGroup)"},
      {ldap_user_cache_validity, 120},
      {ldap_group_cache_validity, 220},
      {ldap_encrypt, tls},
      {ldap_tls_verify, soft}
     ]
    ),
    Expected = Parsed#state{
      host = "foobar",
      eldap_id = "mod_shared_roster_ldap_foobar",
      servers = ["ldap.server"],
      port = 1234,
      base = "the=base",
      group_attr = "somegroupAttr",
      group_desc = "othergroupAttr",
      user_desc = "displayName",
      user_uid = "cn",
      uid = "someMemberAttr",
      uid_format = "%u,memberFormat",
      dn = "the=root",
      password = "thepass",
      filter = "(&(&(someMemberAttr=%u,memberFormat)(somegroupAttr=%g))(x=y))",
      ufilter = "(&(&(someMemberAttr=%u,memberFormat)(somegroupAttr=*))(x=y))",
      gfilter = "(&(&(someMemberAttr=*,memberFormat)(somegroupAttr=%g))(x=y))",
      rfilter = "(objectClass=posixGroup)",
      user_cache_validity = 120,
      group_cache_validity = 220,
      tls_options = [{encrypt, tls}, {tls_verify, soft}]
    },
    assert_states_equal(Expected, Parsed),
    ok.

% Test parsing of ldap_gfilter.
parse_gfilter_options_test() ->
    % When ldap_gfilter is specified, it should be used.
    Parsed = parse_options("foobar", [
      {ldap_groupattr, "somegroupAttr"},
      {ldap_memberattr, "someMemberAttr"},
      {ldap_memberattr_format, "%u,memberFormat"},
      {ldap_filter, "(x=y)"},
      {ldap_gfilter, "(letgroup=%g)"},
      {ldap_rfilter, "(objectClass=posixGroup)"}
     ]
    ),
    Expected = Parsed#state{
      group_attr = "somegroupAttr",
      group_desc = "somegroupAttr",
      uid = "someMemberAttr",
      uid_format = "%u,memberFormat",
      filter = "(&(&(someMemberAttr=%u,memberFormat)(somegroupAttr=%g))(x=y))",
      ufilter = "(&(&(someMemberAttr=%u,memberFormat)(somegroupAttr=*))(x=y))",
      gfilter = "(&(letgroup=%g)(x=y))",
      rfilter = "(objectClass=posixGroup)"
    },
    assert_states_equal(Expected, Parsed),
    ok.

% Test parsing of ldap_ufilter.
parse_ufilter_options_test() ->
    % When ldap_ufilter is specified, it should be used.
    Parsed = parse_options("foobar", [
      {ldap_groupattr, "somegroupAttr"},
      {ldap_memberattr, "someMemberAttr"},
      {ldap_memberattr_format, "%u,memberFormat"},
      {ldap_filter, "(x=y)"},
      {ldap_ufilter, "(letuser=%u)"},
      {ldap_rfilter, "(objectClass=posixGroup)"}
     ]
    ),
    Expected = Parsed#state{
      group_attr = "somegroupAttr",
      uid = "someMemberAttr",
      uid_format = "%u,memberFormat",
      filter = "(&(&(someMemberAttr=%u,memberFormat)(somegroupAttr=%g))(x=y))",
      gfilter = "(&(&(someMemberAttr=*,memberFormat)(somegroupAttr=%g))(x=y))",
      ufilter = "(&(letuser=%u)(x=y))",
      rfilter = "(objectClass=posixGroup)"
    },
    assert_states_equal(Expected, Parsed),
    ok.

% Test port selection based on TLS options.
parse_port_tests() ->
    parse_tls_port_test(),
    parse_starttls_port_test(),
    ok.

parse_tls_port_test() ->
    Parsed = parse_options("ahostname", [
      {ldap_encrypt, tls}
    ]),
    Expected = Parsed#state{
        port = ?LDAPS_PORT
    },
    assert_states_equal(Expected, Parsed),
    ok.

parse_starttls_port_test() ->
    Parsed = parse_options("ahostname", [
      {ldap_encrypt, starttls}
    ]),
    Expected = Parsed#state{
        port = ?LDAP_PORT
    },
    assert_states_equal(Expected, Parsed),
    ok.

% Test default values.
parse_no_options_test() ->
    Parsed = parse_options("ahostname", []),
    Expected = Parsed#state{
        uid = "memberUid",
        group_attr = "cn",
        group_desc = "cn",
        user_desc = "cn",
        user_uid = "cn",
        uid_format = "%u",
        filter = "(&(memberUid=%u)(cn=%g))",
        ufilter = "(&(memberUid=%u)(cn=*))",
        gfilter = "(&(memberUid=*)(cn=%g))",
        rfilter = undefined,
        user_cache_validity = 5*60,
        group_cache_validity = 5*60,
        port = ?LDAP_PORT,
        tls_options = [{encrypt, undefined}, {tls_verify, undefined}]
    },
    assert_states_equal(Expected, Parsed),
    ok.

%
% Helper functions
%

% Run with_fresh_cache after having mocked away
% mod_shared_roster_ldap_helpers:eldap_search to return desired entries.
%
% This function can test both user and group cache, but only ONE AT A TIME.
run_with_fresh_cache_test(CacheType, Cached, Freshness, Check) ->
    State0 = #state{
        ufilter = "(a ufilter with %u)",
        gfilter = "(a gfilter with %g)",
        host = "ahost",
        eldap_id = an_eldap_id,
        base = a_base,
        user_desc = "userdesc_attr",
        group_desc = "groupdesc_attr",
        group_attr = "groupid_attr",
        user_uid = "useruid_attr",
        uid = uid_attr
    },
    Dict = case Cached of
        cached -> entries_dict;
        uncached -> undefined
    end,
    {State, CacheFreshnessFunction, Filter, SearchArgs, DictFunction, DictArgs} = case CacheType of
        user ->
            {State0#state{cached_users = Dict},
             users_cache_fresh,
             "(a ufilter with *)",
             ["userdesc_attr", "useruid_attr"],
             user_entries_to_dict,
             ["useruid_attr", "userdesc_attr", entries_list]};
        group ->
            CheckFun = case Check of
              on -> ignore;
              off-> fun ?MODULE:true_fun/2
            end,
            {State0#state{cached_groups = Dict, auth_check = Check},
             groups_cache_fresh,
             "(a gfilter with *)",
             ["groupid_attr", "groupdesc_attr", uid_attr],
             group_entries_to_dict,
             ["groupid_attr", "groupdesc_attr", uid_attr, "ahost", ignore, CheckFun, entries_list]}
    end,
    % Mock away mod_shared_roster_ldap_helpers:
    M = mock:new(),
    % First, cache is checked for freshness.
    mock:strict(M, mod_shared_roster_ldap_helpers, CacheFreshnessFunction,
        [State, 1234567],
        {return, Freshness}
    ),
    % eldap_search returns what we want, but only when cache is empty or stale.
    if
        (Cached == uncached) or (Freshness == stale) ->
            mock:strict(M, mod_shared_roster_ldap_helpers, eldap_search,
                [State, [Filter], SearchArgs],
                {return, entries_list}
            ),
            N = length(DictArgs),
            mock:strict(M, mod_shared_roster_ldap_helpers, DictFunction,
                % check if all args (apart from atoms 'ignore') are as expected
                N, fun(Called) -> understanding_checker(Called, DictArgs) end,
                {return, entries_dict}
            );
        true -> no_search
    end,
    mock:replay(M),
    ChangedState = case CacheType of
        user ->
            State#state{cached_users = entries_dict, cached_users_timestamp = 1234567};
        group ->
            State#state{cached_groups = entries_dict, cached_groups_timestamp = 1234567}
    end,
    ExpectedState = case Cached of
        uncached -> ChangedState;
	cached ->
            case Freshness of
                fresh -> State;
                stale -> ChangedState
            end
    end,
    assert_states_equal(with_fresh_cache(CacheType, State, 1234567), ExpectedState),
    mock:verify(M),
    ok.

% Compares an expected list of arguments to the list really provided. However
% ignores arguments which match up with atom 'ignore'.
% This is useful for ignoring local funs.
understanding_checker(Called, Expected) ->
    N = length(Expected),
    lists:all(
        fun(I) ->
            {Real, Hoped} = {lists:nth(I, Called), lists:nth(I, Expected)},
            case Hoped of
                ignore -> true;
                Real -> true; % match
                _ -> io:format("Unexpected arg ~p is not what we hoped: ~p~n", [Real, Hoped]), false
            end
        end,
        lists:seq(1, N)
    ).

run_get_user_name_test(UsersDict, ExpectedReply) ->
    StateIn = StateOut = #state{cached_users = UsersDict},
    % Mock away mod_shared_roster_ldap_helpers:
    M = mock:new(),
    mock:stub(M, mod_shared_roster_ldap_helpers, now,
        [],
        {return, {0,1234567,0}}
    ),
    % Pretend cache is fresh to avoid other mocking.
    mock:strict(M, mod_shared_roster_ldap_helpers, users_cache_fresh,
        [StateIn, 1234567],
        {return, fresh}
    ),
    mock:replay(M),
    {reply, Reply, ReturnedState} = handle_call({get_user_name, "auser"}, apid, StateIn),
    assert_states_equal(ReturnedState, StateOut),
    assert_equals(Reply, ExpectedReply),
    mock:verify(M),
    ok.

run_get_group_name_test(GroupsDict, ExpectedReply) ->
    StateIn = StateOut = #state{cached_groups = GroupsDict},
    % Mock away mod_shared_roster_ldap_helpers:
    M = mock:new(),
    mock:stub(M, mod_shared_roster_ldap_helpers, now,
        [],
        {return, {0,1234567,0}}
    ),
    % Pretend cache is fresh to avoid other mocking.
    mock:strict(M, mod_shared_roster_ldap_helpers, groups_cache_fresh,
        [StateIn, 1234567],
        {return, fresh}
    ),
    mock:replay(M),
    {reply, Reply, ReturnedState} = handle_call({get_group_name, "agroup"}, apid, StateIn),
    assert_states_equal(ReturnedState, StateOut),
    assert_equals(Reply, ExpectedReply),
    mock:verify(M),
    ok.

run_get_group_users_test(GroupsDict, ExpectedReply) ->
    StateIn = StateOut = #state{cached_groups = GroupsDict},
    % Mock away mod_shared_roster_ldap_helpers:
    M = mock:new(),
    mock:stub(M, mod_shared_roster_ldap_helpers, now,
        [],
        {return, {0,1234567,0}}
    ),
    % Pretend cache is fresh to avoid other mocking.
    mock:strict(M, mod_shared_roster_ldap_helpers, groups_cache_fresh,
        [StateIn, 1234567],
        {return, fresh}
    ),
    mock:replay(M),
    {reply, Reply, ReturnedState} = handle_call({get_group_users, "agroup"}, apid, StateIn),
    assert_states_equal(ReturnedState, StateOut),
    assert_equals(Reply, ExpectedReply),
    mock:verify(M),
    ok.

% Helper to run get_jid_info after having mocked get_user_to_groups_map.
run_get_jid_info_test(Args, ExpectedUS, ReturnMap) ->
    M = mock:new(),
    mock:strict(M, mod_shared_roster_ldap_helpers, get_user_to_groups_map,
        [ExpectedUS, false], {return, ReturnMap}
    ),
    mock:replay(M),
    Ret = apply(?MODULE, get_jid_info, Args),
    mock:verify(M),
    Ret.

% Run mod_shared_roster_ldap_helpers:eldap_search after having mocked away
% eldap_filter:parse and eldap:search to return desired results.
run_eldap_search_test(Config, ExpectedReply) ->
    State = #state{
        eldap_id = an_eldap_id,
        base = a_base
    },
    % Mock away eldap_filter and eldap so that their functions return the
    % values we want.
    {MFilter, MSearch} = {mock:new(), mock:new()},
    % Prepare args for recording and replaying
    ExpectFilterArgs = ["(a gfilter)", [{"%g", "agroup"}]],
    ExpectSearchArgs = [an_eldap_id, [
        {base, a_base},
        {filter, afilter},
        {attributes, ["groupdesc_attr"]}
    ]],
    % The Config list should start with the least specific instructions first.
    case Config of
        {okParse, SearchResult} ->
            mock:strict(MFilter, eldap_filter, parse, ExpectFilterArgs,
                        {return, {ok, afilter}}
            ),
            mock:strict(MSearch, eldap, search, ExpectSearchArgs,
                        {return, SearchResult}
            );
        {failParse} ->
            mock:strict(MFilter, eldap_filter, parse, ExpectFilterArgs,
                        {return, error}
            )
            % MSearch mock should expect no call after a failed parse
    end,
    mock:replay(MFilter),
    mock:replay(MSearch),
    assert_equals(
	mod_shared_roster_ldap_helpers:eldap_search(
            State,
            ExpectFilterArgs,
            ["groupdesc_attr"]
        ),
        ExpectedReply
    ),
    mock:verify(MFilter),
    mock:verify(MSearch),
    ok.

% Return a new roster record for given Owner and destination with name
make_roster_item(Owner, Dest, Name) ->
    {U, S} = Owner,
    {U1, S1} = Dest,
    #roster{usj = {U, S, {U1, S1, ""}},
        us = Owner,
        jid = {U1, S1, ""},
        name = Name,
        subscription = both}.

% Set up mock gen_server processes for the _current_ module, which will take
% calls as described in Servers list, then run the function in the _specified_
% module, and verify the mocks.
% Each element in Servers is a server name and a list of {Request, Response}
% pairs which describe what Response to return given some Request input
% parameters.
test_gen_server_action(Module, Function, Args, Servers) ->
    % For each {Server, [{Request, Response},...]} generate
    % a registered mock process
    Tags = lists:foldl(fun(#gen_server_program{sname=Server, instr=ReqRes}, A) ->
        Tag = gen_mod:get_module_proc(Server, ?MODULE),
        {ok, Mock} = gen_server_mock:new(),
        lists:foreach(fun(#gen_server_instr{req=Request, resp=Response}) ->
            gen_server_mock:expect_call(Mock, fun(Req, _From, State) when Req =:= Request -> {ok, Response, State} end)
          end, ReqRes),
        register(Tag, Mock),
        [Tag|A]
      end, [], Servers),
    % Run the action
    Ret = apply(Module, Function, Args),
    % Verify all mocks and unregister them
    lists:foreach(fun(Tag) ->
        case whereis(Tag) of
            undefined -> erlang:error(mock_disappeared);
            Mock ->
                gen_server_mock:assert_expectations(Mock),
                unregister(Tag)
        end
      end, Tags),
    % Return what the action returned
    Ret.

% Just like test_gen_server_action/4, but for a single server.
test_gen_server_action(Module, Function, Args, Server, ReqRes) ->
    test_gen_server_action(Module, Function, Args,
        [#gen_server_program{sname=Server, instr=ReqRes}]
    ).

% Run provided TestFun after having mocked away
% ejabberd_config:get_local_option() to return values according to the provided
% Config
run_confparse_test(TestFun, Config) ->
    % Mock away ejabberd_config so that get_local_option returns the value we want.
    M = mock:new(),
    % The Config list should start with the least specific instructions first.
    lists:foreach(fun({Args, Ret}) ->
        mock:stub(M, ejabberd_config, get_local_option, Args, {return, Ret})
      end,
      Config
    ),
    mock:replay(M),
    TestFun(),
    mock:verify(M),
    ok.

% For this to work, ejabberd.beam has to be in the path, and its get_so_path()
% must point at a directory which contains stringprep_drv.so
% Normally get_so_path() will return ".", but in case it does not you can force
% it by setting EJABBERD_SO_PATH=. in the environment.
prepare_stringprep() ->
    stringprep_sup:start_link(),
    ok.

% Run TestFun after having mocked away ejabberd_config:get_local_option() to
% return undefined for any argument.
run_confparse_test(TestFun) -> run_confparse_test(TestFun, [{1, undefined}]).

%
% Assertion utility functions
%

% Match One against Another and print a useful message when they don't, before
% throwing an exception.
assert_equals(One, Another, Message) ->
    try One = Another of
        _ -> ok
    catch
        _:_ ->
            io:format("~s:~n ~p~n !=~n ~p~n", [Message, One, Another]),
            erlang:error(badmatch)
    end.

% Call assert_equals/3 with a boilerplate message.
assert_equals(One, Another) ->
    assert_equals(One, Another, "Comparison failed.").

% Compare state records and fail with a useful message if they don't match.
assert_states_equal(Expected, Parsed) ->
    % Apparently it's impossible to iterate over record's fields, so we have to
    % do the following.
    ?assert_state_field_equals(Expected, Parsed, host),
    ?assert_state_field_equals(Expected, Parsed, eldap_id),
    ?assert_state_field_equals(Expected, Parsed, servers),
    ?assert_state_field_equals(Expected, Parsed, port),
    ?assert_state_field_equals(Expected, Parsed, dn),
    ?assert_state_field_equals(Expected, Parsed, base),
    ?assert_state_field_equals(Expected, Parsed, password),
    ?assert_state_field_equals(Expected, Parsed, uid),
    ?assert_state_field_equals(Expected, Parsed, group_attr),
    ?assert_state_field_equals(Expected, Parsed, group_desc),
    ?assert_state_field_equals(Expected, Parsed, user_desc),
    ?assert_state_field_equals(Expected, Parsed, user_uid),
    ?assert_state_field_equals(Expected, Parsed, uid_format),
    ?assert_state_field_equals(Expected, Parsed, filter),
    ?assert_state_field_equals(Expected, Parsed, ufilter),
    ?assert_state_field_equals(Expected, Parsed, rfilter),
    ?assert_state_field_equals(Expected, Parsed, gfilter),
    ?assert_state_field_equals(Expected, Parsed, cached_users),
    ?assert_state_field_equals(Expected, Parsed, cached_users_timestamp),
    ?assert_state_field_equals(Expected, Parsed, user_cache_validity),
    ?assert_state_field_equals(Expected, Parsed, cached_groups),
    ?assert_state_field_equals(Expected, Parsed, cached_groups_timestamp),
    ?assert_state_field_equals(Expected, Parsed, group_cache_validity),
    ?assert_state_field_equals(Expected, Parsed, tls_options),
    % Just in case the record definition gets extended behind our back, do a
    % catch-all comparison for safety. It's message won't be detailed, though.
    assert_equals(Expected, Parsed, "Fallback record comparison failed. Please update assert_states_equal()."),
    ok.

% Compare roster records and fail with a useful message if they don't match.
assert_rosters_equal(Expected, Parsed) ->
    % Apparently it's impossible to iterate over record's fields, so we have to
    % do the following.
    ?assert_roster_field_equals(Expected, Parsed, usj),
    ?assert_roster_field_equals(Expected, Parsed, us),
    ?assert_roster_field_equals(Expected, Parsed, jid),
    ?assert_roster_field_equals(Expected, Parsed, name),
    ?assert_roster_field_equals(Expected, Parsed, subscription),
    ?assert_roster_field_equals(Expected, Parsed, ask),
    ?assert_roster_field_equals(Expected, Parsed, groups),
    ?assert_roster_field_equals(Expected, Parsed, askmessage),
    ?assert_roster_field_equals(Expected, Parsed, xs),
    % Just in case the record definition gets extended behind our back, do a
    % catch-all comparison for safety. It's message won't be detailed, though.
    assert_equals(Expected, Parsed, "Fallback roster comparison failed. Please update assert_roster_equal()."),
    ok.

% Compare two dictionaries and fail with a useful message if they don't match.
assert_dicts_equal(One, Two) ->
    assert_equals(lists:sort(dict:to_list(One)), lists:sort(dict:to_list(Two))),
    ok.

