# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Philippe Normand <philippe@fluendo.com>
#          Guido Amoruso <guidonte@fluendo.com>


from elisa.plugins.pigment.widgets.progressbar import ProgressBar
from elisa.plugins.pigment.widgets.box import HBox
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.widgets.const import *
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.theme import Theme
import pgm


class Indicator(Widget):

    def __init__(self, empty_indicator, full_indicator):
        super(Indicator, self).__init__()

        self._empty_img = Image()
        self._empty_img.set_from_image(empty_indicator)
        self._empty_img.bg_a = 0
        self._empty_img.visible = False
        self.add(self._empty_img, forward_signals=False)
        self._empty_img.position = (0, 0, 0)
        self._empty_img.size = (1.0, 1.0)

        self._full_img = Image()
        self._full_img.bg_a = 0
        self._full_img.set_from_image(full_indicator)
        self._full_img.visible = False
        self.add(self._full_img, forward_signals=False)
        self._full_img.position = (0, 0, 0)
        self._full_img.size = (1.0, 1.0)

    def fill(self):
        self._full_img.visible = True
        self._empty_img.visible = False

    def empty(self):
        self._full_img.visible = False
        self._empty_img.visible = True

class IndicatorProgressBar(ProgressBar):

    def __init__(self):
        super(IndicatorProgressBar, self).__init__()

        self.background = HBox()
        self.background.spacing = 0.01

        # we don't need the cursor widget: we want to stick to the ProgressBar
        # API, but not actually using much of its implementation
        self.remove(self.cursor)

        self._full_indicator_resource = None
        self._full_resource_img = Image()
        self.add(self._full_resource_img)
        self._full_resource_img.bg_a = 0
        self._full_resource_img.visible = False

        self._empty_indicator_resource = None
        self._empty_resource_img = Image()
        self.add(self._empty_resource_img)
        self._empty_resource_img.bg_a = 0
        self._empty_resource_img.visible = False

    def _set_empty_indicator_resource(self, indicator_resource):
        self._empty_indicator_resource = indicator_resource

        if indicator_resource:
            theme = Theme.get_default()
            image_file = theme.get_resource(indicator_resource)
            self._empty_resource_img.set_from_file(image_file)

    def _get_empty_indicator_resource(self):
        return self._empty_indicator_resource

    empty_indicator_resource = property(fget=_get_empty_indicator_resource,
                                        fset=_set_empty_indicator_resource)

    def _set_full_indicator_resource(self, indicator_resource):
        self._full_indicator_resource = indicator_resource
        if indicator_resource:
            theme = Theme.get_default()
            image_file = theme.get_resource(indicator_resource)
            self._full_resource_img.set_from_file(image_file)

    def _get_full_indicator_resource(self):
        return self._full_indicator_resource

    full_indicator_resource = property(fget=_get_full_indicator_resource,
                                       fset=_set_full_indicator_resource)

    def _layout(self):
        if self._empty_resource_img and self._full_resource_img:
            if len(self.background) > self.items_number:
                # too many indicators, let's remove some
                while len(self.background) > self.items_number:
                    self.background.pop()
            else:
                # fill background with indicators
                idx = self._current_index
                while len(self.background) < self.items_number:
                    indicator = Indicator(self._empty_resource_img,
                                          self._full_resource_img)
                    self.background.pack_start(indicator, True)
                    if idx < self._current_index:
                        indicator.fill()
                    else:
                        indicator.empty()
                    indicator.visible = True
                    idx += 1

    def items_number__set(self, nb):
        if nb <= 0:
            nb = 1

        self._items_number = nb

        if self._current_index > nb:
            self._current_index = nb

        self._layout()

    items_number = property(ProgressBar.items_number__get, items_number__set)

    def do_index_changed(self, current, previous):
        if current > previous:
            for indicator in self.background._children[previous:current]:
                indicator.fill()
        if current < previous:
            for indicator in self.background._children[current:previous+1]:
                indicator.empty()

    # Not using the cursor widget
    def _update_cursor_position(self):
        pass

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        widget = cls(*args)
        widget.visible = True

        return widget

if __name__ == "__main__":
    import logging
    import sys

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    try:
        empty_indicator_resource = sys.argv[1]
        full_indicator_resource = sys.argv[2]
    except IndexError:
        empty_indicator_resource = 'elisa.plugins.poblesec.file'
        full_indicator_resource = 'elisa.plugins.poblesec.folder'

    progressbar = IndicatorProgressBar.demo()
    progressbar.empty_indicator_resource = empty_indicator_resource
    progressbar.full_indicator_resource = full_indicator_resource
    progressbar.size = (1, 0.2)
    progressbar.position = (0, 0, 0)
    progressbar.items_number = 7
    progressbar.current_index = 3
    try:
        __IPYTHON__
    except NameError:
        pgm.main()
