# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import gobject
from cPickle import dump, load, UnpicklingError
from elisa.plugins.pigment.widgets import const
from elisa.plugins.pigment.widgets.style import Style

from elisa.core import log

try:
    from elisa.extern.coherence import inotify
except ImportError:
    inotify = None

from pkg_resources import resource_filename
import pkg_resources

import logging
import copy
import pprint
import os

from cssutils import CSSParser

_DEFAULT_THEME = None

STYLES_FILENAME = 'styles.conf'
RESOURCES_FILENAME = 'resources.conf'


class ResourceNotExisting(Exception):
    pass


def makeTheme(module_name, styles_conf, resources_conf):
    theme = Theme(module_name=module_name, styles_conf=styles_conf,
                  resources_conf=resources_conf)
    return theme


class Theme(gobject.GObject):
    """
    A theme object, that adds styles to widgets properties and to stock
    resources.

    A basic theme will be built from the default configuration files, providing
    necessary style information for the widgets: without that, widgets won't
    work.

    @ivar widget_styles:   the styles for the widgets, for each state
    @type widget_styles:   dict of states to
                           L{elisa.plugins.pigment.widgets.Style}s
    @ivar stock_resources: the map of resource names to file paths
    @type stock_resources: a dict strings to strings
    @ivar fallback_themes: a dictionary of plugin names to Themes, caching
                           information necessary to do the right fallback
                           for missing resources
    @type fallback_themes: dictionary of strings to
                           L{elisa.core.components.theme.Theme}
    """

    __gsignals__ = {
        'styles-updated': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                          ()),
        'resources-updated': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                              ()),
    }

    default_theme = None

    _cache = {}

    def __init__(self, module_name=None, styles_conf=None, resources_conf=None):
        """
        DOCME
        """
        gobject.GObject.__init__(self)
        self._init(module_name, styles_conf, resources_conf)

    def _init(self, module_name=None, styles_conf=None, resources_conf=None):
        # use a common _init() method that we can call from __init__ and from
        # __setstate__
        self.widget_styles = {}
        self.stock_resources = {}
        self.fallback_themes = {}
        self.fallback_theme = None

        if styles_conf:
            self._init_styles(styles_conf)
        if module_name and resources_conf:
            self._init_resources(module_name, resources_conf)

        # needed so that stop_monitoring knows what to unregister from inotify
        self._styles_conf = styles_conf
        self._resources_conf = resources_conf
        self._module_name = module_name

        self.start_monitoring()

    def clean(self):
        self.stop_monitoring()

        for theme in self.fallback_themes.itervalues():
            if theme != None:
                theme.disconnect_by_func(self._fallback_theme_styles_updated)
                theme.disconnect_by_func(self._fallback_theme_resources_updated)

        self.widget_styles = {}
        self.stock_resources = {}
        self.fallback_themes = {}
        self.fallback_theme = None

    def _fallback_theme_styles_updated(self, fallback_theme):
        self.fallback_theme.update(fallback_theme)
        self.emit('styles-updated')

    def _fallback_theme_resources_updated(self, fallback_theme):
        self.fallback_theme.update(fallback_theme)
        self.emit('resources-updated')

    def _notify_cb(self, iwp, filename, mask, parameter=None):
        inotify_handler = inotify.INotify()
        mask_strings = inotify_handler.flag_to_human(mask)
        if 'modify' in mask_strings or 'moved_to' in mask_strings:
            if self._styles_conf and self._styles_conf.endswith(filename):
                self._init_styles(self._styles_conf)
                self.emit('styles-updated')
            if self._resources_conf and self._resources_conf.endswith(filename):
                self._init_resources(self._module_name, self._resources_conf)
                self.emit('resources-updated')

    def start_monitoring(self):
        if not inotify:
            return

        inotify_handler = inotify.INotify()

        if self._styles_conf:
            styles_dir = unicode(os.path.dirname(self._styles_conf))
            inotify_handler.watch(styles_dir, auto_add=False,
                                  callbacks=(self._notify_cb, None),
                                  recursive=False)

        if self._resources_conf:
            resources_dir = unicode(os.path.dirname(self._resources_conf))
            inotify_handler.watch(resources_dir, auto_add=False,
                                  callbacks=(self._notify_cb, None),
                                  recursive=False)

    def stop_monitoring(self):
        if not inotify:
            return

        inotify_handler = inotify.INotify()

        if self._styles_conf:
            styles_dir = unicode(os.path.dirname(self._styles_conf))
            inotify_handler.ignore(styles_dir)

        if self._resources_conf:
            resources_dir = unicode(os.path.dirname(self._resources_conf))
            inotify_handler.ignore(resources_dir)

        inotify_handler.release()

    def __reduce__(self):
        return makeTheme, (self._module_name, self._styles_conf,
                           self._resources_conf), self.__getstate__()

    def __getstate__(self):
        widget_styles = {}
        for widget, states in self.widget_styles.iteritems():
            for state, style in states.iteritems():
                widget_styles.setdefault(widget, dict())[state] = style._properties

        return {'widget_styles': widget_styles,
                'stock_resources': self.stock_resources}

    def __setstate__(self, pickled):
        self._init()

        widget_styles = {}
        for widget, states in pickled['widget_styles'].iteritems():
            for state, style in states.iteritems():
                widget_styles.setdefault(widget, dict())[state] = Style(**style)

        self.widget_styles = widget_styles
        self.stock_resources = pickled['stock_resources']

    def _load_css(self, filename):
        parser = CSSParser(loglevel=logging.ERROR)
        try:
            # new api in cssutils 0.9.5b3
            css = parser.parseFile(filename)
        except AttributeError:
            css = parser.parse(filename)

        return css

    def _get_cache_directory(self):
        # put this in a method so it can be overridden in tests
        from elisa.core.default_config import CONFIG_DIR
        cache_dir = os.path.join(CONFIG_DIR, 'css')
        if not os.path.exists(cache_dir):
            os.makedirs(cache_dir)

        return cache_dir

    def _get_cached_file(self, filename):
        cache_dir = self._get_cache_directory()
        cached_file = os.path.join(cache_dir,
                filename.replace(os.sep, '_').replace(':', '_'))
        # append the modification time of the original file in order to keep
        # the cache always synchronised: if the content of the file changes,
        # its modification time too and we will be looking for a different
        # cache file
        mtime = os.stat(filename).st_mtime
        cached_file += str(int(mtime))
        return cached_file

    def _get_from_cache(self, filename):
        try:
            theme = self._cache[filename]
        except KeyError:
            theme = None
            # try to load the cache from disk
            cached_filename = self._get_cached_file(filename)
            if not os.path.exists(cached_filename):
                raise KeyError()

        # cache is valid
        if theme is None:
            # load the cache from disk
            cached_file = file(cached_filename)
            try:
                # Nesting try... except in try... finally to work with
                # python < 2.5
                try:
                    theme = load(cached_file)
                    if not isinstance(theme, Theme):
                        raise KeyError()
                except UnpicklingError:
                    raise KeyError()
                except TypeError:
                    # the object can't be reconstructed by pickle.
                    # Let's trigger a regenerate of the cache
                    raise KeyError()
            finally:
                cached_file.close()

            self._cache[filename] = theme

        return theme

    def _add_to_cache(self, filename, parsed):
        self._cache[filename] = parsed
        cached_filename = self._get_cached_file(filename)
        cached_file = file(cached_filename, 'w')
        try:
            dump(parsed, cached_file)
        finally:
            cached_file.close()

    def _guess_style_value_type(self, value, key=None):
        # Try to guess the type of a style property based on heuristics on the
        # key and the form of the value. Return the value typed if successful,
        # or the value as a string (leading and trailing double quotes
        # stripped) if no match can be found.
        # Pigment constants (pgm.*) are a special case. Since they cannot be
        # serialized (needed by the CSS cache), they are returned as a pseudo
        # property (string, e.g. 'pgm.left') and should be transformed (e.g. to
        # 'pgm.TEXT_ALIGN_LEFT') and eval()'ed by the widgets.

        # Try to match an integer
        try:
            typed = int(value)
            return typed
        except ValueError:
            pass

        # Try to match a floating point number
        try:
            typed = float(value)
            return typed
        except ValueError:
            pass

        # Try to match a tuple
        lst = value.split(',')
        if len(lst) in (2, 3, 4):
            # Try to match a tuple of integers
            try:
                lst = tuple(map(int, lst))
                return lst
            except ValueError:
                pass

            # Try to match a tuple of floating point numbers
            try:
                lst = tuple(map(float, lst))
                return lst
            except ValueError:
                pass

        if key is not None:
            # Try to match a pigment image or text property
            image_properties = ('alignment', 'layout', 'interp')
            text_properties = ('ellipsize', 'alignment', 'wrap', 'gravity',
                               'stretch', 'style', 'variant', 'weight')
            pgm_properties = set(image_properties + text_properties)
            for pgm_property in pgm_properties:
                if pgm_property in key:
                    typed = 'pgm.%s' % value.strip()
                    return typed

        return value.strip().strip('"')

    def _get_styles(self, filename):
        try:
            return self._get_from_cache(filename)
        except KeyError:
            theme = Theme()
            css = self._load_css(filename)
            for rule in css.cssRules:
                if rule.selectorText == 'defaults':
                    continue

                if ':' in rule.selectorText:
                    widget, state = map(str, rule.selectorText.split(':'))
                    state = getattr(const, state.upper(), const.STATE_NORMAL)
                else:
                    widget, state = str(rule.selectorText), const.STATE_NORMAL

                if widget not in theme.widget_styles:
                    theme.widget_styles[widget] = {}

                props = {}
                for prop in rule.style:
                    value = self._guess_style_value_type(prop.value, prop.name)
                    props[str(prop.name)] = value

                style = Style(**props)
                widget_states = theme.widget_styles[widget]
                if state not in widget_states:
                    widget_states[state] = style
                else:
                    widget_states[state].update(style)

            self._add_to_cache(filename, theme)
            return theme

    def _get_resources(self, module_name, filename):
        try:
            return self._get_from_cache(filename)
        except KeyError:
            theme = Theme()
            css = self._load_css(filename)

            # allow redefinitions of "defaults" sections
            basedir = '.'
            for rule in css.cssRules:
                if rule.selectorText == 'defaults':
                    bdir = rule.style.getProperty('basedir')
                    if bdir is not None:
                        basedir = bdir.value.strip('"\'')
                else:
                    resource_prefix = str(rule.selectorText)

                    for prop in rule.style:
                        filepath = os.path.normpath(os.path.join(basedir, prop.value.strip('"\'')))
                        resource_name = ".".join([resource_prefix, str(prop.name)])
                        try:
                            absolute_path = resource_filename(module_name, filepath)
                            theme.stock_resources[resource_name] = absolute_path
                        except:
                            pass

            self._add_to_cache(filename, theme)

            return theme

    def _init_styles(self, styles_conf):
        """
        Read the configuration file and fill up the widgets styles.
        """
        theme = self._get_styles(styles_conf)
        self.update(theme)

    def _init_resources(self, module_name, resources_conf):
        """
        Read the configuration file and fill up the stock resources.
        """
        theme = self._get_resources(module_name, resources_conf)
        self.update(theme)

    @classmethod
    def load_from_module(cls, module_name):
        """
        Build a L{elisa.plugins.pigment.widgets.Theme} object using the
        'styles.conf' and 'resources.conf' files found in the specified module.

        @param module_name: the module to search, in the absolute dotted
                            notation
        @type module_name:  C{str}

        @return:            the new theme, or C{None}
        @rtype:             L{elisa.plugins.pigment.widgets.Theme}
        """
        log_category = 'theme'

        try:
            styles_conf = resource_filename(module_name, STYLES_FILENAME)
        except (ImportError, KeyError):
            # ImportError raised when resource not found in uninstalled plugin
            # KeyError raised when resource not found in an egg plugin
            styles_conf = ''
        if os.path.isfile(styles_conf):
            log.debug(log_category,
                      "Loading plugin theme from: %s" % styles_conf)
        else:
            log.debug(log_category, "Cannot find theme file: %s" % styles_conf)
            styles_conf = None

        try:
            resources_conf = resource_filename(module_name, RESOURCES_FILENAME)
        except (ImportError, KeyError):
            # ImportError raised when resource not found in uninstalled plugin
            # KeyError raised when resource not found in an egg plugin:
            resources_conf = ''
        if os.path.isfile(resources_conf):
            log.debug(log_category,
                      "Loading resource theme from: %s" % resources_conf)
        else:
            log.debug(log_category,
                      "Cannot find resource file: %s" % resources_conf)
            resources_conf = None

        if resources_conf is None and styles_conf is None:
            return None

        module_theme = cls(module_name=module_name,
                           styles_conf=styles_conf,
                           resources_conf=resources_conf)
        return module_theme

    def update(self, other):
        """
        Merge in-place another theme.

        @param other: the theme from which to update
        @type other: L{elisa.plugins.pigment.widgets.Theme}
        """
        self.merge(other, inplace=True)

    def merge(self, other, inplace=False):
        """
        Merge with another theme, returning a new one.

        The new theme will have all the "properties" of the current style, with
        replaced values from the second, plus further "properties" coming from
        the other theme.

        @param other:   the theme to merge
        @type other:    L{elisa.plugins.pigment.widgets.Theme}
        @param inplace: whether to build another theme, or update the current
                        one
        @type inplace:  C{bool}

        @return:        the new theme
        @rtype:         L{elisa.plugins.pigment.widgets.Theme}
        """

        if inplace:
            new = self
        else:
            new = Theme()

        if not inplace:
            for key, value in self.stock_resources.items():
                new.stock_resources[key] = value
            for widget, styles in self.widget_styles.items():
                new.widget_styles[widget] = copy.deepcopy(styles)

        for key, value in other.stock_resources.items():
            new.stock_resources[key] = value
        for widget, styles in other.widget_styles.items():
            if widget not in new.widget_styles:
                new.widget_styles[widget] = other.widget_styles[widget]
            else:
                for state, style in styles.items():
                    new.widget_styles[widget][state] = style.merge(other.widget_styles[widget][state])

        return new

    def get_style_for_widget(self, widget, state=const.STATE_NORMAL, search=True):
        """
        Get the style for a widget class in the specified state.

        @param widget: the classname of the widget
        @type widget:  C{str}
        @param state:  the state for which we want to retrieve the style
                       (one of L{elisa.plugins.pigment.widgets.const}.STATE_*)
        @type state:   C{int}

        @return:       the associated style, or C{None}
        @rtype:        L{elisa.plugins.pigment.widgets.Style}
        """
        if not widget or not state:
            return None

        widget_styles = self.widget_styles.get(widget)
        if widget_styles is None and self.fallback_theme is not None:
            widget_styles = self.fallback_theme.widget_styles.get(widget)

        if widget_styles is None:
            if not search:
                return None

            return self.lookup(widget, 'style', state)

        return widget_styles.get(state)

    def get_resource(self, name, search=True):
        """
        Get the named resources, doing a lookup into the plugins' defaults if
        not found, or None.

        @param name:   the name of the resource
        @type name:    C{str}
        @param search: whether to do the lookup into the plugins defaults
        @type search:  C{bool}
        """
        media = self.stock_resources.get(name)
        if not media and self.fallback_theme:
            media = self.fallback_theme.stock_resources.get(name)

        if media:
            return media

        if not search:
            raise ResourceNotExisting(name)

        return self.lookup(name, 'resource')

    def lookup(self, name, type, state=const.STATE_NORMAL):
        """
        Dynamically search for the named resource ('style' or 'resource').

        If a suitable module is found during the search, a L{Theme} object will
        be built and cached for later use: it will be stored in a dictionary
        indexed by absolute module names (in the Python dotted notation).

        @param name:  the full qualified name to look for
                      (e.g.: 'elisa.plugins.pigment.widgets.button.Button'
                      or 'elisa.plugins.shelf.icon')
        @type name:   C{str}
        @param type:  'style' or 'resource'
        @type type:   C{str}
        @param state: the state of the widget. Only used if type == 'style'.
                      One of L{elisa.plugins.pigment.widgets.const}.STATE_*
        @type state:  C{int}
        @return:      the resource found (filepath or style), if any
        @rtype:       C{str} or L{elisa.plugins.pigment.widgets.Style} or
                      C{None}
        """
        if type == 'style' and ':' in name:
            name, state = name.split(':')
            state = getattr(const, state, const.STATE_NORMAL)

        module_name, resource_name = name.rsplit('.', 1)
        if module_name:
            try:
                module_theme = self.fallback_themes[module_name]
            except KeyError:
                module_theme = None
            else:
                # we already searched for the theme, without finding it
                if not self.fallback_themes[module_name]:
                    return
            if module_theme is None:
                module_theme = Theme.load_from_module(module_name)
                # remember that we couldn't find the theme
                self.fallback_themes[module_name] = module_theme

                if not module_theme:
                    return
                else:
                    module_theme.connect('styles-updated',
                                         self._fallback_theme_styles_updated)
                    module_theme.connect('resources-updated',
                                         self._fallback_theme_resources_updated)

                    if not self.fallback_theme:
                        self.fallback_theme = module_theme
                    else:
                        self.fallback_theme.merge(module_theme, inplace=True)

            if type == 'resource':
                #item = module_theme.get_resource(name.split('.')[-1], False)
                item = module_theme.get_resource(name, False)
            elif type == 'style':
                item = module_theme.get_style_for_widget(name, state, False)

            return item

    def __repr__(self):
        r = {}
        r['Theme'] = [{'widget_styles': self.widget_styles},
                      {'stock_resources': self.stock_resources}]

        return pprint.pformat(r)

    @classmethod
    def get_default(cls):
        """Get the default theme."""
        global _DEFAULT_THEME

        if not _DEFAULT_THEME:
            _DEFAULT_THEME = cls()

        return _DEFAULT_THEME

    @staticmethod
    def set_default(theme):
        """Set the default theme."""
        global _DEFAULT_THEME

        if _DEFAULT_THEME != None:
            _DEFAULT_THEME.clean()
        _DEFAULT_THEME = theme


gobject.type_register(Theme)
