# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Module responsible for managing Elisa config file upgrades
"""

__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'

from elisa.core import log, config, media_uri
import os
from distutils.version import LooseVersion


class ConfigUpgrader(log.Loggable):

    def __init__(self, current_config, #current_version,
                 default_config):
        log.Loggable.__init__(self)
        self._current_config = current_config
        #self._current_version = current_version
        self._current_version = self._get_version(current_config)
        self._default_config = default_config
        self._new_config_fname = 'elisa_new.conf'

    def _get_version(self, cfg):
        version = cfg.get_option('version', section='general')
        if version is None:
            version = '0.1.7'
            
        cfg_version = LooseVersion(version)
        return cfg_version
    
    def update_for(self, version):
        cfg = self._current_config
        version_str = '.'.join([str(i) for i in version])
        updated = False
        self.info("Trying to upgrade from %s to %s", self._current_version,
                  version_str)
        
        if self._current_version == '0.1.7':
            cfg = self._update_0_1_to_0_3_1()
            if version in ((0, 3, 1, 1), (0, 3, 2)):
                cfg = self._update_0_3_1_to_0_3_2(cfg)
                updated = True
                
        if self._current_version == '0.3.1':
            if version in ((0, 3, 1, 1),(0, 3, 2)):
                cfg = self._update_0_3_1_to_0_3_2(cfg)
                updated = True
                
        if updated:
            full_path = os.path.join(cfg.get_config_dir(),
                                     self._current_config.get_filename())
            cfg.set_filename(full_path)
            if os.path.exists(self._new_config_fname):
                os.unlink(self._new_config_fname)
            self.info("Updated config to %s", cfg.get_filename())
        else:
            self.info("No update needed")
        return cfg

    def _backup(self, version=None, cfg=None):
        if version is None:
            if self._current_version is None:
                version = (0, 1, 7)
            else:
                version = str(self._current_version).split('.')
        
        version = '.'.join([str(i) for i in version])

        if not cfg:
            cfg = self._current_config
            
        full_path = os.path.join(cfg.get_config_dir(),
                                 'elisa_%s.bak' % version)
        self.info("Doing a backup to the old config on %s", full_path)
        cfg.write(full_path)

    def _update_0_3_1_to_0_3_2(self, cfg=None):
        self.info('Upgrading from 0.3.1 to 0.3.2')
        self._backup((0, 3, 1), cfg)
        if not cfg:
            cfg = self._current_config

        def upgrade_paths(opt_name, mapping, section='general'):
            
            old_option = cfg.get_option(opt_name,section=section)
            for old_path, new_path in mapping.iteritems():
                if old_path in old_option:
                    old_option.remove(old_path)
                    old_option.append(new_path)

                old_path_section = cfg.get_section(old_path)
                if old_path_section:
                    cfg.set_section(new_path, old_path_section)
                    cfg.del_section(old_path)
                    
            cfg.set_option(opt_name, old_option, section=section)
            
        # upgrade media_providers paths
        media_providers = {'base:upnp_media': 'coherence_plugin:upnp_media',
                           'base:elisa_media': 'media_good:elisa_media',
                           'base:gnomevfs_media': 'media_good:gnomevfs_media',
                           'base:ipod_media': 'media_bad:ipod_media',
                           'base:mmslist_media': 'media_good:mmslist_media',
                           'base:shoutcast_media': 'media_ugly:shoutcast_media',
                           'base:daap_media': 'media_bad:daap_media',
                           'base:audiocd': 'audiocd:audiocd_media'}
        upgrade_paths('media_providers', media_providers)

        # new media_providers
        new_media_providers = ['fspot:fspot_media','stage6:stage_media',
                               'media_ugly:shoutcast_media',
                               'flickr:flickr_media']
        media_providers = cfg.get_option('media_providers', section='general')
        for mp in new_media_providers:
            if mp not in media_providers:
                media_providers.append(mp)
        cfg.set_option('media_providers', media_providers)

        # upgrade metadata_providers paths
        metadata_providers = {'base:gst_metadata':'media_good:gst_metadata',
                              'base:taglib_metadata': 'media_good:taglib_metadata',
                              'base:cover_in_dir':'media_good:cover_in_dir',
                              'base:cover_cache': 'media_good:cover_cache',
                              'base:amazon_covers':'media_good:amazon_covers'}
        upgrade_paths('metadata_providers', metadata_providers)
        
        # upgrade service_providers paths
        service_providers = {'lastfm:audioscrobbler_service':'services_good:lastfm_scrobbler',
                             'base:hal_service':'hal:hal_service',
                             'base:twisted_pb':'services_good:twisted_pb',
                             'base:http_server':'services_good:http_server',
                             'base:coherence_service':'coherence_plugin:coherence_service',
                             'base:upnp_media_server':'coherence_plugin:upnp_media_server',
                             'base:upnp_media_renderer':'coherence_plugin:upnp_media_renderer'
                             }
        upgrade_paths('service_providers', service_providers)

        # new service_providers
        new_service_providers = ['coherence_plugin:coherence_service',
                                 ]
        service_providers = cfg.get_option('service_providers',
                                           section='general')
        for sp in new_service_providers:
            if sp not in service_providers:
                service_providers.append(sp)
        cfg.set_option('service_providers', service_providers)

        # upgrade input_providers paths
        input_providers = {'base:lirc_input':'input_good:lirc_input',
                           'base:raw_input': 'input_good:raw_input',
                           'base:bluetooth_input':'input_bad:bluetooth_input'}
        upgrade_paths('input_providers', input_providers, 'backend1')
        upgrade_paths('input_providers', input_providers, 'frontend1')

        # upgrade main menu
        menu_activities = ['base:audio_activity', 'base:video_activity',
                           'base:image_activity', 'base:config_activity',
                           'base:service_activity']
        cfg.set_option('menu_activities', menu_activities,
                       section='base:main_menu_activity')

        # upgrade services menu
        service_activities = ['service:about_activity']
        cfg.set_option('service_activities', service_activities,
                       section='base:service_activity')
        cfg.del_section('service:service_activity')

        # upgrade player and dvd controller
        cfg.del_section('base:player_controller')
        cfg.del_section('dvd:dvd_player_controller')
        
        cfg.set_option('version', '0.3.2', section='general')
        return cfg
            
    def _update_0_1_to_0_3_1(self):
        self.info('Upgrading from 0.1.x to 0.3.1')
        self._backup()
        cfg = config.Config(self._new_config_fname, self._default_config)
        
        # media locations options have migrated to activities
        activity_map = {'plugins.music': 'base:audio_activity',
                        'plugins.movies': 'base:video_activity',
                        'plugins.pictures': 'base:image_activity'
                        }
        for old_section, new_section in activity_map.iteritems():
            locations = self._current_config.get_option('locations',
                                                        section=old_section,
                                                        default=[])
            new_locations = []
            location_labels = {}
            for location in locations:

                # meta:// URIs not supported anymore
                if location.startswith('meta'):
                    continue

                # FIXME: location/* not supported yet
                if location.endswith('*'):
                    location = location[:-1]

                location = location.decode('utf-8')
                location_uri = media_uri.MediaUri(location)

                # label parameter removed
                label = location_uri.get_param('label')
                if label:
                    location_uri.del_param('label')
                    location_labels[str(location_uri)] = label
                    
                new_locations.append(str(location_uri))
                
            cfg.set_option('locations', new_locations, section=new_section)
            activity_section = cfg.get_section(new_section)
            for location, label in location_labels.iteritems():
                activity_section[location] = {'label': str(label)}

        # start_fullscreen option moved to elisa_controller
        start_fullscreen = self._current_config.get_option('start_fullscreen',
                                                           default='0')
        controller_section = cfg.get_section('poblenou:elisa_controller')
        controller_section['start_fullscreen'] = str(start_fullscreen)

        # media_manager
        media_manager_opts = self._current_config.get_section('media_manager')
        media_scanner_section = cfg.get_section('media_scanner')
        try:
            media_scanner_section['enabled'] = str(media_manager_opts['enable_cache'])
        except KeyError:
            self.warning("enable_cache option not found in old config")

        try:
            media_scanner_section['database'] = media_manager_opts['db_name']
        except KeyError:
            self.warning("db_name option not found in old config")
            
        for timebased in ('fivemin', 'hourly', 'daily', 'weekly'):
            opt_name = '%s_location_updates' % timebased
            try:
                media_scanner_section[opt_name] = media_manager_opts[opt_name]
            except KeyError:
                self.warning("%s option not found in old config", opt_name)
                continue
        return cfg
