# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.tests.elisa_test_case import ElisaTestCase
from elisa.core import media_db, media_uri, db_backend
import os
DB_DUMP="""\


"""

class TestMediaDB(ElisaTestCase):
    first_load = True
    
    def __init__(self, methodName='runTest'):
        ElisaTestCase.__init__(self, methodName=methodName)
        self._db_file = '/tmp/test.db'

        if os.path.exists(self._db_file):
            os.unlink(self._db_file)
        
        try:
            backend = db_backend.DBBackend(db_backend='sqlite',
                                           database=self._db_file)
        except db_backend.DBBackendError, error:
            self.skip = error
                
    def setUp(self):
        ElisaTestCase.setUp(self)

        self._tmp_uri = media_uri.MediaUri(u'file:///tmp/')

        backend = db_backend.DBBackend(db_backend='sqlite',
                                       database=self._db_file)
        backend.sql_execute(DB_DUMP)
        self._db = media_db.MediaDB(backend, True)

    def tearDown(self):
        # avoid "database lock" issues
        self._db.close()

            
    def test_reset(self):

        version_row = self._db._backend.sql_execute("select version from core_meta")
        self.failUnless(version_row)
        version = version_row[0].version
        self.assertEqual(version, media_db.CURRENT_DB_VERSION)


    def test_add_source(self):
        """ Add a new media source in the database
        """
        source = self._db.add_source(self._tmp_uri, 'tmp')
        self.failUnless(source)

    def test_hide_show_source(self):
        """ Update source's "available" flag
        """
        self._db.add_source(self._tmp_uri, 'tmp')
        source = self._db.get_source_for_uri(self._tmp_uri)
        self.failUnless(source)
        self.failUnless(self._db.hide_source(source))
        self.failUnless(self._db.show_source(source))

    def test_add_remove_media_node(self):
        source = self._db.add_source(self._tmp_uri, 'tmp')
        source = self._db.get_source_for_uri(self._tmp_uri)
        self.failUnless(source)

        # TODO: test metadata keyword in add_media_node
        m_uri = self._tmp_uri.join(u'foo.avi')

        added = self._db.add_media(m_uri, "foo.avi", source,
                                                   'video')
        self.assertEquals(added, True)

        media = self._db.get_media_information(m_uri)
        self.failUnless(media)
        self.assertEquals(media.uri, unicode(m_uri))
        self.assertEquals(media.short_name, 'foo.avi')
        media_id = media.id
        self.failUnless(self._db.del_media_node(media))

        media = self._db.get_media_with_id(media_id)
        self.assertEquals(media.deleted, 1)


    def test_get_source_for_uri(self):
        """ Find in which media source the given uri is registered with.
        """
        source = self._db.add_source(self._tmp_uri, 'tmp')
        source = self._db.get_source_for_uri(self._tmp_uri)
        self.failUnless(source)
        self.assertEquals(source.uri, repr(self._tmp_uri))

    def test_get_media_information(self):
        """ Find in database the media corresponding with the given URI.

        The URI has to be referenced in the "media" table.

        @param uri: the URI to search in the "media" table
        @type uri:  L{elisa.core.media_uri.MediaUri}
        @rtype:     L{elisa.extern.db_row.DBRow}
        """
        source = self._db.add_source(self._tmp_uri, 'tmp')
        m_uri = self._tmp_uri.join('foo.mp3')
        source = self._db.get_source_for_uri(self._tmp_uri)
        added = self._db.add_media(m_uri, "foo.mp3", source,
                                                   'file', format='audio')
        self.assertEquals(added, True)
     
        foo_media = self._db.get_media_information(m_uri)
        self.assertEquals(added, True)
        self.assertEquals(foo_media.uri, 'file:///tmp/foo.mp3')
        self.assertEquals(foo_media.short_name, 'foo.mp3')
        self.assertEquals(foo_media.format, 'audio')
        self.assertEquals(foo_media.typ, 'file')

        # TODO: test metadata

    def test_get_media_with_id(self):
        """ Fetch the media with given id in the database

        @param media_id: the identifier of the Media i have to dig in the db
        @type media_id:  int
        @rtype:          L{elisa.extern.db_row.DBRow}
        """
        source = self._db.add_source(self._tmp_uri, 'tmp')
        m_uri = self._tmp_uri.join('foo.mp3')
        source = self._db.get_source_for_uri(self._tmp_uri)
        added = self._db.add_media(m_uri, "foo.mp3", source,
                                                   'file', format='audio')
        self.assertEquals(added, True)

        media = self._db.get_media_information(m_uri)
        media_id = media.id
        foo_media = self._db.get_media_with_id(media_id)
        self.failUnless(foo_media)

        self.assertEquals(foo_media.uri, repr(m_uri))
        self.assertEquals(foo_media.uri, media.uri)
        self.assertEquals(foo_media.id, media.id)

    def test_update_media(self):
        """ Update some attributes in database of the given media
        """
        source = self._db.add_source(self._tmp_uri, 'tmp')
        m_uri = self._tmp_uri.join('foo.mp3')
        source = self._db.get_source_for_uri(self._tmp_uri)

        media = self._db.get_media_information(m_uri)
        self._db.del_media_node(media, force_drop=True)

        added = self._db.add_media(m_uri, "foo.mp3", source,
                                                   'file', format='audio')
        self.assertEquals(added, True)
        media = self._db.get_media_information(m_uri)

        self._db.update_media_metadata(media, artist='foo', album='bar')
        infos = self._db.get_media_information(m_uri)
        self.failUnless(infos)
        self.assertEquals(infos.artist, 'foo')
        self.assertEquals(infos.album, 'bar')
