# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Exception handler that's able to extract detailled error informations
and write them to a file.
"""

__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'

import cgitb, os, sys
import traceback, tempfile

class ExceptionHook(cgitb.Hook):
    """ Exception handler with detailled error reporting support.

    Extends L{cgitb.Hook}, add a boolean display option indicating
    wether errors should be displayed on stderr or not. Additionnally,
    each detailled error report is written to a file on disk on
    L{logdir} directory.

    Use it like this::

      >>> sys.excepthook = ExceptionHook(format='text', logdir='/tmp',
      ...                                file=sys.stderr, display=False)
      
    """
    
    def handle(self, info=None):
        """ add-on to default cgitb.handle():

        - if logdir doesn't exists, create it
        - if logdir is set, write error report in a text or html file,
          depending on self.format
        """
        import traceback
        
        if self.logdir is not None and not os.path.exists(self.logdir):
            try:
                os.makedirs(self.logdir)
            except OSError, e:
                # avoid to have same error in the future by setting logdir
                # to None
                self.logdir = None
                self.warning("Could not make directory %r: %r", self.logdir, e)
                traceback.print_exc()

        info = info or sys.exc_info()
        if self.format == "html":
            self.file.write(cgitb.reset())

        formatter = (self.format=="html") and cgitb.html or cgitb.text
        plain = False
        try:
            doc = formatter(info, self.context)
        except:
            doc = ''.join(traceback.format_exception(*info))
            plain = True

        if self.display:
            self.file.write(doc + '\n')
        else:
            self.file.write('A problem occurred in Elisa.\n')

        if self.logdir is not None:
            suffix = ['.txt', '.html'][self.format=="html"]
            (fd, path) = tempfile.mkstemp(suffix=suffix, prefix="elisa_",
                                          dir=self.logdir)
            try:
                file = os.fdopen(fd, 'w')
                file.write(doc)
                file.close()
                msg = '%s contains the description of this error.' % path
            except:
                msg = 'Tried to save traceback to %s, but failed.' % path
            self.file.write(msg + '\n')
        try:
            self.file.flush()
        except:
            pass
