# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
DAAP MediaProvider component
"""


__maintainer__ = 'Benjamin Kampmann <benjamin@fluendo.com>'
__maintainer2__ = 'Philippe Normand <philippe@fluendo.com>'

from elisa.base_components.media_provider import MediaProvider
from elisa.core import media_uri
from elisa.core import common
from elisa.core.bus import bus_message
from elisa.core.component import InitializeFailure
from elisa.core.observers.dict import DictObservable

import socket
import gst
import gobject

import daap

try:
    import dbus
    if getattr(dbus, 'version', (0,0,0)) >= (0,41,0):
        import dbus.glib
except ImportError:
    dbus = None

if dbus:
    try:
        import avahi
    except ImportError:
        avahi = None
else:
    avahi = None

from twisted.internet import defer, threads

"""
TODO:

- add a "copy to local library" feature?
- implement URIHandler interface in DaapSource so that daap:// URIs are supported in playbin
  See http://bugzilla.gnome.org/show_bug.cgi?id=339279

"""

class DaapSource(gst.BaseSrc):
    __gsttemplates__ = (
        gst.PadTemplate("src",
                        gst.PAD_SRC,
                        gst.PAD_ALWAYS,
                        gst.caps_new_any()),
        )

    __gstdetails__ = ("DAAP plugin", "Foo/Bar", "Read data on DAAP shares",
                      "Philippe Normand <philippe@fluendo.com>")

    blocksize = 4096
    fd = None

    def __init__(self, name):
        self.__gobject_init__()
        self.curoffset = 0
        self._libs = {}
        self.cnx = daap.DAAPClient()
        self.set_name(name)

    def get_library(self, host, port):
        if (host, port) not in self._libs:
            self.cnx.connect(host, port)
            session = self.cnx.login()
            library = session.library()
            self._libs[(host, port)] = library
        return self._libs.get((host, port))

    def set_property(self, name, value):
        if name == 'uri':
            parsed = media_uri.Uri(value).get_parts()
            port = int(parsed['port'] or  '3689')
            library = self.get_library(parsed['host'], port)
            track_id = int(parsed['params'].get('id','1'))
            track = [ t for t in library.tracks()
                      if t.id == track_id ][0]
            self.fd = track.request()

    def do_create(self, offset, size):
##         if offset != self.curoffset:
##             self.fd.seek(offset, 0)
        data = self.fd.read(self.blocksize)
        if data:
            self.curoffset += len(data)
            return gst.FLOW_OK, gst.Buffer(data)
        else:
            return gst.FLOW_UNEXPECTED, None

gobject.type_register(DaapSource)

## r = gst.element_register(type=DaapSource, elementname="daapsrc",
##                          rank=gst.RANK_PRIMARY)


class AvahiMonitor:

    def __init__(self):
        self._callbacks = {'new-service':  [],
                           'remove-service': []
                           }
        self.bus = dbus.SystemBus()
        avahi_bus = self.bus.get_object(avahi.DBUS_NAME, avahi.DBUS_PATH_SERVER)
        self.server = dbus.Interface(avahi_bus, avahi.DBUS_INTERFACE_SERVER)

        stype = '_daap._tcp'
        domain = 'local'
        self._plugged = {}
        avahi_browser = self.server.ServiceBrowserNew(avahi.IF_UNSPEC,
                                                      avahi.PROTO_UNSPEC,
                                                      stype, domain,
                                                      dbus.UInt32(0))
        obj = self.bus.get_object(avahi.DBUS_NAME, avahi_browser)
        self.browser = dbus.Interface(obj, avahi.DBUS_INTERFACE_SERVICE_BROWSER)

    def start(self):
        self.browser.connect_to_signal('ItemNew', self.new_service)
        self.browser.connect_to_signal('ItemRemove', self.remove_service)

    def stop(self):
        self.bus.close()

    def new_service(self, interface, protocol, name, type, domain, flags):
        service = self.server.ResolveService(interface, protocol,
                                             name, type, domain,
                                             avahi.PROTO_UNSPEC, dbus.UInt32(0))
        address, port = service[-4:-2]
        name = str(service[2])
        for cb in self._callbacks['new-service']:
            self._plugged[name] = (address,port)
            cb(name, address, port)

    def remove_service(self, interface, protocol, name, type, domain,server):
        address, port = self._plugged[name]
        for cb in self._callbacks['remove-service']:
            cb(name, address, port)


    def add_callback(self, sig_name, callback):
        self._callbacks[sig_name].append(callback)

    def remove_callback(self, sig_name, callback):
        self._callback[sig_name].remove(callback)


class DaapMedia(MediaProvider):

    def __init__(self):
        MediaProvider.__init__(self)
        self._libraries = {}
        self._monitor = None

    def initialize(self):
        if not dbus:
            if not avahi:
                self.warning("Please install python-avahi to get Avahi support")
            self.warning("python-dbus is missing...")
            self.warning("Avahi support disabled. DAAP shares won't appear automagically")
        else:
            try:
                self._monitor = AvahiMonitor()
                self._monitor.add_callback('new-service', self._add_location)
                self._monitor.add_callback('remove-service',
                                           self._remove_location)
                self._monitor.start()
            except Exception, ex:
                error_msg = "Couln't initialize Avahi monitor: %s" % str(ex)
                raise InitializeFailure(self.name, error_msg)

    def _add_location(self, name, address, port):
        self.info("New DAAP share at %s:%s : %s" % (address, port, name))
        uri = 'daap://%s:%s/' % (address, port)
        action_type = bus_message.LocalNetworkLocation.ActionType.LOCATION_ADDED
        msg = bus_message.LocalNetworkLocation(action_type, name, 'daap',
                                               uri, ['audio',])
        common.application.bus.send_message(msg)

    def _remove_location(self, name, address, port):
        self.info("DAAP share at %s:%s disappeared" % (address, port))
        uri = 'daap://%s:%s/' % (address, port)
        action_type = bus_message.LocalNetworkLocation.ActionType.LOCATION_REMOVED
        msg = bus_message.LocalNetworkLocation(action_type, name, 'daap',
                                               uri, ['audio',])
        common.application.bus.send_message(msg)

    def _get_library(self, uri):
        self.debug("Retrieving DAAP library at %r", uri)

        host = uri.host
        if uri.port:
            port = int(uri.port)
        else:
            port = 3689

        address = (host, port)
        if address not in self._libraries:
            connection  = daap.DAAPClient()
            try:
                connection.connect(*address)
            except Exception, exc:
                # FIXME: prezise the Exception
                self.info("Couldn't connect to %s : %s" % (address,str(exc)))
            else:
                try:
                    # auth isn't supported yet. Just log in
                    session = connection.login()
                    library = session.library()

                    self._libraries[address] = self._build_library(library.tracks())
                except (socket.timeout, socket.error):
                    pass
                except daap.DAAPError, error:
                    self.warning("Error logging into %s: %s" % (address,
                                                                str(error)))
                    raise daap.DAAPError("Error loggin into %s: %s" % (address,
                                                                       str(error)))

        return self._libraries.get(address)

    def _build_library(self, tracks):
        lib = {}
        for track in tracks:
            if None in (track.artist, track.album):
                continue
            if not lib.get(track.artist):
                lib[track.artist] = {}
            if not lib[track.artist].get(track.album):
                lib[track.artist][track.album] = {}
            if not lib[track.artist][track.album].get(track.id):
                lib[track.artist][track.album][track.id] = (track.name)
        return lib


    def scannable_uri_schemes__get(self):
        return {}

    def supported_uri_schemes__get(self):
        return {'daap' : 0}

    def blocking_get_media_type(self, uri):
        if not uri.get_param('id'):
            media_type = {'file_type': 'directory', 'mime_type': ''}
        else:
            # DAAP only supports audio media, see DMAP for picture media
            media_type = {'file_type': 'audio', 'mime_type': ''}
        return media_type

    def blocking_is_directory(self, uri):
        return not uri.get_param('id')

    def get_direct_children(self, uri, l):
        d = threads.deferToThread(self.blocking_get_direct_children, uri, l)
        return d

    def blocking_get_direct_children(self, uri, children):
        self.debug("Retrieving children of: %s", uri)

        lib = self._get_library(uri)
        if lib:
            artist = uri.get_param('artist', None)
            album = uri.get_param('album', None)
            self.debug("URI artist and album are: %s - %s" % (artist,album))

            if artist != None and album != None:
                # process songs of given album of the artist
                for track_id, track in lib.get(artist).get(album).iteritems():
                    metadata = DictObservable()
                    child_uri = media_uri.MediaUri(uri.parent)
                    child_dict = {'id': track_id, 'artist': artist,
                                  'album': album, 'track': track}
                    child_uri.set_params(child_dict)
                    child_uri.label = track
                    ## FIXME: remove this ugly hack:
                    ## UGLY HACK, so that the albums are not shown for
                    ## track-items
                    metadata['song_artist'] = artist
                    metadata['song_album'] = album
                    metadata['song'] = track
                    self.debug("Appendind %r with metadata %s" % (child_uri,
                                                                metadata))
                    children.append((child_uri, metadata))
            elif artist != None and album == None:
                # list albums of given artist
                for album_name, album in lib.get(artist,{}).iteritems():
                    metadata = DictObservable()
                    child_uri = media_uri.MediaUri(uri.parent)
                    metadata['artist'] = artist
                    metadata['album'] = album_name
                    ## Don't use a DictObservable for params!
                    child_uri.set_params({'artist' : artist,
                                          'album' : album_name})
                    child_uri.label = album_name
                    self.debug("Appendind %r with metadata %s" % (child_uri,
                                                                metadata))

                    children.append((child_uri, metadata))
            else:
                # list artists
                for artist in lib.keys():
                    metadata = DictObservable()
                    child_uri = media_uri.MediaUri(uri.parent)
                    child_uri.set_params({'artist': artist})
                    child_uri.label = artist
                    metadata['artist'] = artist
                    self.debug("Appendind %r with metadata %s" % (child_uri,
                                                                metadata))

                    children.append((child_uri,metadata))

        self.debug("Retrieved children: %r", children)
        return children

    def blocking_has_children_with_types(self, uri, media_types):
        has_children = False
        if not uri.get_param('id'):
            has_children = len(set(['audio',
                                    'directory']).intersection(media_types)) > 0
        return has_children
