# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.base_components.view import View

import pgm
from pypgmtools.graph.group import Group
from pypgmtools.graph.image import Image
from pypgmtools.timing import implicit
from elisa.core import player, common

from twisted.internet import defer, reactor

from elisa.core import plugin_registry
NodeControllerClass = plugin_registry.get_component_class('base:node_controller')


import os
import gtk.gdk, gobject


class ElisaView(View):

    supported_controllers = ('poblenou:elisa_controller')
    default_associations = (
    ('poblenou:node_controller','poblenou:node_view'),
    ('poblenou:tree_controller','poblenou:tree_view'),
    ('base:player_controller','poblenou:player_view'),
    ('dvd:dvd_player_controller','poblenou:dvd_player_view'),
    ('base:slideshow_controller','poblenou:slideshow_view'),
    )

    def __init__(self):
        View.__init__(self)
        self.context_path = 'pigment:pigment_context'
        self.menu = None
        self.player = None
        self.dvd_player = None
        self.slideshow = None
        self.context_handle = None
        self._background = None
        self._animated_player = None
        self._animated_dvd_player = None
        self._animated_slideshow = None
        self._animated_fade = None

    def frontend_changed(self, previous_frontend, new_frontend):
        if new_frontend == None:
            return

        if new_frontend != previous_frontend:
            canvas = new_frontend.context.canvas
            group = Group(canvas, pgm.DRAWABLE_MIDDLE)
            group.opacity = 255
            group.visible = True
            self.context_handle = group

            # connects to the canvas_resized signal of the new context
            new_frontend.context.canvas_resized.connect(self.canvas_resized)

            # FIXME: is this the right place to do that?
            icon_path = self.frontend.theme.get_media("elisa_icon",
                                                      "poblenou:theme/elisa.png")
            try:
                icon = gtk.gdk.pixbuf_new_from_file(icon_path)
            except gobject.GError:
                self.warning("Icon file not found: %r", icon_path)
            else:
                self.frontend.context.viewport_handle.icon = icon



    def canvas_resized(self, size):
        canvas = self.frontend.context.canvas

        if self._background != None:
            self._background.size = canvas.size

        # FIXME: not everything gets resized on the fly yet, please fix it


    def controller_changed(self):
        self._create_fade()
        self._create_background()
        self._create_menu()
        self._create_player()
        self._create_dvd_player()
        self._create_slideshow()

        #self.menu.player_image = self.player.background
        # FIXME: next two lines useless ?
        self._animated_player.opacity = 255
        self._animated_dvd_player.opacity = 255
        self._animated_slideshow.opacity = 255

        self.menu.controller.focus()
        self.menu.controller.selected = True
        
        self.frontend.context.viewport_handle.fullscreen = self.controller.fullscreen

        # FIXME: this is not the nicest solution; the fade in should start
        # exactly when all the children views are ready (pictures loaded, etc.)
        reactor.callLater(1.0, self._fade_in)

    def _create_player(self):
        self.player = self.create_child_view(self.controller.player)
        self.context_handle.add(self.player.context_handle)
        self._unzoom_widget(self.player.context_handle)

        self._animated_player = implicit.AnimatedObject(self.player.context_handle)
        self._animated_player.mode = implicit.REPLACE
        self._animated_player.setup_next_animations(duration = 500,
                                                    transformation = implicit.DECELERATE)

    def _create_dvd_player(self):
        self.dvd_player = self.create_child_view(self.controller.dvd_player)
        self.context_handle.add(self.dvd_player.context_handle)
        self._unzoom_widget(self.dvd_player.context_handle)

        self._animated_dvd_player = implicit.AnimatedObject(self.dvd_player.context_handle)
        self._animated_dvd_player.mode = implicit.REPLACE
        self._animated_dvd_player.setup_next_animations(duration = 500,
                                                    transformation = implicit.DECELERATE)

    def _create_slideshow(self):
        self.slideshow = self.create_child_view(self.controller.slideshow)
        self.context_handle.add(self.slideshow.context_handle)
        self._unzoom_widget(self.slideshow.context_handle)

        self._animated_slideshow = implicit.AnimatedObject(self.slideshow.context_handle)
        self._animated_slideshow.mode = implicit.REPLACE
        self._animated_slideshow.setup_next_animations(duration = 500,
                                          transformation = implicit.DECELERATE)

    def _create_menu(self):
        self.menu = self.create_child_view(self.controller.menu)
        self.context_handle.add(self.menu.context_handle)
        self.menu.context_handle.opacity = 0
        self.menu.context_handle.visible = True

        self._animated_menu = implicit.AnimatedObject(self.menu.context_handle)
        self._animated_menu.mode = implicit.REPLACE
        self._animated_menu.setup_next_animations(duration = 500,
                                                  transformation = implicit.DECELERATE)

    def _create_background(self):
        canvas = self.frontend.context.canvas

        background_path = self.frontend.theme.get_media("background",
                                "poblenou:theme/background.png")
        fd = os.open(background_path, os.O_RDONLY)

        self._background = Image()
        self._background.bg_a = 0
        self._background.size = canvas.size
        self._background.layout = pgm.IMAGE_ZOOMED
        self._background.set_from_fd(fd)
        self._background.set_name('background')
        

        canvas.add(pgm.DRAWABLE_FAR, self._background)

        self._background.visible = True

        self._animated_background = implicit.AnimatedObject(self._background)
        self._animated_background.mode = implicit.REPLACE
        self._animated_background.setup_next_animations(duration = 500,
                                           transformation = implicit.DECELERATE)

    def _create_fade(self):
        background_path = self.frontend.theme.get_media("shade",
                                "poblenou:theme/shade.png")
        fd = os.open(background_path, os.O_RDONLY)

        self._fade = Image()
        self._fade.bg_a = 0
        self._fade.size = (4.0, 3.0)
        self._fade.set_from_fd(fd)
        self._fade.set_name(background_path)

        canvas = self.frontend.context.canvas
        
        self._fade.opacity  = 255
        self._fade.visible  = True
        self._fade.bg_color = (0, 0, 0, 0)
        self._fade.position = (0.0, 0.0, 20.0)
        self._fade.width    = canvas.width
        self._fade.height   = canvas.height*2.0
        self._fade.layout   = pgm.IMAGE_FILLED
        self._animated_fade = implicit.AnimatedObject(self._fade)
        self.context_handle.add(self._fade)


    def _fade_in(self):
        self.debug("Fading in")
        canvas = self.frontend.context.viewport_handle.get_canvas()
        self._animated_fade.setup_next_animations(duration = 1000,
                                           transformation = implicit.DECELERATE)
        self._animated_fade.position = (0.0, -canvas.height*2.0, 20.0)

    def clean(self):
        # fading out that takes 1 second
        self.debug("Fading out")
        self._animated_fade.setup_next_animations(duration = 1000)
        self._animated_fade.position = (0.0, 0.0, 20.0)

        dfr = defer.Deferred()

        def finish(dfr):
            dfr.callback(None)

        # FIXME: would be better with triggers in the animation framework
        reactor.callLater(1, finish, dfr)

        return dfr

    def _unzoom_widget(self, widget):
        canvas = self.frontend.context.canvas

        widget.size = (canvas.width/2.0, canvas.height/2.0)
        widget.position = (canvas.width/4.0, canvas.height/30.0, 1.0)

        self._animated_menu.opacity = 255
        self._animated_background.opacity = 255

    def _zoom_widget(self, widget):
        self._unzoom_widget(widget)
        widget.opacity = 0

        widget.size = self.frontend.context.canvas.size
        widget.position = (0.0, 0.0, 1.0)

        self._animated_menu.opacity = 0
        self._animated_background.opacity = 0
        widget.opacity = 255


    def last_animated_panel_widget_selected__get(self):
        activity_name = self.controller.last_panel_focused.model.activity.name
        if activity_name == 'dvd_player_activity':
            return self._animated_dvd_player
        if activity_name == 'slideshow_activity':
            return self._animated_slideshow
        
        return self._animated_player
        

    def attribute_set(self, key, old_value, new_value):
        if key == 'focused':
            last_panel = self.last_animated_panel_widget_selected
            if self.menu.controller.focused:
                self._unzoom_widget(last_panel)

                # FIXME: big hack to hide the dock
                if last_panel == self._animated_player:
                    self.player.osd_hide()
                elif last_panel == self._animated_dvd_player:
                    self.dvd_player.player_dock.hide()
                    
            else: 
                 self._zoom_widget(last_panel)

        elif key == 'fullscreen':
            self.frontend.context.viewport_handle.fullscreen = new_value
        
        elif key == 'menu_controller_focused':
            last_panel = self.last_animated_panel_widget_selected
            
            if self._is_root_menu(new_value):
                focused_model = self.controller.last_panel_focused.model
                activity_name = focused_model.activity.name
                last_panel.opacity = 0
                if activity_name == 'player_activity':
                    if self.controller.player.model.media_type != 'audio':
                        focused_model.state = player.STATES.PAUSED
                    self.player.osd_hide()
                elif activity_name == 'dvd_player_activity':
                    if focused_model.state == player.STATES.PLAYING:
                        focused_model.state = player.STATES.PAUSED
                    self.player.player_dock.hide()
                elif activity_name == 'slideshow_activity':
                    focused_model.playing = False
            else:
                # retrieve the menu activity we're currently browsing.
                if len(self.menu.controller) > 0:
                    index = self.menu.controller.current_index
                    menu_activity = self.menu.controller[index].model.activity
                    
                    # we want no preview in audio & service activities
                    if menu_activity.name in ('video_activity',
                                              'dvd_activity',
                                              'image_activity',
                                              'player_activity'
                                              ):

                            last_panel.opacity = 255
                        
        elif key == 'theme_path':
            try:
                application = common.application
                theme = application.plugin_registry.create_component(new_value)
                application.config.set_option('theme',
                                               new_value,
                                               section=self.name)
                self.frontend.theme = theme
            except:
                self.warning("Cannot create theme %r for frontend %r", new_value,
                             self.name)
                raise



    def _is_root_menu(self, controller): 
        if isinstance(controller, NodeControllerClass) and isinstance(controller.parent, NodeControllerClass):
            return False
        return True
        
