def substract(range1, range2):
    """
    Compute the difference between two ranges (L{range1}[0], L{range1}[1]) and
    (L{range2}[0], L{range2}[1]) (range1 - range2).

    range1 cannot contain strictly range2. (eg.: (10, 20), (13, 17) is not
    allowed however (10, 20), (10, 17) is)).

    @param range1: boundaries of the first range
    @type range1:  tuple of int
    @param range2: boundaries of the second range
    @type range2:  tuple of int

    @rtype: tuple of int or None

    @raises: IndexError if range2 is included in range1
    @raises: IndexError if range1 or range2 are not valid ranges
    """
    # helpers
    def is_range(range):
        return range[0] <= range[1]

    def contains(range1, range2):
        return range1[0] <= range2[0] and range1[1] >= range2[1]

    def strict_contains(range1, range2):
        return range1[0] < range2[0] and range1[1] > range2[1]

    # ranges validity
    if not is_range(range1) or not is_range(range2):
        raise IndexError

    if strict_contains(range1, range2):
        raise IndexError

    # empty intersection
    if range2[1] < range1[0] or range2[0] > range1[1]:
        return range1

    # range1 included in range2
    if contains(range2, range1):
        return None

    # range2 included in range1
    if contains(range1, range2):
        if range2[0] > range1[0]:
            inf = range1[0]
            sup = range2[0]-1
        else:
            inf = range2[1]+1
            sup = range1[1]

        return (inf, sup)

    # TODO: the general case is covered by the previous one
    # general case
    if range2[0] >= range1[0]:
        inf = range1[0]
        sup = range2[0]-1

    if range2[1] <= range1[1]:
        inf = range2[1]+1
        sup = range1[1]

    return (inf, sup)

def cache(total, cached, index):
    """
    Given a L{total} number of elements and a number of L{cached} elements
    amongst them, compute which elements are currently cached.

    @param total:  total number of elements
    @type total:   int
    @param cached: number of cached elements; inferior to L{total} and odd
    @type cached:  int
    @param index:  the current index; must be between 0
                   and L{total}-1
    @type index:   int
    @rtype:        tuple of int
    """
    if total == 0:
        return None

    half = cached/2
    inf = max(index-half, 0)
    sup = min(index+half, total-1)
    return (inf, sup)

def update_index(total, cached, old_index, new_index):
    """
    Given a L{total} number of elements and a number of L{cached} elements
    amongst them, compute which currently cached elements have to be removed
    and which ones have to be added for a change of index.
    It returns two ranges: the first one for the removed elements, the second
    one for the added elements (eg.: ((1,3), (5,6)), ((1,3), None)...).

    @param total:     total number of elements
    @type total:      int
    @param cached:    number of cached elements; inferior to L{total} and odd
    @type cached:     int
    @param old_index: the current index before the change; must be between 0
                      and L{total}-1
    @type old_index:  int
    @param new_index: the current index after the change; must be between 0
                      and L{total}-1
    @type new_index:  int

    @rtype: tuple of tuple of int
    """
    old_cache = cache(total, cached, old_index)
    new_cache = cache(total, cached, new_index)

    to_add = substract(new_cache, old_cache)
    to_remove = substract(old_cache, new_cache)

    return (to_remove, to_add)

def insert(total, cached, index, inserted):
    """
    Given a L{total} number of elements and a number of L{cached} elements
    amongst them, compute the index of the cached element that has to be
    removed for an insertion at index L{inserted}. Returns None if no element
    has to be removed.

    @param total:     total number of elements
    @type total:      int
    @param cached:    number of cached elements; inferior to L{total} and odd
    @type cached:     int
    @param index:     the current index; must be between 0 and L{total}-1
    @type index:      int
    @param inserted:  index of the newly inserted element; must be between 0
                      and L{total}
    @type inserted:   int

    @rtype: int
    """
    range = cache(total, cached, index)
    if range == None:
        return None

    if total < cached/2+1:
        return None

    if inserted < range[0] or inserted > range[1]:
        return None

    if inserted < index:
        return range[0]
    else:
        return range[1]

def remove(total, cached, index, removed):
    """
    Given a L{total} number of elements and a number of L{cached} elements
    amongst them, compute the index of the cached element that has to be
    added for a removal at index L{removed}. Returns None if no element
    has to be added.

    @param total:     total number of elements
    @type total:      int
    @param cached:    number of cached elements; inferior to L{total} and odd
    @type cached:     int
    @param index:     the current index; must be between 0 and L{total}-1
    @type index:      int
    @param removed:   index of the newly removed element; must be between 0
                      and L{total}
    @type removed:    int

    @rtype: int
    """
    return insert(total, cached, index, removed)


class CacheList(list):
    """
    A proxy list which purpose is to delay creation of its elements. In order
    to achieve that the items inserted are callable that return the actual
    elements. At any time a given number of elements are really instantiated
    (cached).
    Their choice is driven by this number and the L{current_index} in the list.
    They are chosen so that the item at L{current_index} and those around it
    are instantiated.

    Example: for 3 cached elements in a list of 10 with current_index = 4, the
             instantiated elements are the 3rd, 4th and 5th (3, 4, 5).
    
    @ivar current_index: index of the item currently selected
    @type current_index: int
    """

    def __init__(self, cached, real_list):
        """
        @param cached:    number of cached elements; must be odd
        @type cached:     int
        @param real_list: list to progressively fill; must be empty
        @type real_list:  list
        """
        list.__init__(self)
        self._cached = cached
        self._real_list = real_list
        self._current_index = 0

    def _in_real_list(self, key):
        return key - max(self._current_index, self._cached/2) + \
               self._cached/2

    def insert(self, position, element):
        list.insert(self, position, element)
        if len(self._real_list) < self._cached/2+1:
            self._real_list.insert(position, element())
        else:
            to_remove = insert(len(self), self._cached, \
                               self._current_index, position)
            if to_remove != None:
                if to_remove <= self._current_index:
                    self._real_list.pop(0)
                else:
                    self._real_list.pop(len(self._real_list)-1)
                self._real_list.insert(self._in_real_list(position), element())

        self._update_real_index()

    def _update_real_index(self):
        # FIXME: selected_item does not exist everywhere
        try:
            index = self._current_index
            self._real_list.selected_item = self._in_real_list(index)
        except:
            pass

    def pop(self, position=-1):
        list.pop(self, position)
        to_add = remove(len(self), self._cached, \
                        self._current_index, position)
        
        to_remove = self._in_real_list(position)

        if to_remove >= 0 and to_remove < len(self._real_list):
            self._real_list.pop(to_remove)

        if to_add != None:
            element = self[to_add]
            self._real_list.insert(self._in_real_list(to_add), element())

        self._update_real_index()

    def append(self, element):
        self.insert(len(self), element)

    def extend(self, elements):
        for e in elements:
            self.append(e)

    def remove(self, element):
        self.pop(self.index(element))

    def reverse(self):
        # FIXME: to implement
        raise NotImplementedError

    def sort(self, key=None):
        raise NotImplementedError

    def __iadd__(self, elements):
        # FIXME: to implement
        raise NotImplementedError

    def __imul__(self, mult):
        # FIXME: to implement
        raise NotImplementedError

    def current_index__set(self, index):
        if index < 0 or index >= len(self):
            return

        old_index = self._current_index
        new_index = index

        delta = new_index - old_index
        to_remove, to_add = update_index(len(self), self._cached, \
                                         old_index, new_index)

        if to_add != None:
            for i in range(to_add[0], to_add[1]+1):
                element = self[i]
                if delta >= 1:
                    self._real_list.append(element())
                else:
                    self._real_list.insert(0, element())
        if to_remove != None:
            for i in range(to_remove[0], to_remove[1]+1):
                if delta >= 1:
                    self._real_list.pop(0)
                else:
                    self._real_list.pop(-1)

        self._current_index = index
        self._update_real_index()

    def current_index__get(self):
        return self._current_index

    current_index = property(current_index__get, current_index__set)

