# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.core import plugin_registry, common
TreeViewClass = plugin_registry.get_component_class('base:tree_view')

import pgm
from pypgmtools.widgets.top_level_menu import *
from pypgmtools.timing import implicit
from pypgmtools.graph.group import Group
from pypgmtools.graph.text import Text
from pypgmtools.graph.image import Image

from twisted.internet import reactor

from elisa.extern.translation import Translatable


class TreeView(TreeViewClass):

    supported_controllers = ('poblenou:tree_controller',)
    default_associations = (
    ('poblenou:node_controller', 'poblenou:node_view'),
    )

    def __init__(self):
        TreeViewClass.__init__(self)
        self.context_path = 'pigment:pigment_context'

        # description widget
        self._description = None
        self._description_string = ""
        # the descriptive string of the current selected item is displayed
        # after a given time
        self._description_delay = 0.200
        self._description_delayed = None

        # menu widget
        self._menu = None

        # arrow widget that indicates that an item contains more items
        self._arrow = None
        self._arrow_delay = 0.200
        self._arrow_delayed = None

        # invisible drawable used to capture mouse click and return to the
        # previous menu level
        self._previous_menu_zone = None

        # invisible drawable used to capture mouse drag on the menu levels
        self._drag_level_zone = None
        self._drag_started = False
        self._drag_start_position = None
        self._drag_start_index = 0

        # button giving an hint on how to go to the previous menu
        self._back_button = None

        # drawable for loading and empty directory user feedback
        self._loading = None

        # dictionary of drawables that will be reused many times thus avoiding
        # reloading the pictures from disk; this cache is cleaned when self gets
        # destroyed
        self.icons_cache = {}

    def get_icon_drawable(self, icon_theme_name):
        if self.icons_cache.has_key(icon_theme_name):
            return self.icons_cache[icon_theme_name]

        self.debug("drawable for icon %s not in cache; creating it" \
                   % icon_theme_name)

        img = pgm.Image()
        image_path = self.frontend.theme.get_media(icon_theme_name)
        img.set_from_fd(os.open(image_path, os.O_RDONLY))
        img.set_name(icon_theme_name)
        self.icons_cache[icon_theme_name] = img

        self.debug("drawables cache now containing %s drawables" \
                    % len(self.icons_cache))
 
        # FIXME: Crashes if the Image is not in the canvas: see Pigment
        # ticket #163
        canvas = self.frontend.context.canvas
        canvas.add(pgm.DRAWABLE_MIDDLE, img)
       
        return img

    def frontend_changed(self, previous_frontend, new_frontend):
        if new_frontend == None:
            return

        # FIXME: re-bind the widgets to the new canvas

        # connects to the canvas_resized signal of the new context
        if new_frontend != previous_frontend:
            new_frontend.context.canvas_resized.connect(self.canvas_resized)
            new_frontend.theme_changed.connect(self.theme_changed)

    def canvas_resized(self, size):
        canvas = self.frontend.context.canvas

        if self._description != None:
            self._scale_description()
        if self._arrow != None:
            self._scale_arrow()
        if self._menu != None:
            self._scale_menu()
        if self._previous_menu_zone != None:
            self._scale_previous_menu_zone()
        if self._drag_level_zone != None:
            self._scale_drag_level_zone()
        if self._back_button != None:
            self._scale_back_button()
        if self._loading != None:
            self._scale_loading()
            
    def theme_changed(self, new_theme):
        index = 0
        for child_controller in self.controller:
            icon, blurred, reflected = self._get_child_icons(child_controller)
            self._menu.update_images(index, icon, blurred, reflected)
            index += 1

        # update the icon drawables cache
        for icon_theme_name, drawable in self.icons_cache.iteritems():
            icon_path = self.frontend.theme.get_media(icon_theme_name)
            drawable.set_from_fd(os.open(icon_path, os.O_RDONLY))
            drawable.set_name('icon_theme_name')
 
    def controller_changed(self):
        canvas = self.frontend.context.canvas
        self.root_group = Group(canvas, pgm.DRAWABLE_MIDDLE)
        self.context_handle = self.root_group

        self._create_menu()
        self._create_description()
        self._create_arrow()
        self._create_previous_menu_zone()
        self._create_drag_level_zone()
        self._create_back_button()
        self._create_loading()

        TreeViewClass.controller_changed(self)

    def _create_menu(self):
        canvas = self.frontend.context.canvas
        # FIXME: move these 2 to canvas_resized
        menu_selected_position = (canvas.width*0.475, canvas.height*(-0.13), -10.0)
        menu_selected_size = (canvas.width, canvas.width)

        self._menu = TopLevelMenu(canvas, pgm.DRAWABLE_MIDDLE,
                                  width                 = canvas.width,
                                  height                = canvas.height,
                                  font_family           = "Nimbus Sans L",
                                  selected_position     = menu_selected_position,
                                  selected_size         = menu_selected_size,
                                  selected_opacity      = 32,
                                  mode                  = CAROUSEL,
                                  path_steps            = 1,
                                  duration              = 500,
                                  transformation        = implicit.DECELERATE)

        self._menu.bg_color = (0, 0, 0, 0)
        self._menu.opacity = 255
        self.root_group.add(self._menu)

        self._animated_menu = implicit.AnimatedObject(self._menu)
        self._animated_menu.mode = implicit.REPLACE
        self._animated_menu.setup_next_animations(duration = 550,
                                                  transformation = implicit.DECELERATE)
        self._scale_menu()
        self._menu.visible = True
        
        
    def _scale_menu(self):
        # set canvas size dependent properties of menu (size, position)
        canvas = self.frontend.context.canvas
        self._menu.width = canvas.width
        self._menu.height = canvas.height
 
        self._unselected_menu_position = ((canvas.width-self._menu.width)/2.0,
                                          (canvas.height-self._menu.height)/2.0,
                                          -500.0)
        self._selected_menu_position = ((canvas.width-self._menu.width)/2.0,
                                        (canvas.height-self._menu.height)/2.0,
                                         0.0)
        
        self._animated_menu.stop_animations()
        if self.controller.selected:
            self._menu.position = self._selected_menu_position
        else:
            self._menu.position = self._unselected_menu_position

        self._menu.layout()
        self._menu.update()
      
    def _create_description(self):
        # description text
        self._description = Text()
        self._description.label = ""
        self._description.set_name('(empty)')
        self._description.font_family = "Nimbus Sans L"
        self._description.bg_color = (0, 0, 0, 0)
        self._description.alignment = pgm.TEXT_ALIGN_CENTER
        self._description.ellipsize = pgm.TEXT_ELLIPSIZE_END
        self._description.opacity = 255
        
        self._description.connect("clicked", self._description_clicked)

        self._animated_description = implicit.AnimatedObject(self._description)
        self._animated_description.mode = implicit.REPLACE
        self._animated_description.setup_next_animations(duration = 400)

        self._scale_description()
        self._description.visible = True

        self.root_group.add(self._description)

    def _scale_description(self):
        # set canvas size dependent properties of description (size, position,
        # font_height)
        canvas = self.frontend.context.canvas
        self._description.font_height = canvas.height*0.073
        self._description.size = (canvas.width/2.0, canvas.height*0.11)
        self._description.x = (canvas.width - self._description.width)/2.0
        self._description.y = canvas.height * (1.0 - 0.025) - \
                              self._description.height
        self._description.z = 0.0

    def display_description(self, description):
        # hide the currently displayed description
        if self._description.opacity != 0:
            self._animated_description.stop_animations()
            self._description.opacity = 0

        if isinstance(description, Translatable):
            trans = common.application.translator
            text = trans.translateTranslatable(description,
                                               self.frontend.languages)
        else:
            text = description

        self._description_string = text

        # (re)schedule the display of description
        if self._description_delayed != None and self._description_delayed.active():
            self._description_delayed.reset(self._description_delay)
        else:
            self._description_delayed = reactor.callLater(self._description_delay, self._show_description)

    def _show_description(self):
        # show the loaded description
        self._description_delayed = None
        self._description.label = self._description_string
        self._description.set_name(self._description_string)
        self._animated_description.opacity = 255

    def _create_arrow(self):
        # arrow indicating that more items are in the current selected item
        self._arrow = Image()
        self._arrow.bg_color = (0, 0, 0, 0)
        self._arrow.layout = pgm.IMAGE_SCALED
        self._arrow.alignment = pgm.IMAGE_CENTER
        self._arrow.opacity = 0

        self._arrow.connect("clicked", self._description_clicked)

        arrow_path = self.frontend.theme.get_media("down_arrow_icon")
        self._arrow.set_from_fd(os.open(arrow_path, os.O_RDONLY))
        self._arrow.set_name('down_arrow_icon')

        self._animated_arrow = implicit.AnimatedObject(self._arrow)
        self._animated_arrow.setup_next_animations(duration = 350)
        self._animated_arrow.setup_next_animations(attribute = "y",
                                                   duration = 350,
                                                   repeat_count = implicit.INFINITE,
                                                   repeat_behavior = implicit.REVERSE,
                                                   transformation = implicit.SMOOTH)
        self._scale_arrow()
        self._arrow.visible = True
        self.root_group.add(self._arrow)

    def _scale_arrow(self):
        canvas = self.frontend.context.canvas
        self._arrow.font_height = canvas.height*0.073
        # FIXME: height and y
        self._arrow.size = (canvas.width, canvas.height*0.03)
        self._arrow.x = (canvas.width - self._arrow.width)/2.0
        self._arrow.y = canvas.height * (1.0 - 0.01) - self._arrow.height
        self._arrow.z = 0.0
       
    def display_arrow(self, visible):
        # hide the arrow
        if self._arrow.opacity != 0:
            self._animated_arrow.stop_animations()
            self._arrow.opacity = 0
            self._scale_arrow()

        # (re)schedule the display of the arrow
        if self._arrow_delayed != None and self._arrow_delayed.active():
            if visible:
                self._arrow_delayed.reset(self._arrow_delay)
            else:
                self._arrow_delayed.cancel()
        elif visible:
                self._arrow_delayed = reactor.callLater(self._arrow_delay, self._show_arrow)

    def _show_arrow(self):
        self._animated_arrow.opacity = 255
        self._animated_arrow.y += 0.01


    def _create_back_button(self):
        # button giving an hint on how to go to the previous menu
        self._back_button = Image()
        self._back_button.bg_color = (0, 0, 0, 0)
        self._back_button.layout = pgm.IMAGE_SCALED
        self._back_button.alignment = pgm.IMAGE_CENTER
        self._back_button.opacity = 0

        back_button_path = self.frontend.theme.get_media("back_button")
        self._back_button.set_from_fd(os.open(back_button_path, os.O_RDONLY))
        self._back_button.set_name("back_button")

        self._animated_back_button = implicit.AnimatedObject(self._back_button)
        self._animated_back_button.mode = implicit.REPLACE
        self._animated_back_button.setup_next_animations(duration = 400)

        self._scale_back_button()
        self._back_button.visible = True

        self.root_group.add(self._back_button)
    
    def _scale_back_button(self):
        canvas = self.frontend.context.canvas
        back_size = canvas.height*0.1
        offset = canvas.height*0.02
        self._back_button.size = (back_size, back_size)
        self._back_button.x = canvas.width - self._back_button.width - offset
        self._back_button.y = offset
        self._back_button.z = 0.0

    def display_back_button(self, visible):
        if visible:
            self._animated_back_button.opacity = 255
        else:
            self._animated_back_button.opacity = 0

    def _create_loading(self):
        self._loading = Image()

        loading_icon = self.frontend.theme.get_media("loading_icon")
        self._loading.set_from_fd(os.open(loading_icon, os.O_RDONLY))
        self._loading.set_name("loading_icon")

        self._loading.bg_color = (0, 0, 0, 0)
        self._loading.opacity = 1
        self._loading.layout = pgm.IMAGE_SCALED
        self._loading.alignment = pgm.IMAGE_CENTER

        self._animated_loading = implicit.AnimatedObject(self._loading)
        self._animated_loading.mode = implicit.REPLACE
        self._animated_loading.setup_next_animations(attribute="opacity",
                                                     duration = 200)
        params = {"duration": 350, \
                  "repeat_count": implicit.INFINITE, \
                  "repeat_behavior": implicit.REVERSE, \
                  "transformation": implicit.SMOOTH}
        self._animated_loading.setup_next_animations(attribute="width",
                                                     **params)
        self._animated_loading.setup_next_animations(attribute="height",
                                                     **params)
        self._animated_loading.setup_next_animations(attribute="x",
                                                     **params)
        self._animated_loading.setup_next_animations(attribute="y",
                                                     **params)

        self._scale_loading()
        self._loading.visible = True

        self.root_group.add(self._loading)

    def _scale_loading(self):
        canvas = self.frontend.context.canvas
        self._loading.width = canvas.width/6.0
        self._loading.height = self._loading.width*3.0/4.0
        self._loading.position = (canvas.width/2.0-self._loading.width/2.0,
                                  canvas.height*(1.0-0.05)-self._loading.height,
                                  0.0)

    def display_loading(self, value):
        if value:
            loading_icon = self.frontend.theme.get_media("loading_icon")
            self._loading.set_from_fd(os.open(loading_icon, os.O_RDONLY))
            self._loading.set_name("loading_icon")

            self._animated_loading.opacity = 255
            delta = 0.05
            self._animated_loading.x -= delta/2.0
            self._animated_loading.y -= delta/2.0
            self._animated_loading.width += delta
            self._animated_loading.height += delta
        else:
            self._animated_loading.stop_animations()
            self._animated_loading.opacity = 1
            self._scale_loading()

    def display_empty(self, value):
        if value:
            empty_icon = self.frontend.theme.get_media("empty_icon")
            self._loading.set_from_fd(os.open(empty_icon, os.O_RDONLY))
            self._loading.set_name("empty_icon")
            self._animated_loading.stop_animations()
            self._animated_loading.opacity = 255
        else:
            self._animated_loading.stop_animations()
            self._animated_loading.opacity = 1
            self._scale_loading()

 
    def _create_previous_menu_zone(self):
        # invisible drawable used to capture mouse click and return to the
        # previous menu level
        self._previous_menu_zone = Image()
        self._previous_menu_zone.opacity = 0
        self._previous_menu_zone.position = (0.0, 0.0, 0.0)

        self._scale_previous_menu_zone()
        self._previous_menu_zone.visible = True

        canvas = self.frontend.context.canvas
        canvas.add(pgm.DRAWABLE_FAR, self._previous_menu_zone)

        self._previous_menu_zone.connect('clicked',
                                         self._previous_menu_zone_clicked)

        self._previous_menu_zone.connect('drag_begin', self._drag_level_zone_drag_begin)
        self._previous_menu_zone.connect('drag_motion', self._drag_level_zone_drag_motion)
        self._previous_menu_zone.connect('drag_end', self._drag_level_zone_drag_end)

    def _scale_previous_menu_zone(self):
        canvas = self.frontend.context.canvas
        self._previous_menu_zone.size = (canvas.width, canvas.height * 0.75)


    def _create_drag_level_zone(self):
        # invisible drawable used to capture mouse drag on the menu levels
        self._drag_level_zone = Image()
        self._drag_level_zone.opacity = 0

        self._scale_drag_level_zone()
        self._drag_level_zone.visible = True
        
        canvas = self.frontend.context.canvas
        canvas.add(pgm.DRAWABLE_FAR, self._drag_level_zone)

        self._drag_level_zone.connect('drag_begin', self._drag_level_zone_drag_begin)
        self._drag_level_zone.connect('drag_motion', self._drag_level_zone_drag_motion)
        self._drag_level_zone.connect('drag_end', self._drag_level_zone_drag_end)

    def _scale_drag_level_zone(self):
        canvas = self.frontend.context.canvas
        self._drag_level_zone.size = (canvas.width, canvas.height * 0.25)
        self._drag_level_zone.x = 0.0
        self._drag_level_zone.y = canvas.height - self._drag_level_zone.height
        self._drag_level_zone.z = 0.0

    def _drag_level_zone_drag_begin(self, drawable, x, y, z, button, time):
        
        if self.controller.selected_controller is not self.controller.backend.focused_controller:
            return False
        
        if button == pgm.BUTTON_LEFT:
            self._drag_started = True
            self._drag_start_position = (x, y)
            level_controller = self.controller.selected_controller
            self._drag_start_index = level_controller.current_index
            return True
           
    def _drag_level_zone_drag_end(self, drawable, x, y, z, button, time):
        if button == pgm.BUTTON_LEFT and self._drag_started:
            self._drag_started = False

            dx = self._drag_start_position[0] - x
            dy = self._drag_start_position[1] - y

            level_controller = self.controller.selected_controller
            if self._drag_start_index == level_controller.current_index:
                if dy > 0.05:
                    # go up one level if we are not currently at the root level
                    if self.controller.selected_controller != self.controller:
                        self.controller.selected_controller.exit_node()
                    return True

                elif dy < -0.05:
                    # go down one level
                    self.controller.selected_controller.enter_node()
                    return True
            
        return False

    def _drag_level_zone_drag_motion(self, drawable, x, y, z, button, time):
        if self._drag_started == True:
            dx = self._drag_start_position[0] - x

            # drag the current level
            level_controller = self.controller.selected_controller

            if level_controller == self.controller:
                # root menu
                visible_items = 4
                increment = int(dx/float(self._drag_level_zone.width)*visible_items)
                level_controller.current_index = (self._drag_start_index + \
                                                  increment) % len(level_controller)
            else:
                # other menu levels
                visible_items = 11
                increment = int(dx/float(self._drag_level_zone.width)*visible_items)
                level_controller.current_index = self._drag_start_index+increment

            return True

    def attribute_set(self, key, old_value, new_value):
        if key == 'current_index':
            self.debug("new index for %s: %s" % (id(self), new_value))
            text = self[self.controller.current_index].controller.model.text
            self.display_description(text)
            self._menu.focused_item = new_value

        elif key == 'selected':
            self.debug("%s (un)selected: %s" % (id(self), new_value))
            if new_value:
                # display the menu
                if self._menu.is_selected():
                    self._menu.select()
                self._animated_menu.position = self._selected_menu_position

                # empty the description
                text = self[self.controller.current_index].controller.model.text
                self.display_description(text)
                self.display_arrow(False)
                self.display_back_button(False)

                # selected child
                child_index = self.controller.current_index
                if child_index >= 0 and child_index < len(self):
                    self[child_index].hide()

            else:
                # hide the menu
                if not self._menu.is_selected():
                    self._menu.select()
                self._animated_menu.position = self._unselected_menu_position

                if self.frontend.context.touchscreen:
                    self.display_back_button(True)
        
    def _get_child_icons(self, child_controller):
        # retrieve the paths to the icon, its blurred version and its reflected
        # version
        icon = child_controller.model.theme_icon
        icon_path = self.frontend.theme.get_media(icon)

        blurred_icon = icon[0:icon.find('_')] + "_blur_icon"
        blurred_icon_path = self.frontend.theme.get_media(blurred_icon)

        reflected_icon = icon[0:icon.find('_')] + "_reflected_icon"
        reflected_icon_path = self.frontend.theme.get_media(reflected_icon)

        return (icon_path, blurred_icon_path, reflected_icon_path)

    def child_view_creating(self, view, controller, position):
        TreeViewClass.child_view_creating(self, view, controller, position)

        # add a child view representation to the menu
        icon, blurred, reflected = self._get_child_icons(controller)
        if isinstance(controller.model.text, Translatable):
            trans = common.application.translator
            text = trans.translateTranslatable(controller.model.text,
                                                self.frontend.languages)
        else:
            text = controller.model.text
        self._menu.insert(position, icon, blurred, reflected, text)
        
        # FIXME: hack to access drawables from the top menu; a better solution
        # would imply an API addition to the top menu
        for widget in self._menu._item_handles[position]:
            widget.connect('clicked', self._entry_clicked, controller)
        
        self._menu.layout()
        self._menu.update()

    def _entry_clicked(self, drawable, x, y, z, button, time, controller):
        self.debug("%s clicked" % controller.model.text)
        
        if self.controller.selected_controller == self.controller:
            # the root menu is currently selected
            current_index = self.controller.current_index
            clicked_index = self.controller.index(controller)
            if current_index != clicked_index:
                # rotate the menu to the clicked entry
                self.controller.current_index = clicked_index
            else:
                # enter the clicked entry of the menu
                self.controller.enter_node()

            return True

        return False

    def _previous_menu_zone_clicked(self, drawable, x, y, z, button, time):
        self.debug("Previous menu level zone clicked")

        # do not process the click if there was dragging
        # happening
        if self._drag_level_zone_drag_end(drawable, x, y, z, button, time):
            return True

        if self.controller.selected_controller != self.controller:
            self.controller.selected_controller.exit_node()
            return True

        return False

    def _description_clicked(self, drawable, x, y, z, button, time):
        if not self._animated_description.visible or self._animated_description.opacity == 0 or \
           not self.root_group.visible or self.root_group.opacity == 0 or \
           self._description_string =="":
            return False
        
        self.controller.root.selected_controller.activate_node()
        
        return True


