# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'
__maintainer2__ = 'Lionel Martin <lionel@fluendo.com>'


from elisa.base_components.controller import Controller
from elisa.core import player
from elisa.core import common
from elisa.core.input_event import *
from elisa.core import plugin_registry

ListModelClass = plugin_registry.get_component_class('base:list_model')

class PlayerController(Controller):

    supported_models = ('base:player_model')
    default_associations = (
    ('base:list_model','base:list_controller'),
    )

    state = None
    uri = None
    seek_to = 0
    playlist = None
    volume = 1.0
    muted = False
    stopped = False
    pause_requested = False

    # FIXME: add a documentation for the configuration options
    default_config = {'volume_increment_step': '0.02',
                      'volume_decrement_step': '0.02',
                      'seek_forward_step': '30',
                      'seek_backward_step': '15',
                      'show_status_on_ok': '0'
                      }
    
    def __init__(self):
        Controller.__init__(self)
        self._position_when_seek = -1

    def initialize(self):
        config = self.config
        self._show_status_on_ok = int(config['show_status_on_ok'])
        self._volume_increment_step = float(config['volume_increment_step'])
        self._volume_decrement_step = float(config['volume_decrement_step'])
        forward_step = int(config['seek_forward_step'])
        backward_step = int(config['seek_backward_step'])
        self._seek_forward_step = forward_step * 1000000000
        self._seek_backward_step = backward_step * 1000000000
        
    def model_changed(self, old_model, new_model):
        self.uri = self.model.uri
        self.state = self.model.state
        self.playlist = self.create_child_controller(self.model.playlist)

    def handle_input(self, input_event):
        self.debug("Handling input event with action %r", input_event.action)
        
        if input_event.action == EventAction.PLAY:
            self.stopped = False
            self.model.state = player.STATES.PLAYING
        
        elif input_event.action == EventAction.PAUSE:
            self.stopped = False
            self.model.state = player.STATES.PAUSED
        
        elif input_event.action == EventAction.STOP:
            self.stopped = True
            self.model.state = player.STATES.STOPPED
        
        elif input_event.action in (EventAction.GO_UP, EventAction.VOL_UP):
            self.model.volume = min(2.0, self.model.volume + \
                                         self._volume_increment_step)
            
        elif input_event.action in (EventAction.GO_DOWN,
                                    EventAction.VOL_DOWN):
            self.model.volume = max(0.0, self.model.volume - \
                                         self._volume_decrement_step)
            
        elif input_event.action in (EventAction.GO_RIGHT,
                                    EventAction.SEEK_FORWARD):
            if self._position_when_seek > 0 and \
                   self._position_when_seek == self.model.position:
                seek_to = self.model.seek_to + self._seek_forward_step
            else:
                self._position_when_seek = self.model.position
                seek_to = self.model.position + (self._seek_forward_step * 2)
            self.model.seek_to = seek_to
        elif input_event.action == EventAction.MUTE:
            self.model.muted = not self.model.muted
            
        elif input_event.action == EventAction.NEXT:
            # next track in playlist
            uri = self.model.uri            
            if uri != None and \
                   isinstance(self.model.playlist, ListModelClass) and \
                   self.model.state in (player.STATES.PAUSED,
                                        player.STATES.PLAYING):
                self._next_track(uri)

        elif input_event.action == EventAction.PREVIOUS:
            # previous track in playlist
            uri = self.model.uri            
            if uri != None and \
                   isinstance(self.model.playlist, ListModelClass) and \
                   self.model.state in (player.STATES.PAUSED,
                                        player.STATES.PLAYING):
                self._previous_track(uri)
            
        elif input_event.action in (EventAction.GO_LEFT,
                                    EventAction.SEEK_BACKWARD):
            if self._position_when_seek > 0 and \
                   self._position_when_seek == self.model.position:
                seek_to = self.model.seek_to - self._seek_backward_step
            else:
                self._position_when_seek = self.model.position
                seek_to = self.model.position - self._seek_backward_step
            self.model.seek_to = seek_to
        
        elif input_event.action == EventAction.OK:
            if self._show_status_on_ok:
                # First time user presses OK we remember it and next time
                # we toggle player model's state

                if self.model.state == player.STATES.PLAYING:
                    self.model.pause_requested = True

                elif self.model.state == player.STATES.PAUSED:
                    self.model.state = player.STATES.PLAYING
            else:
                if self.model.state == player.STATES.PAUSED:
                    self.model.state = player.STATES.PLAYING
                else:
                    self.model.state = player.STATES.PAUSED
        return False

    def attribute_set(self, key, old_value, new_value):
        self.debug("Attribute %r set to %r", key, new_value)
        Controller.attribute_set(self, key, old_value, new_value)
        if key == 'state':
            if self.state == new_value:
                return

            if self.model.volume != self.volume:
                self.volume = self.model.volume
            if new_value == player.STATES.PLAYING:
                self.state_changed(new_value)
            elif new_value == player.STATES.PAUSED:
                self.pause_requested = False
                self.state_changed(new_value)
            elif new_value == player.STATES.STOPPED:
                self.pause_requested = False
                self._position_when_seek = -1
                self.state_changed(new_value)
        elif key == 'uri':
            self.state = None
            self.uri = new_value
        elif key == 'seek_to':
            self.seek_to = new_value
        elif key == 'volume':
            self.volume = new_value
        elif key == 'muted':
            self.muted = new_value
        elif key == 'pause_requested':
            self.pause_requested = new_value

    def _next_track(self, uri):
        success = False
        if uri in self.model.playlist:
            current_index = self.model.playlist.index(self.model.uri)
            new_index = current_index + 1
            if len(self.model.playlist) > 1:
                if new_index < len(self.model.playlist):
                    new_uri = self.model.playlist[new_index]
                    self.model.uri = new_uri
                    self.state = player.STATES.PLAYING
                    self.model.state = player.STATES.PLAYING
                    success = True
        return success

    def _previous_track(self, uri):
        success = False
        if uri in self.model.playlist:
            current_index = self.model.playlist.index(self.model.uri)
            new_index = current_index - 1
            if new_index > -1:
                new_uri = self.model.playlist[new_index]
                self.model.uri = new_uri
                self.state = player.STATES.PLAYING
                self.model.state = player.STATES.PLAYING
                success = True
        return success
        

    def state_changed(self, state):
        uri = self.model.uri
        if uri != None:           
            # playlist management
            
            if state == player.STATES.STOPPED and \
                   isinstance(self.model.playlist, ListModelClass) \
                        and not self.stopped:
                if self._next_track(uri):
                    return
            if state == player.STATES.PLAYING and not self.focused \
                   and not self.model.preview_mode:
                self.focus()
            
            elif self.focused and state == player.STATES.STOPPED:
                # FIXME : ugly. need a window like system ?
                self.parent.show_menu()
 
        self.state = state
