# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


"""
Media data access support
"""


__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'

from elisa.core import common, component, media_uri
from elisa.core.utils import network
from elisa.core.bus import bus_message
from elisa.base_components.media_provider import MediaProvider
from elisa.extern import upnp_content_directory
from twisted.internet import defer, threads


class UPnPMedia(MediaProvider):
    """


    TODO:

      - implement blocking_next_location
    """
    name = "upnp_media"

    def __init__(self):
        MediaProvider.__init__(self)
        self.cache = {}
        self.device_cache = {}

    def initialize(self):
        MediaProvider.initialize(self)
        common.application.bus.register(self._bus_message_received,
                                        bus_message.ComponentsLoaded)

    def _bus_message_received(self, msg, sender):
        import louie
        msg = bus_message.CoherenceDevice('Coherence.UPnP.ControlPoint.MediaServer.detected',
                                          self._detected_media_server,
                                          louie.Any)
        common.application.bus.send_message(msg)

        msg = bus_message.CoherenceDevice('Coherence.UPnP.ControlPoint.MediaServer.removed',
                                          self._removed_media_server,
                                          louie.Any)
        common.application.bus.send_message(msg)

    def _service_in_db(self, control_url):
        uri = "upnp%s" % control_url[4:]
        media_manager = common.application.media_manager
        if media_manager.enabled:
            media = media_manager.get_source_for_uri(uri)
        else:
            media = None
        return media is not None

    def _get_upnp_target(self, uri):
        new_uri = media_uri.MediaUri(uri)
        if uri.scheme == 'upnp':
            new_uri.scheme = 'http'
        return new_uri

    def _get_control_url(self, upnp_uri):
        url = media_uri.MediaUri(upnp_uri)
        url.fragment = u''
        return url

    def _cache_enabled(self):
        media_manager = common.application.media_manager
        return media_manager.enabled

    def _detected_media_server(self, client, usn):
        device = client.device
        friendly_name = device.get_friendly_name()

        self.info('Browsing services on device %s' % friendly_name)
        self.debug('Device USN is: %r', usn)

        services = device.get_services()
        self.info("Device has %s services" % len(services))
        for service in services:
            self.info("Service type: %s" % service.get_type())
            cds1 = "urn:schemas-upnp-org:service:ContentDirectory:1"
            cds2 = "urn:schemas-upnp-org:service:ContentDirectory:2"
            if service.get_type() in (cds1, cds2):
                #service.subscribe()
                control_url = service.get_control_url()
                if not self._service_in_db(control_url):
                    self.device_cache[usn] = (control_url, friendly_name)
                    dfr = self._browse(client, 0, requested_count=0,
                                       friendly_name=friendly_name,
                                       parent_id=-1)
                    dfr.addCallback(self._build_menus, friendly_name,
                                    control_url, 0)
                else:
                    self._build_menus(None, friendly_name, control_url, 0)

    def _removed_media_server(self, usn):
        if usn in self.device_cache:
            control_url, friendly_name = self.device_cache[usn]
            del self.device_cache[usn]
            self.info("Device %s disappeared", friendly_name)
            uri = "upnp:%s#0" % control_url[5:]
            action_type = bus_message.MediaLocation.ActionType.LOCATION_REMOVED
            msg = bus_message.LocalNetworkLocation(action_type,
                                                   str(friendly_name),
                                                   'upnp', uri)
            common.application.bus.send_message(msg)

    def _browse(self, client, container_id, starting_index=0,
                requested_count=0, friendly_name="", parent_id=None):
        """ Create an UPnP client and return a Deferred
        """
        try:
            client = client.content_directory
        except:
            pass
        control_url = client.url

        self.debug("Browsing %r", control_url)

        if hasattr(client, 'new_browse'):
            browse = client.new_browse
        else:
            browse = client.browse
            
        dfr = browse(container_id, starting_index=starting_index,
                     requested_count=requested_count)
        dfr.addCallback(self._add_to_cache, container_id, control_url,
                        friendly_name, client, parent_id)
        return dfr

    def _add_to_cache(self, results, container_id, control_url, friendly_name,
                      client, parent_id, child=None):
        if not child:
            root = upnp_content_directory.buildHierarchy(results['items'],
                                                         container_id,
                                                         friendly_name, False)
            self.store_in_cache(control_url, container_id, friendly_name,
                                root, client)
            if container_id != 0:
                root.parentID = parent_id

        else:
            root = child



        self.store_in_cache(control_url, root.id, friendly_name, root,client)
        for child in root.children:
            if isinstance(child, upnp_content_directory.Folder):
                self._add_to_cache(None, container_id, control_url,
                                   friendly_name, client, parent_id, child=child)
            else:
##                 for url in child.urls.keys():
##                     self.store_in_cache(url,None,
##                                         friendly_name, child,client)
                self.store_in_cache(control_url, child.id, friendly_name,
                                    child,client)
        return root.children

    def _get_cached_item_with_id(self, id):
        for item_id, (uri, item, client) in self.cache.iteritems():
            if item.id == id:
                return (uri, item.name, item)
        return (None, None, None)

    def store_in_cache(self, control_url, container_id, friendly_name, data,
                       client=None):
        if container_id is not None:
            uri = "%s#%s" % (control_url, container_id)
        else:
            uri = control_url
        #data.parentID = container_id
        #if uri not in self.cache or (len(data.children) > len(self.cache[uri][1].children) and self.cache[uri][1].children != data.children):
        self.debug("Storing %s (%s) to cache", uri, data.name)#, data.parentID)
        self.cache[uri] = (friendly_name, data,client)

    def get_from_cache(self, control_url, container_id):
        uri = "%s#%s" % (control_url, container_id)
        return self.cache.get(uri,(None,None,None))

    def _build_menus(self, results, friendly_name, control_url, container_id):
        """
        Build a menu according browsing results and dispatch it to the plugins
        """
        upnp_uri = "upnp:%s#%s" % (control_url[5:], container_id)
        #new_device.emit(str(friendly_name), upnp_uri, None)
        action_type = bus_message.MediaLocation.ActionType.LOCATION_ADDED
        msg = bus_message.LocalNetworkLocation(action_type, str(friendly_name),
                                               'upnp', upnp_uri)
        common.application.bus.send_message(msg)

    def scannable_uri_schemes__get(self):
        # TODO: remove this method, when we have a good http media_provider
        #       => grant scan access to the media_scanner
        return {}

    def supported_uri_schemes__get(self):
        return {'upnp': 0}

    def blocking_get_media_type(self, uri):
        file_type = 'directory'
        mime_type = ''
        name, container, dummy = self.cache.get(str(self._get_upnp_target(uri)),
                                                (None,None, None))
        if isinstance(container, upnp_content_directory.Item):
            file_type = container.get_media_type()
                        
        media_type = {'file_type': file_type, 'mime_type': mime_type}
        return media_type

    def get_real_uri(self, uri):
        real_uri = None
        name, container, dummy = self.cache.get(str(self._get_upnp_target(uri)),
                                                (None,None, None))
        if isinstance(container, upnp_content_directory.Item):
            real_uri = media_uri.MediaUri(self._get_best_uri(container))
        return real_uri
    
    def _get_parent(self, for_uri):
        parent = None
        searched_uri = media_uri.MediaUri(for_uri)
        searched_uri.fragment = None

        def find_parent(folder_uri, cached_item):
            p = None

            if isinstance(cached_item, upnp_content_directory.Folder):
                for child in cached_item.children:
                    if isinstance(child, upnp_content_directory.Item):
                        urls = child.get_urls()
                        keys = urls.keys()
                        keys.sort()
                        child_uri = keys[0]
                        if child_uri == str(searched_uri):
                            p = folder_uri
                    else:
                        f_uri = media_uri.MediaUri(folder_uri)
                        f_uri.fragment = child.id
                        p = find_parent(f_uri, child)
                    if p:
                        break
            return p

        if for_uri.scheme == 'http':
            for uri, (name, cached_item, dummy) in self.cache.iteritems():
                parent = find_parent(uri, cached_item)
                if parent:
                    parent = media_uri.MediaUri(uri.replace('http','upnp'))
                    break

        else:
            for_uri = self._get_upnp_target(for_uri)
            name, cached_item, dummy = self.cache.get(str(for_uri),
                                                      (None, None, None))
            if cached_item:
                found = None
                for uri, (short_name, item, d) in self.cache.iteritems():
                    if item.id == cached_item.parentID:
                        parent = media_uri.MediaUri(uri.replace('http','upnp'))
                        break
        return parent

    def blocking_is_directory(self, uri):
        is_dir = True
        upnp_uri = self._get_upnp_target(uri)
        friendly_name, folder, client = self.cache.get(upnp_uri,(None,None,
                                                                 None))
        if folder:
            is_dir = isinstance(folder, upnp.content_directory.Folder)
        return is_dir
    
    def has_children_with_types(self, uri, media_types):
        dfr = self.get_direct_children(uri, [])

        def check_media_type(media_type):
            file_type = media_type.get('file_type')
            return file_type in media_types

        def final(result):
            has_children = False
            for dummy, r in result:
                if r:
                    has_children = True
                    break
            return has_children

        def check_children(result):
            dfrs = []
            for child in result:
                d = self.get_media_type(child[0])
                d.addCallback(check_media_type)
                dfrs.append(d)
            dfr2 = defer.DeferredList(dfrs)
            dfr2.addCallback(final)
            return dfr2

        dfr.addCallback(check_children)
        return dfr


    def get_direct_children(self, uri, children):
        dfr = defer.Deferred()
        uri = self._get_upnp_target(uri)
        control_url = self._get_control_url(uri)
        friendly_name, folder, client = self.cache.get(str(uri),(None,None,None))

        if folder and isinstance(folder, upnp_content_directory.Folder):
            if folder.children:
                folder_children = folder.children
                self._process_children(folder_children, control_url, children)
                dfr.callback(children)
            else:
                dfr2 = self._browse(client, folder.id,
                                    friendly_name=folder.name,
                                    parent_id=folder.parentID)
                dfr2.addCallback(self._process_children, control_url, children,
                                 dfr)
        else:
            dfr.callback(children)
        return dfr

    def _process_children(self, folder_children, control_url, children,
                          dfr=None):
        for child in folder_children:
            child_uri = "upnp:%s#%s" % (control_url[5:], child.id)
            is_dir = isinstance(child, upnp_content_directory.Folder)
            
            if is_dir:
                child_type = 'directory'
            else:
                child_type = 'file'

            child_uri = media_uri.MediaUri(child_uri)
            if isinstance(child.name, str):
                child.name = child.name.decode('utf-8')
            child_uri.label = child.name.encode('utf-8')
            children.append((child_uri,{}))

        if dfr:
            dfr.callback(children)

        return children

    def _get_best_uri(self, item):
        my_address = network.get_host_address()
        child_uri = None
        for uri, protocolInfo in item.urls_by_protocol.iteritems():
            remote_protocol,remote_network,remote_content_format,dummy = protocolInfo.split(':')
            if remote_network == my_address:
                child_uri = uri
                break

        if not child_uri:
            urls = item.get_urls()
            keys = filter(lambda u: not u.startswith('file'), urls.keys())
            keys.sort()
            child_uri = keys[0]
        return child_uri

    def open(self, uri, mode=None):
        htt_uri = media_uri.MediaUri(uri)
        http_uri.scheme = 'http'
        return common.application.media_manager.open(http_uri, mode=mode)

    def next_location(self, uri, root=None):
        upnp_uri = self._get_upnp_target(uri)
        control_url = self._get_control_url(upnp_uri)
        friendly_name, folder, client = self.cache.get(str(upnp_uri),
                                                       (None,None,None))
        
        def strip(child):
            if uri.scheme == 'http':
                next_uri = media_uri.MediaUri(root)
            else:
                next_uri = media_uri.MediaUri(uri)
            next_uri.fragment = child.id
            next_uri.label = child.name
            return next_uri

        def get_first_child(children):
            first_child = None
            if children:
                first_child = children[0]
                first_child = strip(first_child)
            return first_child


        def go_up(p_uri, p_folder, cur_f):
            idx = 0
            for child in p_folder.children:
                if child.id == cur_f.id:
                    break
                idx += 1
            if idx == len(p_folder.children) -1:
                p = self._get_parent(p_uri)
                r = None
                if not p:
                    p = root
                fname, parent_folder, c = self.cache.get(str(self._get_upnp_target(p)),
                                                         (None,None,None))
                r = go_up(p, parent_folder, p_folder)
            else:
                try:
                    r = p_folder.children[idx+1]
                except IndexError:
                    r = None
            return r

        dfr = defer.Deferred()

        if folder and isinstance(folder, upnp_content_directory.Folder):
            if len(folder.children):
                dfr.callback(get_first_child(folder.children))
            else:
                dfr = self._browse(client, folder.id,friendly_name=folder.name,
                                   parent_id=folder.parentID)
                dfr.addCallback(get_first_child)

        else:
            item = folder
            parent_uri = self._get_parent(uri)
            if parent_uri:
                fname, parent_folder, c = self.cache.get(str(self._get_upnp_target(parent_uri)),
                                                         (None,None,None))
                
                idx = parent_folder.children.index(item)
                try:
                    next_child = parent_folder.children[idx+1]
                except IndexError:
                    # let's go up level
                    next_child = go_up(parent_uri, parent_folder, item)

                if isinstance(next_child, upnp_content_directory.Folder):
                    if len(next_child.children):
                        dfr.callback(get_first_child(next_child.children))
                    else:
                        dfr = self._browse(client, next_child.id,
                                           friendly_name=next_child.name,
                                           parent_id=next_child.parentID)
                        dfr.addCallback(get_first_child)

                elif isinstance(next_child, upnp_content_directory.Item):
                    dfr.callback(strip(next_child))
                else:
                    dfr.callback(None)
        return dfr
