/* $Id: e2_fs_mount.c 557 2007-07-24 11:59:11Z tpgww $

Copyright (C) 2004-2007 Florian Zaehringer <flo.zaehringer@web.de>
Copyright (C) 1999-2004 Bill Wilson <bill@gkrellm.net>

This file is part of emelFM2.
emelFM2 is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

emelFM2 is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emelFM2; see the file GPL. If not, contact the Free Software
Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

/**
@file src/filesystem/e2_fs_mount.c
@brief filesystem mountpoint related functions

Functions dealing with fstab / mtab have in part been sourced from gkrellm
(http://gkrellm.net) written by Bill Wilson.
*/

#include "emelfm2.h"

#ifdef E2_FS_MOUNTABLE

#include <string.h>
#include <unistd.h>

//shorthand for bsd-like os's supported here
#if defined(__FreeBSD__) || defined(__OpenBSD__) || defined(__NetBSD__)
#define __E2BSD__
#endif

//in general, we support [u]mount completion
#define E2_FS_MOUNTABLE
// OSX 10.4 ("Tiger") automounts all devices. Unlike its predecessors, it uses statvfs()
#if defined(darwin)
#include <sys/types.h>
#include <sys/param.h>
#ifdef HAVE_SYS_STATVFS_H
#undef E2_FS_MOUNTABLE
#endif
#endif
//other exclusions go here ...

# if defined(__linux__)
#  include <mntent.h>
#  include <fstab.h>

# elif defined(__E2BSD__)	//maybe this should just be defined(__FreeBSD__) || defined(__OpenBSD__)
#  include <sys/param.h>
#  include <sys/mount.h>
#  include <fstab.h>

# elif defined(__solaris__) || defined(sco)
//CHECKME sco ?? svr5 in general ??
#  include <sys/mnttab.h>
#  include <vfstab.h>

# else
#  include <mntent.h>
#  include <fstab.h>

/*Some systems use statfs() to provide information about mounted
file systems, other systems use statvfs(). The header files used with
these functions vary between systems.
e.g.
HPUX 	  statfs() with sys/vfs.h
SunOS   statfs() with sys/vfs.h
Solaris  statvfs() with sys/statvfs.h */

#  include <sys/types.h>
#  include <sys/param.h>
#  ifdef HAVE_SYS_STATVFS_H
#   include <sys/statvfs.h>
#  endif
#  ifdef HAVE_SYS_VFS_H
#   include <sys/vfs.h>
#  endif

/*#ifdef HAVE_SYS_STATVFS_H
#define STATFS statvfs
#else
#define STATFS statfs
#endif

...
     struct STATFS sfs;
...
     rc = STATFS(filename, &sfs);
*/
# endif	//which os-specifis includes

//test which mounted devices should be included in the list
//of unmountable ones
//CHECKME does this need to be os-specific ? if so, how ?
# if defined(__linux__) || defined(__E2BSD__)
#  define E2_TEST_MOUNTED_DEVICE \
		if (   strcmp (dev, "none") \
			&& strcmp (dev, "/dev") \
			&& strcmp (type, "proc") \
			&& strcmp (type, "sysfs") \
			&& strcmp (type, "fuse") \
			&& strcmp (dir, G_DIR_SEPARATOR_S)) /*no need to support umounting '/' */ \

# else
#  define E2_TEST_MOUNTED_DEVICE \
		if (   strcmp (dev, "none") \
			&& strcmp (dev, "/dev") \
			&& strcmp (type, "sysfs") \
			&& strcmp (type, "proc") \
			&& strcmp (dir, G_DIR_SEPARATOR_S)) /*no need to support umounting '/' */ \

# endif
/*	these should be handled by dev="none"
			&& strcmp (type, "devpts")
			&& strcmp (type, "proc")
			&& strcmp (type, "usbdevfs")
			&& strcmp (type, "usbfs")
			&& strcmp (type, "sysfs")
			&& strstr (dir, "/dev/") */
/* permission-checking disabled pending doing it properly, if not forever
			gchar *opt = mounted->mnt_opts;
			if (!permcheck || _e2_complete_mount_permission (dir, opt)) */
/*for bsd's, which mount options are relevant ? how do we check them ?
		mounted[i].f_flags	//mount flags
		mounted[i].f_owner	//owner
*/
/* there may be things blocking an unmount eg monitoring with dnotify */

//finesse mounted device name before inclusion in the list of
//unmountable devices
# define E2_PROCESS_MOUNTED_DEVICE
//strip any trailing / from the mountpoint name
//			s = strrchr (dir, G_DIR_SEPARATOR);
//			if (s != NULL && s != dir && *(s+1) == '\0')
//				*s = '\0';

//test which unmounted devices should be included in the list
//of mountable ones
//CHECKME does this need to be os-specific ? if so, how ?
# define E2_TEST_UNMOUNTED_DEVICE \
		if (   strcmp (dev, "none") \
			&& strcmp (type, "swap") \
			&& strcmp (type, "proc") \
			&& strcmp (type, "sysfs") \
			&& strcmp (type, "ignore") \
			&& strcmp (dir, G_DIR_SEPARATOR_S)	/*no need to support mounting '/' */ \
			&& type[0] != '\0' )
/*
//			&& g_file_test (dir, G_FILE_TEST_IS_DIR) don't bother checking for this (slow)
    these should be handled by dev="none"
			&& strcmp (type, "devpts")
			&& strcmp (type, "proc")
			&& strcmp (type, "sysfs")
			&& strcmp (type, "usbdevfs")
			&& strcmp (type, "usbfs")
			&& strstr (dir, "/dev/") == NULL
CHECKME sometimes, test may need to include
	strncmp (dev, "/dev", 4)
	realpath (dev, realdev) != NULL
	gchar realdev[PATH_MAX];
*/
/* permission-checking disabled pending doing it properly, if not forever
			gchar *opt = fs.vfs_mntopts;
			if (!permcheck || _e2_complete_mount_permission (dir, opt)) */

//finesse unmounted device name before inclusion in the list of
//mountable devices
# define E2_PROCESS_UNMOUNTED_DEVICE
			//strip any trailing / from the mountpoint name
//			s = strrchr (dir, G_DIR_SEPARATOR);
//			if (s != NULL && s != dir && *(s+1) == '\0')
//				*s = '\0';

//guint uid;
//time_t fstab_mtime;

/**
@brief check whether @a localpath is a filesystem mountpoint

@param localpath localised absolute path string

@return TRUE if @a localpath is a mountpoint
*/
gboolean e2_fs_mount_is_mountpoint (gchar *localpath)
{
	struct stat sb;
	if (e2_fs_lstat (localpath, &sb E2_ERR_NONE()) || !S_ISDIR (sb.st_mode))
		return FALSE;

	gboolean matched = FALSE;
	gchar *utf = F_FILENAME_FROM_LOCALE (localpath);
	GList *points = e2_fs_mount_get_mounts_list ();
	if (points != NULL)
	{
		matched = (g_list_find_custom (points, utf, (GCompareFunc) e2_list_strcmp) != NULL);
		e2_list_free_with_data (&points);
	}
#if defined(__linux__) || defined(__FreeBSD__)
	if (!matched)
	{
		points = e2_fs_mount_get_fusemounts_list ();
		if (points != NULL)
		{
			matched = (g_list_find_custom (points, utf, (GCompareFunc) e2_list_strcmp) != NULL);
			e2_list_free_with_data (&points);
		}
	}
#endif
	F_FREE (utf);
	return matched;
}
/**
@brief add @a dir to list @a mounts

The string is copied, with conversion to utf-8 if need be

@param dir localised string with path of mountpoint to be added
@param points store of pointer to the list to be updated

@return
*/
static void _e2_fs_mount_add_to_list (gchar *dir, GList **points)
{
	gchar *utf = D_FILENAME_FROM_LOCALE (dir);
	*points = g_list_append (*points, utf);
}

#if defined(__linux__) || defined(__FreeBSD__)
/**
@brief create list of fuse mountpoints
The list is created by the appropriate os-specifc protocol.
Mountpoints are stored as utf8 strings
@param mounts_list pointer to list for storing results
@return list of utf-8 mountpoints, or NULL
*/
GList *e2_fs_mount_get_fusemounts_list (void)
{
	GList *mounts_list = NULL;
	//FIXME lock the data source while accessing it
#if defined(__linux__)
	struct mntent *mounted;
	E2_FILE *f;
/* When the linux proc filesystem is mounted, the files /etc/mtab and /proc/mounts
	have very similar contents. The former has somewhat more information (CHECKME),
	such as the mount options used, but is not necessarily up-to-date */
	if ((f = setmntent ("/proc/mounts", "r")) == NULL)
		if ((f = setmntent ("/etc/mtab", "r")) == NULL)
			return NULL;	//FIXME report an error

	while ((mounted = getmntent (f)) != NULL)
	{
		if (!strcmp (mounted->mnt_type, "fuse"))
			_e2_fs_mount_add_to_list (mounted->mnt_dir, &mounts_list);
	}
	endmntent (f);
#elif defined(__FreeBSD__)
	struct statfs *mounted;

	//NOTE: data provided by this is not process- or thread-safe
	gint i, count = getmntinfo (&mounted, MNT_NOWAIT);

	for (i=0; i<count; i++)
	{
		if (!strcmp (mounted[i].f_fstypename, "fuse"))
			_e2_fs_mount_add_to_list (mounted[i].f_mntonname, &mounts_list);
	}
#endif
	return mounts_list;
}
#endif //which OS
/* *
@brief replace escaped character codes in @a buf
@param buf buffer with string from fs/mtab, presumably locale-encoded
@return
*/
/* UNUSED
static void _e2_fs_mount_fix_fstab_name (gchar *buf)
{
	gchar *rp, *wp;

	if (buf[0] == '\0')
		return;
	rp = buf;
	wp = buf;
	do	// This loop same as in libc6 getmntent()
		if (rp[0] == '\\' && rp[1] == '0' && rp[2] == '4' && rp[3] == '0')
		{
			*wp++ = ' ';		// \040 is a SPACE
			rp += 3;
		}
		else if (rp[0] == '\\' && rp[1] == '0' && rp[2] == '1' && rp[3] == '2')
		{
			*wp++ = '\t';		// \012 is a TAB
			rp += 3;
		}
		else if (rp[0] == '\\' && rp[1] == '\\')
		{
			*wp++ = '\\';		// \\ is a \
			rp += 1;
		}
		else
			*wp++ = *rp;
	while (*rp++ != '\0');
} */
/* *
@brief check whether current user is permitted to [u]mount mountpoint @a device
@param device mountpoint path-string, presumably locale-encoded
@param options is a string with the (4th) parameters string from a line in fs/mtab
@return TRUE if permitted
*/
/* UNUSED FIXME, if this is used, needs to account for sudo & group membership etc
 and work properly for all os's
static gboolean _e2_fs_mount_permission (gchar* device, gchar *options)
{
	if (uid == 0 	//root can always do it
		|| strstr (options, "dev,") != NULL //CHECKME _I(
		|| (strstr (options, "user") != NULL && strstr (options, "nouser") == NULL)) //CHECKME _I(
		return TRUE;
#if defined (__linux__)
	struct stat my_stat;
	//no need to localise *device before stat
	return (e2_fs_stat (device, &my_stat E2_ERR_NONE()) == 0
		&& my_stat.st_uid == uid
		&& strstr (options, "owner") != NULL //CHECKME _I(
	   );
#else
	return FALSE;
#endif
} */
/**
@brief create list of mounted partitions

The list is created by the appropriate os-specifc protocol

@return list of mounted partitions (utf-8 strings), or NULL
*/
GList *e2_fs_mount_get_mounts_list (void)	//gboolean permcheck)
{
	GList *mounts_list = NULL;
	//FIXME lock the data source while accessing it
#if defined(__linux__)
	struct mntent *mounted;
	E2_FILE *f;
/* When the linux proc filesystem is mounted, the files /etc/mtab and /proc/mounts
	have very similar contents. The former has somewhat more information (CHECKME),
	such as the mount options used, but is not necessarily up-to-date */
	if ((f = setmntent ("/proc/mounts", "r")) == NULL)
		if ((f = setmntent ("/etc/mtab", "r")) == NULL)
			return NULL;

	while ((mounted = getmntent (f)) != NULL)
	{
		//check the mounted device meets our needs
		//need to take a copy if string is altered
		gchar *dev = mounted->mnt_fsname;
		gchar *dir = mounted->mnt_dir;
		gchar *type = mounted->mnt_type;
		E2_TEST_MOUNTED_DEVICE
		{
			E2_PROCESS_MOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &mounts_list);
		}
	}
	endmntent (f);
#elif defined(__E2BSD__)
	struct statfs *mounted;

	//NOTE: data provided by this is not process- or thread-safe
	gint i, count = getmntinfo (&mounted, MNT_NOWAIT);

	for (i=0; i<count; i++)
	{
		//check the mounted device meets our needs
		//need to take a copy if string is altered
		gchar *dev = mounted[i].f_mntfromname;
		gchar *dir = mounted[i].f_mntonname;
		gchar *type = mounted[i].f_fstypename;
		E2_TEST_MOUNTED_DEVICE
		{
			E2_PROCESS_MOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &mounts_list);
		}
	}
#elif defined(__solaris__) || defined(sco)
	//CHECKME sco?
	struct mnttab mounted;
	E2_FILE *f;

	if ((f = e2_fs_open_stream ("/etc/mnttab", "r")) == NULL)
		return NULL;

	while (getmntent (f, &mounted) == 0)
	{
		//check the mounted device meets our needs
		//need to take a copy if string is altered
		gchar *dev = mounted.mnt_special;
		gchar *dir = mounted.mnt_mountp;
//		gchar *type = mounted.mnt_fstype;
		E2_TEST_MOUNTED_DEVICE
		{
			E2_PROCESS_MOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &mounts_list);
		}
	}
	e2_fs_close_stream (f);
#elif defined(hpux)
	struct mntent *mounted;
	E2_FILE *f;

	const gchar *mtab = (e2_fs_access ("/etc/pfs_mtab", F_OK E2_ERR_PTR())) ?
		"/etc/mtab" : "/etc/pfs_mtab";
	if ((f = setmntent (mtab, "r")) == NULL)
		return NULL;

	while ((mounted = getmntent (f)) != NULL)
	{
		//check the mounted device meets our needs
		//need to take a copy if string is altered
		gchar *dev = mounted->mnt_fsname;
		gchar *dir = mounted->mnt_dir;
//		gchar *type = mounted->mnt_type;
		E2_TEST_MOUNTED_DEVICE
		{
			E2_PROCESS_MOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &mounts_list);
		}
	}
	endmntent (f);
#elif defined(__svr4__)
	struct mnttab mounted;
	E2_FILE *f;

	if ((f = e2_fs_open_stream ("/etc/mtab", "r") == NULL)
		return NULL;

	while (getmntent (f, &mounted) ==0)
	{
		gchar *dev = mounted.f_mntfromname;
		gchar *dir = mounted.f_mntonname;
//		gchar *type = mounted.f_fstypename;
		E2_TEST_MOUNTED_DEVICE
		{
			E2_PROCESS_MOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &mounts_list);
		}
	}
	e2_fs_close_stream (f);
#else
//other unix-like os's do it like linux, but without /proc/mount
//CHECKME sometimes with a different filename ??
	struct mntent *mounted;
	E2_FILE *f;

	if ((f = setmntent ("/etc/mtab", "r")) == NULL)
		return NULL;

	while ((mounted = getmntent (f)) != NULL)
	{
		//check the mounted device meets our needs
		//need to take a copy if string is altered
		gchar *dev = mounted->mnt_fsname;
		gchar *dir = mounted->mnt_dir;
//		gchar *type = mounted->mnt_type;
		E2_TEST_MOUNTED_DEVICE
		{
			E2_PROCESS_MOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &mounts_list);
		}
	}
	endmntent (f);
#endif	//which OS
	return mounts_list;
}
/**
@brief create list of mountable partitions

The list is created by the appropriate os-specifc protocol

@return list of partitions (utf-8 strings), or NULL
*/
GList *e2_fs_mount_get_mountable_list (void)	//gboolean permcheck)
{
	GList *fstab_list = NULL;
	//CHECKME lock data file while accessing it ??
#if defined(__E2BSD__) || defined(__linux__)
    struct fstab *fs;

	if (!setfsent ())
		return NULL;

	while ((fs = getfsent ()) != NULL)
	{
		gchar *dev = fs->fs_spec;
		gchar *dir = fs->fs_file;
		gchar *type = fs->fs_vfstype;
/*		copy strings if they need to be modified
		g_strlcpy (dev, fs->fs_spec, sizeof(dev));
		g_strlcpy (dir, fs->fs_file, sizeof(dir));
		g_strlcpy (type, fs->fs_vfstype, sizeof(type));
		_e2_complete_mount_fix_fstab_name (dev);
		_e2_complete_mount_fix_fstab_name (dir);
		_e2_complete_mount_fix_fstab_name (type); */
		E2_TEST_UNMOUNTED_DEVICE
		{
			E2_PROCESS_UNMOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &fstab_list);
		}
    }
    endfsent ();
#elif defined(__solaris__) || defined(sco)
	//CHECKME sco
    struct vfstab fs;
	E2_FILE *f;

	if ((f = e2_fs_open_stream ("/etc/vfstab", "r")) == NULL)
		return NULL;

	while (getvfsent (f, &fs) == 0)
	{
		gchar *dev = fs.vfs_special;
		gchar *dir = fs.vfs_mountp;
//		gchar *type = fs.vfs_fstype;
		E2_TEST_UNMOUNTED_DEVICE
		{
			E2_PROCESS_UNMOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &fstab_list);
		}
	}
	e2_fs_close_stream (f);
#elif defined(hpux)
    struct mntent *mountable;
	E2_FILE *f;

    //handle PFS if necessary
    const gchar *fstab = (access("/etc/pfs_fstab", F_OK)) ?
		"/etc/fstab" : "/etc/pfs_fstab";

	if ((f = setmntent (fstab, "r")) == NULL)
		return NULL;

	while ((mountable = getmntent (f)) != NULL)
	{
		gchar *dev = mountable->mnt_fsname;
		gchar *type = mountable->mnt_type;
		E2_TEST_UNMOUNTED_DEVICE
		{
			E2_PROCESS_UNMOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &fstab_list);
		}
	}
	endmntent (f);
#elif defined(__svr4__)
	struct mnttab mountable;
	E2_FILE *f;

	if ((f = e2_fs_open_stream ("/etc/fstab", "r")) == NULL)
		return NULL;

	while (getmntent (f, &mountable) == 0)
	{
		gchar *dev = mountable.mnt_special;
		gchar *dir = mountable.mnt_mountp;
		E2_TEST_UNMOUNTED_DEVICE
		{
			E2_PROCESS_UNMOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &fstab_list);
		}
	}
	e2_fs_close_stream (f);
#else
	struct mntent *mountable;
	E2_FILE *f;

	if ((f = setmntent ("/etc/fstab", "r" )) == NULL)
		return NULL;

	while ((mountable = getmntent (f)) != NULL)	//data is not thread- or process-safe
	{
		gchar *dev = mountable->mnt_fsname;
		gchar *dir = mountable->mnt_dir;
//		gchar *type = mountable->mnt_type;
//		gchar *opts = mountable->mnt_opts;
		E2_TEST_UNMOUNTED_DEVICE
		{
			E2_PROCESS_UNMOUNTED_DEVICE
			_e2_fs_mount_add_to_list (dir, &fstab_list);
		}
	}
	endmntent (f);
#endif	//which OS
	return fstab_list;
}

/**
@brief register mountpoint-related actions
@return
*/
void e2_fs_mount_actions_register (void)
{
	gchar *action = g_strconcat (_A(56),".",_A(24), NULL);
	e2_action_register (action, E2_ACTION_TYPE_ITEM,
		e2_menu_create_mounts_menu, NULL, TRUE,
		E2_ACTION_EXCLUDE_MENU,
		NULL);
}
#endif //def E2_FS_MOUNTABLE
