/* $Id: e2_widget.c 544 2007-07-20 12:48:20Z tpgww $

Copyright (C) 2003-2007 tooar <tooar@gmx.net>
Portions copyright (C) 1999 Michael Clark.

This file is part of emelFM2.
emelFM2 is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

emelFM2 is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emelFM2; see the file GPL. If not, contact the Free Software
Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

/**
@file src/utils/e2_widget.c
@brief miscelleanous GtkWidget utility functions

This file contains various miscelleanous GtkWidget utility functions.
*/

#include "emelfm2.h"
#include <string.h>
#include "e2_widget.h"
#include "e2_dialog.h"

#ifdef USE_GTK2_12
typedef struct
{
	gboolean oneactive;
	gchar *tip1;
	gchar *tip2;
} E2_TipsToggle;

static void _e2_widget_destroy_ttips (E2_TipsToggle *data)
{
	g_free (data->tip1);
	g_free (data->tip2);
	DEMALLOCATE (E2_TipsToggle, data);
}
/**
@brief setup for a button to have 2 alternate tooltips

@param widget the button widget
@param initialtip initially-displayed tip, utf-8, my have markup
@param othertip alternate tip, utf-8, my have markup

@return
*/
void e2_widget_set_toggletip (GtkWidget *widget, const gchar *initialtip,
	const gchar *othertip)
{
	gboolean QQQQTIPFIXME;
//a tooltip window doesn't exist until the tip is ready to show
	E2_TipsToggle *data = MALLOCATE (E2_TipsToggle);
	if (data != NULL)
	{
		data->tip1 = g_strdup (initialtip);
		data->tip2 = g_strdup (othertip);
		data->oneactive = TRUE;
		g_object_set_data_full (G_OBJECT (widget), "btn-toggle-data", data,
			(GDestroyNotify) _e2_widget_destroy_ttips )
	}
	gtk_widget_set_tooltip_text (widget, initialtip);
}
/**
@brief toggle the visiblity of 2 tip labels setup by e2_widget_set_toggletip()

@param widget the button widget

@return
*/
void e2_widget_swap_tooltip (GtkWidget *widget)
{
	gboolean QQQQTIPFIXME;
	E2_TipsToggle *data = g_object_get_data (G_OBJECT (widget), "btn-toggle-data");
	if (data != NULL)
	{
		if (data->oneactive)
		{
			gtk_widget_set_tooltip_text (widget, data->tip2);
			data->oneactive = FALSE;
		}
		else
		{
			gtk_widget_set_tooltip_text (widget, data->tip1);
			data->oneactive = TRUE;
		}
	}
}
#else
void e2_widget_set_tooltip (GtkTooltips *tooltips, GtkWidget *widget, gchar *text)
{
	if (text == NULL)
		return;
	if (tooltips == NULL)
		tooltips = app.tooltips; //use default tooltip group
	gtk_tooltips_set_tip (tooltips, widget, text, NULL);
}
void e2_widget_swap_tooltip (GtkWidget *widget)
{
	GtkTooltipsData *tips = gtk_tooltips_data_get (widget);
	gchar *tmp = tips->tip_text;
	tips->tip_text = tips->tip_private;
	tips->tip_private = tmp;
}
#endif
/**
@brief get image widget for an icon

@param icon gtk-stock-item name, or localised custom-icon filename with or without path
@param size enumerator of the desired image size

@return Image widget, or NULL
*/
GtkWidget *e2_widget_get_icon (gchar *icon, GtkIconSize size)
{
	if ((icon == NULL) || (*icon == '\0'))
		return NULL;
	GtkWidget *image = NULL;

#ifdef E2_IMAGECACHE
	//get cached image, after adding it to the cache if not there already
	E2_Image *cache_img = e2_cache_image_get (icon, size);
	image = gtk_image_new_from_pixbuf (cache_img->pixbuf);
#else
	if (e2_utils_check_stock_icon (icon))
		image = gtk_image_new_from_stock (icon, size);
	else
	{
		GdkPixbuf *pixbuf;
		gchar *fullname;
		if (g_path_is_absolute (icon))
			fullname = icon;
		else
		{
			gchar *localpath = e2_utils_get_icons_path (TRUE);
			fullname = e2_utils_strcat (localpath, icon);
			g_free (localpath);
		}

		if ((pixbuf = gdk_pixbuf_new_from_file (fullname, NULL)) != NULL)
		{
			GdkPixbuf *pixbuf2;
			gint width, height;
			if (!gtk_icon_size_lookup (size, &width, &height))
			{
				width = 16; height = 16;
			}
			pixbuf2 = gdk_pixbuf_scale_simple (pixbuf, width, height,
				GDK_INTERP_BILINEAR);
			image = gtk_image_new_from_pixbuf (pixbuf2);
			g_object_unref (pixbuf2);
			g_object_unref (pixbuf);
		}

		if (fullname != icon)
			g_free (fullname);
	}
#endif
	return image;
}

GtkWidget *e2_widget_add_mid_label (GtkWidget *box, gchar *text, gfloat align,
	gboolean exp, guint pad)
{
	return e2_widget_add_label (box, text, align, 0.5, exp, pad);
}

GtkWidget *e2_widget_add_label (GtkWidget *box, gchar *text,
	gfloat xalign, gfloat yalign, gboolean exp, guint pad)
{
	GtkWidget *label = gtk_label_new (NULL);
	if (box != NULL)
		gtk_box_pack_start (GTK_BOX (box), label, exp, TRUE, pad);
	gtk_label_set_markup (GTK_LABEL (label), text);
//	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label), xalign, yalign);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_selectable (GTK_LABEL (label), FALSE);
	gtk_widget_show (label);
	return label;
}
/**
@brief callback for entry widget key-press, to shorten string in @a entry
<Shift>Delete removes from cursor to end
@param entry the entry widget
@param event pointer to event data struct
@param data UNUSED data specified when callback was connnected

@return TRUE if the key was <Shift>Delete
*/
static gboolean _e2_widget_entry_keypress_cb (GtkWidget *entry, GdkEventKey *event,
	gpointer data)
{
	if (event->keyval == GDK_Delete)
	{
		guint modifiers = gtk_accelerator_get_default_mod_mask ();
		if ((event->state & modifiers) == GDK_SHIFT_MASK)
		{
			gint start = gtk_editable_get_position (GTK_EDITABLE (entry));
			gtk_editable_delete_text (GTK_EDITABLE (entry), start, -1);
			return TRUE;
		}
	}
	return FALSE;
}

GtkWidget *e2_widget_add_entry (GtkWidget *box, gchar *init_text,
	gboolean exp, gboolean select_text)
{
	GtkWidget *entry;
	entry = gtk_entry_new ();
	if (init_text != NULL)
		gtk_entry_set_text (GTK_ENTRY (entry), init_text);
	if (select_text)
		gtk_editable_select_region (GTK_EDITABLE (entry), 0, -1);
	gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
	g_signal_connect (G_OBJECT (entry), "key-press-event",
		G_CALLBACK (_e2_widget_entry_keypress_cb), NULL);
	if (box != NULL)
		gtk_box_pack_start (GTK_BOX (box), entry, exp, TRUE, E2_PADDING);
	gtk_widget_grab_focus (entry);
	gtk_widget_show (entry);

	return entry;
}

GtkWidget *e2_widget_get_box (gboolean vertical, gboolean homogen, gint spacing)
{
	GtkWidget *box;
	if (vertical)
		box = gtk_vbox_new (homogen, spacing);
	else
		box = gtk_hbox_new (homogen, spacing);
	return box;
}

GtkWidget *e2_widget_add_box (GtkWidget *parent, gboolean exp, guint pad,
	gboolean vertical, gboolean homogen, gint spacing)
{
	GtkWidget *box = e2_widget_get_box (vertical, homogen, spacing);
	gtk_box_pack_start (GTK_BOX (parent), box, exp, TRUE, pad);
	gtk_widget_show (box);
	return box;
}
/**
@brief  create a GtkScrolledWindow without a shadow

@param h_policy  scrolled window horizontal scrolling policy
@param v_policy  scrolled window vertical scrolling policy

@return  the scrolled window widget
*/
GtkWidget *e2_widget_get_sw_plain (GtkPolicyType h_policy, GtkPolicyType v_policy)
{
	return e2_widget_get_sw (h_policy, v_policy, GTK_SHADOW_NONE);
}
/**
@brief  create a GtkScrolledWindow

This function creates a GtkScrolledWindow. The parameters @a h_policy and
@a v_policy are used to set the scrolling policy of the scrolled window.
The shadow type is set to @a shadow.

@param h_policy  scrolled window horizontal scrolling policy
@param v_policy  scrolled window vertical scrolling policy
@param shadow  scrolled window shadow type

@return  the scrolled window widget
*/
GtkWidget *e2_widget_get_sw (GtkPolicyType h_policy, GtkPolicyType v_policy,
	GtkShadowType shadow)
{
 	GtkWidget *scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
		h_policy, v_policy);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window),
		shadow);
 	return scrolled_window;
}
/**
@brief  create a GtkScrolledWindow and add it to a GtkBox

This function creates a GtkScrolledWindow using e2_widget_get_sw_plain() and
adds it to @a box. The parameters @a h_policy and @a v_policy are used as
arguments for e2_widget_get_sw_plain(). @a exp and @a pad are used as arguments
for gtk_box_pack_start() to adjust the packing of the scrolled window.

@param box  the parent box to pack the scrolled window to
@param h_policy  scrolled window horizontal scrolling policy
@param v_policy  scrolled window vertical scrolling policy
@param exp what to do with additional free space in the box
@param pad  number of pixels to pad the scrolled window in the box

@return  the scrolled window
*/
GtkWidget *e2_widget_add_sw (GtkWidget *box, GtkPolicyType h_policy,
	GtkPolicyType v_policy, gboolean exp, guint pad)
{
	GtkWidget *scrolled_window = e2_widget_get_sw_plain (h_policy, v_policy);
	gtk_box_pack_start (GTK_BOX (box), scrolled_window, exp, TRUE, pad);
	gtk_widget_show (scrolled_window);
	return scrolled_window;
}
/**
@brief add a child with a viewport to a GtkScrolledWindow

In contrast to the @a e2_widget_[add|get]* functions, this function does not
create a scrolled window. It only adds a viewport to the scrolled window
@a sw and then adds @a child to the viewport. his is done by calling
gtk_scrolled_window_add_with_viewport(). The shadow type of the viewport is
set to GTK_SHADOW_NONE.

@param sw  the scrolled window
@param child  the child that should be added to the scrolled window

@return
*/
void e2_widget_sw_add_with_viewport (GtkWidget *sw, GtkWidget *child)
{
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sw), child);
	gtk_viewport_set_shadow_type (GTK_VIEWPORT (GTK_BIN (sw)->child),
		GTK_SHADOW_NONE);
}

GtkWidget *e2_widget_add_separator (GtkWidget *box, gboolean exp, guint pad)
{
	GtkWidget *sep;
	if (GTK_IS_HBOX (box))
	{
		sep = gtk_vseparator_new ();
		gtk_box_pack_start (GTK_BOX (box), sep, exp, TRUE, pad);
	} else if (GTK_IS_VBOX (box))
	{
		sep = gtk_hseparator_new ();
		gtk_box_pack_start (GTK_BOX (box), sep, exp, TRUE, pad);
	} else
		return NULL;
	gtk_widget_show (sep);
	return sep;
}

GtkWidget *e2_widget_add_table (
	GtkWidget *box,
	gint rows,
	gint cols,
	gboolean homogen,
	gboolean exp,
	guint pad)
{
	GtkWidget *table;
	table = gtk_table_new (rows, cols, homogen);
	gtk_box_pack_start (GTK_BOX (box), table, exp, TRUE, pad);
	gtk_widget_show (table);

	return table;
}

GtkWidget *e2_widget_add_mid_label_to_table (
	GtkWidget *table,
	gchar *text,
	gfloat xalign,
	gint left, gint right, gint top, gint bottom)
{
	GtkWidget *label;
	label = gtk_label_new (text);
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label), xalign, 0.5);
//	gtk_label_set_selectable (GTK_LABEL (label), FALSE);
	gtk_table_attach_defaults (GTK_TABLE (table), label, left, right, top, bottom);
	gtk_widget_show (label);

	return label;
}
/* UNUSED
GtkWidget *e2_widget_add_entry_to_table (
	GtkWidget *table,
	gchar *init_text,
	gint left, gint right, gint top, gint bottom)
{
	GtkWidget *entry;
	entry = gtk_entry_new ();
	if (init_text != NULL)
		gtk_entry_set_text (GTK_ENTRY (entry), init_text);
	gtk_table_attach_defaults (GTK_TABLE (table), entry, left, right, top, bottom);
	gtk_widget_show (entry);

	return entry;
}
UNUSED
GtkWidget *e2_widget_add_framed_table (GtkWidget *box, gchar *title, gint rows, gint cols,
	gboolean exp, guint pad)
{
	GtkWidget *frame, *table;

	frame = gtk_frame_new (title);
	gtk_box_pack_start (GTK_BOX (box), frame, exp, TRUE, pad);
	gtk_widget_show (frame);

	table = gtk_table_new (rows, cols, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), E2_PADDING_XSMALL);
	gtk_table_set_row_spacings (GTK_TABLE (table), E2_PADDING);
	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_widget_show (table);

	return table;
}
UNUSED
GtkWidget *e2_widget_add_framed_widget (GtkWidget *box, gchar *title,
	GtkWidget *widget, gboolean exp, guint pad)
{
	GtkWidget *frame;

	frame = gtk_frame_new (title);
	gtk_box_pack_start (GTK_BOX (box), frame, exp, TRUE, pad);
	gtk_container_add (GTK_CONTAINER (frame), widget);
	gtk_widget_show (widget);
	gtk_widget_show (frame);

	return frame;
}
*/
GtkWidget *e2_widget_add_frame (GtkWidget *box, gboolean fill, guint pad,
	gchar *title, gboolean stretch)
{
	gchar *real_title = NULL;
	if (title != NULL)
		real_title = g_strconcat (" ", title, " ", NULL);
	GtkWidget *frame = gtk_frame_new (real_title);
	if (title != NULL)
	{
		g_free (real_title);
		GtkWidget *label = gtk_frame_get_label_widget (GTK_FRAME (frame));
		gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	}
	gtk_box_pack_start (GTK_BOX (box), frame, fill, fill, pad);
	return frame;
}

GtkWidget *e2_widget_add_eventbox (GtkWidget *box, gboolean fill, guint pad)
{
	GtkWidget *event = gtk_event_box_new ();
	gtk_box_pack_start (GTK_BOX (box), event, fill, fill, pad);
	return event;
}
/**
@brief create a GtkNotebook

This function create a GtkNotebook and connects @a func to the
"switch-patch" signal of the GtkNotebook.

@param func  the switch-patch callback function or NULL
@param data  data pointer for the callback function

@return  a new GtkNotebook
*/
GtkWidget *e2_widget_get_notebook (gpointer func, gpointer data)
{
	GtkWidget *notebook = gtk_notebook_new ();
	if (func != NULL)
		g_signal_connect (G_OBJECT (notebook), "switch-page", G_CALLBACK (func), data);
	return notebook;

}
/**
@brief  create a notebook and add it to a box

This function creates a new GtkNotebook using e2_widget_get_notebook() and
adds it to @a box.

@param box  the box to add the notebook to
@param fill  fill parameter for the box child
@param pad  pad parameter for the box child
@param func  the switch-patch callback function for the notebook or NULL
@param data  data pointer for the callback function

@return  the notebook that has been added to @a box
*/
GtkWidget *e2_widget_add_notebook (GtkWidget *box, gboolean fill, guint pad,
	gpointer func, gpointer data)
{
	GtkWidget *notebook = e2_widget_get_notebook (func, data);
	gtk_box_pack_start (GTK_BOX (box), notebook, fill, fill, pad);
	return notebook;
}
/**
@brief  create a notebook page and add it to @a notebook

@param dialog the dialog to which the notebook belongs
@param notebook the notebook widget
@param name string with name for the tab

@return  the vbox into which tab content can be placed
*/
GtkWidget *e2_widget_add_notebook_page
	(GtkWidget *dialog, GtkWidget *notebook, gchar *name)
{
	GtkWidget *outerbox = e2_widget_get_box (TRUE, FALSE, 0);
	GtkWidget *label = gtk_label_new (name);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), outerbox, label);
	GtkWidget *scrolled = e2_widget_add_sw (outerbox,
		GTK_POLICY_NEVER, GTK_POLICY_NEVER, TRUE, 0);
	GtkWidget *vbox = e2_widget_get_box (TRUE, FALSE, 0);
	e2_widget_sw_add_with_viewport (scrolled, vbox);
	g_signal_connect (G_OBJECT (GTK_DIALOG (dialog)), "show",
		G_CALLBACK (e2_dialog_show_cb), scrolled);
	return vbox;
}
/**
@brief  set font for @a widget to @a font_string

@param widget widget using the font
@param font_string string naming the font

*/
void e2_widget_set_font (GtkWidget *widget, const gchar *font_string)
{
	PangoFontDescription *font_desc;
	font_desc = pango_font_description_from_string (font_string);
	gtk_widget_modify_font (widget, font_desc);
	pango_font_description_free (font_desc);
}
/**
@brief get approximate size of a character used in @a widget

This retrieves approximate width and/or height of a character in the font
used in @a widget. So it should be called only after a font has been assigned
@a height may not be the whole distance between lines (e.g. spacing applies)

@param widget the widget to be evaluated
@param width pointer to int to store the character pixel-width, or NULL
@param height pointer to int to store the character pixel-height, or NULL

*/
void e2_widget_get_font_pixels (GtkWidget *widget, gint *width, gint *height)
{
	PangoContext *context = gtk_widget_get_pango_context (widget);
	PangoFontMetrics *metrics = pango_context_get_metrics (context,
			widget->style->font_desc, pango_context_get_language (context));
	if (width != NULL)
		*width = PANGO_PIXELS (pango_font_metrics_get_approximate_char_width
			(metrics));
	if (height != NULL)
		*height = PANGO_PIXELS (pango_font_metrics_get_ascent (metrics) +
				pango_font_metrics_get_descent (metrics));
	pango_font_metrics_unref (metrics);
/*
	PangoFontDescription *font_desc = gtk_widget_get_style (widget)->font_desc;
	*height = pango_font_description_get_size (font_desc) / PANGO_SCALE * DPI / 72; //(pixels/in/points/in);
	height = size of the font in points, scaled by PANGO_SCALE,
	(i.e. a size value of 10 * PANGO_SCALE is a 10 point font.
	For screen display, a logical DPI of 96 is common, what is it actually ?

	or, if
	pango_font_description_get_size_is_absolute(), returned size is pixels * PANGO_SCALE
	*height = pango_font_description_get_size (font_desc) / PANGO_SCALE

	PangoLanguage *language = pango_context_get_language (context);
	const gchar *defstr =  pango_language_get_sample_string (language);
	gint count = g_utf8_strlen (defstr, -1);

	gint _width, _height;
	PangoLayout *layout;
//	layout = gtk_widget_create_pango_layout (widget, "m");
	layout = gtk_widget_create_pango_layout (widget, defstr);
//	layout = gtk_widget_create_pango_layout (widget, _("abcdefghijklmnopqrstuvwxyz.ABCDEFGHIJKLMNOPQRSTUVWXYZ"));
//	layout = gtk_widget_create_pango_layout (widget, _("abcd efgh ABCD EFGH"));
	pango_layout_get_pixel_size (layout, &_width, &_height);
//	g_object_unref (layout);
//	if (width != NULL)
//		*width = _width;
//		*width = _width/53;
//		*width = _width/19;
//	if (height != NULL)
//		*height = _height;
	_width /= count;

	layout = gtk_widget_create_pango_layout (widget, ("a\na\n"));
	pango_layout_set_single_paragraph_mode (layout, FALSE);
	gint debug = PANGO_PIXELS (pango_layout_get_spacing (layout));
	g_object_unref (layout);
*/
}
