C Copyright 1981-2007 ECMWF
C 
C Licensed under the GNU Lesser General Public License which
C incorporates the terms and conditions of version 3 of the GNU
C General Public License.
C See LICENSE and gpl-3.0.txt for details.
C

      INTEGER FUNCTION INTUVU( PVYIN, PDVIN, INLEN,
     X                         PUOUT, PVOUT, OUTLEN)
C
C---->
C**** INTUVU
C
C     Purpose
C     -------
C
C     Interpolate unpacked input vorticity and divergence field to
C     unpacked U and V fields.
C
C
C     Interface
C     ---------
C
C     IRET = INTUVU( PVYIN, PDVIN, INLEN, PUOUT,PVOUT,OUTLEN)
C
C     Input
C     -----
C
C     PVYIN  - Input vorticity field  (unpacked array).
C     PDVIN  - Input divergence field (unpacked array).
C     INLEN  - Input field length (words).
C
C
C     Output
C     ------
C
C     PUOUT  - Output U field (unpacked array).
C     PVOUT  - Output V field (unpacked array).
C     OUTLEN - Output field length (words).
C
C
C     Method
C     ------
C
C     Convert spectral vorticity/divergence to spectral U/V and then
C     interpolate U and V to output fields.
C
C
C     Externals
C     ---------
C
C     JVOD2UV - Converts spectral vorticity/divergence to spectral U/V.
C     JMEMHAN - Allocate scratch memory.
C     INTFAU  - Prepare to interpolate unpacked input field.
C     INTFBU  - Interpolate unpacked input field.
C     INTLOG  - Log error message.
C
C
C     Author
C     ------
C
C     J.D.Chambers     ECMWF     Feb 1995
C
C     J.D.Chambers     ECMWF        Feb 1997
C     Allow for 64-bit pointers
C
C----<
C
      IMPLICIT NONE
C
C     Function arguments
C
      INTEGER INLEN, OUTLEN
      REAL PVYIN(INLEN), PDVIN(INLEN), PUOUT(*), PVOUT(*)
C
#include "parim.h"
#include "nifld.common"
#include "nofld.common"
#include "grfixed.h"
#include "intf.h"
C
C     Parameters
C
      INTEGER JPROUTINE, JPALLOC, JPDEALL, JPSCR3, JPSCR4, JPSCR5
      PARAMETER (JPROUTINE = 27000 )
      PARAMETER (JPALLOC = 1) 
      PARAMETER (JPDEALL = 0) 
      PARAMETER (JPSCR3 = 3) 
      PARAMETER (JPSCR4 = 4) 
      PARAMETER (JPSCR5 = 5) 
C
C     Local variables
C
      REAL EW, NS
      LOGICAL LOLDWIND, LSPECUV
      INTEGER IERR, KPR, ISZUV, ISIZE, NOLD, IPDIV, IDIVOFF
      INTEGER NEXT, LOOP, MTRUNC, NTRUNC, NTROLD, NTROLD2, IPVORT
C
      LOGICAL LFIRST, LNEWUV
      CHARACTER*3 EXTRA
      DATA LFIRST/.TRUE./, LNEWUV/.TRUE./, EXTRA/'NO '/
      SAVE LFIRST, LNEWUV
C
      DATA NTROLD/-1/, NTROLD2/-1/
      SAVE NTROLD, NTROLD2
      INTEGER IP_U, IP_V
#ifdef POINTER_64
      INTEGER*8 IZNFLDO
#endif
      REAL ZNFLDO
      POINTER ( IZNFLDO, ZNFLDO )
      DIMENSION ZNFLDO( 1 )
#ifdef POINTER_64
      INTEGER*8 IUV
#endif
      REAL UV
      POINTER ( IUV, UV )
      DIMENSION UV( 1 )
C
C     Externals
C
      INTEGER INTFAU, INTFBU, AURESOL
C
C     -----------------------------------------------------------------|
C*    Section 1.   Initialise
C     -----------------------------------------------------------------|
C
  100 CONTINUE
      INTUVU = 0
      IERR = 0
      KPR = 0
C
      IF( LFIRST ) THEN
        CALL GETENV('IGNORE_UV_EXTRA_MODE', EXTRA)
        IF((EXTRA(1:1).EQ.'Y').OR.(EXTRA(1:1).EQ.'y')) LNEWUV = .FALSE.
        IF( LNEWUV ) THEN
          CALL INTLOG(JP_DEBUG,
     X      'INTUVU: IGNORE_UV_EXTRA_MODE not turned on',JPQUIET)
        ELSE
          CALL INTLOG(JP_DEBUG,
     X      'INTUVU: IGNORE_UV_EXTRA_MODE turned on',JPQUIET)
        ENDIF
        LFIRST = .FALSE.
      ENDIF
C
      NOLD = NIRESO
C
      LSPECUV = (NOREPR.EQ.JPSPHERE).OR.(NOREPR.EQ.JPSPHROT)
C
C     -----------------------------------------------------------------|
C*    Section 2. Convert spectral vorticity/divergence to spectral U/V
C     -----------------------------------------------------------------|
C
  200 CONTINUE
C
      IF( LSPECUV ) THEN
C
C       Spectral U and V for Tn are to be generated from vorticity
C       and divergence spectral T(n-1)
C
        IF( LARESOL.AND.LNEWUV ) THEN
          IF( (NOGRID(1).NE.0).AND.(NOGRID(2).NE.0) ) THEN
            EW = FLOAT(NOGRID(1))/PPMULT
            NS = FLOAT(NOGRID(2))/PPMULT
            NTRUNC = AURESOL(NS,EW) - 1
          ELSE IF( NOGAUSS.NE.0 ) THEN
            EW = 90.0/FLOAT(NOGAUSS)
            NS = EW
            NTRUNC = AURESOL(NS,EW) - 1
          ELSE IF( LNORESO ) THEN
            NTRUNC = NORESO - 1
          ELSE
            NTRUNC = NIRESO - 1
          ENDIF
          IF( NTRUNC.GT.(NIRESO-1) ) NTRUNC = NIRESO - 1
C
        ELSE IF( LNORESO ) THEN
          NTRUNC = NORESO - 1
        ELSE
          NTRUNC = NIRESO - 1
        ENDIF
C
        IF( LNEWUV ) THEN
          MTRUNC = NTRUNC + 1
        ELSE
          NTRUNC = NTRUNC + 1
          MTRUNC = NTRUNC
        ENDIF
C
C     -----------------------------------------------------------------|
C       Use old-style processing if IGNORE_UV_EXTRA_MODE = Y
C     -----------------------------------------------------------------|
C
        IF( .NOT.LNEWUV ) THEN
C
          CALL INTLOG(JP_DEBUG,'INTUVU: vo/div truncation = ', NIRESO)
C
C         Get scratch memory for U and V spectral fields.
C         U and V memory areas are adjacent.
C
          ISZUV = (NIRESO+1)*(NIRESO+2)
          IP_U = 1
          IP_V = 1 + ISZUV
          CALL JMEMHAN( JPSCR3, IUV, ISZUV*2, JPALLOC, IERR)
          IF ( IERR .NE. 0 ) THEN
            CALL INTLOG(JP_ERROR,'INTUVU: Memory allocn fail',JPQUIET)
            INTUVU = IERR
            GOTO 900
          ENDIF
C
C         Generate U and V with same truncation as input fields.
C
          CALL INTLOG(JP_DEBUG,
     X      'INTUVU: Make intermediate U/V with truncation = ', NIRESO)
C
          CALL JVOD2UV(PVYIN,PDVIN,NIRESO,UV(IP_U),UV(IP_V),NIRESO)
C
C         Is the output a truncated spectral field?
C
          IF( LNORESO ) THEN
C
            CALL INTLOG(JP_DEBUG,
     X        'INTUVU: Produce spectral output with truncation',NORESO)
C
            ISIZE = (NORESO+1)*(NORESO+2)
            CALL JMEMHAN( JPSCR5, IZNFLDO, ISIZE, JPALLOC, IERR)
            IF( IERR.NE.0 ) THEN
              CALL INTLOG(JP_FATAL,'INTUVU: Get scratch fail',JPQUIET)
              INTUVU = JPROUTINE + 2
              GOTO 900
            ENDIF
C
            CALL SH2SH( UV(IP_U), NIRESO, ZNFLDO, NORESO )
            DO LOOP = 1, ISIZE
              PUOUT(LOOP) = ZNFLDO(LOOP)
            ENDDO
C
            CALL SH2SH( UV(IP_V), NIRESO, ZNFLDO, NORESO )
            DO LOOP = 1, ISIZE
              PVOUT(LOOP) = ZNFLDO(LOOP)
            ENDDO
C
            NIRESO = NORESO
C
            GOTO 900
C
          ENDIF
C
C     -----------------------------------------------------------------|
C       Use new-style processing if IGNORE_UV_EXTRA_MODE not set
C     -----------------------------------------------------------------|
C
        ELSE
C
          CALL INTLOG(JP_DEBUG,'INTUVU: vo/div truncation = ', NTRUNC)
          CALL INTLOG(JP_DEBUG,'INTUVU: U/V truncation    = ', MTRUNC)
C
C         Truncate vorticity and divergence to correspond to U/V
C
          ISIZE =  (MTRUNC+1)*(MTRUNC+2)
          CALL JMEMHAN( JPSCR5, IZNFLDO, ISIZE*2, JPALLOC, IERR)
          IF( IERR.NE.0 ) THEN
            CALL INTLOG(JP_ERROR,
     X        'INTUVU: Scratch memory type 5 allocn failed.',JPQUIET)
            INTUVU = JPROUTINE + 4
            GOTO 900
          ENDIF
C
          IPVORT = 1
          CALL SH2SH( PVYIN, NIRESO, ZNFLDO, NTRUNC )
C
          IPDIV  = IPVORT + (NIRESO+1)*(NIRESO+2)
          IDIVOFF = 1 + (NTRUNC+1)*(NTRUNC+2)
          CALL SH2SH( PDVIN, NIRESO, ZNFLDO(IDIVOFF), NTRUNC )
C
C         Get scratch memory for U and V spectral fields.
C         U and V memory areas are adjacent.
C
          ISZUV = (MTRUNC+1)*(MTRUNC+2)
          IP_U = 1
          IP_V = 1 + ISZUV
C
          ISIZE = ISZUV*2
          CALL JMEMHAN( JPSCR3, IUV, ISIZE, JPALLOC, IERR)
          IF ( IERR .NE. 0 ) THEN
          CALL INTLOG(JP_ERROR,
     X      'INTUVU: Scratch memory type 3 allocation failed.',JPQUIET)
          INTUVU = IERR
          GOTO 900
        ENDIF
C
C       Generate U and V spectral fields
C
          CALL JVOD2UV(ZNFLDO(IPVORT),ZNFLDO(IDIVOFF),NTRUNC,
     X                 UV(IP_U),UV(IP_V),MTRUNC)
C
          ISIZE = (NORESO+1)*(NORESO+2)
          CALL JMEMHAN( JPSCR5, IZNFLDO, ISIZE*2, JPALLOC, IERR)
          IF ( IERR .NE. 0 ) THEN
            CALL INTLOG(JP_FATAL,'INTUVU: Get scratch fail',JPQUIET)
            INTUVU = JPROUTINE + 2
            GOTO 900
          ENDIF
C
          CALL SH2SH( UV(IP_U), NIRESO, ZNFLDO, NORESO )
          DO LOOP = 1, ISIZE
            PUOUT(LOOP) = ZNFLDO(LOOP)
          ENDDO
C
          CALL SH2SH( UV(IP_V), NIRESO, ZNFLDO, NORESO )
          DO LOOP = 1, ISIZE
            PVOUT(LOOP) = ZNFLDO(LOOP)
          ENDDO
C
          GOTO 900
C
        ENDIF
C
      ENDIF
C
C     -----------------------------------------------------------------|
C*    Section 3.   Generate grid point GRIB format U and V fields.
C     -----------------------------------------------------------------|
C
  300 CONTINUE
C
C     Spectral U and V for Tn are to be generated from vorticity
C     and divergence spectral T(n-1)
C
C     See whether or not the 'autoresol' flag is set.
C     If not, use the input truncation.
C
      IF( LARESOL ) THEN
        IF( (NOREPR.EQ.JPREGULAR).OR.(NOREPR.EQ.JPREGROT) ) THEN
          EW = FLOAT(NOGRID(1))/PPMULT
          NS = FLOAT(NOGRID(2))/PPMULT
        ELSE
          EW = 90.0/FLOAT(NOGAUSS)
          NS = EW
        ENDIF
        NTRUNC = AURESOL(EW,NS)
        IF( NTRUNC.NE.NTROLD ) THEN
          NTROLD = NTRUNC
          CALL INTLOG(JP_WARN,
     X      'INTUVU: Resolution automatically set to ', NTRUNC)
        ENDIF
      ELSE IF( LNORESO ) THEN
        NTRUNC = NORESO
      ELSE
        NTRUNC = NIRESO
      ENDIF
C
C     Check whether the output resolution is greater than the input
C
      IF( NTRUNC.GT.NIRESO ) THEN
C
C       Issue warning if the output resolution was user-supplied
C
        IF( .NOT.LARESOL ) THEN
C
C         Revert to the input truncation
C
          IF( NIRESO.NE.NTROLD2 ) THEN
            CALL INTLOG(JP_WARN,
     X        'INTUVU: spectral -> grid point interpolation',JPQUIET)
            CALL INTLOG(JP_WARN,
     X        'INTUVU: User supplied resolution = ',NTRUNC)
            CALL INTLOG(JP_WARN,
     X        'INTUVU: Input field resolution   = ',NIRESO)
            CALL INTLOG(JP_WARN,
     X        'INTUVU: User supplied resolution ignored',JPQUIET)
            CALL INTLOG(JP_WARN,
     X        'INTUVU: Input field resolution has been used',JPQUIET)
            NTROLD2 = NIRESO
          ENDIF
          NTRUNC = NIRESO
C
        ELSE
C
C         Revert to the input truncation
C
          NTRUNC = NIRESO
          IF( NTRUNC.NE.NTROLD2 ) THEN
            NTROLD2 = NTRUNC
            CALL INTLOG(JP_WARN,
     X        'INTUVU: Automatic resolution selectn too high',JPQUIET)
            CALL INTLOG(JP_WARN,
     X        'INTUVU: Resolution reset to input resolution: ',NTRUNC)
          ENDIF
C
        ENDIF
      ENDIF
C
C     IF extra mode is in use, adjust the calculated truncation.
C
      MTRUNC = NTRUNC
      IF( LNEWUV ) NTRUNC = MTRUNC - 1
C
      CALL INTLOG(JP_DEBUG,'INTUVU: vo/div truncation = ', NTRUNC)
      CALL INTLOG(JP_DEBUG,'INTUVU: U/V truncation    = ', MTRUNC)
C
      ISIZE = (MTRUNC+1)*(MTRUNC+2)
      CALL JMEMHAN( JPSCR5, IZNFLDO, ISIZE*2, JPALLOC, IERR)
      IF( IERR.NE.0 ) THEN
        CALL INTLOG(JP_FATAL,
     X    'INTUVU: Get scratch space failed',JPQUIET)
        INTUVU = JPROUTINE + 2
        GOTO 900
      ENDIF
C
C     Adjust the vorticity and divergence by one wave number before
C     conversion to U and V
C
      IPVORT = 1
      IDIVOFF = 1 + (NTRUNC+1)*(NTRUNC+2)
C
      CALL SH2SH( PVYIN, NIRESO, ZNFLDO, NTRUNC )
C
      CALL SH2SH( PDVIN, NIRESO, ZNFLDO(IDIVOFF), NTRUNC )
C
C     Get scratch memory for U and V spectral fields.
C     U and V memory areas are adjacent.
C
      ISZUV = (MTRUNC+1)*(MTRUNC+2)
      IP_U  = 1
      IP_V  = IP_U + ISZUV
C
      ISIZE = ISZUV*2
      CALL JMEMHAN( JPSCR3, IUV, ISIZE, JPALLOC, IERR)
      IF( IERR.NE.0 ) THEN
        CALL INTLOG(JP_ERROR,
     X    'INTUVU: Scratch memory type 3 allocation failed.',JPQUIET)
        INTUVU = IERR
        GOTO 900
      ENDIF
C
C     Generate U and V spectral fields
C
      CALL JVOD2UV(ZNFLDO(IPVORT),ZNFLDO(IDIVOFF),NTRUNC,
     X             UV(IP_U),UV(IP_V),MTRUNC)
C
      NIRESO = MTRUNC
C
C     -----------------------------------------------------------------|
C*    Section 4.   Interpolate U field.
C     -----------------------------------------------------------------|
C
  400 CONTINUE
C
      NIFORM = 0
      NIPARAM = JP_U
      LWIND = .TRUE.
      LOLDWIND = LWINDSET
      LWINDSET = .TRUE.
C
      IERR = INTFAU( UV(IP_U), ISZUV, PUOUT, OUTLEN)
      IF ( IERR .NE. 0 ) THEN
        CALL INTLOG(JP_ERROR,
     X    'INTUVU: Prepare to interpolate failed.',JPQUIET)
        INTUVU = IERR
        GOTO 900
      ENDIF
C
      IERR = INTFBU( UV(IP_U), ISZUV, PUOUT, OUTLEN)
C
      IF ( IERR .NE. 0 ) THEN
        CALL INTLOG(JP_ERROR,'INTUVU: Interpolation failed.',JPQUIET)
        INTUVU = IERR
        GOTO 900
      ENDIF
C
C     -----------------------------------------------------------------|
C*    Section 5.   Interpolate V field.
C     -----------------------------------------------------------------|
C
  500 CONTINUE
C
      NIPARAM = JP_V
      IERR = INTFAU( UV(IP_V), ISZUV, PVOUT, OUTLEN)
      IF ( IERR .NE. 0 ) THEN
        CALL INTLOG(JP_ERROR,
     X    'INTUVU: Prepare to interpolate failed.',JPQUIET)
        INTUVU = IERR
        GOTO 900
      ENDIF
C
      IERR = INTFBU( UV(IP_V), ISZUV, PVOUT, OUTLEN)
C
      IF ( IERR .NE. 0 ) THEN
        CALL INTLOG(JP_ERROR,
     X    'INTUVU: Prepare to interpolate failed.',JPQUIET)
        INTUVU = IERR
        GOTO 900
      ENDIF
C
C     -----------------------------------------------------------------|
C*    Section 9.   Closedown.
C     -----------------------------------------------------------------|
C
  900 CONTINUE
C
C     Clear change flags for next product processing
      LCHANGE = .FALSE.
      LSMCHNG = .FALSE.
      LWINDSET = LOLDWIND
C
      NIRESO = NOLD
C
      RETURN
      END
