/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */


#include <stdio.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <signal.h>
#include "ipaddrfile.h"
#include "lockfile.h"

 /*
   * break out next entry from the file with the list of client ips 
   * the entry is in host order
   */
  static int mygetipaddrent(struct nbd_statfile * self, unsigned long *ipaddr) {

    unsigned short p[4];

    if (!self->fp)
       return -1;

    while (1) {
      int count = fscanf(self->fp, " %hd.%hd.%hd.%hd\n",
                    &p[0],&p[1],&p[2],&p[3]);
      if (count >= 4)
        break;
      count = fscanf(self->fp,"%*[^\n]\n");
      if (count < 0) 
          return -1;         // eof
      // go round again
    }

    // PTB store in host order
    if (ipaddr)
      *ipaddr = (((((p[0] << 8) | p[1]) << 8) | p[2]) << 8) | p[3];
    return 0;
  }

  /*
   * reset the ip address file pointer to first entry
   */
  static int mysetipaddr(struct nbd_statfile * self) {

     if (!self->fp)
       return -1;

     rewind(self->fp);
     return 0;
  }

  /*
   * check for particular ip in ip address file. Ip is in host order
   */
  static int mygetipaddr(struct nbd_statfile * self, unsigned long ipaddr) {

    // save visiting ipaddr in /var/state/nbd/server-id.ipaddr

    if (!self->fp) 
         return -1;

    self->reset(self);

    while(1) {
      unsigned long otheripaddr;
      unsigned short q[4];

      if (self->next(self,&otheripaddr) < 0)
         break;

      q[3] = otheripaddr         & 0xff;
      q[2] = (otheripaddr >> 8)  & 0xff;
      q[1] = (otheripaddr >> 16) & 0xff;
      q[0] = (otheripaddr >> 24) & 0xff;

      fprintf(stderr, "checking %hd.%hd.%hd.%hd\n", q[0],q[1],q[2],q[3]);

      if (otheripaddr == ipaddr)
         return 0;
    }
    return -1;
  }


  static int myaddipaddr(struct nbd_statfile * self, unsigned long ipaddr) {

    // save visiting ipaddr (host order) in /var/state/nbd/server-id.ipaddr

    unsigned short q[4];
    short i;

    if (!self->fp) 
         return -1;
    self->fp = freopen(self->statfile, "a+", self->fp);
    if (!self->fp)
         return -1;
                
    // PTB get always starts at the beginning itself
    if (self->get(self,ipaddr) >= 0)
       return 0;
   
    // PTB go to the end
    clearerr(self->fp);
    fseek(self->fp,0,SEEK_END);

    for (i = 3; i >= 0; i--) {
       q[i] = ipaddr & 0xff;
       ipaddr >>= 8;
    }

    fprintf(self->fp,"%hd.%hd.%hd.%hd\n",q[0],q[1],q[2],q[3]);
    //MSG("added new session from %hd.%hd.%hd.%hd to %s\n",
    //        q[0],q[1],q[2],q[3], self->statfile);

    return 0;
  }

  /*
   * Do "dot locking" via the FS so that scripts can see it too.
   */
  static int mylock(struct nbd_statfile * self) {

      const char suffix[] = "lock";
      char * lockfile = malloc(strlen(self->statfile) + 1 + strlen(suffix) + 1);
      int err = 0;

      if (!lockfile)
        return -1;

      sprintf(lockfile, "%s.%s", self->statfile, suffix);

      err = dolock(lockfile);
      free(lockfile);
      return err;
  }
  static int myunlock(struct nbd_statfile * self) {

      const char suffix[] = "lock";
      char * lockfile = malloc(strlen(self->statfile) + 1 + strlen(suffix) + 1);
      int err;

      if (!lockfile)
        return -1;

      sprintf(lockfile, "%s.%s", self->statfile, suffix);

      err = dounlock(lockfile);
      free(lockfile);
      return err;
  }

  static int myopen(struct nbd_statfile * self) {
      if (!self->fp)
        self->fp = fopen(self->statfile,"r");
      if (!self->fp)
        self->fp = fopen(self->statfile,"a+");
      if (!self->fp)
        return -1;
      if (fseek(self->fp,0,SEEK_SET) != 0)
        return -1;
      return 0;
  }
  static int myclose(struct nbd_statfile * self) {
      if (self->fp) {
        int err = fclose(self->fp);
        self->fp = NULL;
        return err;
      }
      return 0;
  }

  // delete ipaddr (host order) from statfile
  static int mydelipaddr(struct nbd_statfile * self, unsigned long ipaddr) {

    char * statfilebak;
    FILE * sfbak, *sfnew;

    if (!self->fp) 
         return -1;
    if (self->get(self,ipaddr) < 0)
       return 0;
    statfilebak = malloc(strlen(self->statfile) + 5);
    if (!statfilebak)
       return -1;
    sprintf(statfilebak, "%s.bak", self->statfile);
    unlink(statfilebak);
    if (link(self->statfile, statfilebak) < 0)
       return -1;
    if (unlink(self->statfile) < 0)
       return -1;
    sfbak = self->fp;
    sfnew = fopen(self->statfile,"w+");
    if (!sfnew) {
       if (link(statfilebak, self->statfile) >= 0) {
         unlink(statfilebak);
       }
       fclose(self->fp);
       return -1;
    }
    // PTB OK, all set, new in pfnew and old in pfbak
    self->fp = sfbak;
    self->reset(self);
    while (1) {
       unsigned long other_ipaddr;

       // PTB read from old
       self->fp = sfbak;
       if (self->next(self, &other_ipaddr) < 0)
         break;

       if (other_ipaddr == ipaddr)
         continue;

       // PTB write to new
       self->fp = sfnew;
       self->add(self, other_ipaddr);
    }
    fclose(sfbak);
    unlink(statfilebak);
    self->fp = sfnew;
    return 0;
  }

  int initipaddrfile(struct nbd_statfile * self, char * statfile) {
      self->statfile = statfile;
      self->add      = myaddipaddr;
      self->del      = mydelipaddr;
      self->get      = mygetipaddr;
      self->next     = mygetipaddrent;
      self->reset    = mysetipaddr;
      self->open     = myopen;
      self->close    = myclose;
      self->lock     = mylock;
      self->unlock   = myunlock;
      return 0;
  }

