/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */


#include <stdio.h>
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include "pidfile.h"

  static int
  openpidfile(struct nbd_pidfile *self, char * how) {

       self->pf = fopen(self->pidfile, how);

       if (!self->pf)
         return -1;

       return 0;
  }

  static int
  closepidfile(struct nbd_pidfile *self) {
       int err;
       if (!self->pf)
         return -1;
       err = fclose(self->pf);
       self->pf = NULL;
       return err;
  }

  // PTB *re*make our pidfile with our pid inside after fork
  static int
  mymkpidfile(struct nbd_pidfile * self) {

    char c;
    int mypid = getpid();
    int err;

    // truncate
    err = openpidfile(self, "w+");
    if (err < 0)
      return err;

    fprintf(self->pf, "%-5u\n", mypid);

    // PTB fill with blanks to eof??
    while (fread(&c, 1, 1, self->pf) >= 1) {
      fseek(self->pf, -1, SEEK_CUR);
      c = ' ';
      fwrite(&c, 1, 1, self->pf);
    }

    closepidfile(self);
    return 0;
  }

  // PTB *re*check our pidfile with our pid inside after fork
  static int
  readpidfile(struct nbd_pidfile * self) {

       int otherpid = 0;

       if (openpidfile(self, "r") < 0)
         return -1;

       if (fscanf(self->pf, "%u\n", &otherpid) < 1) {
           closepidfile(self);
           return -1;
       }
       closepidfile(self);
       return otherpid;
  }

  static int
  unlockpidfile(struct nbd_pidfile *self) {
       int err;
       if (!self->pidfile)
         return -1;

       if (readpidfile(self) != getpid())
         return -1;
       if (self->pf)
         closepidfile(self);
       err = unlink(self->pidfile);
       self->pf = NULL;
       return err;
  }

  // PTB *re*check our pidfile with our pid inside after fork
  static int
  myckpidfile(struct nbd_pidfile * self) {

       int otherpid;
       int mypid = getpid();
       int myppid = getppid();

       otherpid = readpidfile(self);

       if (otherpid == -1 || otherpid == mypid || otherpid == myppid)  {
           return 0;
       }
       if (kill (otherpid, 0) < 0) {
           return 0;
       }

       // PTB bad news, pidfile contains valid other pid
       return -EBUSY;
  }

  
  static int
  lockpidfile(struct nbd_pidfile * self) {
       int err;
       
       err = myckpidfile(self);
       if (err < 0) 
           return err; // failed to get lock
       err = mymkpidfile(self);
       return err;
  }


  int initpidfile(struct nbd_pidfile *self, char * pidfile) {

       self->pidfile = pidfile;
       self->read  = readpidfile;
       self->lock  = lockpidfile;
       self->unlock
                   = unlockpidfile;

       if (!self->pidfile)
         return -1;

       return 0;
  }

