/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */


#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <setjmp.h>
#include <sys/errno.h>
#include <sys/socket.h>
#include <netdb.h>
#include <syslog.h>
#include <stdio.h>

#ifdef USING_SSL
#include "sslincl.h"
#endif

#include "alarm.h"
#include "stream.h"
#include "cliserv.h"
#include "socket.h"
#include "select.h"

#ifndef MY_NAME
#define MY_NAME "stream2"
#endif

extern int debug_level;


#ifdef USING_SSL

   inline int ssl_check(SSL * con, int res) {

             switch (SSL_get_error(con,res)){ 
               case SSL_ERROR_WANT_WRITE:
               case SSL_ERROR_WANT_X509_LOOKUP:
               case SSL_ERROR_SYSCALL:
               case SSL_ERROR_ZERO_RETURN:
               case SSL_ERROR_SSL:
                 return -1;
              }
              return 0;
   }

   static inline int sslreadnet3(struct nbd_stream *self, char *buf, int len) {

      int res;
      DEBUG("SSL_read will get %d bytes from net\n",len);
      res = SSL_read(self->con, buf, len);
      if (ssl_check(self->con, res) < 0) {
            PERR("ssl_check hit some kind of catastrophe. Exiting ...\n");
            exit(1);
      }
      DEBUG("SSL_read got %d bytes from net\n",len);
      return res;
   }
   static inline int sslwritenet3(struct nbd_stream *self, char *buf, int len)
   {
        int res;
        DEBUG("(%d) SSL_write will send %d bytes to connection 0x%x\n",
            self->i,len,(unsigned)self->con);
        res = SSL_write(self->con, buf, len);
        if (ssl_check(self->con, res) < 0) {
           PERR("ssl_check hit some kind of catastrophe. Exiting ...\n");
           exit(1);
         }
         DEBUG("SSL_write sent %d bytes to net\n",len);
         return res;
   }
#endif


   static int readnet3(struct nbd_stream *self, char *buf, int len)
   {
      int res;
      unsigned count = 0;
      fd_set fds, xfds;

      while (len > 0) {

           read:
           DEBUG("*");

#ifdef USING_SSL
           if (using_ssl) {
             res = sslreadnet(self,buf,len);        
           } else
#endif
           {
             fd_set * rfds = & fds, * efds = & xfds;

             FD_ZERO(efds); FD_SET(self->sock,efds);
             FD_ZERO(rfds); FD_SET(self->sock,rfds);

             res =
               microselect(self->sock+1,rfds,NULL,efds,self->timeout*1000000);

             if (res < 0)
                // PTB  timeout - maybe -ETIMEDOUT
                return res;

             // PTB select returned 1. We won't block on read.

             res = -1;

             if (FD_ISSET(self->sock,efds)) {
               // PTB some kind of network error predicted

               res = read(self->sock, buf, len);

               if (res >= 0) {
                 // PTB but it did not happen!
                 if (self->errs++ < 1)
                   PERR("Read returned %d after select predicted err!\n", res);
                 goto next;
               }
               // PTB the expected error did happen, so we report it
               self->errs = 0;
               goto read_error;
             }

             if (FD_ISSET(self->sock,rfds)) {
               // PTB we expect to do a read without error

               res = read(self->sock, buf, len);

               if (res == 0) {
                 // PTB well, almost. The other side closed on us. This
                 // PTB is not always an error, but it sometimes is.
                 if (self->errs++ < 1)
                   PERR("Read returned %d after select predicted read!\n", res);
                 // PTB let's introduce a 1s delay before retrying after 0B
                 microsleep(1000000);
                 goto next;
               }

               if (res < 0) {
                 // PTB a real error occurred
                 if (self->errs++ < 1)
                   PERR("Read returned %d after select predicted read!\n", res);
                 res = -errno;
                 errno = 0;
                 goto read_error;
               }

               // PTB we actually read some bytes, as expected!
               self->errs = 0;
               goto next;
             }

             // PTB completely impossible. FDISSET can't return anything else
             PERR("impossible read case from FD_ISSET. Universe corrupt!\n");
             exit(1);

           } // eblck

           // PTB good case
           self->errs = 0;
           goto next;

      read_error:
           // PTB bother to analyse why we errored
           switch (res) {
             default:      // PTB unknown read error
               PERR("Read failed in readnet: %m\n");
               return res;
               break;
             case 0:       // PTB end of file. Socket closed at other end.
               PERR("Read got 0 bytes in readnet: %m\n");
               return -EIO;
               break;
             case -EINTR:  // PTB interrupted before read() read a byte
             case -EAGAIN: // PTB read asks for a second chance
               // PTB try 10 times at 1s intervals then fail
               if (self->errs < 10) {
                 microsleep(1000000);
                 goto read;
               }
               // PTB give up, fail it and reset count
               PERR("Read eventually timed out in readnet: %m\n");
               return -ETIME;
               break;
           }
      next:
         len   -= res;
         buf   += res;
         count += res;

      } // ewhile

      return count;
    }


  static int writenet3(struct nbd_stream *self, char *buf, int len)
   {
      int res;
      unsigned count = 0;
      fd_set fds, xfds;

      while (len > 0) {

           write:
           DEBUG("+");

#ifdef USING_SSL
           if (using_ssl) {
             sslwritenet3(self, buf, len);
           } else
#endif
           {
             fd_set * wfds = & fds, * efds = & xfds;

             FD_ZERO(efds); FD_SET(self->sock,efds);
             FD_ZERO(wfds); FD_SET(self->sock,wfds);
             
             res =
               microselect(self->sock+1,NULL,wfds,efds,self->timeout*1000000);

             if (res < 0)
                // PTB timeout
                return res;

             // PTB select returned 1. We won't block on write.

             res = -1;

             if (FD_ISSET(self->sock,efds)) {
               // PTB some kind of network error predicted
               res = write(self->sock, buf, len);
               if (res >= 0) {
                 // PTB but the error never materialized. Amazing.
                 if (self->errs++ < 1)
                   PERR("Write returned %d after select predicted err!\n", res);
                 goto next;
               }
               goto write_error;
             }

             if (FD_ISSET(self->sock,wfds)) {

               // PTB we expect to write safely
               res = write(self->sock, buf, len);

               if (res == 0) {
                 // PTB but somehow we didn't manage it!
                 if (self->errs++ < 1)
                   PERR("Write returned %d after select predicted write!\n",
                            res);
                 microsleep(1000000);
                 goto next;
               }

               if (res < 0) {
                 // PTB we errored on write afeter all!
                 if (self->errs++ < 1)
                   PERR("Write returned %d after select predicted write!\n",
                           res);
                 res = -errno;
                 errno = 0;
                 goto write_error;
               }

               // PTB we actually wrote some bytes, as expected!
               self->errs = 0;
               goto next;
             }

             // PTB unreachable
             PERR("impossible write case from FD_ISSET. Universe corrupt!\n");
             exit(1);
 
           } // ewhile

           // PTB good case. 
           self->errs = 0;
           goto next;

        write_error:

           // PTB bother to analyse why we errored
           switch (res) {
             default:      // PTB unknown write error
               PERR("Write failed in writenet: %m\n");
               return res;
               break;
               if (res != -EAGAIN && res != -EWOULDBLOCK) {
                 PERR("Write failed in writenet: %m\n");
                 return -ETIME;
               }
             case 0:       // PTB end of file. Socket closed at other end.
               PERR("Write sent 0 bytes in writenet: %m\n");
               return -EIO;
               break;
             case -EINTR:  // PTB interrupted before read() read a byte
             case -EAGAIN: // PTB read asks for a second chance
               // PTB try 10 times at 1s intervals then fail

               if (self->errs < 10) {
                 microsleep(1000000);
                 goto write;
               }
               // PTB give up, fail it and reset count
               PERR("Write timed out in writenet: %m\n");
               self->errs = 0;
               return -ETIME;
               break;
           }

      next:
         len   -= res;
         buf   += res;
         count += res;

      } // ewhile

      return count;
   }


int initstream2(struct nbd_stream *self, int timeout, void * using_ssl) {
    self->timeout = timeout;
    self->sock = -1;
    self->port = -1;
    self->hostname = NULL;
    self->errs = 0;
    self->flags = 0;

    self->close = NULL;
#ifdef USING_SSL
    self->con = NULL;
    self->ctx = (SSL_CTX **)using_ssl;
    if (using_ssl) {
      self->read  = readssl3;
      self->write = writessl3;
      self->open  = NULL;
    } else
#endif
    {
      self->read  = readnet3;
      self->write = writenet3;
      self->open  = NULL;
    }
    self->reopen  = NULL;
    self->magic   = NBD_STREAM_MAGIC;
    self->flags  |= NBD_STREAM_INITIALIZED;
    DEBUG("initialized stream %x\n", (unsigned)self);

    return 0;
}

