#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvobj.h"
#include "edvdevices.h"
#include "edvstatusbar.h"
#include "browser.h"
#include "browsercb.h"
#include "browseropcb.h"
#include "browserdirtree.h"
#include "browsercontents.h"
#include "browserdnd.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvlistseek.h"
#include "edvcfglist.h"
#include "config.h"



void EDVBrowserContentsItemDestroyCB(gpointer data);
void EDVBrowserDirTreeItemDestroyCB(gpointer data);

static gint EDVBrowserCListColumnSortDateNexus(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
);
static gint EDVBrowserCListColumnSortDateAccessCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVBrowserCListColumnSortDateModifyCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVBrowserCListColumnSortDateChangeCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);

gint EDVBrowserDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
void EDVBrowserDestroyCB(GtkObject *object, gpointer data);

gint EDVBrowserKeyEventCB(
         GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint EDVBrowserButtonPressEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
);

void EDVBrowserHandleChildAttachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void EDVBrowserHandleChildDetachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

void EDVBrowserTreeSelectRowCB(
        GtkCTree *ctree, GtkCTreeNode *node, gint column,
        gpointer data
);
void EDVBrowserTreeUnselectRowCB(
        GtkCTree *ctree, GtkCTreeNode *node, gint column,
        gpointer data
);
void EDVBrowserTreeExpandCB(
        GtkCTree *ctree, GtkCTreeNode *node, gpointer data
);
void EDVBrowserTreeCollapseCB(
        GtkCTree *ctree, GtkCTreeNode *node, gpointer data
);

void EDVBrowserResizeColumnCB(
	GtkCList *clist, gint column, gint width, gpointer data
);
void EDVBrowserClickColumnCB(
	GtkCList *clist, gint column, gpointer data
);
void EDVBrowserSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
);
void EDVBrowserUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
);


void EDVBrowserComboActivateCB(GtkWidget *widget, gpointer data);

void EDVBrowserWriteProtectChangedCB(
	edv_browser_struct *browser, gbool state
);
void EDVBrowserObjectAddedNotifyCB(
        edv_browser_struct *browser, const gchar *path,
	const struct stat *lstat_buf
);
void EDVBrowserObjectModifiedNotifyCB(
        edv_browser_struct *browser, const gchar *path,
	const gchar *new_path,
	const struct stat *lstat_buf
);
void EDVBrowserObjectRemovedNotifyCB(
        edv_browser_struct *browser, const gchar *path
);

void EDVBrowserMountNotifyCB(
	edv_browser_struct *browser,
	gint dev_num, edv_device_struct *dev_ptr,
	gbool is_mounted
);

void EDVBrowserRecycledObjectAddedNotifyCB(
        edv_browser_struct *browser, guint index
);
void EDVBrowserRecycledObjectRemovedNotifyCB(
        edv_browser_struct *browser, guint index
);

void EDVBrowserReconfiguredNotifyCB(edv_browser_struct *browser);

void EDVBrowserMimeTypeAddedCB(
        edv_browser_struct *browser,
        gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVBrowserMimeTypeModifiedCB(
        edv_browser_struct *browser,
        gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVBrowserMimeTypeRemovedCB(
        edv_browser_struct *browser, gint mt_num
);

void EDVBrowserDeviceAddedCB(
        edv_browser_struct *browser,
        gint dev_num, edv_device_struct *dev_ptr
);
void EDVBrowserDeviceModifiedCB(
        edv_browser_struct *browser,
        gint dev_num, edv_device_struct *dev_ptr
);
void EDVBrowserDeviceRemovedCB(
	edv_browser_struct *browser, gint dev_num
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Browser contents GtkCList item "destroy" signal callback.
 */
void EDVBrowserContentsItemDestroyCB(gpointer data)
{
        EDVObjectDelete((edv_object_struct *)data);
}

/*
 *	Browser directory GtkCTree item "destroy" signal callback.
 */
void EDVBrowserDirTreeItemDestroyCB(gpointer data)
{
	EDVObjectDelete((edv_object_struct *)data);
}


/*
 *	Returns the sort code for the rows sorted by date.
 *
 *	The disk object structures are obtained from each row and the
 *	dates are compared.
 */
static gint EDVBrowserCListColumnSortDateNexus(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
)
{
        gint sort_column;
        const GtkCListRow *row1 = (const GtkCListRow *)ptr1;
        const GtkCListRow *row2 = (const GtkCListRow *)ptr2;
	edv_object_struct *obj1, *obj2;


        if((clist == NULL) || (row1 == NULL) || (row2 == NULL))
            return(-1);

        sort_column = clist->sort_column;
        if((sort_column < 0) || (sort_column >= clist->columns))
            return(-1);

	obj1 = (edv_object_struct *)row1->data;
	obj2 = (edv_object_struct *)row2->data;
	if((obj1 == NULL) || (obj2 == NULL))
	    return(-1);

	switch(sort_code)
	{
	  case 0:	/* Access time. */
	    if(obj1->access_time <= obj2->access_time)
                    return((gint)(obj1->access_time < obj2->access_time));
                else
                    return(-1);
	    break;

          case 1:       /* Modify time. */
            if(obj1->modify_time <= obj2->modify_time)
                    return((gint)(obj1->modify_time < obj2->modify_time));
                else
                    return(-1);
            break;

          case 2:       /* Change time. */
            if(obj1->change_time <= obj2->change_time)
                    return((gint)(obj1->change_time < obj2->change_time));
                else
                    return(-1);
            break;
	}

	return(-1);
}

static gint EDVBrowserCListColumnSortDateAccessCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVBrowserCListColumnSortDateNexus(
	    clist, ptr1, ptr2, 0
	));
}

static gint EDVBrowserCListColumnSortDateModifyCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
        return(EDVBrowserCListColumnSortDateNexus(
            clist, ptr1, ptr2, 1
        ));
}

static gint EDVBrowserCListColumnSortDateChangeCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
        return(EDVBrowserCListColumnSortDateNexus(
            clist, ptr1, ptr2, 2
        ));
}


/*
 *	Browser toplevel GtkWidget "delete_event" signal callback.
 */
gint EDVBrowserDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_browser_struct *browser = (edv_browser_struct *)data;
        if(browser == NULL)
            return(TRUE);

	if(browser->processing)
	    return(TRUE);

	EDVBrowserOPClose(browser);

	return(TRUE);
}

/*
 *	Browser toplevel GtkWidget "destroy" signal callback.
 */
void EDVBrowserDestroyCB(GtkObject *object, gpointer data)
{
	return;
}


/*
 *	Browser "key_press_event" or "key_release_event" signal callback.
 */
gint EDVBrowserKeyEventCB(
         GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
        static gbool reenterent = FALSE;
	gint status = FALSE;
	gint etype;
	guint keyval, state;
	gbool press;
	edv_core_struct *core_ptr;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((widget == NULL) || (key == NULL) || (browser == NULL))
            return(status);

	if(browser->processing)
	    return(status);

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	    return(status);

        if(reenterent)
            return(status);
        else
            reenterent = TRUE;

	/* Get event type. */
	etype = key->type;

	/* Get other key event values. */
        press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

/* Macro to emit a signal stop for a key press or release depending
 * on the current event's type.
 */
#define DO_STOP_KEY_SIGNAL_EMIT	\
{ \
 gtk_signal_emit_stop_by_name( \
  GTK_OBJECT(widget), \
  press ? "key_press_event" : "key_release_event" \
 ); \
}

/* Macro to clamp the GtkAdjustment adj and emit a "value_changed"
 * signal.
 */
#define DO_ADJ_CLAMP_EMIT	\
{ \
 if(adj->value > (adj->upper - adj->page_size)) \
  adj->value = adj->upper - adj->page_size; \
\
 if(adj->value < adj->lower) \
  adj->value = adj->lower; \
\
 gtk_signal_emit_by_name( \
  GTK_OBJECT(adj), "value_changed" \
 ); \
}

/* Macro to clamp the GtkCList clist's focus_row. */
#define DO_CLIST_FOCUS_ROW_CLAMP	\
{ \
 if(clist->focus_row >= clist->rows) \
  clist->focus_row = clist->rows - 1; \
 if(clist->focus_row < 0) \
  clist->focus_row = 0; \
\
/* gtk_widget_queue_draw(GTK_WIDGET(clist)); */ \
}


        /* Check which ctree this signal is for. */

	/* Directory ctree. */
        if(widget == browser->directory_ctree)
        {
	    GList *glist;
	    GtkCTreeNode *node;
            GtkCList *clist = GTK_CLIST(widget);
	    GtkCTree *ctree = GTK_CTREE(widget);


            /* Get last selected node. */
	    node = EDVCTreeGetSelectedLast(ctree, NULL);

            /* Handle by key value. */
            switch(keyval)
            {
              case GDK_Return:
              case GDK_KP_Enter:
              case GDK_ISO_Enter:
              case GDK_3270_Enter:
		/* Handle only if control key modifier is not held, so
		 * that accelerator keys will get handled properly.
		 */
		if(!(state & GDK_CONTROL_MASK))
		{
		    DO_STOP_KEY_SIGNAL_EMIT
		    status = TRUE;
		}
                break;

              case GDK_Delete:
                if(press)
                {
                    EDVBrowserOPDelete(browser);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

	      case GDK_BackSpace:
		if(press)
		{
		    EDVBrowserOPGoToParent(browser);
		}
		DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_space:
              case GDK_KP_Space:
		node = gtk_ctree_node_nth(ctree, clist->focus_row);
                if((node != NULL) && press)
                {
		    gbool already_selected = FALSE;

		    /* Check if this node is already selected. */
		    glist = clist->selection;
		    while(glist != NULL)
		    {
			if(node == (GtkCTreeNode *)glist->data)
			{
			    already_selected = TRUE;
			    break;
			}
			glist = glist->next;
		    }

                    gtk_clist_freeze(clist);
		    if(already_selected)
			gtk_ctree_unselect(ctree, node);
		    else
			gtk_ctree_select(ctree, node);
                    gtk_clist_thaw(clist);
		}
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
                break;

	      case GDK_Up:
	      case GDK_KP_Up:
                if(state & GDK_CONTROL_MASK)
                {
		    /* Get adjustment and scroll up. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value -= adj->step_increment;
			DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
		    DO_STOP_KEY_SIGNAL_EMIT
		    status = TRUE;
                }
		else
		{
		    if(press)
                    {
                        gtk_clist_freeze(clist);
			clist->focus_row--;
			DO_CLIST_FOCUS_ROW_CLAMP

                        if(gtk_clist_row_is_visible(
                            clist, clist->focus_row) != GTK_VISIBILITY_FULL
                        )
                            gtk_clist_moveto(
                                clist,
                                clist->focus_row, -1,	/* Row, column. */
                                0.5, 0.0                /* Row, column. */
                            );
                        gtk_clist_thaw(clist);
		    }
		    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
		}
                break;

              case GDK_Down:
              case GDK_KP_Down:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll down. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value += adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                else
                {
                    GtkAdjustment *adj = clist->hadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        clist->focus_row++;
                        DO_CLIST_FOCUS_ROW_CLAMP

                        if(gtk_clist_row_is_visible(
                            clist, clist->focus_row) != GTK_VISIBILITY_FULL
                        )
			    gtk_clist_moveto(
				clist,
				clist->focus_row, -1,	/* Row, column. */
				0.5, 0.0		/* Row, column. */
			    );
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

              case GDK_Left:
              case GDK_KP_Left:
                if(1)
                {
                    /* Get adjustment and scroll left. */
                    GtkAdjustment *adj = clist->hadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        if(state & GDK_CONTROL_MASK)
                            adj->value -= adj->step_increment * 4;
                        else if(state & GDK_SHIFT_MASK)
                            adj->value = adj->lower;
                        else
                            adj->value -= adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Right:
              case GDK_KP_Right:
                if(1)
                {
                    /* Get adjustment and scroll right. */
                    GtkAdjustment *adj = clist->hadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        if(state & GDK_CONTROL_MASK)
                            adj->value += adj->step_increment * 4;
                        else if(state & GDK_SHIFT_MASK)
                            adj->value = adj->upper - adj->page_size;
                        else
                            adj->value += adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Page_Up:
	      case GDK_KP_Page_Up:
                if(1)
                {
                    /* Get adjustment and scroll up one page. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value -= adj->page_increment;
			DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
		    }
		}
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
                break;

              case GDK_Page_Down:
	      case GDK_KP_Page_Down:
                if(1)
                {
                    /* Get adjustment and scroll down one page. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value += adj->page_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Home:
	      case GDK_KP_Home:
                if(1)
                {
                    /* Get adjustment and scroll all the way up. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value = adj->lower;
			DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
		}
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
                break;

              case GDK_End:
              case GDK_KP_End:
                if(1)
                {
                    /* Get adjustment and scroll all the way down. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value = adj->upper - adj->page_size;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

	      default:
#if 0
/* This does not work well for the tree list, since it is not
 * sorted alphabetically.
 */
		/* For all other alphanumeric character keys and while
		 * no modifier keys are held, attempt to seek to the
		 * item who's name starts with the letter of the key
		 * that was pressed.
		 */
                if(isalnum((char)keyval) &&
                   ((state == 0x00000000) || (state == GDK_SHIFT_MASK))
                )
		{
		    gint column_type_name = 0;

		    gtk_clist_freeze(clist);
		    EDVCListSeekCharacter(
			clist, column_type_name, 0,
                        (gchar)((state & GDK_SHIFT_MASK) ?
                            toupper((char)keyval) : keyval
                        )
		    );
		    gtk_clist_thaw(clist);

		    DO_STOP_KEY_SIGNAL_EMIT
		    status = TRUE;
		}
#endif
		break;
            }
	}
	/* Contents clist. */
        else if(widget == browser->contents_clist)
        {
	    gint row;
	    GList *glist;
	    GtkCList *clist = GTK_CLIST(widget);


	    /* Get last selected row. */
	    row = EDVCListGetSelectedLast(clist, NULL);

	    /* Handle by key value. */
	    switch(keyval)
	    {
	      case GDK_Return:
	      case GDK_KP_Enter:
	      case GDK_ISO_Enter:
	      case GDK_3270_Enter:
                /* Handle only if control key modifier is not held, so
                 * that accelerator keys will get handled properly.
                 */
                if(!(state & GDK_CONTROL_MASK))
		{
		    if(press)
			EDVBrowserContentsDoOpenObject(
			    browser, row, 0, state
			);
		    DO_STOP_KEY_SIGNAL_EMIT
		    status = TRUE;
		}
		break;

              case GDK_Delete:
                if(press)
                {
                    EDVBrowserOPDelete(browser);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_space:
              case GDK_KP_Space:
		row = clist->focus_row;
                if((row >= 0) && (row < clist->rows) && press)
                {
                    gbool already_selected = FALSE;

                    /* Check if this row is already selected. */
                    glist = clist->selection;
                    while(glist != NULL)
                    {
                        if(row == (gint)glist->data)
                        {
                            already_selected = TRUE;
                            break;
                        }
                        glist = glist->next;
                    }

                    gtk_clist_freeze(clist);
		    if(already_selected)
			gtk_clist_unselect_row(clist, row, 0);
		    else
			gtk_clist_select_row(clist, row, 0);
                    gtk_clist_thaw(clist);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Up:
              case GDK_KP_Up:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll up. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value -= adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                else
                {
                    if(press)
                    {
			gint prev_focus_row = clist->focus_row;

                        gtk_clist_freeze(clist);
                        clist->focus_row--;
                        DO_CLIST_FOCUS_ROW_CLAMP

                        if(gtk_clist_row_is_visible(
                            clist, clist->focus_row) != GTK_VISIBILITY_FULL
                        )
                            gtk_clist_moveto(
                                clist,
                                clist->focus_row, -1,	/* Row, column. */
                                0.5, 0.0                /* Row, column. */
                            );

                        if(state & GDK_SHIFT_MASK)
                        {
                            gtk_clist_select_row(
                                clist, prev_focus_row, 0
                            );
                            gtk_clist_select_row(
                                clist, clist->focus_row, 0
                            );
                        }
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

              case GDK_Down:
              case GDK_KP_Down:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll down. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value += adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                else
                {
                    if(press)
                    {
                        gint prev_focus_row = clist->focus_row;

                        gtk_clist_freeze(clist);
                        clist->focus_row++;
                        DO_CLIST_FOCUS_ROW_CLAMP

                        if(gtk_clist_row_is_visible(
                            clist, clist->focus_row) != GTK_VISIBILITY_FULL
                        )
                            gtk_clist_moveto(
                                clist,
                                clist->focus_row, -1,	/* Row, column. */
                                0.5, 0.0                /* Row, column. */
                            );

			if(state & GDK_SHIFT_MASK)
			{
			    gtk_clist_select_row(
				clist, prev_focus_row, 0
			    );
                            gtk_clist_select_row(
                                clist, clist->focus_row, 0
                            );
			}
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

              case GDK_Left:
              case GDK_KP_Left:
                if(1)
                {
                    /* Get adjustment and scroll left. */
                    GtkAdjustment *adj = clist->hadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
			if(state & GDK_CONTROL_MASK)
			    adj->value -= adj->step_increment * 4;
			else if(state & GDK_SHIFT_MASK)
			    adj->value = adj->lower;
			else
			    adj->value -= adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Right:
              case GDK_KP_Right:
                if(1)
                {
                    /* Get adjustment and scroll right. */
                    GtkAdjustment *adj = clist->hadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        if(state & GDK_CONTROL_MASK)
                            adj->value += adj->step_increment * 4;
                        else if(state & GDK_SHIFT_MASK)
                            adj->value = adj->upper - adj->page_size;
                        else
                            adj->value += adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Page_Up:
              case GDK_KP_Page_Up:
                if(1)
                {
                    /* Get adjustment and scroll up one page. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value -= adj->page_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Page_Down:
              case GDK_KP_Page_Down:
                if(1)
                {
                    /* Get adjustment and scroll down one page. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value += adj->page_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Home:
              case GDK_KP_Home:
                if(1)
                {
                    /* Get adjustment and scroll all the way up. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value = adj->lower;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_End:
              case GDK_KP_End:
                if(1)
                {
                    /* Get adjustment and scroll all the way down. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value = adj->upper - adj->page_size;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              default:
                /* For all other alphanumeric character keys and while
                 * no modifier keys are held, attempt to seek to the
                 * item who's name starts with the letter of the key
                 * that was pressed.
                 */
                if(isalnum((char)keyval) &&
		   ((state == 0x00000000) || (state == GDK_SHIFT_MASK))
		)
                {
                    /* Get the column that is displaying the object
                     * name as column_type_name.
                     */
                    gint i, column_type_name = -1;
                    edv_intlist_struct *column_type_intlist =
                        EDVCFGItemListGetValueIntList(
                            core_ptr->cfg_list,
                            EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
                        );
                    if(column_type_intlist != NULL)
                    {
                        /* Iterate through column type intlist. */
                        for(i = 0; i < column_type_intlist->total; i++)
                        {
                            /* Is column i displaying the name? */
                            if(column_type_intlist->i[i] == EDV_BROWSER_COLUMN_TYPE_NAME)
                            {
                                column_type_name = i;
                                break;
                            }
                        }
                    }

                    gtk_clist_freeze(clist);
                    EDVCListSeekCharacter(
                        clist, column_type_name, 0,
			(gchar)((state & GDK_SHIFT_MASK) ?
			    toupper((char)keyval) : keyval
			)
                    );
                    gtk_clist_thaw(clist);

                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

	    }
	}

#undef DO_CLIST_FOCUS_ROW_CLAMP
#undef DO_ADJ_CLAMP_EMIT
#undef DO_STOP_KEY_SIGNAL_EMIT

	reenterent = FALSE;
	return(status);
}

/*
 *	Browser GtkWidget "button_press_event" signal callback.
 */
gint EDVBrowserButtonPressEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	static gbool reenterent = FALSE;
	gint status = FALSE;
	gint etype;
	guint state;
	edv_core_struct *core_ptr;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((widget == NULL) || (button == NULL) || (browser == NULL))
            return(status);

        if(browser->processing)
            return(status);

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	    return(status);

	if(reenterent)
	    return(status);
	else
	    reenterent = TRUE;

	/* Get event type. */
	etype = button->type;

	state = button->state;

        /* Check which ctree this signal is for. */
        if(widget == browser->directory_ctree)
        {
            gint row, column, nodes_selected = 0;
	    GList *glist;
            GtkCTreeNode *selected_node = NULL;
	    GtkCList *clist = GTK_CLIST(widget);
	    GtkCTree *ctree = GTK_CTREE(widget);
	    GtkCTreeNode *node = EDVNodeGetByCoordinates(
		ctree,
		button->x,
		button->y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
                clist->column_title_area.height +
                clist->column_title_area.y : 0)
	    );


            /* Find row and column based on given coordinates. */
            if(!gtk_clist_get_selection_info(
                clist, button->x, button->y, &row, &column
            ))
            {
                row = -1;
                column = 0;
            }

	    /* Get number of selected rows and highest selected row. */
	    glist = clist->selection;
	    while(glist != NULL)
            {
                selected_node = (GtkCTreeNode *)glist->data;
		if(selected_node != NULL)
		    nodes_selected++;
                glist = glist->next;
            }

	    /* Handle by button number. */
	    switch(button->button)
	    {
	      case 3:
		if(etype == GDK_BUTTON_PRESS)
		{
		    GtkMenu *menu;

		    /* Select item before mapping menu? */
		    if(EDVCFGItemListGetValueI(
			core_ptr->cfg_list, EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS
		    ) && (node != NULL))
		    {
			/* Select the node that the button was pressed over. */
			gtk_clist_freeze(clist);
			if((row >= 0) && (row < clist->rows))
			    clist->focus_row = row;
			gtk_ctree_select(ctree, node);
			gtk_clist_thaw(clist);
		    }

		    /* Update all menus and map right click menu. */
		    EDVBrowserUpdateMenus(browser);
                    menu = (GtkMenu *)browser->directory_ctree_menu;
		    if(menu != NULL)
			gtk_menu_popup(
			    menu, NULL, NULL,
			    NULL, NULL,
			    button->button, button->time
			);
		}
		status = TRUE;
		break;

	      case 2:
                if(etype == GDK_BUTTON_PRESS)
                {
		    if(node != NULL)
			EDVBrowserDirTreeDoFPromptRename(browser, node);
		}
                status = TRUE;
		break;
	    }

	    if(etype == GDK_BUTTON_PRESS)
		gtk_widget_grab_focus(widget);
	}
	else if(widget == browser->contents_clist)
        {
	    gint row, column;
	    gint rows_selected = 0, selected_row = -1;
	    GList *glist;
	    GtkCList *clist = GTK_CLIST(widget);


            /* Find row and column based on given coordinates. */
	    if(!gtk_clist_get_selection_info(
		clist, button->x, button->y, &row, &column
	    ))
            {
                row = -1;
                column = 0;
            }

	    /* Get number of selected rows and highest selected row. */
	    glist = clist->selection;
	    while(glist != NULL)
	    {
		rows_selected++;
		selected_row = (gint)glist->data;
		glist = glist->next;
	    }

	    /* Handle by button number. */
	    switch(button->button)
	    {
	      case 3:
		if(etype == GDK_BUTTON_PRESS)
		{
		    GtkMenu *menu;

		    /* Select item before mapping menu? */
		    if(EDVCFGItemListGetValueI(
			core_ptr->cfg_list, EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS
                    ) && (row >= 0) && (row < clist->rows))
		    {
			/* Select the row that the button was pressed over.
			 * if no key modifiers are held then this will also
			 * unselect all previously selected rows.
			 */
			gtk_clist_freeze(clist);
			if(!(button->state & GDK_CONTROL_MASK) &&
                           !(button->state & GDK_SHIFT_MASK)
			)
			    gtk_clist_unselect_all(clist);
			clist->focus_row = row;
			gtk_clist_select_row(clist, row, 0);
			gtk_clist_thaw(clist);
		    }

                    /* Update all menus and map right click menu. */
		    EDVBrowserUpdateMenus(browser);
                    menu = (GtkMenu *)browser->contents_clist_menu;
                    if(menu != NULL)
                        gtk_menu_popup(
                            menu, NULL, NULL,
                            NULL, NULL,
                            button->button, button->time
                        );
		}
                status = TRUE;
		break;

	      case 2:
		if(etype == GDK_BUTTON_PRESS)
		{
		    if((row >= 0) && (row < clist->rows))
			EDVBrowserContentsDoFPromptRename(
			    browser, row, column
			);
		}
                status = TRUE;
		break;

	      case 1:
		/* Double click? */
		if(etype == GDK_2BUTTON_PRESS)
		{
                    if((row >= 0) && (row < clist->rows))
		    {
			EDVBrowserContentsDoOpenObject(
			    browser, row, column, state
			);
			status = TRUE;
		    }
		}
		break;
	    }

            if(etype == GDK_BUTTON_PRESS)
                gtk_widget_grab_focus(widget);
	}

	reenterent = FALSE;
	return(status);
}

/*
 *	GtkHandleBox "child_attached" signal callback.
 */
void EDVBrowserHandleChildAttachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
        edv_browser_struct *browser = (edv_browser_struct *)data;
	if((handle_box == NULL) || (browser == NULL))
	    return;

	gtk_widget_queue_resize(
	    gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}

/*
 *      GtkHandleBox "child_detached" signal callback.
 */
void EDVBrowserHandleChildDetachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((handle_box == NULL) || (browser == NULL))
            return;

        gtk_widget_queue_resize(
            gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
        );
}


/*
 *	Browser GtkCTree "tree_select_row" signal callback.
 */
void EDVBrowserTreeSelectRowCB(
        GtkCTree *ctree, GtkCTreeNode *node, gint column,
        gpointer data
)
{
        static gbool reenterent = FALSE;
	edv_core_struct *core_ptr;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((ctree == NULL) || (browser == NULL))
            return;

        if(browser->processing)
            return;

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which ctree this signal is for. */
        if(GTK_WIDGET(ctree) == browser->directory_ctree)
        {
	    /* Change in selection? */
	    if((browser->directory_ctree_selected_node != node) &&
	       (node != NULL)
	    )
	    {
		gint row, column, objects_in_directory;
		edv_object_struct *obj;
		GtkCList *clist;


                EDVBrowserSetBusy(browser, TRUE);
		GUIBlockInput(browser->toplevel, TRUE);
                browser->processing = TRUE;

                /* Get disk object of selected node. */
                obj = (edv_object_struct *)gtk_ctree_node_get_row_data(
                    ctree, node
                );

		/* Get row and column index based on the tree node. */
		clist = GTK_CLIST(ctree);
		if(EDVNodeGetIndex(ctree, node, &row, &column))
		{
		    /* Check if selected row is fully visible, if not then
		     * adjust scroll position to try and make it visible.
		     */
		    if(gtk_clist_row_is_visible(clist, row) !=
			GTK_VISIBILITY_FULL
		    )
			gtk_clist_moveto(
			    clist,
			    row, -1,		/* Row, column. */
			    0.5, 0.0		/* Row, column. */
			);
		}

		/* Update current selected node on the directory ctree. */
                browser->directory_ctree_selected_node = node;

		/* Need to last selected contents clist item as none, this
		 * is so that operations will focus on the selected node
		 * on the directory ctree.
		 */
		browser->contents_clist_selected_row = -1;

		/* Match device index number from core structure's list of
		 * devices who's mount path matches the selected disk
		 * object's path.
		 */
		if(obj != NULL)
		    EDVeviceMatchListByMountPath(
			core_ptr->device, core_ptr->total_devices,
			&browser->selected_dev_num,
			obj->full_path
		    );


                /* Update DND icon for directory ctree. */
                EDVBrowserDirTreeDNDSetIcon(
                    browser, row
                );


		/* Get contents clist. */
		clist = (GtkCList *)browser->contents_clist;
		if((obj != NULL) && (clist != NULL))
		{
		    /* Update value of title, location combo (do not
		     * record history), and location icon.
		     */
		    EDVBrowserSetTitle(browser, obj->full_path);
		    EDVBrowserSetLocation(
			browser, obj->full_path, FALSE
		    );
		    EDVBrowserUpdateLocationIcon(browser, obj->full_path);

		    /* Update contents clist. */
		    gtk_clist_freeze(clist);
		    EDVBrowserContentsDoUpdate(browser, obj->full_path, TRUE);
		    gtk_clist_thaw(clist);

		    /* Get number of objects in the directory by the number
		     * of rows in the just updated contents clist minus one
		     * (to get rid of the parent directory notation row).
		     */
		    objects_in_directory = MAX(clist->rows - 1, 0);


                    /* Update status bar message. */
                    if(obj->name != NULL)
                    {
                        gchar *buf = g_strdup_printf(
 "Directory \"%s\" selected (containing %i objects)",
			    obj->name, objects_in_directory
			);

                        EDVStatusBarMessage(
                            browser->status_bar, buf, FALSE
                        );

                        g_free(buf);
                    }
                    else
                    {
                        EDVStatusBarMessage(
                            browser->status_bar,
                            "Directory with NULL name selected",
                            FALSE
                        );
                    }
		}

		browser->processing = FALSE;
                GUIBlockInput(browser->toplevel, FALSE);
                EDVBrowserSetBusy(browser, FALSE);

		EDVBrowserUpdateMenus(browser);
	    }
	}

	reenterent = FALSE;
}

/*
 *      Browser GtkCTree "tree_unselect_row" signal callback.
 */
void EDVBrowserTreeUnselectRowCB(
        GtkCTree *ctree, GtkCTreeNode *node, gint column,
        gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((ctree == NULL) || (browser == NULL))
            return;

        if(browser->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which ctree this signal is for. */
        if(GTK_WIDGET(ctree) == browser->directory_ctree)
        {
	    if(browser->directory_ctree_selected_node != NULL)
	    {
/*
		EDVBrowserSetBusy(browser, TRUE);
		GUIBlockInput(browser->toplevel, TRUE);
		browser->processing = TRUE;
 */
		/* Update currently selected node to NULL, indicating that
		 * no row is selected. This is known because only one row
		 * may be selected at any given time.
		 */
		browser->directory_ctree_selected_node = NULL;

		/* Mark that no device is selected. */
		browser->selected_dev_num = -1;

/*
		browser->processing = FALSE;
		GUIBlockInput(browser->toplevel, FALSE);
		EDVBrowserSetBusy(browser, FALSE);
 */
                EDVBrowserSetTitle(browser, NULL);
		EDVBrowserUpdateMenus(browser);
	    }
        }

        reenterent = FALSE;
}

/*
 *	Browser GtkCTree "tree_expand" signal callback.
 */
void EDVBrowserTreeExpandCB(
        GtkCTree *ctree, GtkCTreeNode *node, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((ctree == NULL) || (browser == NULL))
            return;

	if(browser->processing)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	/* Check which ctree this signal is for. */
	if(GTK_WIDGET(ctree) == browser->directory_ctree)
	{
	    EDVBrowserSetBusy(browser, TRUE);
	    GUIBlockInput(browser->toplevel, TRUE);
	    browser->processing = TRUE;

	    /* Expand node and get childs of each directory in this
	     * node.
	     */
	    gtk_clist_freeze(GTK_CLIST(ctree));
	    EDVBrowserDirTreeDoExpand(browser, node, TRUE);
            gtk_clist_thaw(GTK_CLIST(ctree));

	    browser->processing = FALSE;
            GUIBlockInput(browser->toplevel, FALSE);
            EDVBrowserSetBusy(browser, FALSE);

	    EDVBrowserUpdateMenus(browser);
	}

        reenterent = FALSE;
}

/*
 *      Browser GtkCTree "tree_collapse" signal callback.
 */
void EDVBrowserTreeCollapseCB(
        GtkCTree *ctree, GtkCTreeNode *node, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((ctree == NULL) || (browser == NULL))
            return;

        if(browser->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which ctree this signal is for. */
        if(GTK_WIDGET(ctree) == browser->directory_ctree)
        {
/*
            EDVBrowserSetBusy(browser, TRUE);
            GUIBlockInput(browser->toplevel, TRUE);
	    browser->processing = TRUE;

            gtk_clist_freeze(GTK_CLIST(ctree));



            gtk_clist_thaw(GTK_CLIST(ctree));

	    browser->processing = FALSE;
            GUIBlockInput(browser->toplevel, FALSE);
            EDVBrowserSetBusy(browser, FALSE);
 */

	    EDVBrowserUpdateMenus(browser);
	}

        reenterent = FALSE;
}


/*
 *	Browser GtkCList "resize_column" signal callback.
 */
void EDVBrowserResizeColumnCB(
        GtkCList *clist, gint column, gint width, gpointer data
)
{
        static gbool reenterent = FALSE;
	edv_core_struct *core_ptr;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((clist == NULL) || (browser == NULL))
            return;

        if(browser->processing)
            return;

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == browser->contents_clist)
        {
	    gint column_type = -1;
	    edv_intlist_struct *column_types_intlist, *column_width_intlist;


	    /* Get column_type from the given column index. */
	    column_types_intlist = EDVCFGItemListGetValueIntList(
		core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	    );
	    if(column_types_intlist != NULL)
	    {
		if((column >= 0) && (column < column_types_intlist->total))
		    column_type = column_types_intlist->i[column];
	    }

	    /* Get column widths intlist. */
	    column_width_intlist = EDVCFGItemListGetValueIntList(
		core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN_WIDTH
	    );
	    if(column_width_intlist != NULL)
	    {
		/* Need to increase column_width_intlist allocation? */
		if((column_type >= column_width_intlist->total) &&
                   (column_type >= 0)
		)
		{
		    gint n;
		    gint prev_total = column_width_intlist->total;

		    /* Increase array allocation. */
		    column_width_intlist->total = column_type + 1;
		    column_width_intlist->i = (gint *)g_realloc(
			column_width_intlist->i,
			column_width_intlist->total * sizeof(gint)
		    );
		    /* Reset newly allocated indexes. */
		    for(n = prev_total; n < column_width_intlist->total; n++)
			column_width_intlist->i[n] = 0;
		}

		/* Got column type in bounds as index on column width
		 * intlist?
		 */
		if((column_type >= 0) &&
                   (column_type < column_width_intlist->total)
		)
		{
		    /* Record new width of column on the column width
		     * intlist.
		     */
		    column_width_intlist->i[column_type] = width;
		}
	    }
        }

        reenterent = FALSE;
}

/*
 *	Browser GtkCList "click_column" signal callback.
 */
void EDVBrowserClickColumnCB(
        GtkCList *clist, gint column, gpointer data
)
{
        static gbool reenterent = FALSE;
	edv_core_struct *core_ptr;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((clist == NULL) || (browser == NULL))
            return;

        if(browser->processing)
            return;

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == browser->contents_clist)
        {
	    edv_intlist_struct *column_types_intlist;
            GtkCListCompareFunc cmp_func = NULL;
            GtkCListCompareFunc cmp_func_str =
                 (GtkCListCompareFunc)EDVCListColumnSortStringCB;
            GtkCListCompareFunc cmp_func_num =
                (GtkCListCompareFunc)EDVCListColumnSortNumericCB;


            EDVBrowserSetBusy(browser, TRUE);
            GUIBlockInput(browser->toplevel, TRUE);
            browser->processing = TRUE;	    


	    /* Get column types mapping. */
	    column_types_intlist = EDVCFGItemListGetValueIntList(
		core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	    );
	    if(column_types_intlist != NULL)
	    {
		if((column >= 0) && (column < column_types_intlist->total))
		{
		    gint column_type = column_types_intlist->i[column];
		    switch(column_type)
		    {
		      case EDV_BROWSER_COLUMN_TYPE_NAME:
			cmp_func = cmp_func_str;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_SIZE:
                        cmp_func = cmp_func_num;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_TYPE:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_PERMISSIONS:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_OWNER:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_GROUP:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_DATE_ACCESS:
                        cmp_func = EDVBrowserCListColumnSortDateAccessCB;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_DATE_MODIFIED:
                        cmp_func = EDVBrowserCListColumnSortDateModifyCB;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_DATE_CHANGED:
                        cmp_func = EDVBrowserCListColumnSortDateChangeCB;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_HARD_LINKS:
                        cmp_func = cmp_func_num;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_LINKED_TO:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_DEVICE:
                        cmp_func = cmp_func_num;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_INODE:
                        cmp_func = cmp_func_num;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_DEVICE_TYPE:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_BLOCK_SIZE:
                        cmp_func = cmp_func_num;
                        break;
                      case EDV_BROWSER_COLUMN_TYPE_BLOCKS:
                        cmp_func = cmp_func_num;
                        break;
		    }
		}
	    }


	    gtk_clist_freeze(clist);

	    /* Set sort column settings on the clist. */
	    if(column != clist->sort_column)
		gtk_clist_set_sort_column(clist, column);
	    else
		gtk_clist_set_sort_type(
		    clist,
		    (clist->sort_type == GTK_SORT_ASCENDING) ?
			GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
		);
            if(cmp_func != NULL)
                gtk_clist_set_compare_func(clist, cmp_func);

	    /* Sort rows, this will call the GtkCList column sort
	     * callbacks.
	     */
	    gtk_clist_sort(clist);

	    gtk_clist_thaw(clist);

            browser->processing = FALSE;
            GUIBlockInput(browser->toplevel, FALSE);
            EDVBrowserSetBusy(browser, FALSE);

            EDVBrowserUpdateMenus(browser);
        }

        reenterent = FALSE;
}

/*
 *	Browser GtkCList "select_row" signal callback.
 */
void EDVBrowserSelectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((clist == NULL) || (browser == NULL))
            return;

        if(browser->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == browser->contents_clist)
        {
	    gint total_selected = 0;
	    GList *glist;
	    edv_object_struct *obj;


	    /* Get total number of objects selected. */
	    glist = clist->selection;
	    while(glist != NULL)
	    {
		total_selected++;
		glist = glist->next;
	    }


	    /* Update selected row. */
	    browser->contents_clist_selected_row = row;

	    /* Need to mark last selected node on the directory
	     * ctree as NULL. This is a hint that operations should
	     * focus on the selected items on the  ontents clist.
	     */
	    browser->directory_ctree_selected_node = NULL;

	    /* Mark that no device is selected. */
	    browser->selected_dev_num = -1;


	    /* Update DND icon for contents clist. */
	    EDVBrowserContentsDNDSetIcon(browser, row, column);

	    /* Get disk object structure from selected row. */
	    obj = (edv_object_struct *)gtk_clist_get_row_data(
		clist, row
	    );
	    if(obj != NULL)
	    {
		/* Update status bar message. */
                if(obj->name != NULL)
                {
                    gchar *buf;
		    const gchar *type_str = "Object";
		    gchar size_str[80];


		    /* Get object type and size strings. */
		    *size_str = '\0';
		    switch(obj->type)
		    {
		      case EDV_OBJECT_TYPE_FILE:
			type_str = "File";
			sprintf(size_str, " (%s bytes)",
			    EDVGetObjectSizeStr(
				(edv_core_struct *)browser->core_ptr,
				obj->size
			    )
			);
			break;
                      case EDV_OBJECT_TYPE_DIRECTORY:
                        type_str = "Directory";
                        break;
                      case EDV_OBJECT_TYPE_LINK:
                        type_str = "Link";
                        break;
                      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
                        type_str = "Block device";
                        break;
                      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
                        type_str = "Character device";
                        break;
                      case EDV_OBJECT_TYPE_FIFO:
                        type_str = "FIFO Pipe";
                        break;
                      case EDV_OBJECT_TYPE_SOCKET:
                        type_str = "Socket";
                        break;
                    }

		    /* Format status bar message. */
		    if(total_selected > 1)
			buf = g_strdup_printf(
			    "%i objects selected",
			    total_selected
			);
		    else if(!strcmp(obj->name, ".."))
			buf = g_strdup_printf(
			    "Parent directory selected"
			);
		    else
			buf = g_strdup_printf(
			    "%s \"%s\" selected%s",
			    type_str, obj->name, size_str
			);

		    /* Set status bar message. */
                    EDVStatusBarMessage(
                        browser->status_bar, buf, FALSE
                    );

		    /* Deallocate status bar message. */
		    g_free(buf);
                }
                else
                {
                    EDVStatusBarMessage(
                        browser->status_bar,
                        "Object with NULL name selected",
                        FALSE
                    );
                }
	    }

	    /* Check if selected row is fully visible, if not then adjust
	     * scroll position to try and make it visible.
	     */
	    if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    row, -1,	/* Row, column. */
		    0.5, 0.0	/* Row, column. */
		);

            EDVBrowserUpdateMenus(browser);
	}

	reenterent = FALSE;
}

/*
 *	Browser GtkCList "unselect_row" signal callback.
 */
void EDVBrowserUnselectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((clist == NULL) || (browser == NULL))
            return;

        if(browser->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == browser->contents_clist)
        {



            EDVBrowserUpdateMenus(browser);
        }

        reenterent = FALSE;
}


/*
 *	Location combo activate callback.
 */
void EDVBrowserComboActivateCB(GtkWidget *widget, gpointer data)
{
	static gbool reenterent = FALSE;
	GtkCombo *combo = (GtkCombo *)widget;
	edv_browser_struct *browser = (edv_browser_struct *)data;
        if((widget == NULL) || (browser == NULL))
            return;

	if(browser->processing)
	    return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;

	/* Check which widget was activated. */
	if(widget == browser->location_combo)
	{
	    gchar *new_path = EDVCopyEvaluateInputPath(
		NULL,		/* No parent path, imply use toplevel. */
		gtk_entry_get_text(GTK_ENTRY(combo->entry))
	    );
	    if(new_path != NULL)
	    {
		GtkCList *clist;


		EDVBrowserSetBusy(browser, TRUE);
                GUIBlockInput(browser->toplevel, TRUE);

		/* Do not mark as processing or else
		 * EDVBrowserDirTreeDoSelectPath() will not be able to
		 * expand nodes because the expand signal callback will
		 * detect the processing and not expand the node.
		 */

		/* Update location combo, this is really just to record
		 * the new value as history. The combo will be updated
		 * again (without recording history) if the new path is
		 * valid and selected in the callbacks triggered by
		 * EDVBrowserDirTreeDoSelectPath() farther below.
		 */
		EDVBrowserSetLocation(browser, new_path, TRUE);

		/* Update selected node on directory ctree. */
		clist = (GtkCList *)browser->directory_ctree;
		if(clist != NULL)
		{
		    /* Attempt to select the node who's disk object path
		     * matches new_path. Recurse and expand as needed.
		     *
		     * The ctree will be freeze'ed and thawed by the
		     * callbacks each time it is expanded so we do not
		     * need to freeze or thaw it here.
		     *
		     * Also, the location prompt will be updated again
		     * (but its history will not be re-recorded) when the
		     * the node is selected.
		     */
		    EDVBrowserDirTreeDoSelectPath(browser, new_path);
		}


                GUIBlockInput(browser->toplevel, FALSE);
		EDVBrowserSetBusy(browser, FALSE);


		/* Deallocate copy of the new path. */
		g_free(new_path);
		new_path = NULL;
	    }
	}

	reenterent = FALSE;
}



/*
 *	Called whenever the global write protect has changed.
 *
 *	The new state is given as state.
 */
void EDVBrowserWriteProtectChangedCB(
        edv_browser_struct *browser, gbool state
)
{
        if(browser == NULL)
            return;

        if(browser->processing)
            return;

	EDVBrowserUpdateMenus(browser);
}

/*
 *	Called whenever a disk object has been added.
 */
void EDVBrowserObjectAddedNotifyCB(
        edv_browser_struct *browser, const gchar *path,
        const struct stat *lstat_buf
)
{
	if(browser == NULL)
	    return;

        if(browser->processing)
            return;

	/* Notify contents clist and directory ctree about added object. */
        EDVBrowserContentsObjectAddedNotify(
            browser, path, lstat_buf
        );
	EDVBrowserDirTreeObjectAddedNotify(
	    browser, path, lstat_buf
	);

/*	EDVBrowserUpdateMenus(browser); */
}

/*
 *      Called whenever a disk object has had its properties modified.
 */
void EDVBrowserObjectModifiedNotifyCB(
        edv_browser_struct *browser, const gchar *path,
	const gchar *new_path,
        const struct stat *lstat_buf
)
{
        if(browser == NULL)
            return;

	if(browser->processing)
	    return;

	/* Check current location first, if the old path matches the
	 * current location then the current location needs to be updated
	 * to reflect the new path before notifying the contents clist
	 * and directory ctree.
	 */
	if(path != NULL)
	{
	    const gchar *cstrptr;


	    if(new_path == NULL)
		new_path = path;

	    /* Check if the old path matches the current location, if it
	     * does then the current location needs to be updated to the
	     * new path.
	     */
	    cstrptr = EDVBrowserCurrentLocation(browser);
	    if((cstrptr != NULL) ? !strcmp(cstrptr, path) : FALSE)
	    {
		/* Old path matches current location, change values
		 * reflecting the current location to the value of
		 * new_path.
		 */
		EDVBrowserSetTitle(browser, new_path);
		EDVBrowserSetLocation(browser, new_path, FALSE);
                EDVBrowserUpdateLocationIcon(browser, new_path);
	    }
	}

	/* Notify contents clist and directory ctree about modified object. */
        EDVBrowserContentsObjectModifiedNotify(
            browser, path, new_path, lstat_buf
        );
        EDVBrowserDirTreeObjectModifiedNotify(
            browser, path, new_path, lstat_buf
        );

/*	EDVBrowserUpdateMenus(browser); */
}

/*
 *	Called whenever a disk object has been removed.
 */
void EDVBrowserObjectRemovedNotifyCB(
        edv_browser_struct *browser, const gchar *path
)
{
        if(browser == NULL)
            return;

        if(browser->processing)
            return;

        /* Notify contents clist and directory ctree about removed object. */
        EDVBrowserContentsObjectRemovedNotify(browser, path);
        EDVBrowserDirTreeObjectRemovedNotify(browser, path);

	/* Check if the path that was removed is the current location
	 * (after letting contents clist and directory ctree check on it
	 * first), if the path that was removed is the location then
	 * set the current location to be the parent of the path that
	 * was removed.
	 */
        if(path != NULL)
        {
	    const gchar *cstrptr;

            /* Check if the removed object path matches the current
	     * location. If it matches then the current location needs to
	     * be updated to reflect that.
	     */
            cstrptr = EDVBrowserCurrentLocation(browser);
            if((cstrptr != NULL) ? !strcmp(cstrptr, path) : FALSE)
            {
                /* Removed object path matches the current location, so
		 * technically the current location needs to be cleared
		 * however we'll instead just change the current location
		 * value to reflect the value of the parent path of the
		 * removed object path.
                 */
		cstrptr = GetParentDir(path);
		if(cstrptr != NULL)
		{
		    EDVBrowserSetTitle(browser, cstrptr);
		    EDVBrowserSetLocation(browser, cstrptr, FALSE);
                    EDVBrowserUpdateLocationIcon(browser, cstrptr);
		}
            }
        }

/*	EDVBrowserUpdateMenus(browser); */
}


/*
 *      Called whenever a device has been mounted or unmounted.
 */
void EDVBrowserMountNotifyCB(
	edv_browser_struct *browser,
	gint dev_num, edv_device_struct *dev_ptr,
	gbool is_mounted
)
{
        if(browser == NULL)
            return;

        if(browser->processing)
            return;

        /* Notify contents clist and directory ctree about mount. */
	EDVBrowserContentsMountNotify(browser, dev_ptr, is_mounted);
        EDVBrowserDirTreeMountNotify(browser, dev_ptr, is_mounted);


        EDVBrowserUpdateMenus(browser);
}


/*
 *	Called whenever an object has been added to the recycle bin.
 */
void EDVBrowserRecycledObjectAddedNotifyCB(
        edv_browser_struct *browser, guint index
)
{
        edv_core_struct *core_ptr;


	if(browser == NULL)
	    return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;


	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects.
	 */
        if(core_ptr->last_recbin_items != browser->last_recbin_items)
	    EDVBrowserUpdateMenus(browser);
}

/*
 *	Called whenever an object has been removed from the recycle bin.
 */
void EDVBrowserRecycledObjectRemovedNotifyCB(
        edv_browser_struct *browser, guint index
)
{
	edv_core_struct *core_ptr;


        if(browser == NULL)
            return;

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	    return;


        /* There is not much interest when a recycled object has been
         * added or removed. Only the menus need to be updated if there
         * is a change in the number of recycled objects.
         */
	if(core_ptr->last_recbin_items != browser->last_recbin_items)
	    EDVBrowserUpdateMenus(browser);
}


/*
 *	Called whenever the global configuration has changed.
 */
void EDVBrowserReconfiguredNotifyCB(edv_browser_struct *browser)
{
	const gchar *cstrptr;
	gchar *cur_path;
	GtkWidget *w;
	GtkCList *clist;
	edv_status_bar_struct *status_bar;
	edv_core_struct *core_ptr;


        if(browser == NULL)
            return;

        if(browser->processing)
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;


        /* Reset last state markers so that resources get explicitly
         * checked due to reconfiguring.
         */
	browser->last_recbin_items = -1;
	browser->last_write_protect_state = -1;


	/* Get copy of current location path. */
	cstrptr = EDVBrowserCurrentLocation(browser);
	cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


	/* Update title. */
	EDVBrowserSetTitle(browser, cur_path);

	/* Regenerate tool bar. */
	EDVBrowserToolbarRegenerate(browser);


	/* Show tool bar? */
	w = browser->tool_bar_handle;
	if(w != NULL)
	{
	    browser->tool_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_SHOW_TOOL_BAR
            );
	    if(browser->tool_bar_map_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

        /* Show location bar? */
        w = browser->location_bar_handle;
        if(w != NULL)
        {
            browser->location_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_SHOW_LOCATION_BAR
            );
            if(browser->location_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show mount bar? */
        w = browser->mount_bar_handle;
        if(w != NULL)
        {
            browser->mount_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_SHOW_MOUNT_BAR
            );
            if(browser->mount_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show find bar? */
        w = browser->find_bar_handle;
        if(w != NULL)
        {
            browser->find_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_SHOW_FIND_BAR
            );
            if(browser->find_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show status bar? */
        status_bar = browser->status_bar;
        if(status_bar != NULL)
        {
            browser->status_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_SHOW_STATUS_BAR
            );
            if(browser->status_bar_map_state)
                EDVStatusBarMap(status_bar);
            else
                EDVStatusBarUnmap(status_bar);
        }

        /* Reget listings. */

	/* Update all nodes and remove any nodes that reffer to non-existing
	 * disk objects on the directory ctree.
	 */
	EDVBrowserDirTreeDoSync(browser, NULL);

	/* Reget listing for contents clist. */
        clist = (GtkCList *)browser->contents_clist;
        if(clist != NULL)
        {
            gtk_clist_freeze(clist);
            EDVBrowserContentsResetRows(browser);
            gtk_clist_thaw(clist);
        }


	/* Update menus. */
        EDVBrowserUpdateMenus(browser);


	/* Notify browser's toplevel widget to resize. */
	w = browser->toplevel;
	if(w != NULL)
	    gtk_widget_queue_resize(w);


	/* Deallocate copy of current location path. */
	g_free(cur_path);
	cur_path = NULL;
}


/*
 *      Called whenever a MIME Type has been added to the core
 *      structure's list of MIME Types.
 */
void EDVBrowserMimeTypeAddedCB(
        edv_browser_struct *browser,
        gint mt_num, edv_mimetype_struct *mt_ptr
)
{
        /* Treat a MIME Type added the same as it would be for a MIME Type
         * modified, forward signal to the MIME Type modified callback.
         */
        EDVBrowserMimeTypeModifiedCB(browser, mt_num, mt_ptr);
}

/*
 *      Called whenever a MIME Type has been modified on the core
 *      structure's list of MIME Types.
 */
void EDVBrowserMimeTypeModifiedCB(
        edv_browser_struct *browser,
        gint mt_num, edv_mimetype_struct *mt_ptr
)
{
        const gchar *cstrptr;
        gchar *cur_path;
        GtkCList *clist;
        edv_core_struct *core_ptr;


        if(browser == NULL)
            return;

        if(browser->processing)
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;


        /* Get copy of current location path. */
        cstrptr = EDVBrowserCurrentLocation(browser);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;



        /* Reget listings. */

	/* Update all nodes and remove any nodes that reffer to non-existing
	 * disk objects on the directory ctree.
	 */
        EDVBrowserDirTreeDoSync(browser, NULL);

        /* Reset displayed row values of contents clist. */
        clist = (GtkCList *)browser->contents_clist;
        if(clist != NULL)
        {
            gtk_clist_freeze(clist);
            EDVBrowserContentsResetRows(browser);
            gtk_clist_thaw(clist);
        }

/*	EDVBrowserUpdateMenus(browser); */

        /* Deallocate copy of current location path. */
        g_free(cur_path);
        cur_path = NULL;
}

/*
 *      Called whenever a MIME Type has been removed from the core
 *      structure's list of MIME Types.
 */
void EDVBrowserMimeTypeRemovedCB(
        edv_browser_struct *browser, gint mt_num
)
{
        const gchar *cstrptr;
        gchar *cur_path;
        GtkCList *clist;
        edv_core_struct *core_ptr;


        if(browser == NULL)
            return;

        if(browser->processing)
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;


        /* Get copy of current location path. */
        cstrptr = EDVBrowserCurrentLocation(browser);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;



	/* Reget listings. */

        /* Update all nodes and remove any nodes that reffer to non-existing
         * disk objects on the directory ctree.
         */
        EDVBrowserDirTreeDoSync(browser, NULL);

        /* Reset displayed row values of contents clist. */
        clist = (GtkCList *)browser->contents_clist;
        if(clist != NULL)
        {
            gtk_clist_freeze(clist);
            EDVBrowserContentsResetRows(browser);
            gtk_clist_thaw(clist);
        }

/*	EDVBrowserUpdateMenus(browser); */

        /* Deallocate copy of current location path. */
        g_free(cur_path);
        cur_path = NULL;
}


/*
 *      Called whenever a device has been added to the core
 *      structure's list of devices.
 */
void EDVBrowserDeviceAddedCB(
        edv_browser_struct *browser,
        gint dev_num, edv_device_struct *dev_ptr
)
{
	/* Treat a device added the same as it would be for a device
	 * modified, forward signal to the device modified callback.
	 */
	EDVBrowserDeviceModifiedCB(browser, dev_num, dev_ptr);
}

/*
 *	Called whenever a device has been modified on the core
 *	structure's list of devices.
 */
void EDVBrowserDeviceModifiedCB(
        edv_browser_struct *browser,
        gint dev_num, edv_device_struct *dev_ptr
)
{
	const gchar *cstrptr;
	gchar *cur_path;
        GtkCList *clist;
        edv_core_struct *core_ptr;


        if(browser == NULL)
            return;

        if(browser->processing)
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;


        /* Get copy of current location path. */
        cstrptr = EDVBrowserCurrentLocation(browser);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;



        /* Reget listings. */

        /* Update all nodes and remove any nodes that reffer to non-existing
         * disk objects on the directory ctree.
         */
        EDVBrowserDirTreeDoSync(browser, NULL);

        /* Reset displayed row values of contents clist. */
        clist = (GtkCList *)browser->contents_clist;
        if(clist != NULL)
        {
            gtk_clist_freeze(clist);
	    EDVBrowserContentsResetRows(browser);
            gtk_clist_thaw(clist);
        }

        /* Update menus. */
        EDVBrowserUpdateMenus(browser);

        /* Deallocate copy of current location path. */
        g_free(cur_path);
        cur_path = NULL;
}

/*
 *      Called whenever a device has been removed from the core
 *      structure's list of devices.
 */
void EDVBrowserDeviceRemovedCB(
        edv_browser_struct *browser, gint dev_num
)
{
        const gchar *cstrptr;
        gchar *cur_path;
        GtkCList *clist;
	edv_mountbar_struct *mb;
        edv_core_struct *core_ptr;


        if(browser == NULL)
            return;

        if(browser->processing)
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;


        /* Get copy of current location path. */
        cstrptr = EDVBrowserCurrentLocation(browser);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


	/* Check if mount bar is currently referencing this device, if it
	 * is then its selected device needs to be set to -1.
	 */
	mb = browser->mountbar;
	if(mb != NULL)
	{
	    if(mb->selected_dev_num == dev_num)
		mb->selected_dev_num = -1;
	    /* Mount bar will be updated further below when menus are
	     * updated.
	     */
	}


        /* Reget listings. */

        /* Update all nodes and remove any nodes that reffer to non-existing
         * disk objects on the directory ctree.
         */
        EDVBrowserDirTreeDoSync(browser, NULL);

        /* Reset displayed row values of contents clist. */
        clist = (GtkCList *)browser->contents_clist;
        if(clist != NULL)
        {
            gtk_clist_freeze(clist);
            EDVBrowserContentsResetRows(browser);
            gtk_clist_thaw(clist);
        }

        /* Update menus. */
        EDVBrowserUpdateMenus(browser);

        /* Deallocate copy of current location path. */
        g_free(cur_path);
        cur_path = NULL;
}
