#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"
#include "fprompt.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvobj.h"
#include "edvfop.h"
#include "browser.h"
#include "browsercb.h"
#include "browserdirtree.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"


static void EDVBrowserDirTreeFPromptCancelCB(gpointer data);

static void EDVBrowserDirTreeSetNode(
	edv_core_struct *core_ptr, edv_browser_struct *browser,
	GtkCTree *ctree, GtkCTreeNode *node, edv_object_struct *object
);

gint EDVNodeDiskObjectLStat(
        GtkCTree *ctree, GtkCTreeNode *node,
        struct stat *stat_buf
);
gint EDVNodeDiskObjectStat(
        GtkCTree *ctree, GtkCTreeNode *node,
        struct stat *stat_buf
);
gchar *EDVNodeDiskObjectFullPath(GtkCTree *ctree, GtkCTreeNode *node);
GtkCTreeNode *EDVBrowserDirTreeGetVFSToplevel(
        edv_browser_struct *browser
);

GtkCTreeNode *EDVBrowserDirTreeInsertNode(
	edv_browser_struct *browser,
        GtkCTreeNode *parent, GtkCTreeNode *sibling,
	edv_object_struct *object
);

void EDVBrowserDirTreeDoCreateToplevels(
	edv_browser_struct *browser
);
static void EDVBrowserDirTreeDoGetChildrenListIterate(
	edv_core_struct *core_ptr, edv_browser_struct *browser,
	GtkCTree *ctree, GtkCTreeNode *parent,
        gbool update_progress, gbool recurse,
        gbool hide_object_hidden, gbool hide_object_noaccess
);
static void EDVBrowserDirTreeDoGetChildrenList(
	edv_browser_struct *browser, GtkCTreeNode *parent,
	gbool update_progress, gbool recurse
);

void EDVBrowserDirTreeDoRefresh(
        edv_browser_struct *browser, GtkCTreeNode *node,
	gbool update_progress
);
void EDVBrowserDirTreeDoExpand(
        edv_browser_struct *browser, GtkCTreeNode *node,
	gbool update_progress
);

static void EDVBrowserDirTreeDoSyncIterate(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCTree *ctree, GtkCTreeNode *node,
	gbool parent_is_expanded
);
void EDVBrowserDirTreeDoSync(
        edv_browser_struct *browser, GtkCTreeNode *node
);

static void EDVBrowserDirTreeDoSelectPathIterate(
        GtkCTree *ctree, GtkCTreeNode *node, const gchar *path,
        GtkCTreeNode **matched_node
);
void EDVBrowserDirTreeDoSelectPath(
	edv_browser_struct *browser, const gchar *path
);

static void EDVBrowserDirTreeFindNodeByPathIterate(
        GtkCTree *ctree, GtkCTreeNode *node,
        const gchar *path, GtkCTreeNode **matched_node
);
GtkCTreeNode *EDVBrowserDirTreeFindNodeByPath(
	edv_browser_struct *browser, const gchar *path
);

static void EDVBrowserDirTreeFPromptRenameApplyCB(
	gpointer data, const gchar *value
);
void EDVBrowserDirTreeDoFPromptRename(
        edv_browser_struct *browser, GtkCTreeNode *node
);

void EDVBrowserDirTreeObjectAddedNotify(
        edv_browser_struct *browser, const gchar *path,
	const struct stat *lstat_buf
);
void EDVBrowserDirTreeObjectModifiedNotify(
        edv_browser_struct *browser, const gchar *path,
        const gchar *new_name,
        const struct stat *lstat_buf
);
void EDVBrowserDirTreeObjectRemovedNotify(
        edv_browser_struct *browser, const gchar *path
);

void EDVBrowserDirTreeMountNotify(
        edv_browser_struct *browser, edv_device_struct *dev_ptr,
	gbool is_mounted
);



#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	All purpose fprompt cancel callback.
 */
static void EDVBrowserDirTreeFPromptCancelCB(gpointer data)
{
	gpointer *cb_data = (gpointer *)data;
	if(cb_data == NULL)
	    return;

	/* Deallocate callback data. */
	g_free(cb_data);
	cb_data = NULL;
}


/*
 *	Sets the pixmap, style, and text for the node of the given ctree.
 */
static void EDVBrowserDirTreeSetNode(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCTree *ctree, GtkCTreeNode *node, edv_object_struct *object
)
{
	const gchar *text;
	GtkStyle *style;
	GtkCTreeRow *ctree_row_ptr;
	GdkPixmap *pixmap_closed, *pixmap_opened;
	GdkBitmap *mask_closed, *mask_opened;


	if((core_ptr == NULL) || (browser == NULL) ||
	   (ctree == NULL) || (node == NULL) || (object == NULL)
	)
	    return;


	/* Get text from object's name. */
	text = object->name;


	/* Begin checking for which GtkStyle to be used. */
	style = NULL;

	/* Directory not accessable? */
	if(!EDVCheckObjectAccessable(core_ptr, object))
	    style = browser->cell_style[EDV_BROWSER_CELL_STYLE_NO_ACCESS];


	/* Get opened and closed icons for this directory or link object. */
        EDVMatchObjectIcon(
            core_ptr->device, core_ptr->total_devices,
            core_ptr->mimetype, core_ptr->total_mimetypes,
            object->type,
            object->full_path,
	    object->link_valid, object->permissions,
            0,                  /* Small icons. */
            &pixmap_closed, &mask_closed,
            &pixmap_opened, &mask_opened,
	    NULL, NULL
        );
	/* If opened pixmap is not aviailable then set opened pixmap to be
	 * the same as the closed pixmap.
	 */
	if(pixmap_opened == NULL)
	{
	    pixmap_opened = pixmap_closed;
	    mask_opened = mask_closed;
        }


	/* Get pointer ctree row from the given node. */
        ctree_row_ptr = GTK_CTREE_ROW(node);
        if(ctree_row_ptr == NULL)
            return;

	/* Update text? */
        if(text != NULL)
            gtk_ctree_set_node_info(
                ctree, node, text,
                EDV_LIST_PIXMAP_TEXT_SPACING,
                (pixmap_closed != NULL) ? 
		    pixmap_closed : ctree_row_ptr->pixmap_closed,
		(mask_closed != NULL) ?
		    mask_closed : ctree_row_ptr->mask_closed,
		(pixmap_opened != NULL) ?
		    pixmap_opened : ctree_row_ptr->pixmap_opened,
		(mask_opened != NULL) ?
		    mask_opened : ctree_row_ptr->mask_opened,
                ctree_row_ptr->is_leaf,
                ctree_row_ptr->expanded
            );

	/* Set node style. */
	gtk_ctree_node_set_row_style(ctree, node, style);
}


/*
 *	Attempts to lstat() the object specified by the disk object
 *	structure on the given node.
 *
 *	Returns -1 if it cannot be lstat()'ed or 0 on success.
 */
gint EDVNodeDiskObjectLStat(
	GtkCTree *ctree, GtkCTreeNode *node,
	struct stat *stat_buf
)
{
	edv_object_struct *obj;
	struct stat local_stat_buf;


	if((ctree == NULL) || (node == NULL))
	    return(-1);

	/* Get disk object structure from this node. */
	obj = (edv_object_struct *)gtk_ctree_node_get_row_data(
	    ctree, node
	);
	if((obj != NULL) ? (obj->full_path == NULL) : TRUE)
	    return(-1);

	/* Try to lstat() the object. */
	if(lstat(
	    obj->full_path,
	    (stat_buf != NULL) ? stat_buf : &local_stat_buf
	))
	    return(-1);
	else
	    return(0);
}

/*
 *      Attempts to stat() the object specified by the disk object
 *      structure on the given node.
 *
 *      Returns -1 if it cannot be stat()'ed or 0 on success.
 */
gint EDVNodeDiskObjectStat(
        GtkCTree *ctree, GtkCTreeNode *node,
        struct stat *stat_buf
)
{
        edv_object_struct *obj;
        struct stat local_stat_buf;


        if((ctree == NULL) || (node == NULL))
            return(-1);

        /* Get disk object structure from this node. */
        obj = (edv_object_struct *)gtk_ctree_node_get_row_data(
            ctree, node
        );
        if((obj != NULL) ? (obj->full_path == NULL) : TRUE)
            return(-1);

        /* Try to stat() the object. */
        if(stat(
            obj->full_path,
            (stat_buf != NULL) ? stat_buf : &local_stat_buf
        ))
            return(-1);
        else
            return(0);
}

/*
 *	Returns the full path specified on the disk object from the given
 *	node. Can return NULL if there isn't any.
 */
gchar *EDVNodeDiskObjectFullPath(GtkCTree *ctree, GtkCTreeNode *node)
{
        edv_object_struct *obj;


        if((ctree == NULL) || (node == NULL))
            return(NULL);

        /* Get disk object structure from this node. */
        obj = (edv_object_struct *)gtk_ctree_node_get_row_data(
            ctree, node
        );
        if(obj == NULL)
	    return(NULL);

	return(obj->full_path);
}

/*
 *	Returns the node to the toplevel node of the VFS. This is usually
 *	the node representing the toplevel directory "/".
 *
 *	However in the future this may be a different one.
 */
GtkCTreeNode *EDVBrowserDirTreeGetVFSToplevel(
	edv_browser_struct *browser
)
{
	GtkCTree *ctree;


	if(browser == NULL)
	    return(NULL);

	ctree = (GtkCTree *)browser->directory_ctree;
	if(ctree == NULL)
	    return(NULL);

	/* For now just get the first toplevel, later on we may need
	 * to return a node other than the first toplevel if multiple
	 * vfs are to be supported.
	 */
	return(EDVNodeGetToplevel(ctree));
}


/*
 *	Appends a node as a child of the node specified by parent.
 *
 *	If parent is NULL then the node will be a toplevel node.
 *
 *	The given disk object will be transfered as client data to the
 *	new node or deleted on failure. In either case the given object
 *	should not be referenced again.
 */
GtkCTreeNode *EDVBrowserDirTreeInsertNode(
        edv_browser_struct *browser,
	GtkCTreeNode *parent, GtkCTreeNode *sibling,
        edv_object_struct *object
)
{
	gint i, columns;
	edv_core_struct *core_ptr;
        GtkWidget *w;
	GtkCList *clist;
        GtkCTree *ctree;
	GtkCTreeNode *node;
	gchar **name;
	GdkPixmap *pixmap_closed = NULL, *pixmap_opened = NULL;
	GdkBitmap *mask_closed = NULL, *mask_opened = NULL;
	gbool is_leaf = FALSE;


        if((browser == NULL) || (object == NULL))
	{
	    EDVObjectDelete(object);
            return(NULL);
	}

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	{
            EDVObjectDelete(object);
            return(NULL);
        }

        w = browser->directory_ctree;
        if(w == NULL)
	{
            EDVObjectDelete(object);
            return(NULL);
	}
        else
	{
	    clist = GTK_CLIST(w);
            ctree = GTK_CTREE(w);
	}

	/* Get number of columns. */
	columns = MAX(clist->columns, 1);

	/* Allocate new name string array for new node, allocate each
	 * name to be the same through.
	 */
	name = (gchar **)g_malloc(columns * sizeof(gchar *));
	for(i = 0; i < columns; i++)
	    name[i] = g_strdup("");

	/* Insert new node. */
	node = gtk_ctree_insert_node(
	    ctree, parent, sibling, name,
            EDV_LIST_PIXMAP_TEXT_SPACING,
            pixmap_closed, mask_closed,
            pixmap_opened, mask_opened,
            is_leaf,                    /* Is leaf. */
            FALSE                       /* Expanded. */
        );

	/* Deallocate node name string array. */
	for(i = 0; i < columns; i++)
	    g_free(name[i]);
	g_free(name);
	name = NULL;

	/* Failed to insert new node? */
	if(node == NULL)
	{
	    EDVObjectDelete(object);
            return(NULL);
	}

	/* Set text, pixmaps, and style for the new node. */
	EDVBrowserDirTreeSetNode(
	    core_ptr, browser, ctree, node, object
	);

	/* Set node's row data. */
	gtk_ctree_node_set_row_data_full(
	    ctree, node,
	    object, (GtkDestroyNotify)EDVBrowserDirTreeItemDestroyCB
	);
	/* Object pointer has now been transfered to the new node's
	 * data. Mark it NULL so we do not reference it again.
	 */
	object = NULL;


	/* Return newly created node. */
	return(node);
}


/*
 *	Procedure to create toplevel directories.
 */
void EDVBrowserDirTreeDoCreateToplevels(
        edv_browser_struct *browser
)
{
	edv_object_struct *obj;
	const gchar *path;
	GtkWidget *w;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCTreeNode *node;
	struct stat lstat_buf;


	if(browser == NULL)
	    return;

	w = browser->directory_ctree;
	if(w == NULL)
	    return;

	clist = GTK_CLIST(w);
	ctree = GTK_CTREE(w);


	/* Mark as busy but do not mark as processing since callbacks
	 * need to see things as not processing when nodes are expanded
	 * in the EDVBrowserDirTreeDoRefresh() call.
	 */
	EDVBrowserSetBusy(browser, TRUE);

/*	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE); */


	/* Add toplevel. */
	path = "/";
	obj = EDVObjectNew();
	if(!lstat(path, &lstat_buf))
	{
	    EDVObjectSetPath(obj, path);
	    EDVObjectSetStat(obj, &lstat_buf);
	    obj->link_valid = TRUE;		/* Assume it. */
	}
	node = EDVBrowserDirTreeInsertNode(
	    browser, NULL, NULL, obj
	);
	obj = NULL;

	/* Get one level of nodes from this toplevel node. */
	EDVBrowserDirTreeDoGetChildrenList(
	    browser, node, TRUE, FALSE
	);





/*	gtk_clist_set_selection_mode(clist, GTK_SELECTION_BROWSE); */

        EDVBrowserSetBusy(browser, FALSE);
}


/*
 *	This function should be called by
 *	EDVBrowserDirTreeDoGetChildrenList().
 */
static void EDVBrowserDirTreeDoGetChildrenListIterate(
	edv_core_struct *core_ptr, edv_browser_struct *browser,
	GtkCTree *ctree, GtkCTreeNode *parent,
	gbool update_progress, gbool recurse,
	gbool hide_object_hidden, gbool hide_object_noaccess
)
{
	gchar *parent_path;
	edv_object_struct *object;
        gint i, strc;
	gchar **strv;
	const gchar *cstrptr, *cstrptr2;
	struct stat lstat_buf, stat_buf;


	if(parent == NULL)
	    return;

	/* Get disk object of the given parent node. */
	object = (edv_object_struct *)gtk_ctree_node_get_row_data(
	    ctree, parent
	);
	if(object == NULL)
	    return;

	/* Make a copy of the full path from the disk object structure. */
	parent_path = (object->full_path != NULL) ?
	    g_strdup(object->full_path) : NULL;
	if(parent_path == NULL)
	    return;

	/* We don't need the parent disk object structure past this point. */
	object = NULL;


	/* Update status bar indicating which directory we're scanning? */
	if(update_progress)
	{
	    gchar *buf;
	    const gchar *parent_name;


	    parent_name = strrchr(parent_path, DIR_DELIMINATOR);
	    if(parent_name == NULL)
		parent_name = parent_path;
	    else
		parent_name++;

	    buf = g_strdup_printf(
		"Scanning directory: %s",
		parent_name
	    );
	    EDVStatusBarMessage(
		browser->status_bar, buf, FALSE
	    );
	    g_free(buf);
	}


	/* Get listing of contents in the directory specified by
	 * parent_path.
	 */
	strv = GetDirEntNames2(parent_path, &strc);
	if(strv != NULL)
	{
	    gchar *child_path;


	    /* Sort strings. */
	    strv = StringQSort(strv, strc);
	    if(strv == NULL)
	    {
		g_free(parent_path);
		return;
	    }

	    /* Iterate through each string. */
	    child_path = NULL;
	    for(i = 0; i < strc; i++)
	    {
#define FREE_AND_CONTINUE	\
{ \
 g_free(child_path); \
 child_path = NULL; \
\
 g_free(strv[i]); \
 strv[i] = NULL; \
\
 continue; \
}
		cstrptr = strv[i];
		if(cstrptr == NULL)
		    FREE_AND_CONTINUE

		/* Skip special dir notations. */
		if(!strcmp(cstrptr, "..") || !strcmp(cstrptr, "."))
		    FREE_AND_CONTINUE

		cstrptr2 = PrefixPaths(parent_path, cstrptr);
		if(cstrptr2 == NULL)
		    FREE_AND_CONTINUE

		/* Copy full path to child object. */
		child_path = g_strdup(cstrptr2);
		if(child_path == NULL)
		    FREE_AND_CONTINUE

		/* Get local and destination stats of child object. */
		if(lstat(child_path, &lstat_buf))
		    FREE_AND_CONTINUE
		if(stat(child_path, &stat_buf))
		    FREE_AND_CONTINUE


		/* Destination child object is a directory? */
		if(S_ISDIR(stat_buf.st_mode))
		{
		    GtkCTreeNode *node;


		    /* Create new disk object structure and set it up. */
		    object = EDVObjectNew();
		    if(object != NULL)
		    {
			EDVObjectSetPath(object, child_path);
			EDVObjectSetStat(object, &lstat_buf);
			object->link_valid = TRUE;

			/* Begin filter checks. */
			if((hide_object_hidden ?
			    EDVCheckObjectHidden(core_ptr, object) : FALSE) ||
                           (hide_object_noaccess ?
                            !EDVCheckObjectAccessable(core_ptr, object) : FALSE)
			)
			{
			    /* Filter check failed, delete disk object
			     * structure instead of appending it.
			     */
			    EDVObjectDelete(object);
			    object = NULL;
			    node = NULL;
			}
			else
			{
			    /* Create new node using the disk object, note
			     * that the disk object obj will be taken by this
			     * call and we should not reference obj again.
			     */
			    node = EDVBrowserDirTreeInsertNode(
				browser, parent, NULL, object
			    );
			    object = NULL;
			}
		    }
		    else
		    {
			node = NULL;
		    }


		    /* Update progress? */
		    if(update_progress)
			EDVStatusBarProgress(browser->status_bar, -1.0, TRUE);

		    /* Recurse into sub directories? */
		    if(recurse && (node != NULL))
		    {
			EDVBrowserDirTreeDoGetChildrenListIterate(
			    core_ptr, browser, ctree, node, update_progress,
			    FALSE,	/* Do not recurse in this itteration. */
			    hide_object_hidden, hide_object_noaccess
			);
		    }
		}

		FREE_AND_CONTINUE
#undef FREE_AND_CONTINUE
	    }

	    /* At this point all strings in the strv array should be
	     * deallocated. Only the pointer array itself needs to be
	     * deallocated.
	     */
	    g_free(strv);
	    strv = NULL;
	}

	/* Deallocate coppied full path of the parent disk object. */
	g_free(parent_path);
	parent_path = NULL;
}

/*
 *	Gets listing of directories as child nodes of the given node,
 *	the given node may not be NULL.
 *
 *	This will recurse no more than two levels from the given node.
 */
static void EDVBrowserDirTreeDoGetChildrenList(
        edv_browser_struct *browser, GtkCTreeNode *parent,
	gbool update_progress, gbool recurse
)
{
	gbool hide_object_hidden, hide_object_noaccess;
        GtkWidget *w;
        GtkCTree *ctree;
	edv_core_struct *core_ptr;


        if((browser == NULL) || (parent == NULL))
            return;

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	    return;

        w = browser->directory_ctree;
        if(w == NULL)
            return;
        else
            ctree = GTK_CTREE(w);

	hide_object_hidden = !EDVCFGItemListGetValueI(
	    core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN
	);
	hide_object_noaccess = !EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS
        );


        /* Update progress? */
        if(update_progress)
	{
	    EDVStatusBarMessage(
		browser->status_bar,
		"Scanning directories...",
		FALSE
	    );
            EDVStatusBarProgress(browser->status_bar, -1.0, TRUE);
	}

	/* First delete all child nodes from the given node if any. */
	if(1)
	{
	    GtkCTreeNode *node, *node_next;

	    node = EDVNodeGetChild(parent);
	    while(node != NULL)
	    {
		node_next = EDVNodeGetSibling(node);
		gtk_ctree_remove_node(ctree, node);
		node = node_next;
	    }
	}

	/* Get child directories as child nodes of the given parent node,
	 * recursing into each child directories if recurse is TRUE.
	 */
	EDVBrowserDirTreeDoGetChildrenListIterate(
	    core_ptr, browser, ctree, parent,
	    update_progress, recurse,
	    hide_object_hidden, hide_object_noaccess
	);

        /* Reset progress. */
	if(update_progress)
	{
	    EDVStatusBarMessage(browser->status_bar, NULL, FALSE);
	    EDVStatusBarProgress(browser->status_bar, 0.0, FALSE);
	}
}


/*
 *	Procedure to refresh by regetting the list of child nodes from the
 *	given parent node, recursing no more than 2 levels.
 *
 *	Then all nodes (including the given node if it is not NULL) will
 *	have their referenced disk object checked to see if it exists. If
 *	it does not then the node will be removed.
 *
 *	The given node may be removed in this call so it should not be
 *	referenced again by the calling function.
 */
void EDVBrowserDirTreeDoRefresh(
        edv_browser_struct *browser, GtkCTreeNode *node,
	gbool update_progress
)
{
	GtkCList *clist;


	if(browser == NULL)
	    return;

        clist = (GtkCList *)browser->directory_ctree;
        if(clist == NULL)
            return;


	/* If there is no given node then get toplevel node. */
	if(node == NULL)
	    node = EDVBrowserDirTreeGetVFSToplevel(browser);


	/* Remove all child nodes of the given node and then reget all
	 * child nodes, recursing to at most 2 levels.
	 */
	EDVBrowserDirTreeDoGetChildrenList(
	    browser, node, update_progress, TRUE
	);

        /* Check all nodes (including the given node) to see if the
         * referenced disk object exists. Any node that references
         * a disk object that does not exist will be removed, so the
         * given node should be considered invalid after this point.
	 *
	 * All child nodes will be updated on the directory ctree.
         */
        EDVBrowserDirTreeDoSync(browser, NULL);
	node = NULL;
}

/*
 *	Procedure to create child nodes parented to the given node, each
 *	child node will be a child directory or a link that points to a
 *	directory in the directory specified by the given node.
 */
void EDVBrowserDirTreeDoExpand(
	edv_browser_struct *browser, GtkCTreeNode *node,
	gbool update_progress
)
{
	GtkCList *clist;


	if((browser == NULL) || (node == NULL))
	    return;

        clist = (GtkCList *)browser->directory_ctree;
        if(clist == NULL)
            return;

	/* Get child directories as child nodes of the given expanded
	 * node, recursing at most 2 levels.
	 */
	EDVBrowserDirTreeDoGetChildrenList(
	    browser, node, update_progress, TRUE
	);
}


/*
 *	Called by EDVBrowserDirTreeDoSync() or itself to scan all child
 *	nodes of the given node (which may not be NULL) and remove any
 *	nodes that reffer to a non-existant disk object.
 */
static void EDVBrowserDirTreeDoSyncIterate(
	edv_core_struct *core_ptr, edv_browser_struct *browser,
	GtkCTree *ctree, GtkCTreeNode *node,
	gbool parent_is_expanded
)
{
	GtkCTreeRow *row_ptr;
	GtkCTreeNode *child_node, *next_node;
	struct stat lstat_buf, stat_buf;


	/* Iterate through given node and its siblings. */
	while(node != NULL)
	{
	    /* Get ctree row pointer from this node. */
	    row_ptr = GTK_CTREE_ROW(node);

	    /* Check if this node has its own child node, if it does then
	     * we need to recurse into it first.
	     */
	    child_node = EDVNodeGetChild(node);
	    if(child_node != NULL)
	    {
		/* Child node exists for this node, so recurse into child
		 * node and sync all its subsequent nodes.
		 */
		EDVBrowserDirTreeDoSyncIterate(
		    core_ptr, browser, ctree, child_node,
		    row_ptr->expanded
		);
	    }
	    else
	    {
		/* This node had no child nodes before, but we need to
		 * check if it has child nodes on disk now.
		 *
		 * Do this only if this node's parent is expanded to prevent
		 * recursions caused by multiple calls to sync.
		 *
		 * Recurse one level.
		 */
		if(parent_is_expanded)
		    EDVBrowserDirTreeDoGetChildrenList(
			browser, node, FALSE, FALSE
		    );
	    }


            /* Get next sibling node (which may be NULL). */
            next_node = EDVNodeGetSibling(node);


	    /* Get local stats for the disk object referenced by this
	     * node. Check if the disk object cannot be lstat()'ed,
	     * implying it no longer exists.
	     */
	    if(EDVNodeDiskObjectLStat(ctree, node, &lstat_buf))
	    {
		/* Object no longer exists, remove the node and disk
		 * object structure.
		 */
		gtk_ctree_remove_node(ctree, node);
		node = NULL;
		row_ptr = NULL;
	    }
	    else
	    {
		/* Object exists, update local statistics and node value. */
		edv_object_struct *obj = (edv_object_struct *)
		    gtk_ctree_node_get_row_data(ctree, node);

		/* Check if object's destination can be stat'ed. */
		if(EDVNodeDiskObjectStat(ctree, node, &stat_buf))
		{
		    /* Cannot stat destination, node needs to be removed. */
		    gtk_ctree_remove_node(ctree, node);
		    node = NULL;
		    row_ptr = NULL;
		    obj = NULL;
		}
		/* Destination does not goes to a directory? */
		else if(!S_ISDIR(stat_buf.st_mode))
		{
                    /* Destination not a directory any more, node needs to
		     * be removed.
		     */
                    gtk_ctree_remove_node(ctree, node);
                    node = NULL;
		    row_ptr = NULL;
		    obj = NULL;
                }
		else if(obj != NULL)
		{
		    /* All checks passed, update this node and its disk
		     * object structure.
		     */
		    EDVObjectSetStat(obj, &lstat_buf);
		    obj->link_valid = TRUE;
		    EDVBrowserDirTreeSetNode(
			core_ptr, browser, ctree, node, obj
		    );
		}
	    }

	    /* The current node should now be considered invalid. */

	    /* Set current node to the next sibling. */
	    node = next_node;
	}

}

/*
 *	Procedure to check, update icon and text, and remove any 
 *	unaccessable nodes starting on the given node of the directory
 *	ctree.
 *
 *	If the given node is NULL then the starting point will be 
 *	toplevel. All nodes after this call should be rechecked since
 *	any number of nodes may have been removed.
 */
void EDVBrowserDirTreeDoSync(
        edv_browser_struct *browser, GtkCTreeNode *node
)
{
	GtkWidget *w;
	GtkCTree *ctree;
	GtkCTreeRow *row_ptr;


	if(browser == NULL)
	    return;

        w = browser->directory_ctree;
        if(w == NULL)
            return;
        else
            ctree = GTK_CTREE(w);

	/* If given node is NULL, then get toplevel node. */
	if(node == NULL)
	    node = EDVBrowserDirTreeGetVFSToplevel(browser);
	if(node == NULL)
	    return;

	row_ptr = GTK_CTREE_ROW(node);

	/* Iterate through all child nodes and check if they exist,
	 * removing them if they do not and recurse into additional childs
	 * as needed. The given node may no longer be valid after this
	 * point.
	 *
	 * The given node, its siblings, and child nodes will be updated
	 * be updated.
	 */
	EDVBrowserDirTreeDoSyncIterate(
	    (edv_core_struct *)browser->core_ptr, browser,
	    ctree, node, row_ptr->expanded
	);
	node = NULL;
}


/*
 *	Called by EDVBrowserDirTreeDoSelectPath() to seek through all
 *	child directories for a prefix match to the given full path and
 *	recurse into child directories.
 *
 *	The given path is the full path that is intended to be selected
 *	if found.
 */
static void EDVBrowserDirTreeDoSelectPathIterate(
        GtkCTree *ctree, GtkCTreeNode *node, const gchar *path,
	GtkCTreeNode **matched_node
)
{
	const gchar *cur_path;


	/* Got match? If so then we should not continue further. */
	if((matched_node != NULL) ? (*matched_node != NULL) : TRUE)
	    return;

	if(node == NULL)
	    return;

	/* Get path of given node. */
	cur_path = EDVNodeDiskObjectFullPath(ctree, node);
	if(cur_path == NULL)
	    return;

	/* Is cur_path a parent of the given path? */
	if(EDVIsParentPath(cur_path, path))
	{
	    GtkCTreeRow *ctree_row_ptr;


	    /* Yes it is, now check if it is a complete match. */
	    if(!strcmp(path, cur_path))
	    {
		/* It is a complete match, note that we got a complete
		 * match and select this node.
		 */
		if(matched_node != NULL)
		    *matched_node = node;

		return;
	    }
	    else
	    {
		/* Not a complete match, but we know we now much recurse
		 * into this node. Get child node and iterate through all
		 * children.
		 */

		/* Get ctree row of current node and expand it as needed.
		 *
		 * Note that this may call the expand callback and that
		 * wil load more child nodes.
		 */
		ctree_row_ptr = GTK_CTREE_ROW(node);
		if(!ctree_row_ptr->expanded)
		    gtk_ctree_expand(ctree, node);

		/* Get child node and iterate through children. */
		node = EDVNodeGetChild(node);
		while(node != NULL)
		{
		    /* Check this node, recurse if it is a match. */
		    EDVBrowserDirTreeDoSelectPathIterate(
			ctree, node, path, matched_node
		    );
		    /* Got match? If so then we should not continue further. */
		    if((matched_node != NULL) ? (*matched_node != NULL) : TRUE)
			return;

		    /* Get sibling node. */
		    node = EDVNodeGetSibling(node);
		}
	    }
	}
}

/*
 *	Procedure to select (if possible or as much of) the given
 *	full path to a directory.
 *
 *	Nodes will be expanded and updated as needed.
 */
void EDVBrowserDirTreeDoSelectPath(
        edv_browser_struct *browser, const gchar *path
)
{
        GtkWidget *w;
	GtkCList *clist;
        GtkCTree *ctree;
	gchar *full_path;
	GtkCTreeNode *node, **matched_node;


	if((browser == NULL) || (path == NULL))
	    return;

        w = browser->directory_ctree;
        if(w == NULL)
            return;

	ctree = GTK_CTREE(w);
	clist = GTK_CLIST(w);


	/* Path must be absolute. */
	if(!ISPATHABSOLUTE(path))
	    return;

	/* Make a copy of the given path as the full path. */
	full_path = g_strdup(path);
	if(full_path == NULL)
	    return;

	/* Strip tailing deliminators. */
	StripPath(full_path);


	/* Allocate marker variable to pass around and denote if we got
	 * a full match.
	 */
	matched_node = (GtkCTreeNode **)g_malloc(sizeof(GtkCTreeNode *));
	*matched_node = NULL;



	/* Get VFS toplevel node to start search from. */
	node = EDVBrowserDirTreeGetVFSToplevel(browser);

	/* Begin searching for path, expanding (thus updating nodes) and
	 * selecting the final node if found.
	 */
	EDVBrowserDirTreeDoSelectPathIterate(
	    ctree, node, full_path, matched_node
	);



	/* Got match? */
	if(*matched_node != NULL)
	    gtk_ctree_select(ctree, *matched_node);



	/* Deallocate matched node marker. */
	g_free(matched_node);
	matched_node = NULL;

	/* Deallocate full path. */
	g_free(full_path);
	full_path = NULL;
}


/*
 *	Called by EDVBrowserDirTreeFindNodeByPath() or itself to check
 *	child nodes of the given node, recursing if a prefix match is
 *	found and returning with *got_match = TRUE if a perfect match is
 *	made.
 */
static void EDVBrowserDirTreeFindNodeByPathIterate(
        GtkCTree *ctree, GtkCTreeNode *node,
        const gchar *path, GtkCTreeNode **matched_node
)
{
	const gchar *cur_path;


        /* Check if the pointed to location of matched_node is not NULL,
	 * if it is not NULL then that implies a full match is already
	 * made.
	 */
        if((matched_node != NULL) ? (*matched_node != NULL) : TRUE)
            return;

        if(node == NULL)
            return;

        /* Get path of given node. */
        cur_path = EDVNodeDiskObjectFullPath(ctree, node);
        if(cur_path == NULL)
            return;

        /* Is cur_path is a prefix of the given path? */
        if(EDVIsParentPath(cur_path, path))
        {
            /* Yes it is, now check if it is a complete match. */
            if(!strcmp(path, cur_path))
            {
		if(matched_node != NULL)
		    *matched_node = node;
		return;
	    }
            else
            {
                /* Not a complete match, just a prefix match. So this
		 * implies we need to recurse into this child's child
		 * nodes to continue the search.
                 */

                /* Get child node and iterate through children. */
                node = EDVNodeGetChild(node);
                while(node != NULL)
                {
                    /* Check this node, recurse if it is a match. */
                    EDVBrowserDirTreeFindNodeByPathIterate(
                        ctree, node, path, matched_node
                    );
                    /* Got match? If so then we should not continue further. */
		    if((matched_node != NULL) ?
			(*matched_node != NULL) : TRUE
		    )
		        return;

                    /* Get sibling node. */
                    node = EDVNodeGetSibling(node);
                }
	    }
	}
}

/*
 *	Searches for a node on the browser's directory ctree that matches
 *	the full path specified by path.
 *
 *	Will not expand or modify any nodes.
 *
 *	Can return NULL on failed match.
 */
GtkCTreeNode *EDVBrowserDirTreeFindNodeByPath(
        edv_browser_struct *browser, const gchar *path
)
{
	gchar *full_path;
        GtkWidget *w;
        GtkCTree *ctree;
	GtkCTreeNode *node;
	GtkCTreeNode **matched_node, *lmatched_node;


	if((browser == NULL) || (path == NULL))
	    return(NULL);

        w = browser->directory_ctree;
        if(w == NULL)
            return(NULL);
        else
            ctree = GTK_CTREE(w);

        /* Path must be absolute. */
        if(!ISPATHABSOLUTE(path))
            return(NULL);

        /* Make a copy of the given path as the full path. */
        full_path = g_strdup(path);
        if(full_path == NULL)
            return(NULL);

	/* Strip tailing deliminators. */
	StripPath(full_path);


        /* Allocate marker variable to pass around and denote if we got
         * a full match by setting its pointed to location to point to
	 * the matched node.
         */
	matched_node = (GtkCTreeNode **)g_malloc(sizeof(GtkCTreeNode *));
        *matched_node = NULL;


        /* Get VFS toplevel node to start search from. */
        node = EDVBrowserDirTreeGetVFSToplevel(browser);

        /* Begin searching for a node who's disk object's fuil path
	 * matches the given full_path. The pointed to location of 
	 * matched_node will point to the node if a match was made.
	 */
	EDVBrowserDirTreeFindNodeByPathIterate(
	    ctree, node, full_path, matched_node
	);

	/* Get matched node (or NULL if none). */
	lmatched_node = *matched_node;

        /* Deallocate matched node pointer. */
        g_free(matched_node);
        matched_node = NULL;

        /* Deallocate full path. */
        g_free(full_path);
        full_path = NULL;

	return(lmatched_node);
}


/*
 *	FPrompt apply callback, set in EDVBrowserDirTreeDoFPromptRename().
 */
static void EDVBrowserDirTreeFPromptRenameApplyCB(
        gpointer data, const gchar *value
)
{
	gpointer *cb_data = (gpointer *)data;
        edv_browser_struct *browser;
	GtkCTree *ctree;
	GtkCTreeNode *node;
        if(cb_data == NULL)
            return;

	/* Get callback data. */
	browser = (edv_browser_struct *)cb_data[0];
	ctree = (GtkCTree *)cb_data[1];
	node = (GtkCTreeNode *)cb_data[2];

	/* Inputs valid? */
	if((browser != NULL) && (ctree != NULL) && (node != NULL) &&
	   (value != NULL)
	)
	{
            edv_core_struct *core_ptr = (edv_core_struct *)browser->core_ptr;
	    edv_object_struct *object = (edv_object_struct *)gtk_ctree_node_get_row_data(
		ctree, node
	    );


            /* Check if the selected object's disk object structure is
             * valid.
             */
            if((object != NULL) ? (object->full_path != NULL) : FALSE)
            {
                gchar *old_full_path = g_strdup(object->full_path);
                gchar *new_obj = NULL;
                const gchar *error_mesg;
                gbool yes_to_all = FALSE;
                gint status;


                /* Perform rename. */
                status = EDVFOPRename(
                    core_ptr, old_full_path, value,
                    &new_obj, browser->toplevel,
                    FALSE, TRUE,
                    &yes_to_all
                );

                /* Unmap progress dialog, it may have been mapped in the
                 * above operation.
                 */
                ProgressDialogBreakQuery(FALSE);
                ProgressDialogSetTransientFor(NULL);

                /* Get error message if any that might have occured in the
                 * above operation.
                 */
                error_mesg = EDVFOPGetError();
                if(error_mesg != NULL)
                {
                    CDialogSetTransientFor(browser->toplevel);
                    CDialogGetResponse(
                        "Operation Error",
                        error_mesg,
                        NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                }

                /* Got new object full path name (implying success)? */
                if((new_obj != NULL) && (old_full_path != NULL))
                {
                    struct stat lstat_buf;
                    const gchar *new_child, *old_child;


                    /* Get child object names. */
                    new_child = strrchr(new_obj, DIR_DELIMINATOR);
                    if(new_child == NULL)
                        new_child = new_obj;
                    else
                        new_child++;

                    old_child = strrchr(old_full_path, DIR_DELIMINATOR);
                    if(old_child == NULL)
                        old_child = old_full_path;
                    else
                        old_child++;

                    /* Get new local statistics for the renamed object. */
                    if(!lstat(new_obj, &lstat_buf))
		    {
			gchar *buf = g_strdup_printf(
			    "Object \"%s\" renamed to \"%s\"",
			    old_child, new_child
			);
			EDVStatusBarMessage(
			    browser->status_bar, buf, FALSE
			);
			g_free(buf);

			/* Emit object modified signal to all windows. */
                        EDVObjectModifiedEmit(
                            core_ptr, old_full_path,
                            new_obj, &lstat_buf
                        );
		    }
                }
		else
		{
		    /* Did not get new object path new_obj, implying failed. */
                    EDVStatusBarMessage(
                        browser->status_bar, "Rename object failed", FALSE
                    );
		}

                /* The disk object structure may now be invalid if the
                 * object modified signal was emitted.
                 */
                object = NULL;

                /* Deallocate coppies of paths. */
                g_free(new_obj);
                g_free(old_full_path);
	    }
	}

	/* Deallocate callback data, it is no longer needed. */
	g_free(cb_data);
	cb_data = NULL;
}

/*
 *	Procedure to map the floating prompt for renaming of the
 *	node specified by node.
 */
void EDVBrowserDirTreeDoFPromptRename(
        edv_browser_struct *browser, GtkCTreeNode *node
)
{
	GtkWidget *w;
        GtkCList *clist;
	GtkCTree *ctree;
	edv_object_struct *object;
	gint row, column;
        gint cx, cy, px, py, pwidth, pheight;


        if((browser == NULL) || (node == NULL))
            return;

	if(FPromptIsQuery())
	    return;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect((edv_core_struct *)browser->core_ptr, TRUE))
            return;

	/* Get directory ctree widget. */
	w = browser->directory_ctree;
	if(w == NULL)
	    return;
        clist = GTK_CLIST(w);
	ctree = GTK_CTREE(w);


        /* Sync data on browser so as to ensure we have the most up to
         * date information to send out.
         */
        EDVBrowserSyncData(browser);


	/* Get row and column of the given node. */
	if(!EDVNodeGetIndex(ctree, node, &row, &column))
	    return;

        /* Get clist cell geometry. */
	if(GUICListGetCellGeometry(
            clist, column, row,
            &cx, &cy, &pwidth, &pheight
        ))
	    return;

	/* Get root window relative coordinates. */
	px = 0;
	py = 0;
	gdk_window_get_deskrelative_origin(
	    clist->clist_window, &px, &py
	);
/*	px += cx + 0; */
	py += cy - 2;	/* Move up a little. */

	/* Modify intended prompt width to just match width of ctree. */
	pwidth = w->allocation.width;
	

        /* Get disk object structure from the directory ctree node. */
	object = (edv_object_struct *)gtk_ctree_node_get_row_data(
	    ctree, node
	);
	if(object == NULL)
	    return;

        /* Check if object name is a special notation that should not be
         * allowed to be renamed.
         */
        if(object->name != NULL)
        {
            const gchar *name = object->name;
            if(!strcmp(name, ".") || !strcmp(name, "..") ||
               !strcmp(name, "/")
            )
                return;
        }

        if(1)
        {
	    gpointer *cb_data = (gpointer *)g_malloc0(
		3 * sizeof(gpointer)
	    );
	    gchar *value = (object->name != NULL) ?
		g_strdup(object->name) : NULL;

	    /* Set up callback data. */
	    if(cb_data != NULL)
	    {
		cb_data[0] = browser;
		cb_data[1] = ctree;
		cb_data[2] = node;
	    }

            /* Map floating prompt to change values. */
            FPromptSetTransientFor(browser->toplevel);
            FPromptSetPosition(px, py);
            FPromptMapQuery(
                NULL,                   /* No label. */
                value,
                NULL,                   /* No tooltip message. */
                FPROMPT_MAP_NO_MOVE,    /* Map code. */
                pwidth, -1,		/* Width and height. */
/*		FPROMPT_FLAG_OK_BTN | FPROMPT_FLAG_CANCEL_BTN, */
		0,	/* No buttons. */
                (gpointer)cb_data,	/* Callback data. */
                NULL,			/* No browse callback. */
		EDVBrowserDirTreeFPromptRenameApplyCB,
                EDVBrowserDirTreeFPromptCancelCB
            );

            /* Do not reference cb_data after this call, it will be passed
             * to the callbacks where it will be deallocated.
             */
            cb_data = NULL;

            /* Deallocate original value. */
	    g_free(value);
	    value = NULL;
        }
}



/*
 *	This should be called whenever a new object has been added, it
 *	will add a new tree node as needed to represent the new object.
 *
 *	The given path must be an absolute path to the object.
 */
void EDVBrowserDirTreeObjectAddedNotify(
        edv_browser_struct *browser, const gchar *path,
	const struct stat *lstat_buf
)
{
	const gchar *cstrptr;
	GtkWidget *w;
	GtkCTree *ctree;
        GtkCTreeNode *node, *parent_node;
        edv_object_struct *object;
	gchar *parent_path;
	struct stat stat_buf;


        if((browser == NULL) || (path == NULL) || (lstat_buf == NULL))
	    return;

        w = browser->directory_ctree;
        if(w == NULL)
            return;
        else
            ctree = GTK_CTREE(w);

	/* Check if ultimate destination is a directory. */
	if(stat(path, &stat_buf))
	    return;
	if(!S_ISDIR(stat_buf.st_mode))
	    return;


	/* Get parent of the given path, we need to use the parent
	 * because that is the only way we know where to create the
	 * new node in.
	 */
	cstrptr = GetParentDir(path);
	/* Make a copy of the parent path of the new disk object. */
	parent_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	if(parent_path == NULL)
	    return;

        /* Look for the parent node that matches the parent of the new
	 * disk object.
	 */
	parent_node = EDVBrowserDirTreeFindNodeByPath(browser, parent_path);
        if(parent_node == NULL)
	{
	    /* Unable to find parent node, this implies we have no way
	     * to create the node for the new object.
	     */
	    g_free(parent_path);
	    return;
	}


	/* Check if node already exists for the given path. */
	node = EDVBrowserDirTreeFindNodeByPath(browser, path);
	if(node != NULL)
	{
	    /* Node already exists for the given path, so just update
	     * the node by regetting child nodes and recursing at most
	     * 2 levels.
	     */
            gtk_clist_freeze(GTK_CLIST(ctree));
	    EDVBrowserDirTreeDoGetChildrenList(
		browser, node, TRUE, TRUE
	    );
            gtk_clist_thaw(GTK_CLIST(ctree));
	}
	else
	{
	    /* Begin adding new node for the new disk object. */

	    /* Create a new disk object structure. */
	    object = EDVObjectNew();
	    if(object != NULL)
	    {
		EDVObjectSetPath(object, path);
		EDVObjectSetStat(object, lstat_buf);
		object->link_valid = TRUE;

		gtk_clist_freeze(GTK_CLIST(ctree));

		/* Add node to the parent_node, this will transfer the
		 * object to the new node. The given disk object structure
		 * should not be referenced again.
		 */
		node = EDVBrowserDirTreeInsertNode(
		    browser, parent_node, NULL, object
		);
		object = NULL;

		/* Created new node successfully? */
		if(node != NULL)
		{
		    /* Get new listing of child disk objects for the new
		     * node, recurse no more than 2 levels.
		     */
		    EDVBrowserDirTreeDoGetChildrenList(
			browser, node, TRUE, TRUE
		    );
		}

		gtk_clist_thaw(GTK_CLIST(ctree));
	    }
	}


	/* Deallocate coppied parent path. */
	g_free(parent_path);
	parent_path = NULL;
}

/*
 *	This should be called whenever a object has been modified, it will
 *	search for the object and then reupdate the matching node.
 *
 *      The given path must be an absolute path to the object and must be
 *      the path of the object's original name. The new_path must be an
 *	absolute path to the object at its new name, the new_path may be
 *	NULL if there was no name change.
 */
void EDVBrowserDirTreeObjectModifiedNotify(
        edv_browser_struct *browser, const gchar *path,
	const gchar *new_path,
	const struct stat *lstat_buf
)
{
        GtkWidget *w;
        GtkCTree *ctree;
	GtkCTreeNode *node;
	edv_object_struct *object;
	gint stat_result;
	struct stat stat_buf;


        if((browser == NULL) || (path == NULL) || (lstat_buf == NULL))
            return;

	if(new_path == NULL)
	    new_path = path;

        w = browser->directory_ctree;
        if(w == NULL)
            return;
        else
            ctree = GTK_CTREE(w);


        /* Get destination stats of the modified object. */
	stat_result = stat(new_path, &stat_buf);

	/* Destination invalid or does not go to a directory? */
	if((!stat_result) ? !S_ISDIR(stat_buf.st_mode) : TRUE)
	{
	    /* It does not go to a directory, so we need to check if there
	     * is an existing node that matches the given old path. If it
	     * exists then it should no longer be listed and must be 
	     * removed.
	     */
            node = EDVBrowserDirTreeFindNodeByPath(browser, path);
	    if(node != NULL)
	    {
		/* Found a node that reffers to the old path of the object
		 * who's destination is no longer a directory, remove it.
		 */
		gtk_ctree_remove_node(ctree, node);
	    }
            return;
	}

	/* Look for a node that matches the old path of the object. */
	node = EDVBrowserDirTreeFindNodeByPath(browser, path);
	if(node == NULL)
	{
	    /* No existing node matches the old path, so check if an
	     * existing node matches the parent of the new path.
	     */
	    GtkCTreeNode *parent_node;
	    gchar *new_parent_path = GetParentDir(new_path);
	    if(new_parent_path != NULL)
		new_parent_path = g_strdup(new_parent_path);

	    /* Find node that matches the full path of the parent of the
	     * new path.
	     */
	    parent_node = EDVBrowserDirTreeFindNodeByPath(
		browser, new_parent_path
	    );
	    if(parent_node != NULL)
	    {
                /* Modified node does not exist but its parent does, which
		 * implies we need to create a new child node for the
		 * matched parent_node.
		 */
                object = EDVObjectNew();
                if(object != NULL)
                {
		    EDVObjectSetPath(object, new_path);
		    EDVObjectSetStat(object, lstat_buf);
		    object->link_valid = TRUE;

		    gtk_clist_freeze(GTK_CLIST(ctree));

		    /* Add node to the parent_node, this will transfer the
		     * object to the new node. The given disk object structure
		     * should not be referenced again.
		     */
		    node = EDVBrowserDirTreeInsertNode(
			browser, parent_node, NULL, object
		    );
		    object = NULL;

		    /* Created new node successfully? */
		    if(node != NULL)
		    {
			/* Get new listing of child disk objects for the new
			 * node, recurse no more than 2 levels.
			 */
			EDVBrowserDirTreeDoGetChildrenList(
			    browser, node, TRUE, TRUE
			);
		    }

		    gtk_clist_thaw(GTK_CLIST(ctree));
		}
	    }

	    /* Deallocate copy of new path's parent. */
	    g_free(new_parent_path);
	}
	else
	{
	    /* Got existing node that matches the old path, need to
	     * update its values.
	     */

	    /* Get disk object structure from matched node. */
	    object = (edv_object_struct *)gtk_ctree_node_get_row_data(
		ctree, node
	    );
	    if(object != NULL)
	    {
		/* Update disk object structure. */
		EDVObjectSetPath(object, new_path);
		EDVObjectSetStat(object, lstat_buf);
		object->link_valid = TRUE;

		gtk_clist_freeze(GTK_CLIST(ctree));

		/* Update node's text, pixmap, and style with respect to
		 * the newly modified disk object.
		 */
		EDVBrowserDirTreeSetNode(
		    (edv_core_struct *)browser->core_ptr, browser,
		    ctree, node, object
		);

		/* Get new listing of child disk objects for the updated
		 * node, recurse no more than 2 levels.
		 */
		EDVBrowserDirTreeDoGetChildrenList(
		    browser, node, TRUE, TRUE
		);

		gtk_clist_thaw(GTK_CLIST(ctree));
	    }
	}
}

/*
 *	This should be called whenever a object has been removed, it will
 *	search for the object and then remove the matching node.
 *
 *	The given path must be an absolute path to the object.
 */
void EDVBrowserDirTreeObjectRemovedNotify(
        edv_browser_struct *browser, const gchar *path
)
{
        GtkWidget *w;
        GtkCTree *ctree;
        GtkCTreeNode *node;


        if((browser == NULL) || (path == NULL))
            return;

        w = browser->directory_ctree;
        if(w == NULL)
            return;
        else
            ctree = GTK_CTREE(w);


        /* Look for a node that matches the given object. */
        node = EDVBrowserDirTreeFindNodeByPath(browser, path);
        if(node == NULL)
            return;

	/* Remove matched node and all of its child nodes. */
	gtk_ctree_remove_node(ctree, node);
}


/*
 *      This should be called whenever a object has been mounted or
 *	unmounted.
 */
void EDVBrowserDirTreeMountNotify(
        edv_browser_struct *browser, edv_device_struct *dev_ptr,
        gbool is_mounted
)
{
	gchar *mount_path;
	GtkWidget *w;
	GtkCTree *ctree;
	GtkCTreeNode *node;
	edv_object_struct *object;
	struct stat lstat_buf;


	if((browser == NULL) || (dev_ptr == NULL))
	    return;

        w = browser->directory_ctree;
        if(w == NULL)
            return;
        else
            ctree = GTK_CTREE(w);

	/* Get copy of mount path from device. */
	mount_path = (dev_ptr->mount_path != NULL) ?
	    g_strdup(dev_ptr->mount_path) : NULL;
	if(mount_path == NULL)
	    return;

	/* Simplify mount path. */
	EDVSimplifyPath(mount_path);


        /* Look for a node that matches the mount path. */
        node = EDVBrowserDirTreeFindNodeByPath(browser, mount_path);
        if(node != NULL)
        {
            /* Got existing node that matches the mount path, need to
             * update its values.
             */

            /* Get disk object structure from matched node. */
            object = (edv_object_struct *)gtk_ctree_node_get_row_data(
                ctree, node
            );
            if((object != NULL) && !lstat(mount_path, &lstat_buf))
            {
                /* Update disk object structure. */
                EDVObjectSetPath(object, mount_path);
                EDVObjectSetStat(object, &lstat_buf);
                object->link_valid = TRUE;

                gtk_clist_freeze(GTK_CLIST(ctree));

                /* Update node's text, pixmap, and style with respect to
                 * the newly modified disk object.
                 */
                EDVBrowserDirTreeSetNode(
                    (edv_core_struct *)browser->core_ptr, browser,
                    ctree, node, object
                );

                /* Get new listing of child disk objects for the updated
                 * node, recurse no more than 2 levels.
                 */
                EDVBrowserDirTreeDoGetChildrenList(
                    browser, node, TRUE, TRUE
                );

                gtk_clist_thaw(GTK_CLIST(ctree));
            }
        }

	/* Deallocate copy of mount path. */
	g_free(mount_path);
	mount_path = NULL;
}
