#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "edvtypes.h"
#include "edvcfg.h"


edv_cfg_item_struct *EDVCFGItemNew(
        gint type, const gchar *parameter
);
gpointer EDVCFGItemGetValue(
	const edv_cfg_item_struct *ci, gint *type_rtn
);
void EDVCFGItemSetValue(edv_cfg_item_struct *ci, const gpointer value);
void EDVCFGItemResetValue(edv_cfg_item_struct *ci);
void EDVCFGItemReset(edv_cfg_item_struct *ci);
void EDVCFGItemDelete(edv_cfg_item_struct *ci);

/* List utilities. */
edv_cfg_item_struct *EDVCFGItemListCopyList(
	const edv_cfg_item_struct *list
);
void EDVCFGItemListDeleteList(edv_cfg_item_struct *list);
gint EDVCFGItemListMatchParameter(
	const edv_cfg_item_struct *list, const gchar *parameter
);

/* Value fetching. */
gpointer EDVCFGItemListMatchGetValue(
	const edv_cfg_item_struct *list, const gchar *parameter,
	gint *type_rtn
);
gint EDVCFGItemListGetValueI(
	const edv_cfg_item_struct *list, const gchar *parameter
);
glong EDVCFGItemListGetValueL(
	const edv_cfg_item_struct *list, const gchar *parameter
);
gulong EDVCFGItemListGetValueUL(
        const edv_cfg_item_struct *list, const gchar *parameter
);
gfloat EDVCFGItemListGetValueF(
	const edv_cfg_item_struct *list, const gchar *parameter
);
gdouble EDVCFGItemListGetValueD(
	const edv_cfg_item_struct *list, const gchar *parameter
);
gchar *EDVCFGItemListGetValueS(
	const edv_cfg_item_struct *list, const gchar *parameter
);
edv_color_struct *EDVCFGItemListGetValueColor(
        const edv_cfg_item_struct *list, const gchar *parameter
);
edv_intlist_struct *EDVCFGItemListGetValueIntList(
        const edv_cfg_item_struct *list, const gchar *parameter
);

/* Value setting. */
edv_cfg_item_struct *EDVCFGItemListSetValue(
        edv_cfg_item_struct *list, const gchar *parameter,
        const gpointer value, gbool create_as_needed
);
edv_cfg_item_struct *EDVCFGItemListSetValueI(
        edv_cfg_item_struct *list, const gchar *parameter,
        gint value, gbool create_as_needed
);
edv_cfg_item_struct *EDVCFGItemListSetValueL(
        edv_cfg_item_struct *list, const gchar *parameter,
        glong value, gbool create_as_needed
);
edv_cfg_item_struct *EDVCFGItemListSetValueUL(
        edv_cfg_item_struct *list, const gchar *parameter,
        gulong value, gbool create_as_needed
);
edv_cfg_item_struct *EDVCFGItemListSetValueF(
        edv_cfg_item_struct *list, const gchar *parameter,
        gfloat value, gbool create_as_needed
);
edv_cfg_item_struct *EDVCFGItemListSetValueS(
        edv_cfg_item_struct *list, const gchar *parameter,
        const gchar *value, gbool create_as_needed
);
edv_cfg_item_struct *EDVCFGItemListSetValueColor(
        edv_cfg_item_struct *list, const gchar *parameter,
        const edv_color_struct *value, gbool create_as_needed
);
edv_cfg_item_struct *EDVCFGItemListSetValueIntList(
        edv_cfg_item_struct *list, const gchar *parameter,
        const edv_intlist_struct *value, gbool create_as_needed
);


#define g_memcpy	memcpy


/*
 *	Allocates a new configuration item structure.
 *
 *	Parameter will be coppied to the parameter member of the
 *	structure however member value will be left NULL.
 */
edv_cfg_item_struct *EDVCFGItemNew(
	gint type, const gchar *parameter
)
{
	edv_cfg_item_struct *ci = (edv_cfg_item_struct *)g_malloc0(
	    sizeof(edv_cfg_item_struct)
	);
	if(ci == NULL)
	    return(ci);

	ci->type = type;
	ci->parameter = (parameter != NULL) ? g_strdup(parameter) : NULL;
	ci->value = NULL;

	return(ci);
}

/*
 *	Returns the given configuration item ci's value and type.
 */
gpointer EDVCFGItemGetValue(
	const edv_cfg_item_struct *ci, gint *type_rtn
)
{
	if(type_rtn != NULL)
	    (*type_rtn) = -1;

	if(ci == NULL)
	    return(NULL);

	if(type_rtn != NULL)
	    (*type_rtn) = ci->type;

	return(ci->value);
}

/*
 *	Sets the given value to the given configuration item ci
 *	depending on ci's type.
 *
 *	The type of data that value points to must match the same
 *	type of data of the ci's type.
 *
 *	Note that ci->value may be reallocated during this call.
 */
void EDVCFGItemSetValue(edv_cfg_item_struct *ci, const gpointer value)
{
	gint size;
	gint8 *ptr_8;
        guint8 *ptr_u8;
        gint16 *ptr_16;
        guint16 *ptr_u16;
        gint32 *ptr_32;
        guint32 *ptr_u32;
	gint64 *ptr_64;
	guint64 *ptr_u64;
	gfloat *ptr_float;
	gdouble *ptr_double;
	edv_color_struct *ptr_color;
	edv_intlist_struct *ptr_intlist;


	if(ci == NULL)
	    return;

	/* Reset value on the cfg item, deallocating it. */
	EDVCFGItemResetValue(ci);

	/* No given value? */
	if(value == NULL)
	    return;

	/* Set by cfg item value data type. */
	switch(ci->type)
	{
	  case EDV_CFG_ITEM_TYPE_NONE:
	    break;

	  case EDV_CFG_ITEM_TYPE_INT8:
	    size = sizeof(gint8);
	    ptr_8 = (gint8 *)g_malloc(size);
	    if(ptr_8 != NULL)
		g_memcpy(ptr_8, value, size);
	    ci->value = ptr_8;
	    break;
          case EDV_CFG_ITEM_TYPE_UINT8:
            size = sizeof(guint8);
            ptr_u8 = (guint8 *)g_malloc(size);
            if(ptr_u8 != NULL)
                g_memcpy(ptr_u8, value, size);
            ci->value = ptr_u8;
            break;

          case EDV_CFG_ITEM_TYPE_INT16:
            size = sizeof(gint16);
            ptr_16 = (gint16 *)g_malloc(size);
            if(ptr_16 != NULL)
                g_memcpy(ptr_16, value, size);
            ci->value = ptr_16;
            break;
          case EDV_CFG_ITEM_TYPE_UINT16:
            size = sizeof(guint16);
            ptr_u16 = (guint16 *)g_malloc(size);
            if(ptr_u16 != NULL)
                g_memcpy(ptr_u16, value, size);
            ci->value = ptr_u16;
            break;

          case EDV_CFG_ITEM_TYPE_INT32:
            size = sizeof(gint32);
            ptr_32 = (gint32 *)g_malloc(size);
            if(ptr_32 != NULL)
                g_memcpy(ptr_32, value, size);
            ci->value = ptr_32;
            break;
          case EDV_CFG_ITEM_TYPE_UINT32:
            size = sizeof(guint32);
            ptr_u32 = (guint32 *)g_malloc(size);
            if(ptr_u32 != NULL)
                g_memcpy(ptr_u32, value, size);
            ci->value = ptr_u32;
            break;

          case EDV_CFG_ITEM_TYPE_INT64:
            size = sizeof(gint64);
            ptr_64 = (gint64 *)g_malloc(size);
            if(ptr_64 != NULL)
                g_memcpy(ptr_64, value, size);
            ci->value = ptr_64;
            break;
          case EDV_CFG_ITEM_TYPE_UINT64:
            size = sizeof(guint64);
            ptr_u64 = (guint64 *)g_malloc(size);
            if(ptr_u64 != NULL)
                g_memcpy(ptr_u64, value, size);
            ci->value = ptr_u64;
            break;

          case EDV_CFG_ITEM_TYPE_FLOAT:
            size = sizeof(gfloat);
            ptr_float = (gfloat *)g_malloc(size);
            if(ptr_float != NULL)
                g_memcpy(ptr_float, value, size);
            ci->value = ptr_float;
            break;
          case EDV_CFG_ITEM_TYPE_DOUBLE:
            size = sizeof(gdouble);
            ptr_double = (gdouble *)g_malloc(size);
            if(ptr_double != NULL)
                g_memcpy(ptr_double, value, size);
            ci->value = ptr_double;
            break;

          case EDV_CFG_ITEM_TYPE_STRING:
            ci->value = g_strdup((const gchar *)value);
            break;

          case EDV_CFG_ITEM_TYPE_COLOR:
            size = sizeof(edv_color_struct);
            ptr_color = (edv_color_struct *)g_malloc(size);
            if(ptr_color != NULL)
                g_memcpy(ptr_color, value, size);
            ci->value = ptr_color;
            break;

          case EDV_CFG_ITEM_TYPE_INTLIST:
            size = sizeof(edv_intlist_struct);
            ptr_intlist = (edv_intlist_struct *)g_malloc(size);
            if(ptr_intlist != NULL)
	    {
		const edv_intlist_struct *src_intlist =
		    (const edv_intlist_struct *)value;

		/* Copy values from source intlist to target intlist. */
		ptr_intlist->total = src_intlist->total;
		if(ptr_intlist->total > 0)
		{
		    ptr_intlist->i = (gint *)g_malloc0(
			ptr_intlist->total * sizeof(gint)
		    );
		    if((ptr_intlist->i != NULL) && (src_intlist->i != NULL))
			g_memcpy(
			    ptr_intlist->i, src_intlist->i,
			    ptr_intlist->total * sizeof(gint)
			);
		}
		else
		{
		    ptr_intlist->i = NULL;
		}
	    }
            ci->value = ptr_intlist;
            break;
	}
}

/*
 *	Deallocates the given configuration item ci's member value
 *	(as needed) and resets it to NULL.
 */
void EDVCFGItemResetValue(edv_cfg_item_struct *ci)
{
	edv_intlist_struct *ptr_intlist;


        if(ci == NULL)
	    return;

	/* No value to reset? */
	if(ci->value == NULL)
	    return;

	/* Deallocate value by type. */
	switch(ci->type)
	{
	  case EDV_CFG_ITEM_TYPE_INTLIST:
	    ptr_intlist = (edv_intlist_struct *)ci->value;
	    g_free(ptr_intlist->i);
	    ptr_intlist->i = NULL;
	    ptr_intlist->total = 0;
	    break;

/* Add support for other value types that are structures and have
 * additional memory to be deallocated here.
 */
	}

	/* Deallocate value and mark as deleted. */
	g_free(ci->value);
	ci->value = NULL;
}

/*
 *	Deallocates all substructures on the given configuration item
 *	ci and sets the type to EDV_CFG_ITEM_TYPE_NONE.
 *
 *	The structure ci itself will not be deallocated.
 */
void EDVCFGItemReset(edv_cfg_item_struct *ci)
{
	if(ci == NULL)
	    return;

	/* Deallocate value. */
	EDVCFGItemResetValue(ci);

	/* Deallocate parameter name. */
	g_free(ci->parameter);
	ci->parameter = NULL;

	/* Reset type. */
	ci->type = EDV_CFG_ITEM_TYPE_NONE;
}

/*
 *	Deallocates all structures on the ci and the structure itself.
 */
void EDVCFGItemDelete(edv_cfg_item_struct *ci)
{
	if(ci == NULL)
	    return;

	/* Deallocate all of the given configuration item's members. */
	EDVCFGItemReset(ci);

	/* Deallocate the configuration item structure itself. */
	g_free(ci);
}


/*
 *	Coppies the given list of sequential configuration items, all
 *	members will be coppied as well.
 */
edv_cfg_item_struct *EDVCFGItemListCopyList(
        const edv_cfg_item_struct *list
)
{
	gint i;
        const edv_cfg_item_struct	*src_list = list,
					*src_cfg_item;
        edv_cfg_item_struct	*tar_list = NULL,
				*tar_cfg_item;

	if(src_list == NULL)
	    return(tar_list);


        /* Iterate through source list, copying each configuration
	 * item to the target list.
	 */
        i = 0;
        while(((&src_list[i])->type != EDV_CFG_ITEM_TYPE_NONE) ||
              ((&src_list[i])->parameter != NULL) ||
              ((&src_list[i])->value != NULL)
        )
        {
	    /* Allocate (append) a new configuration item on the target
	     * list.
	     */
            tar_list = (edv_cfg_item_struct *)g_realloc(
                tar_list,
                (i + 1) * sizeof(edv_cfg_item_struct)
            );
            if(tar_list == NULL)
            {
                i = 0;
                break;
            }

            /* Get pointers to target and source items. */
            tar_cfg_item = &tar_list[i];
            src_cfg_item = &src_list[i];

            /* Reset newly allocated target item. */
            memset(tar_cfg_item, 0x00, sizeof(edv_cfg_item_struct));


            /* Begin copying values from source item to new item. */

	    /* Type. */
            tar_cfg_item->type = src_cfg_item->type;

	    /* Parameter. */
            tar_cfg_item->parameter = (src_cfg_item->parameter != NULL) ?
                g_strdup(src_cfg_item->parameter) : NULL;

	    /* Value. */
	    tar_cfg_item->value = NULL;
	    if(src_cfg_item->value != NULL)
	    {
		switch(tar_cfg_item->type)
		{
		  case EDV_CFG_ITEM_TYPE_INT8:
		  case EDV_CFG_ITEM_TYPE_UINT8:
		  case EDV_CFG_ITEM_TYPE_INT16:
		  case EDV_CFG_ITEM_TYPE_UINT16:
		  case EDV_CFG_ITEM_TYPE_INT32:
		  case EDV_CFG_ITEM_TYPE_UINT32:
		    EDVCFGItemListSetValueI(
                        tar_cfg_item, tar_cfg_item->parameter,
	EDVCFGItemListGetValueI(src_cfg_item, src_cfg_item->parameter),
			FALSE
		    );
		    break;
		  case EDV_CFG_ITEM_TYPE_INT64:
                    EDVCFGItemListSetValueL(
                        tar_cfg_item, tar_cfg_item->parameter,
        EDVCFGItemListGetValueL(src_cfg_item, src_cfg_item->parameter),
                        FALSE
                    );
		    break;
		  case EDV_CFG_ITEM_TYPE_UINT64:
		    EDVCFGItemListSetValueUL(
			tar_cfg_item, tar_cfg_item->parameter,
        EDVCFGItemListGetValueUL(src_cfg_item, src_cfg_item->parameter),
                        FALSE
		    );
		    break;
		  case EDV_CFG_ITEM_TYPE_FLOAT:
		  case EDV_CFG_ITEM_TYPE_DOUBLE:
                    EDVCFGItemListSetValueF(
                        tar_cfg_item, tar_cfg_item->parameter,
        EDVCFGItemListGetValueF(src_cfg_item, src_cfg_item->parameter),
                        FALSE
                    );
                    break;
		  case EDV_CFG_ITEM_TYPE_STRING:
                    EDVCFGItemListSetValueS(
                        tar_cfg_item, tar_cfg_item->parameter,
        EDVCFGItemListGetValueS(src_cfg_item, src_cfg_item->parameter),
                        FALSE
                    );
                    break;
		  case EDV_CFG_ITEM_TYPE_COLOR:
                    EDVCFGItemListSetValueColor(
                        tar_cfg_item, tar_cfg_item->parameter,
			(const edv_color_struct *)src_cfg_item->value,
                        FALSE
                    );
                    break;
		  case EDV_CFG_ITEM_TYPE_INTLIST:
		    EDVCFGItemListSetValueIntList(
                        tar_cfg_item, tar_cfg_item->parameter,
                        (const edv_intlist_struct *)src_cfg_item->value,
                        FALSE
                    );
		    break;
		}
	    }

            /* Go on to next the next item. */
            i++;
        }

        /* Allocate the last item on the target configuration list to be
	 * all 0's.
	 */
	tar_list = (edv_cfg_item_struct *)g_realloc(
	    tar_list,
	    (i + 1) * sizeof(edv_cfg_item_struct)
        );
        if(tar_list != NULL)
        {
            tar_cfg_item = &tar_list[i];
            memset(tar_cfg_item, 0x00, sizeof(edv_cfg_item_struct));
        }

	return(tar_list);
}

/*
 *	Deletes the given list of sequential configuration items, all
 *      members will be deleted as well.
 */
void EDVCFGItemListDeleteList(edv_cfg_item_struct *list)
{
	gint i;


	if(list == NULL)
	    return;

        /* Delete configuration list, each item's parameter and value
         * needs to be deallocated but not each structure (since it is
         * a sequential array of structures).
         */
        i = 0;
	while(((&list[i])->type != EDV_CFG_ITEM_TYPE_NONE) ||
	      ((&list[i])->parameter != NULL) ||
	      ((&list[i])->value != NULL)
	)
	{
	    /* Deallocate parameter string and value. */
	    EDVCFGItemReset(&list[i]);
	    i++;
        }

        /* Deallocate all structures in the configuration list by
         * deallocating the sequential array of structures at the base.
         */
        g_free(list);
}

/*
 *	Scans through the sequential list of configuration items and
 *	returns the index number of the one who's parameter matches the
 *	given parameter (case insensitive).
 *
 *	The last item in the sequential list needs to be all 0.
 *
 *	Returns -1 on failed match or error, otherwise returns
 *	the index of the valid item.
 */
gint EDVCFGItemListMatchParameter(
	const edv_cfg_item_struct *list, const gchar *parameter
)
{
	gint i = 0;
	const edv_cfg_item_struct *ci;


	if((list == NULL) || (parameter == NULL))
	    return(-1);

	do
	{
	    ci = &list[i];

	    if(ci->parameter != NULL)
	    {
		if(!g_strcasecmp(ci->parameter, parameter))
		    return(i);
	    }

	    i++;

	} while((ci->type != 0) && (ci->parameter != NULL));

	return(-1);
}

/*
 *	Returns the pointer to the value and sets the type to the
 *	type of value the matched list item in the sequential list is.
 *
 *      The last item in the sequential list needs to be all 0.
 *
 *      Returns NULL on failed match or error, otherwise returns
 *      the pointer to the value and sets the type code.
 *
 *	The returned pointer should never be deallocated by the calling
 *	function.
 */
gpointer EDVCFGItemListMatchGetValue(
	const edv_cfg_item_struct *list, const gchar *parameter,
	gint *type_rtn
)
{
	gint i;


	if(type_rtn != NULL)
	    (*type_rtn) = -1;

	/* Check if given parameter matches the parameter of a 
	 * configuration item in the given list.
	 */
	i = EDVCFGItemListMatchParameter(list, parameter);
	if(i > -1)
	    return(EDVCFGItemGetValue(&list[i], type_rtn));
	else
	    return(NULL);
}

/*
 *	Returns value as an gint, the parameter must match to an item
 *	who's type matches any of the following:
 *
 *		EDV_CFG_ITEM_TYPE_INT8
 *		EDV_CFG_ITEM_TYPE_UINT8
 *		EDV_CFG_ITEM_TYPE_INT16
 *		EDV_CFG_ITEM_TYPE_UINT16
 *		EDV_CFG_ITEM_TYPE_INT32
 *		EDV_CFG_ITEM_TYPE_UINT32
 *
 *	Does not distinguish error.
 */
gint EDVCFGItemListGetValueI(
        const edv_cfg_item_struct *list, const gchar *parameter
)
{
#define RTN_VAL_T	gint
        RTN_VAL_T rtn_val = (RTN_VAL_T)0;
        gint ci_num;
        const edv_cfg_item_struct *ci_ptr;

	gint8 *ptr_8;
        guint8 *ptr_u8;
        gint16 *ptr_16;
        guint16 *ptr_u16;
        gint32 *ptr_32;
        guint32 *ptr_u32;


        /* Match by parameter. */
        ci_num = EDVCFGItemListMatchParameter(list, parameter);
        if(ci_num < 0)
            return(rtn_val);
        else
            ci_ptr = &list[ci_num];

        /* No value? */
        if(ci_ptr->value == NULL)
            return(rtn_val);

        /* Get return value by cfg item type. */
        switch(ci_ptr->type)
        {
          case EDV_CFG_ITEM_TYPE_INT8: 
            ptr_8 = (gint8 *)ci_ptr->value;  
            rtn_val = (RTN_VAL_T)(*ptr_8); 
            break;
          case EDV_CFG_ITEM_TYPE_UINT8: 
            ptr_u8 = (guint8 *)ci_ptr->value;  
            rtn_val = (RTN_VAL_T)(*ptr_u8); 
            break;

          case EDV_CFG_ITEM_TYPE_INT16:
            ptr_16 = (gint16 *)ci_ptr->value;
            rtn_val = (RTN_VAL_T)(*ptr_16);
            break;
          case EDV_CFG_ITEM_TYPE_UINT16:
            ptr_u16 = (guint16 *)ci_ptr->value;
            rtn_val = (RTN_VAL_T)(*ptr_u16);
            break;

          case EDV_CFG_ITEM_TYPE_INT32:
            ptr_32 = (gint32 *)ci_ptr->value;
            rtn_val = (RTN_VAL_T)(*ptr_32);
            break;
          case EDV_CFG_ITEM_TYPE_UINT32:
            ptr_u32 = (guint32 *)ci_ptr->value;
            rtn_val = (RTN_VAL_T)(*ptr_u32);
            break;
        }

        return(rtn_val);
#undef RTN_VAL_T
}

/*
 *      Returns value as a glong, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_INT8
 *              EDV_CFG_ITEM_TYPE_UINT8
 *              EDV_CFG_ITEM_TYPE_INT16
 *              EDV_CFG_ITEM_TYPE_UINT16
 *              EDV_CFG_ITEM_TYPE_INT32
 *              EDV_CFG_ITEM_TYPE_UINT32
 *		EDV_CFG_ITEM_TYPE_INT64
 *		EDV_CFG_ITEM_TYPE_UINT64
 *
 *      Does not distinguish error.
 */
glong EDVCFGItemListGetValueL(
        const edv_cfg_item_struct *list, const gchar *parameter
)
{
#define RTN_VAL_T	glong
	RTN_VAL_T rtn_val = (RTN_VAL_T)0;
	gint ci_num;
	const edv_cfg_item_struct *ci_ptr;

        gint64 *ptr_64;
        guint64 *ptr_u64;


        /* Match by parameter. */
        ci_num = EDVCFGItemListMatchParameter(list, parameter);
        if(ci_num < 0)
            return(rtn_val);
        else
            ci_ptr = &list[ci_num];

        /* No value? */
        if(ci_ptr->value == NULL)
            return(rtn_val);

        /* Get return value by cfg item type. */
        switch(ci_ptr->type)  
        {
          case EDV_CFG_ITEM_TYPE_INT8:
          case EDV_CFG_ITEM_TYPE_UINT8:
          case EDV_CFG_ITEM_TYPE_INT16:
          case EDV_CFG_ITEM_TYPE_UINT16:
          case EDV_CFG_ITEM_TYPE_INT32:
          case EDV_CFG_ITEM_TYPE_UINT32:
	    rtn_val = (RTN_VAL_T)EDVCFGItemListGetValueI(
		list, parameter
	    );
	    break;

          case EDV_CFG_ITEM_TYPE_INT64:
            ptr_64 = (gint64 *)ci_ptr->value;
            rtn_val = (RTN_VAL_T)(*ptr_64);
            break;
          case EDV_CFG_ITEM_TYPE_UINT64:
            ptr_u64 = (guint64 *)ci_ptr->value;
            rtn_val = (RTN_VAL_T)(*ptr_u64);
            break;
        }

        return(rtn_val);
#undef RTN_VAL_T  
}           

/*
 *      Returns value as a gulong, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_INT8
 *              EDV_CFG_ITEM_TYPE_UINT8
 *              EDV_CFG_ITEM_TYPE_INT16
 *              EDV_CFG_ITEM_TYPE_UINT16
 *              EDV_CFG_ITEM_TYPE_INT32
 *              EDV_CFG_ITEM_TYPE_UINT32
 *              EDV_CFG_ITEM_TYPE_INT64
 *              EDV_CFG_ITEM_TYPE_UINT64
 *
 *      Does not distinguish error.
 */
gulong EDVCFGItemListGetValueUL(
        const edv_cfg_item_struct *list, const gchar *parameter
)
{
#define RTN_VAL_T       gulong
        RTN_VAL_T rtn_val = (RTN_VAL_T)0;
        gint ci_num;
        const edv_cfg_item_struct *ci_ptr;

        gint64 *ptr_64;
        guint64 *ptr_u64;


        /* Match by parameter. */
        ci_num = EDVCFGItemListMatchParameter(list, parameter);
        if(ci_num < 0)
            return(rtn_val);
        else
            ci_ptr = &list[ci_num];

        /* No value? */
        if(ci_ptr->value == NULL)
            return(rtn_val);

        /* Get return value by cfg item type. */
        switch(ci_ptr->type)
        {
          case EDV_CFG_ITEM_TYPE_INT8:
          case EDV_CFG_ITEM_TYPE_UINT8:
          case EDV_CFG_ITEM_TYPE_INT16:
          case EDV_CFG_ITEM_TYPE_UINT16:
          case EDV_CFG_ITEM_TYPE_INT32:
          case EDV_CFG_ITEM_TYPE_UINT32:
            rtn_val = (RTN_VAL_T)EDVCFGItemListGetValueI(
                list, parameter
            );
            break;

          case EDV_CFG_ITEM_TYPE_INT64:
            ptr_64 = (gint64 *)ci_ptr->value;
            rtn_val = (RTN_VAL_T)(*ptr_64);
            break;
          case EDV_CFG_ITEM_TYPE_UINT64:
            ptr_u64 = (guint64 *)ci_ptr->value;
            rtn_val = (RTN_VAL_T)(*ptr_u64);
            break;
        }

        return(rtn_val);
#undef RTN_VAL_T
}

/*
 *      Returns value as a gfloat, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_INT8
 *              EDV_CFG_ITEM_TYPE_UINT8
 *              EDV_CFG_ITEM_TYPE_INT16
 *              EDV_CFG_ITEM_TYPE_UINT16
 *              EDV_CFG_ITEM_TYPE_INT32
 *              EDV_CFG_ITEM_TYPE_UINT32
 *              EDV_CFG_ITEM_TYPE_INT64
 *              EDV_CFG_ITEM_TYPE_UINT64
 *		EDV_CFG_ITEM_TYPE_FLOAT
 *
 *      Does not distinguish error.
 */
float EDVCFGItemListGetValueF(
        const edv_cfg_item_struct *list, const gchar *parameter
)
{
#define RTN_VAL_T	float
        RTN_VAL_T rtn_val = (float)0.0;
	gint ci_num;
        const edv_cfg_item_struct *ci_ptr;

	gfloat *ptr_f;


        /* Match by parameter. */
        ci_num = EDVCFGItemListMatchParameter(list, parameter);
        if(ci_num < 0)  
            return(rtn_val);
        else
            ci_ptr = &list[ci_num];

        /* No value? */
        if(ci_ptr->value == NULL)
            return(rtn_val);

        /* Get return value by cfg item type. */
        switch(ci_ptr->type)
        {
          case EDV_CFG_ITEM_TYPE_INT8:
          case EDV_CFG_ITEM_TYPE_UINT8:
          case EDV_CFG_ITEM_TYPE_INT16:
          case EDV_CFG_ITEM_TYPE_UINT16:
          case EDV_CFG_ITEM_TYPE_INT32:
          case EDV_CFG_ITEM_TYPE_UINT32:
          case EDV_CFG_ITEM_TYPE_INT64:
          case EDV_CFG_ITEM_TYPE_UINT64:
	    rtn_val = (RTN_VAL_T)EDVCFGItemListGetValueL(
		list, parameter
	    );
	    break;

          case EDV_CFG_ITEM_TYPE_FLOAT:
            ptr_f = (float *)ci_ptr->value;
            rtn_val = (RTN_VAL_T)(*ptr_f);
            break;
        }

        return(rtn_val);
#undef RTN_VAL_T
}

/*
 *      Returns value as a gdouble, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_INT8
 *              EDV_CFG_ITEM_TYPE_UINT8 
 *              EDV_CFG_ITEM_TYPE_INT16
 *              EDV_CFG_ITEM_TYPE_UINT16  
 *              EDV_CFG_ITEM_TYPE_INT32
 *              EDV_CFG_ITEM_TYPE_UINT32
 *              EDV_CFG_ITEM_TYPE_INT64
 *              EDV_CFG_ITEM_TYPE_UINT64
 *              EDV_CFG_ITEM_TYPE_FLOAT
 *		EDV_CFG_ITEM_TYPE_DOUBLE
 *
 *      Does not distinguish error.
 */
gdouble EDVCFGItemListGetValueD(
        const edv_cfg_item_struct *list, const gchar *parameter
)
{
#define RTN_VAL_T	gdouble
        RTN_VAL_T rtn_val = (RTN_VAL_T)0.0;
	gint ci_num;
	const edv_cfg_item_struct *ci_ptr;

	gdouble *ptr_d;


        /* Match by parameter. */
        ci_num = EDVCFGItemListMatchParameter(list, parameter);
        if(ci_num < 0)
            return(rtn_val);
        else
            ci_ptr = &list[ci_num];

        /* No value? */
        if(ci_ptr->value == NULL)
            return(rtn_val);

        /* Get return value by cfg item type. */
        switch(ci_ptr->type)
        {
          case EDV_CFG_ITEM_TYPE_INT8:
          case EDV_CFG_ITEM_TYPE_UINT8:
          case EDV_CFG_ITEM_TYPE_INT16:
          case EDV_CFG_ITEM_TYPE_UINT16:
          case EDV_CFG_ITEM_TYPE_INT32:
          case EDV_CFG_ITEM_TYPE_UINT32:
          case EDV_CFG_ITEM_TYPE_INT64:
          case EDV_CFG_ITEM_TYPE_UINT64:
          case EDV_CFG_ITEM_TYPE_FLOAT:
	    rtn_val = (RTN_VAL_T)EDVCFGItemListGetValueF(
		list, parameter
	    );
            break;

          case EDV_CFG_ITEM_TYPE_DOUBLE:
            ptr_d = (double *)ci_ptr->value;
            rtn_val = (RTN_VAL_T)(*ptr_d);
            break;
        }

        return(rtn_val);
#undef RTN_VAL_T
}

/*
 *	Returns a pointer to the value of the matched parameter, the
 *	returned string may not be deallocated as it points directly
 *	to the matched cfg item's value. Can return NULL on error.
 *
 *	Only cfg items are of type EDV_CFG_ITEM_TYPE_STRING will be
 *	matched.
 */
gchar *EDVCFGItemListGetValueS(
        const edv_cfg_item_struct *list, const gchar *parameter
)
{
	gint ci_num;
        const edv_cfg_item_struct *ci_ptr;


        /* Match by parameter. */
        ci_num = EDVCFGItemListMatchParameter(list, parameter);
        if(ci_num < 0)
            return(NULL);
        else
            ci_ptr = &list[ci_num];

        /* Not the correct type or no value? */
        if((ci_ptr->type != EDV_CFG_ITEM_TYPE_STRING) ||
           (ci_ptr->value == NULL)
	)
            return(NULL);
	else
	    return((gchar *)ci_ptr->value);
}

/*
 *      Returns a pointer to the value of the matched parameter, the
 *      returned edv_color_struct may not be deallocated as it
 *	points directly to the matched cfg item's value. Can return
 *	NULL on error.
 *
 *      Only cfg items are of type EDV_CFG_ITEM_TYPE_COLOR will be
 *      matched.
 */
edv_color_struct *EDVCFGItemListGetValueColor(
        const edv_cfg_item_struct *list, const gchar *parameter
)
{
        gint ci_num;
        const edv_cfg_item_struct *ci_ptr;


        /* Match by parameter. */
        ci_num = EDVCFGItemListMatchParameter(list, parameter);
        if(ci_num < 0)
            return(NULL);
        else
            ci_ptr = &list[ci_num];

        /* Not the correct type or no value? */
        if((ci_ptr->type != EDV_CFG_ITEM_TYPE_COLOR) ||
           (ci_ptr->value == NULL)
        )
            return(NULL);
        else
            return((edv_color_struct *)ci_ptr->value);
}

/*
 *      Returns a pointer to the value of the matched parameter, the
 *      returned edv_intlist_struct may not be deallocated as it
 *      points directly to the matched cfg item's value. Can return
 *      NULL on error.
 *
 *      Only cfg items are of type EDV_CFG_ITEM_TYPE_INTLIST will be
 *      matched.
 */
edv_intlist_struct *EDVCFGItemListGetValueIntList(
        const edv_cfg_item_struct *list, const gchar *parameter
)
{
        gint ci_num;
        const edv_cfg_item_struct *ci_ptr;


        /* Match by parameter. */
        ci_num = EDVCFGItemListMatchParameter(list, parameter);
        if(ci_num < 0)
            return(NULL);
        else
            ci_ptr = &list[ci_num];

        /* Not the correct type or no value? */
        if((ci_ptr->type != EDV_CFG_ITEM_TYPE_INTLIST) ||
           (ci_ptr->value == NULL)
        )
            return(NULL);
        else
            return((edv_intlist_struct *)ci_ptr->value);
}


/*
 *      Sets the given value to the sequential list item found match
 *	the given parameter.
 *
 *      The last item in the sequential list needs to be all 0.
 *
 *	Returns the pointer to the new sequential list, if
 *	create_as_needed is false then pointer will be exactly the same
 *	as the list pointer value. If create_as_needed is true then
 *	the returned pointer will be a new pointer to the new
 *	sequential list and the old list will be deallocated.
 *
 *	Can return NULL on error if create_as_needed is true.
 */
edv_cfg_item_struct *EDVCFGItemListSetValue(
	edv_cfg_item_struct *list, const gchar *parameter,
	const gpointer value, gbool create_as_needed
)
{
	gint i;


	if(list == NULL)
	    return(list);

	/* First try to match an existing configuration item in the given
	 * list of configuration items. 
	 */
	i = EDVCFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    /* Matched existing configuration item. */
	    EDVCFGItemSetValue(&list[i], value);
	    return(&list[i]);
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
	    if(create_as_needed)
	    {
		/* Need to work on this. */
		return(NULL);
	    }
	    else
	    {
		/* Not exist and not allowed to create a new one, so
		 * return NULL to indicate failure.
		 */
		return(NULL);
	    }
	}
}

/*
 *      Sets value as a gint, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_INT8
 *              EDV_CFG_ITEM_TYPE_UINT8
 *              EDV_CFG_ITEM_TYPE_INT16
 *              EDV_CFG_ITEM_TYPE_UINT16
 *              EDV_CFG_ITEM_TYPE_INT32
 *              EDV_CFG_ITEM_TYPE_UINT32
 *              EDV_CFG_ITEM_TYPE_INT64
 *              EDV_CFG_ITEM_TYPE_UINT64
 *              EDV_CFG_ITEM_TYPE_FLOAT
 *              EDV_CFG_ITEM_TYPE_DOUBLE
 */
edv_cfg_item_struct *EDVCFGItemListSetValueI(
        edv_cfg_item_struct *list, const gchar *parameter,
        gint value, gbool create_as_needed
)
{
	return(EDVCFGItemListSetValueL(
	    list, parameter, value, create_as_needed
	));
}

/*
 *      Sets value as a gint, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_INT8
 *              EDV_CFG_ITEM_TYPE_UINT8
 *              EDV_CFG_ITEM_TYPE_INT16
 *              EDV_CFG_ITEM_TYPE_UINT16
 *              EDV_CFG_ITEM_TYPE_INT32
 *              EDV_CFG_ITEM_TYPE_UINT32
 *              EDV_CFG_ITEM_TYPE_INT64
 *              EDV_CFG_ITEM_TYPE_UINT64
 *              EDV_CFG_ITEM_TYPE_FLOAT
 *              EDV_CFG_ITEM_TYPE_DOUBLE
 */
edv_cfg_item_struct *EDVCFGItemListSetValueL(
        edv_cfg_item_struct *list, const gchar *parameter,
	glong value, gbool create_as_needed
)
{
        gint i;
	gpointer ptr;
	gint8 v_8;
	guint8 v_u8;
	gint16 v_16;
        guint16 v_u16;
        gint32 v_32;
        guint32 v_u32;
        gint64 v_64;
        guint64 v_u64;
	gfloat v_float;
	gdouble v_double;


        if(list == NULL)
            return(list);

        /* First try to match an existing configuration item in the given
         * list of configuration items.
         */
        i = EDVCFGItemListMatchParameter(list, parameter);
        if(i > -1)
        {
	    ptr = NULL;
	    switch((&list[i])->type)
	    {
	      case EDV_CFG_ITEM_TYPE_INT8:
		v_8 = (gint8)value;
		ptr = (gpointer)&v_8;
		break;
	      case EDV_CFG_ITEM_TYPE_UINT8:
                v_u8 = (guint8)value;
                ptr = (gpointer)&v_u8;
                break;

              case EDV_CFG_ITEM_TYPE_INT16:
                v_16 = (gint16)value;
                ptr = (gpointer)&v_16;
                break;
              case EDV_CFG_ITEM_TYPE_UINT16:
                v_u16 = (guint16)value;
                ptr = (gpointer)&v_u16;
                break;

              case EDV_CFG_ITEM_TYPE_INT32:
                v_32 = (gint32)value;
                ptr = (gpointer)&v_32;
                break;
              case EDV_CFG_ITEM_TYPE_UINT32:
                v_u32 = (guint32)value;
                ptr = (gpointer)&v_u32;
                break;

              case EDV_CFG_ITEM_TYPE_INT64:
                v_64 = (gint64)value;
                ptr = (gpointer)&v_64;
                break;
              case EDV_CFG_ITEM_TYPE_UINT64:
                v_u64 = (guint64)value;
                ptr = (gpointer)&v_u64;
                break;

              case EDV_CFG_ITEM_TYPE_FLOAT:
                v_float = (gfloat)value;
                ptr = (gpointer)&v_float;
                break;
              case EDV_CFG_ITEM_TYPE_DOUBLE:
                v_double = (gdouble)value;
                ptr = (gpointer)&v_double;
                break;
	    }
	    if(ptr == NULL)
	    {
		return(NULL);
	    }
	    else
	    {
		EDVCFGItemSetValue(&list[i], ptr);
		return(&list[i]);
	    }
        }
        else
        {
            /* Does not exist, are we allowed to create one? */
            if(create_as_needed)
            {
                /* Need to work on this. */
                return(NULL);
            }
            else
            {
                /* Not exist and not allowed to create a new one, so
                 * return NULL to indicate failure.
                 */
                return(NULL);
            }
        }
}

/*
 *      Sets value as a gulong, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_INT8
 *              EDV_CFG_ITEM_TYPE_UINT8
 *              EDV_CFG_ITEM_TYPE_INT16
 *              EDV_CFG_ITEM_TYPE_UINT16
 *              EDV_CFG_ITEM_TYPE_INT32
 *              EDV_CFG_ITEM_TYPE_UINT32
 *              EDV_CFG_ITEM_TYPE_INT64
 *              EDV_CFG_ITEM_TYPE_UINT64
 *              EDV_CFG_ITEM_TYPE_FLOAT
 *              EDV_CFG_ITEM_TYPE_DOUBLE
 */
edv_cfg_item_struct *EDVCFGItemListSetValueUL(
        edv_cfg_item_struct *list, const gchar *parameter,
        gulong value, gbool create_as_needed
)
{
        return(EDVCFGItemListSetValueL(
            list, parameter, value, create_as_needed
        ));
}

/*
 *      Sets value as a gfloat, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_INT8
 *              EDV_CFG_ITEM_TYPE_UINT8
 *              EDV_CFG_ITEM_TYPE_INT16
 *              EDV_CFG_ITEM_TYPE_UINT16
 *              EDV_CFG_ITEM_TYPE_INT32
 *              EDV_CFG_ITEM_TYPE_UINT32
 *              EDV_CFG_ITEM_TYPE_INT64
 *              EDV_CFG_ITEM_TYPE_UINT64
 *              EDV_CFG_ITEM_TYPE_FLOAT
 *              EDV_CFG_ITEM_TYPE_DOUBLE
 */
edv_cfg_item_struct *EDVCFGItemListSetValueF(
        edv_cfg_item_struct *list, const gchar *parameter,
        gfloat value, gbool create_as_needed
)
{
        gint i;
        gpointer ptr;
        gint8 v_8;
        guint8 v_u8;
        gint16 v_16;
        guint16 v_u16;
        gint32 v_32;
        guint32 v_u32;
        gint64 v_64;
        guint64 v_u64;
        gfloat v_float;
        gdouble v_double;


        if(list == NULL)
            return(list);

        /* First try to match an existing configuration item in the given
         * list of configuration items.
         */
        i = EDVCFGItemListMatchParameter(list, parameter);
        if(i > -1)
        {
            ptr = NULL;
            switch((&list[i])->type)
            {
              case EDV_CFG_ITEM_TYPE_INT8:
                v_8 = (gint8)value;
                ptr = (gpointer)&v_8;
                break;
              case EDV_CFG_ITEM_TYPE_UINT8:
                v_u8 = (guint8)value;
                ptr = (gpointer)&v_u8;
                break;

              case EDV_CFG_ITEM_TYPE_INT16:
                v_16 = (gint16)value;
                ptr = (gpointer)&v_16;
                break;
              case EDV_CFG_ITEM_TYPE_UINT16:
                v_u16 = (guint16)value;
                ptr = (gpointer)&v_u16;
                break;

              case EDV_CFG_ITEM_TYPE_INT32:
                v_32 = (gint32)value;
                ptr = (gpointer)&v_32;
                break;
              case EDV_CFG_ITEM_TYPE_UINT32:
                v_u32 = (guint32)value;
                ptr = (gpointer)&v_u32;
                break;

              case EDV_CFG_ITEM_TYPE_INT64:
                v_64 = (gint64)value;
                ptr = (gpointer)&v_64;
                break;
              case EDV_CFG_ITEM_TYPE_UINT64:
                v_u64 = (guint64)value;
                ptr = (gpointer)&v_u64;
                break;

              case EDV_CFG_ITEM_TYPE_FLOAT:
                v_float = (gfloat)value;
                ptr = (gpointer)&v_float;
                break;
              case EDV_CFG_ITEM_TYPE_DOUBLE:
                v_double = (gdouble)value;
                ptr = (gpointer)&v_double;
                break;
            }
            if(ptr == NULL)
            {
                return(NULL);
            }
            else
            {
                EDVCFGItemSetValue(&list[i], ptr);
                return(&list[i]);
            }
        }
        else
        {
            /* Does not exist, are we allowed to create one? */
            if(create_as_needed)
            {
                /* Need to work on this. */
                return(NULL);
            }
            else
            {
                /* Not exist and not allowed to create a new one, so
                 * return NULL to indicate failure.
                 */
                return(NULL);
            }
        }
}

/*
 *      Sets value as a string, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_STRING
 */
edv_cfg_item_struct *EDVCFGItemListSetValueS(
        edv_cfg_item_struct *list, const gchar *parameter,
        const gchar *value, gbool create_as_needed
)
{
        gint i;


        if(list == NULL)
            return(list);

        /* First try to match an existing configuration item in the given
         * list of configuration items.
         */
        i = EDVCFGItemListMatchParameter(list, parameter);
        if(i > -1)
        {
	    if((&list[i])->type == EDV_CFG_ITEM_TYPE_STRING)
	    {
		EDVCFGItemSetValue(&list[i], (const gpointer)value);
		return(&list[i]);
	    }
	    else
	    {
		return(NULL);
	    }
        }
        else
        {
            /* Does not exist, are we allowed to create one? */
            if(create_as_needed)
            {
                /* Need to work on this. */
                return(NULL);
            }
            else
            {
                /* Not exist and not allowed to create a new one, so
                 * return NULL to indicate failure.
                 */
                return(NULL);
            }
        }
}

/*
 *      Sets value as a string, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_COLOR
 */
edv_cfg_item_struct *EDVCFGItemListSetValueColor(
        edv_cfg_item_struct *list, const gchar *parameter,
        const edv_color_struct *value, gbool create_as_needed
)
{
        gint i;


        if(list == NULL)
            return(list);

        /* First try to match an existing configuration item in the given
         * list of configuration items.
         */
        i = EDVCFGItemListMatchParameter(list, parameter);
        if(i > -1)
        {
            if((&list[i])->type == EDV_CFG_ITEM_TYPE_COLOR)
            {
                EDVCFGItemSetValue(&list[i], (const gpointer)value);
                return(&list[i]);
            }
            else
            {
                return(NULL);
            }
        }
        else
        {
            /* Does not exist, are we allowed to create one? */
            if(create_as_needed)
            {
                /* Need to work on this. */
                return(NULL);
            }
            else
            {
                /* Not exist and not allowed to create a new one, so
                 * return NULL to indicate failure.
                 */
                return(NULL);
            }
        }
}

/*
 *      Sets value as a string, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              EDV_CFG_ITEM_TYPE_INTLIST
 */
edv_cfg_item_struct *EDVCFGItemListSetValueIntList(
        edv_cfg_item_struct *list, const gchar *parameter,
        const edv_intlist_struct *value, gbool create_as_needed
)
{
        gint i;


        if(list == NULL)
            return(list);

        /* First try to match an existing configuration item in the given
         * list of configuration items.
         */
        i = EDVCFGItemListMatchParameter(list, parameter);
        if(i > -1)
        {
            if((&list[i])->type == EDV_CFG_ITEM_TYPE_INTLIST)
            {
                EDVCFGItemSetValue(&list[i], (const gpointer)value);
                return(&list[i]);
            }
            else
            {
                return(NULL);
            }
        }
        else
        {
            /* Does not exist, are we allowed to create one? */
            if(create_as_needed)
            {
                /* Need to work on this. */
                return(NULL);
            }
            else
            {
                /* Not exist and not allowed to create a new one, so
                 * return NULL to indicate failure.
                 */
                return(NULL);
            }
        }
}

