#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "cdialog.h"
#include "edvtypes.h"
#include "edvcfg.h"
#include "edvcfgfio.h"
#include "config.h"


gint EDVCFGLoadFromFile(
	const gchar *filename, edv_cfg_item_struct *cfg_list
);
gint EDVCFGSaveToFile(
	const gchar *filename, edv_cfg_item_struct *cfg_list
);


/*
 *	Deliminator characters, should be defined in config.h but in
 *	some cases they are not so assume default.
 */
#ifndef EDV_CFG_COMMENT_CHAR
# define EDV_CFG_COMMENT_CHAR		'#'
#endif

#ifndef EDV_CFG_DELIMINATOR_CHAR
# define EDV_CFG_DELIMINATOR_CHAR	'='
#endif


/*
 *	Loads configuration list from the specified file, this will update
 *	the values of the given cfg_list. The last set of values in the
 *	cfg_list must be all zero's.
 */
gint EDVCFGLoadFromFile(
	const gchar *filename, edv_cfg_item_struct *cfg_list
)
{
	gint i, ignore_all_errors = 0;
	FILE *fp;
	struct stat stat_buf;
	gchar *parm = NULL;


	if((filename == NULL) || (cfg_list == NULL))
	    return(-1);

	if(stat(filename, &stat_buf))
	{
/*
	    fprintf(
		stderr,
		"%s: No such file.\n",
		filename
	    );
 */
	    return(-1);
	}
	if(!S_ISREG(stat_buf.st_mode))
	{
            fprintf(
                stderr,
                "%s: Not a file.\n",
                filename
            );
            return(-1);
	}

	/* Open configuration file. */
	fp = FOpen(filename, "rb");
	if(fp == NULL)
	{
/*
	    fprintf(
		stderr,
		"%s: Cannot open.\n",
		filename
	    );
 */
	    return(-1);
	}

	while(1)
	{
	    parm = FSeekNextParm(
		fp, parm,
		EDV_CFG_COMMENT_CHAR, EDV_CFG_DELIMINATOR_CHAR
	    );
	    if(parm == NULL)
		break;

	    i = EDVCFGItemListMatchParameter(cfg_list, parm);
	    if(i < 0)
	    {
		if(!ignore_all_errors)
		{
                    fprintf(
			stderr,
                        "%s: Unsupported parameter `%s'.\n",
                        filename,
                        parm
                    );
		}

		/* No such parameter. */
		FSeekNextLine(fp);
	    }
	    else
	    {
		edv_cfg_item_struct *ci = &cfg_list[i];
		gchar *strptr;
		gint vi[10];
		gdouble vd[10];

		gint8		i8;
		guint8		ui8;
		gint16		i16;
		guint16		ui16;
		gint32 		i32;
		guint32		ui32;
		gint64		i64;
		guint64		ui64;
		gfloat		f;
		gdouble		d;
		edv_color_struct color;
                edv_intlist_struct *intlist;

		switch(ci->type)
		{
		  case EDV_CFG_ITEM_TYPE_INT8:
		    FGetValuesI(fp, vi, 1);
		    i8 = (gint8)vi[0];
		    EDVCFGItemSetValue(ci, &i8);
		    break;
                  case EDV_CFG_ITEM_TYPE_UINT8:
                    FGetValuesI(fp, vi, 1);
                    ui8 = (guint8)vi[0];
                    EDVCFGItemSetValue(ci, &ui8);
                    break;

                  case EDV_CFG_ITEM_TYPE_INT16:
                    FGetValuesI(fp, vi, 1);
                    i16 = (gint16)vi[0];
                    EDVCFGItemSetValue(ci, &i16);
                    break;
                  case EDV_CFG_ITEM_TYPE_UINT16:
                    FGetValuesI(fp, vi, 1);  
                    ui16 = (guint16)vi[0];
                    EDVCFGItemSetValue(ci, &ui16);
                    break;

                  case EDV_CFG_ITEM_TYPE_INT32:
                    FGetValuesI(fp, vi, 1);
                    i32 = (gint32)vi[0];
                    EDVCFGItemSetValue(ci, &i32);
                    break;
                  case EDV_CFG_ITEM_TYPE_UINT32:
                    FGetValuesI(fp, vi, 1);
                    ui32 = (guint32)vi[0];
                    EDVCFGItemSetValue(ci, &ui32);
                    break;

                  case EDV_CFG_ITEM_TYPE_INT64:
                    FGetValuesI(fp, vi, 1);
                    i64 = (gint64)vi[0];
                    EDVCFGItemSetValue(ci, &i64);
                    break;
                  case EDV_CFG_ITEM_TYPE_UINT64:
                    FGetValuesI(fp, vi, 1);
                    ui64 = (guint64)vi[0];
                    EDVCFGItemSetValue(ci, &ui64);
                    break;

                  case EDV_CFG_ITEM_TYPE_FLOAT:
                    FGetValuesF(fp, vd, 1);
                    f = (gfloat)vd[0];
                    EDVCFGItemSetValue(ci, &f);
                    break;
                  case EDV_CFG_ITEM_TYPE_DOUBLE:
                    FGetValuesF(fp, vd, 1);
                    d = (gdouble)vd[0];
                    EDVCFGItemSetValue(ci, &d);
                    break;

		  case EDV_CFG_ITEM_TYPE_STRING:
		    g_free(ci->value);
		    ci->value = FGetString(fp);
		    break;

                  case EDV_CFG_ITEM_TYPE_COLOR:
                    FGetValuesF(fp, vd, 4);
                    color.r = (gfloat)vd[0];
                    color.g = (gfloat)vd[1];
                    color.b = (gfloat)vd[2];
                    color.a = (gfloat)vd[3];
                    EDVCFGItemSetValue(ci, &color);
                    break;

                  case EDV_CFG_ITEM_TYPE_INTLIST:
                    strptr = FGetString(fp);
		    if(strptr != NULL)
		    {
			gint n;


			/* Delete old value on cfg item. */
			EDVCFGItemResetValue(ci);

			/* Allocate a new edv_intlist_struct. */
			intlist = (edv_intlist_struct *)g_malloc0(
			    sizeof(edv_intlist_struct)
			);
			if(intlist != NULL)
			{
			    /* Begin parsing intlist string. */
			    gchar *strptr2 = strptr;
			    while(*strptr2 != '\0')
			    {
                                while(ISBLANK(*strptr2))
                                    strptr2++;

				/* Allocate a new int. */
				n = intlist->total;
				intlist->total = n + 1;
				intlist->i = (gint *)g_realloc(
				    intlist->i,
				    intlist->total * sizeof(gint)
				);
				if(intlist->i == NULL)
				{
				    intlist->total = 0;
				    break;
				}

				/* Get int from current strptr2 position. */
				intlist->i[n] = atoi(strptr2);

				/* Seek strptr2 to next int or end of string. */
				while(!ISBLANK(*strptr2) && (*strptr2 != '\0'))
				    strptr2++;
			    }

			    /* Transfer new intlist to cfg item's value. */
			    ci->value = (gpointer)intlist;
			    intlist = NULL;
			}

			/* Deallocate string, it is no longer needed. */
			g_free(strptr);
			strptr = NULL;
		    }
                    break;


		   default:
		    FSeekNextLine(fp);
		    break;
		}
	    }
	}

	/* Close configuration file. */
	FClose(fp);
	fp = NULL;

	return(0);
}


/*
 *	Saves configuration in the sequential parm list to the
 *	specified file.
 *
 *      The given sequential parm list must have the last structure
 *      unit all 0's to indicate end of the list.
 */
gint EDVCFGSaveToFile(
	const gchar *filename, edv_cfg_item_struct *cfg_list
)
{
	gint i;
	gchar *parent_path;
        FILE *fp;
        struct stat stat_buf;
	edv_cfg_item_struct *ci;
	edv_intlist_struct *intlist;


        if((filename == NULL) || (cfg_list == NULL))
            return(-1);

        if(!stat(filename, &stat_buf))
        {
            if(S_ISDIR(stat_buf.st_mode))
            {
                fprintf(
                    stderr,
                    "%s: Cannot write to a directory.\n",
                    filename
                );
                return(-1);
	    }
        }

	/* Get parent directory and create it as needed. */
	parent_path = GetParentDir(filename);
	parent_path = (parent_path != NULL) ? g_strdup(parent_path) : NULL;
	if(parent_path != NULL)
	{
	    rmkdir(parent_path, S_IRUSR | S_IWUSR | S_IXUSR);
	    g_free(parent_path);
	    parent_path = NULL;
	}

	/* Open configuration file for writing. */
	fp = FOpen(filename, "wb");

	i = 0;
	do
	{
	    ci = &cfg_list[i];

	    if(ci->parameter != NULL)
	    {
		const gchar *parm = ci->parameter;
		const gpointer value = ci->value;

		switch(ci->type)
		{
		  case EDV_CFG_ITEM_TYPE_INT8:
		    fprintf(
			fp,
			"%s %c %i\n",
			parm, EDV_CFG_DELIMINATOR_CHAR,
			(gint8)((value == NULL) ? 0 : *(gint8 *)value)
		    );
		    break;
                  case EDV_CFG_ITEM_TYPE_UINT8:
                    fprintf(
			fp,
                        "%s %c %i\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (guint8)((value == NULL) ? 0 : *(guint8 *)value) 
                    ); 
                    break;

                  case EDV_CFG_ITEM_TYPE_INT16:
                    fprintf(
			fp,
                        "%s %c %i\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (gint16)((value == NULL) ? 0 : *(gint16 *)value)
                    );
                    break;
                  case EDV_CFG_ITEM_TYPE_UINT16:
                    fprintf(
			fp,
                        "%s %c %i\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (guint16)((value == NULL) ? 0 : *(guint16 *)value)
                    );
                    break;

                  case EDV_CFG_ITEM_TYPE_INT32:
                    fprintf(
			fp,
                        "%s %c %i\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (gint32)((value == NULL) ? 0 : *(gint32 *)value)
                    );
                    break;
                  case EDV_CFG_ITEM_TYPE_UINT32:
                    fprintf(
			fp,
                        "%s %c %i\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (guint32)((value == NULL) ? 0 : *(guint32 *)value)
                    );
                    break;

                  case EDV_CFG_ITEM_TYPE_INT64:
                    fprintf(
			fp,
                        "%s %c %ld\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (glong)((value == NULL) ? 0 : *(gint64 *)value)
                    );
                    break;
                  case EDV_CFG_ITEM_TYPE_UINT64:
                    fprintf(
			fp,
                        "%s %c %ld\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (gulong)((value == NULL) ? 0 : *(guint64 *)value)
                    );
                    break;

                  case EDV_CFG_ITEM_TYPE_FLOAT:
                    fprintf(
			fp,
                        "%s %c %f\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (gfloat)((value == NULL) ? 0.0 : *(gfloat *)value)
                    );
                    break;
                  case EDV_CFG_ITEM_TYPE_DOUBLE:
                    fprintf(
			fp,
                        "%s %c %f\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (gdouble)((value == NULL) ? 0.0 : *(gdouble *)value)
                    );
                    break;

                  case EDV_CFG_ITEM_TYPE_STRING:
                    fprintf(
			fp,
                        "%s %c %s\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (gchar *)((value == NULL) ? "" : (gchar *)value)
                    );
                    break;

                  case EDV_CFG_ITEM_TYPE_COLOR:
                    fprintf(
			fp,
                        "%s %c %f %f %f %f\n",
                        parm, EDV_CFG_DELIMINATOR_CHAR,
                        (gfloat)((value == NULL) ? 0.0 :
			    ((edv_color_struct *)value)->r
			),
                        (gfloat)((value == NULL) ? 0.0 :
                            ((edv_color_struct *)value)->g  
                        ),
                        (gfloat)((value == NULL) ? 0.0 :
                            ((edv_color_struct *)value)->b  
                        ),
                        (gfloat)((value == NULL) ? 0.0 :
                            ((edv_color_struct *)value)->a
                        )
                    );
                    break;

                  case EDV_CFG_ITEM_TYPE_INTLIST:
                    fprintf(
                        fp,
                        "%s %c ",
                        parm, EDV_CFG_DELIMINATOR_CHAR
		    );
		    intlist = (edv_intlist_struct *)value;
		    if(intlist != NULL)
		    {
			gint n;

			for(n = 0; n < intlist->total; n++)
			    fprintf(fp, "%i ", intlist->i[n]);
		    }
		    fprintf(fp, "\n");
                    break;


		}
	    }

	    i++;

	} while(!((ci->type == 0) &&
                  (ci->parameter == NULL)
                 )
	);

	/* Close configuration file. */
	FClose(fp);
	fp = NULL;

	return(0);
}
