#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "edvtypes.h"
#include "edvdde.h"


edv_dnd_object_struct *EDVDNDObjectNew(void);
void EDVDNDObjectDelete(edv_dnd_object_struct *dnd_obj);

guint8 *EDVDNDObjectToBuffer(
	const edv_dnd_object_struct **dnd_object, gint total_dnd_objects,
	gint *buf_len
);
edv_dnd_object_struct **EDVDNDBufferToObject(
	const guint8 *buf, gint buf_len,
	gint *total_dnd_objects
);



/*
 *	Allocates a new dnd disk object structure.
 */
edv_dnd_object_struct *EDVDNDObjectNew(void)
{
	return((edv_dnd_object_struct *)g_malloc0(
	    sizeof(edv_dnd_object_struct)
	));
}

/*
 *	Deallocates the dnd disk object structure.
 */
void EDVDNDObjectDelete(edv_dnd_object_struct *dnd_obj)
{
	if(dnd_obj == NULL)
	    return;

        g_free(dnd_obj->protocol);
        g_free(dnd_obj->user);
        g_free(dnd_obj->password);
        g_free(dnd_obj->host);
	g_free(dnd_obj->full_path);

	g_free(dnd_obj);
}


/*
 *	Returns an allocated buffer containing the information described
 *	in the given list of dnd disk object structures.
 */
guint8 *EDVDNDObjectToBuffer(
        const edv_dnd_object_struct **dnd_object, gint total_dnd_objects,
        gint *buf_len
)
{
	gint i, len = 0, cur_pos, url_string_len;
	gchar *url_string, port_str[80];
	guint8 *buf = NULL, *buf_ptr;
	const edv_dnd_object_struct *dnd_obj;


	/* Reset returns. */
	if(buf_len != NULL)
	    *buf_len = len;

	/* No dnd disk object structures in given list? */
	if((dnd_object == NULL) || (total_dnd_objects < 1))
	    return(buf);

	/* Iterate through each object. */
	for(i = 0; i < total_dnd_objects; i++)
	{
	    dnd_obj = dnd_object[i];
	    if(dnd_obj == NULL)
		continue;

	    /* If this DND object has no full path then it must be
	     * skipped.
	     */
	    if(dnd_obj->full_path == NULL)
		continue;
	    if(*dnd_obj->full_path == '\0')
		continue;

	    /* Format port string if a port is given, otherwise
	     * set port string to be an empty string.
	     */
	    if(dnd_obj->port > 0)
		sprintf(port_str, ":%i", dnd_obj->port);
	    else
		*port_str = '\0';

	    /* Generate url string, note that if dnd_obj->full_path is
	     * not an absolute path then a '#' character will be used to
	     * deliminate the host address and full path.
	     */
	    if(dnd_obj->user != NULL)
                url_string = g_strdup_printf(
                    "%s://%s:%s@%s%s%s%s",
                    (dnd_obj->protocol != NULL) ?
                        dnd_obj->protocol : "file",
		    dnd_obj->user, dnd_obj->password,
                    (dnd_obj->host != NULL) ?
                        dnd_obj->host : "localhost",
		    port_str,
                    ISPATHABSOLUTE(dnd_obj->full_path) ? "" : "#",
		    dnd_obj->full_path
                );
	    else 
		url_string = g_strdup_printf(
		    "%s://%s%s%s%s",
		    (dnd_obj->protocol != NULL) ?
			dnd_obj->protocol : "file",
		    (dnd_obj->host != NULL) ?
			dnd_obj->host : "localhost",
		    port_str,
		    ISPATHABSOLUTE(dnd_obj->full_path) ? "" : "#",
		    dnd_obj->full_path
		);
	    if(url_string == NULL)
		continue;

	    /* Calculate length of this url string (including the null
	     * terminating character) and increase buffer allocation to
	     * hold the entire url string plus one null character as a
	     * deliminator.
	     */
	    cur_pos = len;		/* Record current buffer index. */
	    url_string_len = strlen(url_string) + 1;	/* Include null char. */
	    len += url_string_len;	/* Increase total buffer length. */

	    /* Increase buffer allocation. */
	    buf = (guint8 *)g_realloc(buf, len * sizeof(guint8));
	    if(buf == NULL)
	    {
		g_free(url_string);
		len = 0;
		break;
	    }

	    /* Get pointer to starting position of current buffer segment. */
	    buf_ptr = &buf[cur_pos];

	    /* Copy the url string to the current buffer segment, including
	     * the null character as a deliminator.
	     */
	    memcpy(buf_ptr, url_string, url_string_len * sizeof(guint8));


	    /* Deallocate url string. */
	    g_free(url_string);
	}

	/* Update returns. */
	if(buf_len != NULL)
	    *buf_len = len;

	return(buf);
}

/*
 *	Returns an allocated array of dnd disk object structures parssed
 *	from the given buffer.
 */
edv_dnd_object_struct **EDVDNDBufferToObject(
        const guint8 *buf, gint buf_len,
        gint *total_dnd_objects
)
{
	edv_dnd_object_struct **list = NULL, *dnd_obj;
	gchar *strptr, *strptr2;
	gchar *url_string, *protocol, *user, *password, *host, *full_path;
	gint i, port, url_string_len, total = 0;
	const guint8 *buf_ptr, *buf_end;


	/* Reset return. */
	if(total_dnd_objects != NULL)
	    *total_dnd_objects = total;

	if((buf == NULL) || (buf_len <= 0))
	    return(list);

	/* Get pointer to one unit past the end byte, so no pointer
	 * should access buf_end or past it.
	 */
	buf_end = buf + buf_len;

	/* Set buf_ptr to the start of the given buffer and begin
	 * iterating through the buffer using buf_ptr as the current
	 * buffer segment start position.
	 */
	buf_ptr = buf;
	while(buf_ptr < buf_end)
	{
	    /* Calculate the length of the url string (including the null
	     * terminating character) for this buffer segment. Note that
	     * since buf_end > buf_ptr, url_string_len will always be 1
	     * or greater.
	     */
	    url_string_len = (gint)(buf_end - buf_ptr);

	    /* Allocate and copy tempory url string.
	     *
             * Note, some applications will send buf_len to only contain
             * the string characters and not the null terminating byte,
             * in which case we should allocate one more byte just in
             * case.
	     */
	    url_string = (gchar *)g_malloc(
		(url_string_len + 1) * sizeof(gchar)
	    );
	    if(url_string == NULL)
		break;
	    strncpy(url_string, buf_ptr, url_string_len);
	    /* Null terminate the coppied tempory url string in case the
	     * given buffer was corrupt.
	     */
	    url_string[url_string_len] = '\0';

	    /* Now calculate actual length of the url string after
	     * the copy, this time we are sure we do not include the
	     * null terminating character in the count.
	     */
	    url_string_len = strlen(url_string);


	    /* Parse url string, get copies of the protocol, user,
	     * password, host, and full_path.
	     */
	    /* Parse protocol. */
	    strptr = url_string;
	    protocol = (strptr != NULL) ? g_strdup(strptr) : NULL;
	    if(protocol != NULL)
	    {
		strptr2 = strstr(protocol, "://");
		if(strptr2 != NULL)
		    *strptr2 = '\0';
	    }

	    /* Parse user/password or host. */
	    if(strptr != NULL)
		strptr = strstr(strptr, "://");
	    if(strptr != NULL)
		strptr += strlen("://");
	    /* Get entire host string now, we will check if there is
	     * a user/password or port in it later.
	     */
	    host = (strptr != NULL) ? g_strdup(strptr) : NULL;
	    if(host != NULL)
	    {
		strptr2 = strchr(host, '/');
                if(strptr2 != NULL)
                    *strptr2 = '\0';

		/* Double check if there was a # deliminator incase
		 * the upcomming path is not an absolute path.
		 */
                strptr2 = strchr(host, '#');
                if(strptr2 != NULL)
                    *strptr2 = '\0';
	    }
	    /* Check if there is a user/password in the host, in
	     * which case we need to re-parse the host string.
	     */
	    strptr2 = (host != NULL) ? strchr(host, '@') : NULL;
	    if(strptr2 != NULL)
	    {
		gchar *strptr3;

		user = g_strdup(host);
		strptr3 = strchr(user, ':');
		if(strptr3 != NULL)
		{
		    *strptr3 = '\0';
		    password = g_strdup(strptr3 + 1);
		    strptr3 = strchr(password, '@');
		    if(strptr3 != NULL)
			*strptr3 = '\0';
		}
		else
		{
		    password = NULL;
		}

		/* Reparse host, strptr2 should be untouched since
		 * entering this condition.
		 */
		strptr3 = g_strdup(strptr2 + 1);
		g_free(host);
		host = strptr3;
	    }
	    else
	    {
		user = NULL;
		password = NULL;
	    }
	    /* Check if there is a port in the host string? */
	    strptr2 = (host != NULL) ? strchr(host, ':') : NULL;
	    if(strptr2 != NULL)
	    {
		port = atoi(strptr2 + 1);
		*strptr2 = '\0';
	    }
	    else
	    {
		port = 0;
	    }

	    /* Parse full path. */
	    if(strptr != NULL)
	    {
		/* Seek strptr to start of full path. */
		while(*strptr != '\0')
		{
		    if(*strptr == '/')
		    {
			break;
		    }
		    else if(*strptr == '#')
		    {
			/* Seek past the '#' deliminator since it is not
			 * part of the path.
			 */
			strptr++;
			break;
		    }
		    else
		    {
			strptr++;
		    }
		}
	    }
	    full_path = (strptr != NULL) ? g_strdup(strptr) : NULL;


	    /* Begin allocating a new dnd disk object structure. */
	    i = total;
	    total = i + 1;
	    list = (edv_dnd_object_struct **)g_realloc(
		list,
		total * sizeof(edv_dnd_object_struct *)
	    );
	    if(list == NULL)
	    {
		g_free(full_path);
		g_free(host);
                g_free(password);
                g_free(user);
		g_free(protocol);
		g_free(url_string);

		total = 0;
		break;
	    }

	    /* Allocate a new dnd disk object structure and add it to the
	     * list.
	     */
	    list[i] = dnd_obj = EDVDNDObjectNew();
	    if(dnd_obj != NULL)
	    {
		dnd_obj->protocol = (protocol != NULL) ?
		    g_strdup(protocol) : NULL;
                dnd_obj->user = (user != NULL) ?
                    g_strdup(user) : NULL;
                dnd_obj->password = (password != NULL) ?
                    g_strdup(password) : NULL;
                dnd_obj->host = (host != NULL) ?
                    g_strdup(host) : NULL;
		dnd_obj->port = port;
                dnd_obj->full_path = (full_path != NULL) ?
                    g_strdup(full_path) : NULL;
	    }

	    /* Deallocate locally allocated strings. */
	    g_free(full_path);
	    g_free(host);
            g_free(password);
            g_free(user);
	    g_free(protocol);
	    g_free(url_string);

	    /* Increment buffer position to start of next string. */
	    buf_ptr += url_string_len + 1;
	}

	/* Update total. */
        if(total_dnd_objects != NULL)
            *total_dnd_objects = total;

	return(list);
}
