#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fnmatch.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "edvtypes.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "edvarch.h"
#include "edvarchfio.h"
#include "edvfind.h"


static gchar *EDVFindGrepExcerptFile(
        const gchar *path, const gchar *expression,
        gbool case_sensitive
);

/* Archive object finding. */
static void EDVFindArchiveObjectByNameIterate(
        edv_core_struct *core_ptr,
        const gchar *arch_path, const gchar *expression,
        gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer),
        gint *total_matches, gbool *stop_find
);
gint EDVFindArchiveObjectByName(
        edv_core_struct *core_ptr,
        const gchar *arch_path,	const gchar *expression,
        gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer)
);

/* Recycled object finding. */
static void EDVFindRecycledObjectByContentIterate(
	const gchar *recycled_index_file, const gchar *expression,
	gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(
                const gchar *, const struct stat *, const gchar *, gpointer
        ),
        gint *total_matches, gbool *stop_find
);
gint EDVFindRecycledObjectByContent(
        const gchar *recycled_index_file, const gchar *expression,
	gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(
                const gchar *, const struct stat *, const gchar *, gpointer
        )
);

static void EDVFindRecycledObjectByNameIterate(
        const gchar *recycled_index_file, const gchar *expression,
        gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer),
        gint *total_matches, gbool *stop_find
);
gint EDVFindRecycledObjectByName(
        const gchar *recycled_index_file, const gchar *expression,
        gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer)
);


/* Disk object finding. */
static void EDVFindObjectByContentIterate(
        const gchar *path, const gchar *expression,
        gbool recursive, gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(
		const gchar *, const struct stat *, const gchar *, gpointer
	),
        gint *total_matches, gbool *stop_find
);
gint EDVFindObjectByContent(
        const gchar *start_dir, const gchar *expression,
        gbool recursive, gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(
                const gchar *, const struct stat *, const gchar *, gpointer
        )
);

static void EDVFindObjectByNameIterate(
        const gchar *path, const gchar *expression,
        gbool recursive, gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer),
        gint *total_matches, gbool *stop_find
);
gint EDVFindObjectByName(
        const gchar *start_dir, const gchar *expression,
	gbool recursive, gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer)
);


#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))
#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))


/*
 *	Opens the file specified by path and checks for a string
 *	within that matches the given expression.
 *
 *	If a match is made then a dynamically allocated string will be
 *	returned containing the excerpt of the matched line.
 */
static gchar *EDVFindGrepExcerptFile(
	const gchar *path, const gchar *expression,
	gbool case_sensitive
)
{
	glong last_line_start_pos = 0;
	const gchar *expression_ptr;
	gchar *excerpt = NULL;
	gint c;
	FILE *fp;


	if((path == NULL) || (expression == NULL))
	    return(NULL);

	if(*expression == '\0')
	    return(NULL);


	/* Open file for reading. */
	fp = FOpen(path, "rb");
	if(fp == NULL)
	    return(NULL);

	expression_ptr = expression;

	while(1)
	{
	    /* Get next character. */
	    c = fgetc(fp);
	    if(c == EOF)
		break;

	    if(!case_sensitive)
		c = toupper(c);

	    /* If this character marks the end of the line then record
	     * the last line starting position as one character ahead
	     * of the current position.
	     */
	    if(ISCR(c))
		last_line_start_pos = ftell(fp);

	    /* This character matches current character of expression?
	     * If so then increment the expression pointer. Otherwise
	     * reset the expression pointer back to the start of the
	     * expression.
	     */
	    if((c == *expression_ptr) && (c != '\0'))
		expression_ptr++;
	    else
		expression_ptr = expression;

	    /* Matched entire expression? Test if the expression pointer
	     * has been incremented all the way to the null terminating
	     * character which suggests that all characters in the
	     * expression were matched.
	     */
	    if(*expression_ptr == '\0')
	    {
		/* Gather excerpt. */
		if(excerpt == NULL)
		{
		    /* Seek to start of the current line. */
		    fseek(fp, last_line_start_pos, SEEK_SET);
		    excerpt = FGetStringLiteral(fp);
		}
		break;
	    }
	}

	/* Close file. */
	FClose(fp);


	return(excerpt);
}


/*
 *      Called by EDVFindArchiveObjectByName() to perform one find iteration
 *      on the given path.
 */
static void EDVFindArchiveObjectByNameIterate(
        edv_core_struct *core_ptr,
        const gchar *arch_path, const gchar *expression,
        gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer),
        gint *total_matches, gbool *stop_find
)
{
	edv_archive_object_struct **aobj = NULL, *aobj_ptr;
	gint i, total_aobjs = 0;


        if(*stop_find)
            return;

        /* Get listing of all archive objects found in the given
	 * archive.
	 */
	aobj = EDVArchFIOGetListing(
	    core_ptr, arch_path, &total_aobjs,
	    NULL, 0
	);
	/* Iterate through archive objects. */
	for(i = 0; i < total_aobjs; i++)
	{
	    aobj_ptr = aobj[i];
	    if(aobj_ptr == NULL)
		continue;

            /* Do not continue iterating through archive objects if the
             * *stop_find was set to TRUE after a progress check.
             */
            if(*stop_find)
                break;

            /* Call progress callback? */
            if(progress_cb != NULL)
            {
                if(progress_cb(aobj_ptr->name, -1.0, client_data))
                {
                    /* Progress callback returned non-zero indicating
                     * the procedure should stop. So mark stop_find to
                     * be TRUE and do not continue with matching of this
                     * object.
                     */
                    *stop_find = TRUE;
                    continue;
                }
            }

            /* Do match. */
            if(aobj_ptr->name != NULL)
            {
                gchar *s = g_strdup(aobj_ptr->name);

                if(!case_sensitive)
                    strtoupper(s);

                if(!fnmatch(expression, s, 0))
                {
                    *total_matches = (*total_matches) + 1;

		    if(match_cb != NULL)
			match_cb(
			    aobj_ptr->full_path,
			    NULL,		/* No stats. */
			    client_data
			);
                }

                g_free(s);
            }
        }

	/* Deallocate all archive object stat structures. */
        for(i = 0; i < total_aobjs; i++)
	    EDVArchObjectDelete(aobj[i]);
	g_free(aobj);
	aobj = NULL;
	total_aobjs = 0;
}

/*
 *      Searches for a object who's name matches the given expression
 *      in the archive.
 *
 *      If progress_cb is not NULL then it will be called frequently to
 *      update progress. If progress_cb returns non-zero then the
 *      operation will end prematurly.
 *
 *      If match_cb is not NULL then each matched object will be reported
 *      to it.
 *
 *      Returns the number of occurances found.
 */
gint EDVFindArchiveObjectByName(
	edv_core_struct *core_ptr,
        const gchar *arch_path, const gchar *expression,
        gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer)
)
{
        gint i;
        gchar *lexpression;
        gbool *stop_find;
        gint *total_matches;


        if((core_ptr == NULL) || (arch_path == NULL) || (expression == NULL))
            return(0);
        if((*expression == '\0') || !strcmp(expression, "*"))
            return(0);

        /* Convert expression to upper case if not matching case
         * sensitive.
         */
        lexpression = g_strdup(expression);
        if(!case_sensitive)
            strtoupper(lexpression);

        /* Allocate total number of matches variable for passing to
         * iteration calls.
         */
        stop_find = (gbool *)g_malloc(sizeof(gbool));
        *stop_find = FALSE;

        total_matches = (gint *)g_malloc(sizeof(gint));
        *total_matches = 0;


	EDVFindArchiveObjectByNameIterate(
	    core_ptr,
            arch_path, lexpression,
            case_sensitive,
            client_data,
            progress_cb, match_cb,
            total_matches, stop_find
        );


        /* Record total matches. */
        i = *total_matches;

        /* Deallocate local resources. */
        g_free(lexpression);
        lexpression = NULL;

        g_free(stop_find);
        stop_find = NULL;

        g_free(total_matches);
        total_matches = NULL;


        return(i);
}


/*
 *	Called by EDVFindRecycledObjectByContent() to perform one find
 *	iteration on the given path.
 */
static void EDVFindRecycledObjectByContentIterate(
        const gchar *recycled_index_file, const gchar *expression,
	gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(
                const gchar *, const struct stat *, const gchar *, gpointer
        ),
        gint *total_matches, gbool *stop_find
)
{
        const gchar *recycled_dir;
        edv_recbin_index_struct *rbi_ptr;
        edv_recbin_object_struct *obj;


        if(*stop_find)
            return;

        /* Get recycled objects directory from the recycled objects
         * index file path.
         */
        recycled_dir = GetParentDir(recycled_index_file);

        /* Get recycled objects directory index pointer used to iterate
         * through all recycled objects.
         */
        rbi_ptr = EDVRecBinIndexOpen(recycled_index_file);
        while(!EDVRecBinIndexNext(rbi_ptr))
        {
            /* Get recycled object structure obtained from the recycled
             * objects directory index structure. This structure is shared
             * so do not modify or deallocate it.
             */
            obj = rbi_ptr->obj;
            if(obj == NULL)
                continue;

            /* Do not continue iterating through recycled objects if the
             * *stop_find was set to TRUE after a progress check.
             */
            if(*stop_find)
                break;

            /* Call progress callback? */
            if(progress_cb != NULL)
            {
                if(progress_cb(obj->name, -1.0, client_data))
                {
                    /* Progress callback returned non-zero indicating
                     * the procedure should stop. So mark stop_find to
                     * be TRUE and do not continue with matching of this
                     * object.
                     */
                    *stop_find = TRUE;
                    continue;
                }
            }

	    /* Do match. */
	    if(1)
	    {
		gchar *full_path, *excerpt;


		full_path = g_strdup_printf(
		    "%s%c%i",
		    recycled_dir, DIR_DELIMINATOR, obj->index
		);
		excerpt = EDVFindGrepExcerptFile(
		    full_path, expression, case_sensitive
		);
		if(excerpt != NULL)
		{
		    gchar index_str[80];
		    struct stat lstat_buf;

		    /* Got match, increment match count and call match
		     * callback. The path passed should be the object's
		     * name which is a string specifying a recycled object
		     * index number.
		     */
		    *total_matches = (*total_matches) + 1;
		    sprintf(index_str, "%i", obj->index);
		    if((match_cb != NULL) && !lstat(full_path, &lstat_buf))
			match_cb(index_str, &lstat_buf, excerpt, client_data); 

		    /* Deallocate excerpt string, it is no longer needed. */
		    g_free(excerpt);
                }

		g_free(full_path);
	    }
        }

        EDVRecBinIndexClose(rbi_ptr);
}

/*
 *      Searches for recycled object who contains a string that matches
 *	the given expression in the recycled objects directory obtained
 *	from the given recycle objects index file.
 *
 *      If progress_cb is not NULL then it will be called frequently to
 *      update progress. If progress_cb returns non-zero then the
 *      operation will end prematurly.
 *
 *      If match_cb is not NULL then each matched object will be reported
 *      to it.
 *
 *      Returns the number of occurances found.
 */
gint EDVFindRecycledObjectByContent(
	const gchar *recycled_index_file, const gchar *expression,
	gbool case_sensitive,
	gpointer client_data,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(
                const gchar *, const struct stat *, const gchar *, gpointer
        )
)
{
        gint i;
        gchar *lexpression;
        gbool *stop_find;
        gint *total_matches;


        if((recycled_index_file == NULL) || (expression == NULL))
            return(0);
        if(*expression == '\0')
            return(0);

        /* Convert expression to upper case if not matching case
         * sensitive.
         */
        lexpression = g_strdup(expression);
        if(!case_sensitive)
            strtoupper(lexpression);

        /* Allocate total number of matches variable for passing to
         * iteration calls.
         */
        stop_find = (gbool *)g_malloc(sizeof(gbool));
        *stop_find = FALSE;

        total_matches = (gint *)g_malloc(sizeof(gint));
        *total_matches = 0;


	EDVFindRecycledObjectByContentIterate(
	    recycled_index_file, lexpression,
            case_sensitive,
            client_data,
            progress_cb, match_cb,
            total_matches, stop_find
        );


        /* Record total matches. */
        i = *total_matches;

        /* Deallocate local resources. */
        g_free(lexpression);
        lexpression = NULL;

        g_free(stop_find);
        stop_find = NULL;

        g_free(total_matches);
        total_matches = NULL;


        return(i);
}


/*
 *	Called by EDVFindRecycledObjectByName() to perform one find iteration
 *	on the given path.
 */
static void EDVFindRecycledObjectByNameIterate(
        const gchar *recycled_index_file, const gchar *expression,
        gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer),
        gint *total_matches, gbool *stop_find
)
{
	const gchar *recycled_dir;
	edv_recbin_index_struct *rbi_ptr;
	edv_recbin_object_struct *obj;


        if(*stop_find)
            return;

	/* Get recycled objects directory from the recycled objects
	 * index file path.
	 */
	recycled_dir = GetParentDir(recycled_index_file);

        /* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects.
	 */
	rbi_ptr = EDVRecBinIndexOpen(recycled_index_file);
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    /* Get recycled object structure obtained from the recycled
	     * objects directory index structure. This structure is shared
	     * so do not modify or deallocate it.
	     */
	    obj = rbi_ptr->obj;
	    if(obj == NULL)
		continue;

	    /* Do not continue iterating through recycled objects if the
	     * *stop_find was set to TRUE after a progress check.
	     */
	    if(*stop_find)
		break;

	    /* Call progress callback? */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(obj->name, -1.0, client_data))
		{
		    /* Progress callback returned non-zero indicating
		     * the procedure should stop. So mark stop_find to
		     * be TRUE and do not continue with matching of this
		     * object.
		     */
		    *stop_find = TRUE;
		    continue;
		}
	    }

	    /* Do match. */
	    if(obj->name != NULL)
	    {
		gchar *s = g_strdup(obj->name);

		if(!case_sensitive)
		    strtoupper(s);

		if(!fnmatch(expression, s, 0))
		{
		    gchar *full_path;
		    gchar index_str[80];

		    /* Got match, increment match count and call match
		     * callback. The path passed should be the object's
		     * name which is a string specifying a recycled
		     * object index number.
		     */
		    *total_matches = (*total_matches) + 1;

		    sprintf(index_str, "%i", obj->index);
		    full_path = g_strdup_printf(
			"%s%c%i",
			recycled_dir, DIR_DELIMINATOR, obj->index
		    );
		    if((full_path != NULL) && (match_cb != NULL))
		    {
			struct stat lstat_buf;

			if(!lstat(full_path, &lstat_buf))
			    match_cb(index_str, &lstat_buf, client_data);
		    }

		    g_free(full_path);
		}

		g_free(s);
	    }
        }

	EDVRecBinIndexClose(rbi_ptr);
}

/*
 *	Searches for a recycled object who's name matches the given
 *	expression in the recycled objects directory obtained from the
 *	given recycled objects index file.
 *
 *      If progress_cb is not NULL then it will be called frequently to
 *      update progress. If progress_cb returns non-zero then the
 *      operation will end prematurly.
 *
 *      If match_cb is not NULL then each matched object will be reported
 *      to it.
 *
 *      Returns the number of occurances found.
 */
gint EDVFindRecycledObjectByName(
        const gchar *recycled_index_file, const gchar *expression,
        gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer)
)
{
        gint i;
        gchar *lexpression;
        gbool *stop_find;
        gint *total_matches;


        if((recycled_index_file == NULL) || (expression == NULL))
            return(0);
        if((*expression == '\0') || !strcmp(expression, "*"))
            return(0);

        /* Convert expression to upper case if not matching case
         * sensitive.
         */
        lexpression = g_strdup(expression);
        if(!case_sensitive)
            strtoupper(lexpression);

        /* Allocate total number of matches variable for passing to
         * iteration calls.
         */
        stop_find = (gbool *)g_malloc(sizeof(gbool));
        *stop_find = FALSE;

        total_matches = (gint *)g_malloc(sizeof(gint));
        *total_matches = 0;


        EDVFindRecycledObjectByNameIterate(
	    recycled_index_file, lexpression,
            case_sensitive,
            client_data,
            progress_cb, match_cb,
            total_matches, stop_find
        );


        /* Record total matches. */
        i = *total_matches;

        /* Deallocate local resources. */
        g_free(lexpression);
        lexpression = NULL;

        g_free(stop_find);
        stop_find = NULL;

        g_free(total_matches);
        total_matches = NULL;


        return(i);
}


/*
 *	Called by EDVFindObjectByContent() to perform one find iteration
 *	on the given path and its child objects if path is a directory and
 *	recurisive is TRUE.
 */
static void EDVFindObjectByContentIterate(
        const gchar *path, const gchar *expression,
        gbool recursive, gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(
                const gchar *, const struct stat *, const gchar *, gpointer
        ),
        gint *total_matches, gbool *stop_find
)
{
        gint strc;
        gchar **strv;


        if(*stop_find)
            return;

        /* Get listing of child objects if the given path is a directory. */
        strv = GetDirEntNames2(path, &strc);
        if(strv != NULL)
        {
            gint i;
            const gchar *cstrptr, *cstrptr2;
            gchar *child_path = NULL;
            struct stat stat_buf, lstat_buf;


            strv = StringQSort(strv, strc);
            if(strv == NULL)
                return;

            for(i = 0; i < strc; i++)
            {
#define DO_FREE_CONTINUE        \
{ \
 g_free(child_path); \
 child_path = NULL; \
\
 g_free(strv[i]); \
 strv[i] = NULL; \
\
 continue; \
}
                cstrptr = strv[i];
                if(cstrptr == NULL)
                    DO_FREE_CONTINUE

                /* Do not check this object if stop find has been
                 * specified from the progress callback.
                 */
                if(*stop_find)
                    DO_FREE_CONTINUE

                /* Skip special directory notations. */
                if(!strcmp(cstrptr, "..") ||
                   !strcmp(cstrptr, ".")
                )
                    DO_FREE_CONTINUE

                /* Get full path to child object as child_path. */
                cstrptr2 = PrefixPaths(path, cstrptr);
                child_path = (cstrptr2 != NULL) ?
                    g_strdup(cstrptr2) : NULL;
                if(child_path == NULL)
                    DO_FREE_CONTINUE

                /* Call progress callback? */
                if(progress_cb != NULL)
                {
                    if(progress_cb(child_path, -1.0, client_data))
                    {
                        /* Progress callback returned non-zero indicating
                         * the procedure should stop. So mark stop_find to
                         * be TRUE and do not continue with matching of this
                         * object.
                         */
                        *stop_find = TRUE;
                        DO_FREE_CONTINUE
                    }
                }

                /* Do match. */
                if(1)
                {
                    gchar *excerpt = EDVFindGrepExcerptFile(
			child_path, expression,
			case_sensitive
		    );
		    if(excerpt != NULL)
		    {
                        /* Got match, increment match count and call match
                         * callback.
                         */
                        *total_matches = (*total_matches) + 1;
                        if((match_cb != NULL) && !lstat(child_path, &lstat_buf))
                            match_cb(child_path, &lstat_buf, excerpt, client_data);

			/* Deallocate excerpt string, it is no longer needed. */
			g_free(excerpt);
                    }
                }


                /* Get destination stats. */
                if(stat(child_path, &stat_buf))
                    DO_FREE_CONTINUE



                /* Is it a directory and should we recurse into it? */
                if(recursive && S_ISDIR(stat_buf.st_mode))
                {
                    EDVFindObjectByContentIterate(
                        child_path, expression,
                        recursive, case_sensitive,
                        client_data,
                        progress_cb, match_cb,
                        total_matches, stop_find
                    );
                }

                DO_FREE_CONTINUE
#undef DO_FREE_CONTINUE
            }

            g_free(strv);
            strv = NULL;
        }
}

/*
 *	Searches for an object who contains a string that matches the
 *	given expression starting from the directory specified by
 *	start_dir.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindObjectByContent(
        const gchar *start_dir, const gchar *expression,
        gbool recursive, gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(
                const gchar *, const struct stat *, const gchar *, gpointer
        )
)
{
        gint i;
        gchar *lexpression;
        gbool *stop_find;
        gint *total_matches;


        if((start_dir == NULL) || (expression == NULL))
            return(0);
        if(*expression == '\0')
            return(0);

        /* Convert expression to upper case if not matching case
         * sensitive.
         */
        lexpression = g_strdup(expression);
        if(!case_sensitive)
            strtoupper(lexpression);

        /* Allocate total number of matches variable for passing to
         * iteration calls.
         */
        stop_find = (gbool *)g_malloc(sizeof(gbool));
        *stop_find = FALSE;

        total_matches = (gint *)g_malloc(sizeof(gint));
        *total_matches = 0;


        EDVFindObjectByContentIterate(
            start_dir, lexpression,
            recursive, case_sensitive,
            client_data,
            progress_cb, match_cb,
            total_matches, stop_find
        );


        /* Record total matches. */
        i = *total_matches;

        /* Deallocate local resources. */
        g_free(lexpression);
        lexpression = NULL;

        g_free(stop_find);
        stop_find = NULL;

        g_free(total_matches);
        total_matches = NULL;


        return(i);
}


/*
 *	Called by EDVFindObjectByName() to perform one find iteration on
 *	the given path and its child objects if path is a directory and
 *	recurisive is TRUE.
 */
static void EDVFindObjectByNameIterate(
	const gchar *path, const gchar *expression,
	gbool recursive, gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gint *total_matches, gbool *stop_find
)
{
	gint strc;
	gchar **strv;


	if(*stop_find)
	    return;

	/* Get listing of child objects if the given path is a directory. */
	strv = GetDirEntNames2(path, &strc);
	if(strv != NULL)
	{
	    gint i;
	    const gchar *cstrptr, *cstrptr2;
	    gchar *child_path = NULL;
	    struct stat stat_buf, lstat_buf;


	    strv = StringQSort(strv, strc);
	    if(strv == NULL)
		return;

	    for(i = 0; i < strc; i++)
	    {
#define DO_FREE_CONTINUE	\
{ \
 g_free(child_path); \
 child_path = NULL; \
\
 g_free(strv[i]); \
 strv[i] = NULL; \
\
 continue; \
}
		cstrptr = strv[i];
		if(cstrptr == NULL)
		    DO_FREE_CONTINUE

		/* Do not check this object if stop find has been
		 * specified from the progress callback.
		 */
		if(*stop_find)
		    DO_FREE_CONTINUE

		/* Skip special directory notations. */
		if(!strcmp(cstrptr, "..") ||
		   !strcmp(cstrptr, ".")
		)
		    DO_FREE_CONTINUE

		/* Get full path to child object as child_path. */
		cstrptr2 = PrefixPaths(path, cstrptr);
		child_path = (cstrptr2 != NULL) ?
		    g_strdup(cstrptr2) : NULL;
		if(child_path == NULL)
		    DO_FREE_CONTINUE

		/* Call progress callback? */
		if(progress_cb != NULL)
                {
                    if(progress_cb(child_path, -1.0, client_data))
                    {
			/* Progress callback returned non-zero indicating
			 * the procedure should stop. So mark stop_find to
			 * be TRUE and do not continue with matching of this
			 * object.
			 */
                        *stop_find = TRUE;
                        DO_FREE_CONTINUE
                    }
                }

		/* Do match. */
		if(1)
		{
		    gchar *s = g_strdup(cstrptr);

		    if(!case_sensitive)
			strtoupper(s);

		    if(!fnmatch(expression, s, 0))
		    {
			/* Got match, increment match count and call match
			 * callback.
			 */
			*total_matches = (*total_matches) + 1;
			if((match_cb != NULL) && !lstat(child_path, &lstat_buf))
			    match_cb(child_path, &lstat_buf, client_data);
		    }

		    g_free(s);
		}


		/* Get destination stats. */
		if(stat(child_path, &stat_buf))
		    DO_FREE_CONTINUE

		/* Is it a directory and should we recurse into it? */
		if(recursive && S_ISDIR(stat_buf.st_mode))
		{
		    EDVFindObjectByNameIterate(
			child_path, expression,
			recursive, case_sensitive,
			client_data,
			progress_cb, match_cb,
			total_matches, stop_find
		    );
		}

		DO_FREE_CONTINUE
#undef DO_FREE_CONTINUE
	    }

	    g_free(strv);
	    strv = NULL;
	}
}

/*
 *	Searches for an object who's name matches the given expression
 *	starting from the directory specified by start_dir.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindObjectByName(
        const gchar *start_dir, const gchar *expression,
        gbool recursive, gbool case_sensitive,
        gpointer client_data,
        gint (*progress_cb)(const gchar *, gfloat, gpointer),
        void (*match_cb)(const gchar *, const struct stat *, gpointer)
)
{
	gint i;
	gchar *lexpression;
	gbool *stop_find;
	gint *total_matches;


	if((start_dir == NULL) || (expression == NULL))
	    return(0);
	if((*expression == '\0') || !strcmp(expression, "*"))
	    return(0);

	/* Convert expression to upper case if not matching case
	 * sensitive.
	 */
	lexpression = g_strdup(expression);
	if(!case_sensitive)
	    strtoupper(lexpression);

	/* Allocate total number of matches variable for passing to
	 * iteration calls.
	 */
	stop_find = (gbool *)g_malloc(sizeof(gbool));
	*stop_find = FALSE;

	total_matches = (gint *)g_malloc(sizeof(gint));
	*total_matches = 0;


	EDVFindObjectByNameIterate(
	    start_dir, lexpression,
	    recursive, case_sensitive,
	    client_data,
	    progress_cb, match_cb,
	    total_matches, stop_find
	);


	/* Record total matches. */
	i = *total_matches;

	/* Deallocate local resources. */
	g_free(lexpression);
	lexpression = NULL;

	g_free(stop_find);
	stop_find = NULL;

	g_free(total_matches);
	total_matches = NULL;


	return(i);
}
