#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <glib.h>

#include "edvtypes.h"
#include "edvid.h"


/* User ID functions. */
edv_uid_struct *EDVUIDNew(void);
void EDVUIDDelete(edv_uid_struct *uid);

edv_uid_struct **EDVUIDGetSystem(gint *total);

edv_uid_struct *EDVUIDMatchByUID(
        edv_uid_struct **list, gint total,
        gint user_id, gint *n
);
edv_uid_struct *EDVUIDMatchByGID(
        edv_uid_struct **list, gint total,
        gint group_id, gint *n
);
gint EDVUIDNameToUID(
        edv_uid_struct **list, gint total, const gchar *user_name
);
gchar *EDVUIDGetNameFromUID(
        edv_uid_struct **list, gint total,
        gint user_id, gint *n
);
gchar *EDVUIDGetNameFromGID(
        edv_uid_struct **list, gint total,
        gint group_id, gint *n
);

/* Group ID functions. */
edv_gid_struct *EDVGIDNew(void);
void EDVGIDDelete(edv_gid_struct *gid);

edv_gid_struct **EDVGIDGetSystem(gint *total);

edv_gid_struct *EDVGIDMatchByGID(
        edv_gid_struct **list, gint total,
        gint group_id, gint *n
);
gint EDVGIDNameToGID(
        edv_gid_struct **list, gint total, const gchar *group_name
);
gchar *EDVGIDGetNameFromGID(
        edv_gid_struct **list, gint total,
        gint group_id, gint *n
);




/*
 *	Creates a new edv_uid_struct.
 */
edv_uid_struct *EDVUIDNew(void)
{
	edv_uid_struct *uid = (edv_uid_struct *)g_malloc0(
	    sizeof(edv_uid_struct)
	);
	if(uid == NULL)
	    return(uid);

	uid->name = NULL;
	uid->password = NULL;

	uid->user_id = 0;
	uid->group_id = 0;

	uid->government_name = NULL;
	uid->home_directory = NULL;
	uid->shell_program = NULL;

	return(uid);
}

/*
 *	Deallocates all resources of the given uid and deallocates the
 *	structure itself.
 */
void EDVUIDDelete(edv_uid_struct *uid)
{
	if(uid == NULL)
	    return;

	g_free(uid->name);
	g_free(uid->password);

        g_free(uid->government_name);
        g_free(uid->home_directory);
        g_free(uid->shell_program);

	g_free(uid);
}

/*
 *	Returns a dynamically allocated list of uid structures found
 *	on the system.
 *
 *	The calling function must deallocate each structure in the
 *	returned list and the pointer array.
 */
edv_uid_struct **EDVUIDGetSystem(gint *total)
{
	gint i;
	edv_uid_struct **list = NULL, *uid;
        struct passwd *pwent;


	if(total == NULL)
	    return(NULL);

	*total = 0;


        /* Get first entry. */
        pwent = getpwent();

	/* Iterate through each entry. */
        while(pwent != NULL)
        {
	    /* Allocate more pointers. */
	    i = *total;
	    *total = i + 1;
	    list = (edv_uid_struct **)g_realloc(
		list, (*total) * sizeof(edv_uid_struct *)
	    );
	    if(list == NULL)
	    {
		*total = 0;
		break;
	    }

	    /* Allocate a new structure. */
	    list[i] = uid = EDVUIDNew();
	    if(uid != NULL)
	    {
#define SAFE_STRDUP(s)	\
(((s) != NULL) ? g_strdup((const gchar *)(s)) : NULL)

		g_free(uid->name);
		uid->name = SAFE_STRDUP(pwent->pw_name);

                g_free(uid->password);
                uid->password = SAFE_STRDUP(pwent->pw_passwd);

                uid->user_id = pwent->pw_uid;
                uid->group_id = pwent->pw_gid;

                g_free(uid->government_name);
                uid->government_name = SAFE_STRDUP(pwent->pw_gecos);

                g_free(uid->home_directory);
                uid->home_directory = SAFE_STRDUP(pwent->pw_dir);

                g_free(uid->shell_program);
                uid->shell_program = SAFE_STRDUP(pwent->pw_shell);
#undef SAFE_STRDUP
	    }

            /* Get next password entry entry. */
            pwent = getpwent();
        }

        /* Close the password entry file. */
        endpwent();

	return(list);
}


/*
 *	Matches the uid structure in the given list who's user_id matches
 *	the given user_id.
 */
edv_uid_struct *EDVUIDMatchByUID(
        edv_uid_struct **list, gint total,
        gint user_id, gint *n
)
{
	gint i;
	edv_uid_struct *uid;


        if(n != NULL)
            *n = -1;

	for(i = 0; i < total; i++)
	{
	    uid = list[i];
	    if(uid == NULL)
		continue;

	    if(uid->user_id == user_id)
	    {
		if(n != NULL)
		    *n = i;
		return(uid);
	    }
	}

	return(NULL);
}

/*
 *      Matches the uid structure in the given list who's group_id matches
 *      the given group_id.
 */
edv_uid_struct *EDVUIDMatchByGID(
        edv_uid_struct **list, gint total,
        gint group_id, gint *n
)
{
        gint i;
        edv_uid_struct *uid;


        if(n != NULL)
            *n = -1;

        for(i = 0; i < total; i++)
        {
            uid = list[i];
            if(uid == NULL)
                continue;

            if(uid->group_id == group_id)
            {
                if(n != NULL)
                    *n = i;
                return(uid);
            }
        }

        return(NULL);
}

/*
 *      Matches the given user name with a uid structure, returns the
 *	uid value of the matched user name.
 */
gint EDVUIDNameToUID(
        edv_uid_struct **list, gint total, const gchar *user_name
)
{
        gint i;
        edv_uid_struct *uid;


	if(user_name == NULL)
	    return(0);

	/* Match by number? */
	if(isdigit(*user_name))
	{
	    return(atoi(user_name));
	}
	else
	{
	    for(i = 0; i < total; i++)
	    {
		uid = list[i];
		if(uid == NULL)
		    continue;

                if(uid->name == NULL)
		    continue;

		if(!strcmp(uid->name, user_name))
		    return(uid->user_id);
            }
        }

        return(0);
}


/*
 *      Matches and returns the name found on the matched uid structure
 *	matched by the given user_id.
 *
 *	On failed matched, a number string containing the user_id is
 *	returned.
 *
 *	The returned pointer must not be deallocated.
 *	This function never returns NULL.
 */
gchar *EDVUIDGetNameFromUID(
        edv_uid_struct **list, gint total,
        gint user_id, gint *n
)
{
	edv_uid_struct *uid;
        static gchar num_str[80];

	uid = EDVUIDMatchByUID(list, total, user_id, n);
	if((uid != NULL) ? (uid->name != NULL) : FALSE)
	{
	    return(uid->name);
	}
	else
	{
	    sprintf(num_str, "%i", user_id);
	    return(num_str);
	}
}

/*
 *      Matches and returns the name found on the matched uid structure
 *      matched by the given group_id.
 *
 *      On failed matched, a number string containing the group_id is
 *	returned.
 *
 *      The returned pointer must not be deallocated.
 *      This function never returns NULL.
 */
gchar *EDVUIDGetNameFromGID(
        edv_uid_struct **list, gint total,
        gint group_id, gint *n
)
{
        edv_uid_struct *uid;
        static gchar num_str[80];

        uid = EDVUIDMatchByGID(list, total, group_id, n);
        if((uid != NULL) ? (uid->name != NULL) : FALSE)
        {
            return(uid->name);
        }
        else
        {
            sprintf(num_str, "%i", group_id);
            return(num_str);
        }
}



/*
 *	Allocates a new edv_gid_struct.
 */
edv_gid_struct *EDVGIDNew(void)
{
        edv_gid_struct *gid = (edv_gid_struct *)g_malloc0(
            sizeof(edv_gid_struct)
        );
        if(gid == NULL)
            return(gid);

        gid->name = NULL;
        gid->password = NULL;

        gid->group_id = 0;

        gid->group_member = NULL;
	gid->total_group_members = 0;

        return(gid);
}

/*
 *      Deallocates all resources of the given gid and deallocates the
 *      structure itself.
 */
void EDVGIDDelete(edv_gid_struct *gid)
{
	gint i;


        if(gid == NULL)
            return;

        g_free(gid->name);
        g_free(gid->password);

	for(i = 0; i < gid->total_group_members; i++)
	    g_free(gid->group_member[i]);
	g_free(gid->group_member);
	gid->group_member = NULL;
	gid->total_group_members = 0;

        g_free(gid);
}

/*
 *      Returns a dynamically allocated list of gid structures found
 *      on the system.
 *
 *      The calling function must deallocate each structure in the
 *      returned list and the pointer array.
 */
edv_gid_struct **EDVGIDGetSystem(gint *total)
{
        gint i;
        edv_gid_struct **list = NULL, *gid;
	struct group *grent;


        if(total == NULL)
            return(NULL);

        *total = 0;


        /* Get first entry. */
        grent = getgrent();

        /* Iterate through each entry. */
        while(grent != NULL)
        {
            /* Allocate more pointers. */
            i = *total;
            *total = i + 1;
            list = (edv_gid_struct **)g_realloc(
                list, (*total) * sizeof(edv_gid_struct *)
            );
            if(list == NULL)
            {
                *total = 0;
                break;
            }

            /* Allocate a new structure. */
            list[i] = gid = EDVGIDNew();
            if(gid != NULL)
            {
#define SAFE_STRDUP(s)  \
(((s) != NULL) ? g_strdup((const gchar *)(s)) : NULL)

                g_free(gid->name);
                gid->name = SAFE_STRDUP(grent->gr_name);

                g_free(gid->password);
                gid->password = SAFE_STRDUP(grent->gr_passwd);

                gid->group_id = grent->gr_gid;

		/* Any group members to add? */
		if(grent->gr_mem != NULL)
		{
		    gint j = 0, m;
		    for(j = 0; grent->gr_mem[j] != NULL; j++)
		    {
			m = gid->total_group_members;
			gid->total_group_members = m + 1;
			gid->group_member = (gchar **)g_realloc(
			    gid->group_member,
			    gid->total_group_members * sizeof(gchar *)
			);
			if(gid->group_member == NULL)
			{
			    gid->total_group_members = 0;
			    break;
			}
			gid->group_member[m] = SAFE_STRDUP(grent->gr_mem[j]);
		    }
		}
#undef SAFE_STRDUP
            }

            /* Get next group entry. */
            grent = getgrent();
        }

        /* Close the group entry file. */
        endgrent();

        return(list);
}

/*
 *      Matches the gid structure in the given list who's group_id matches
 *      the given group_id.
 */
edv_gid_struct *EDVGIDMatchByGID(
        edv_gid_struct **list, gint total,
        gint group_id, gint *n
)
{
        gint i;
        edv_gid_struct *gid;


        if(n != NULL)
            *n = -1;

        for(i = 0; i < total; i++)
        {
            gid = list[i];
            if(gid == NULL)
                continue;

            if(gid->group_id == group_id)
            {
                if(n != NULL)
                    *n = i;
                return(gid);
            }
        }

        return(NULL);
}

/*
 *      Matches the given group name with a gid structure, returns the
 *      gid value of the matched group name.
 */
int EDVGIDNameToGID(
        edv_gid_struct **list, gint total, const gchar *group_name
)
{
        gint i;
        edv_gid_struct *gid;


        if(group_name == NULL)
            return(0);

        /* Match by number? */
        if(isdigit(*group_name))
        {
            return(atoi(group_name));
        }
        else
        {
            for(i = 0; i < total; i++)
            {
                gid = list[i];
                if(gid == NULL)
                    continue;

                if(gid->name == NULL)
                    continue;

                if(!strcmp(gid->name, group_name))
                    return(gid->group_id);
            }
        }

        return(0);
}


/*
 *      Matches and returns the name found on the matched gid structure
 *      matched by the given group_id.
 *
 *      On failed matched, a number string containing the group_id is
 *      returned.
 *
 *      The returned pointer must not be deallocated.
 *      This function never returns NULL.
 */
gchar *EDVGIDGetNameFromGID(
        edv_gid_struct **list, gint total,
        gint group_id, gint *n
)
{
        edv_gid_struct *gid;
	static gchar num_str[80];

        gid = EDVGIDMatchByGID(list, total, group_id, n);
        if((gid != NULL) ? (gid->name != NULL) : FALSE)
        {
            return(gid->name);
	}
        else
	{
            sprintf(num_str, "%i", group_id);
            return(num_str);
	}
}
