#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "edvcfg.h"
#include "edvinterps.h"
#include "edvcfglist.h"
#include "config.h"


gint EDVInterPSGetLock(edv_cfg_item_struct *cfg_list);
void EDVInterPSMakeLock(
	edv_cfg_item_struct *cfg_list,
	gint p, gbool force
);
void EDVInterPSRemoveLock(edv_cfg_item_struct *cfg_list);

gbool EDVInterPSHaveCommand(edv_cfg_item_struct *cfg_list);
void EDVInterPSSendCommand(
	edv_cfg_item_struct *cfg_list,
	gint p, gchar **cmdv, gint cmdc
);
gchar **EDVInterPSGetCommand(
	edv_cfg_item_struct *cfg_list,
	gint *total
);
void EDVInterPSRemoveCommand(edv_cfg_item_struct *cfg_list);


/*
 *	Returns the pid of the currently running process of Endeavour or
 *	0 if it is not running.  The lock link will be checked and the
 *	actual process will be checked to see if it is actually running.
 */
gint EDVInterPSGetLock(edv_cfg_item_struct *cfg_list)
{
	const gchar *cstrptr;
	gchar *lock_link;
	gchar num_str[80];
	gint bytes_read;
	pid_t p;


	cstrptr = EDVCFGItemListGetValueS(
            cfg_list, EDV_CFG_PARM_DIR_LOCAL
        );
	lock_link = g_strdup_printf(
	    "%s%c%s",
	    (cstrptr != NULL) ? cstrptr : "/",
	    DIR_DELIMINATOR,
	    EDV_INTERPS_LOCK_LINK
	);

	/* Lock link does not exist? */
	bytes_read = readlink(lock_link, num_str, 80);
	if(bytes_read <= 0)
	{
	    g_free(lock_link);
	    return(0);
	}
	if(bytes_read >= 80)
	    bytes_read = 80 - 1;
	num_str[bytes_read] = '\0';

	g_free(lock_link);	/* Don't need this anymore. */
	lock_link = NULL;

	/* Get pid from the link's destination and check if it is
	 * actually running.
	 */
	p = atoi(num_str);
	if(ExecProcessExists(p))
	    return((gint)p);
	else
	    return(0);
}

/*
 *	Creates a new lock link that points to the pid p, the link will
 *	be tested to see if it already exists. If it does not exist or
 *	force is TRUE then a new lock will be created and linked to the
 *	pid of this process.
 */
void EDVInterPSMakeLock(
	edv_cfg_item_struct *cfg_list,
	gint p, gbool force
)
{
        const gchar *cstrptr;
        gchar *lock_link;
        gchar num_str[80];


        cstrptr = EDVCFGItemListGetValueS(
            cfg_list, EDV_CFG_PARM_DIR_LOCAL
        );
        lock_link = g_strdup_printf(
            "%s%c%s",
            (cstrptr != NULL) ? cstrptr : "/",
            DIR_DELIMINATOR,
            EDV_INTERPS_LOCK_LINK
        );
	/* Lock link already exists and not forcing? */
	if(!access(lock_link, F_OK) && !force)
	{
	    g_free(lock_link);
	    return;
	}

	/* Format the link destination, which is the pid of this
	 * process.
	 */
	sprintf(num_str, "%i", p);

	/* (Re)create the lock link. */
	unlink(lock_link);
	symlink(num_str, lock_link);


	g_free(lock_link);
	lock_link = NULL;
}

/*
 *	Removes the lock link.
 */
void EDVInterPSRemoveLock(edv_cfg_item_struct *cfg_list)
{
        const gchar *cstrptr;
        gchar *lock_link;


        cstrptr = EDVCFGItemListGetValueS(
            cfg_list, EDV_CFG_PARM_DIR_LOCAL
        );
        lock_link = g_strdup_printf(
            "%s%c%s",
            (cstrptr != NULL) ? cstrptr : "/",
            DIR_DELIMINATOR,
            EDV_INTERPS_LOCK_LINK
        );

	/* Remove lock link. */
	unlink(lock_link);
	g_free(lock_link);
	lock_link = NULL;
}


/*
 *	Checks if there is a interprocess command file, returns TRUE if
 *	there is one.
 */
gbool EDVInterPSHaveCommand(edv_cfg_item_struct *cfg_list)
{
	gint status;
        const gchar *cstrptr;
        gchar *cmd_file;

        cstrptr = EDVCFGItemListGetValueS(
            cfg_list, EDV_CFG_PARM_DIR_LOCAL
        );
        cmd_file = g_strdup_printf(
            "%s%c%s",
            (cstrptr != NULL) ? cstrptr : "/",
            DIR_DELIMINATOR,
            EDV_INTERPS_CMD_FILE
        );

	status = access(cmd_file, F_OK);

        g_free(cmd_file);

	return(status ? FALSE : TRUE);
}

/*
 *	Sends the given command cmd to the process specified by pid.
 *	First the interprocess command file will be created, containing
 *	the given command cmd, then a signal SIGUSR1 will be sent to the
 *	specified pid.
 *
 *	If a command has already been queued (a command file already
 *	exists) then it will be overwritten).
 *
 *	A newline character '\n' will be appended to the given command.
 */
void EDVInterPSSendCommand(
	edv_cfg_item_struct *cfg_list,
	gint p, gchar **cmdv, gint cmdc
)
{
        const gchar *cstrptr;
	gint i;
        gchar *cmd_file;
        FILE *fp;


        if((p <= 0) || (cmdv == NULL) || (cmdc <= 0))
            return;

        cstrptr = EDVCFGItemListGetValueS(
	    cfg_list, EDV_CFG_PARM_DIR_LOCAL
        );
        cmd_file = g_strdup_printf(
            "%s%c%s",
            (cstrptr != NULL) ? cstrptr : "/",
            DIR_DELIMINATOR,
            EDV_INTERPS_CMD_FILE
        );

	fp = FOpen(cmd_file, "ab");
	if(fp != NULL)
	{
	    for(i = 0; i < cmdc; i++)
	    {
		if(cmdv[i] != NULL)
		    fprintf(fp, "%s\n", cmdv[i]);
	    }
	    FClose(fp);
#ifdef SIGUSR1
	    kill(p, SIGUSR1);
#endif
	}

        g_free(cmd_file);
}

/*
 *	Returns a string containing the oldest (first) command obtained
 *	from the interprocess command file or NULL if there isn't any.
 *
 *	The returned string must be deallocated by the calling function.
 */
gchar **EDVInterPSGetCommand(
	edv_cfg_item_struct *cfg_list,
	gint *total
)
{
	const gchar *cstrptr;
	gchar *cmd_file;
	FILE *fp;
	gchar *buf;
	gint buf_len;
	gchar **cmdv = NULL;
	gint cmdc = 0;
	struct stat stat_buf;


	if(total != NULL)
	    *total = cmdc;

        cstrptr = EDVCFGItemListGetValueS(
            cfg_list, EDV_CFG_PARM_DIR_LOCAL
        );
	cmd_file = g_strdup_printf(
            "%s%c%s",
            (cstrptr != NULL) ? cstrptr : "/",
            DIR_DELIMINATOR,
            EDV_INTERPS_CMD_FILE
        );

        /* Open command file for reading. */
        fp = FOpen(cmd_file, "rb");
        if(fp == NULL)
	{
            g_free(cmd_file);
            return(cmdv);
        }
	if(fstat(fileno(fp), &stat_buf))
	{
	    FClose(fp);
	    g_free(cmd_file);
	    return(cmdv);
	}

	/* Calculate the required buffer length and allocate buffer. */
	buf_len = stat_buf.st_size + 1;
	if(buf_len < 1)
	    buf_len = 1;
	buf = (gchar *)g_malloc(buf_len * sizeof(gchar));
	if(buf == NULL)
	{
	    FClose(fp);
	    g_free(cmd_file);
            return(cmdv);
	}

	/* Read entire contents of command file into buffer and close
	 * the file.
	 */
	if(buf_len > 1)
	    fread(buf, sizeof(gchar), buf_len - 1, fp);
	buf[buf_len - 1] = '\0';
	FClose(fp);

	/* Explode commands at newline deliminators. */
	cmdv = strchrexp(buf, '\n', &cmdc);
	if((cmdv != NULL) && (total != NULL))
	    *total = cmdc;

	g_free(buf);
	g_free(cmd_file);

	return(cmdv);
}

/*
 *	Removes the interprocess command file.
 */
void EDVInterPSRemoveCommand(edv_cfg_item_struct *cfg_list)
{
        const gchar *cstrptr;
        gchar *cmd_file;


        cstrptr = EDVCFGItemListGetValueS(
            cfg_list, EDV_CFG_PARM_DIR_LOCAL
        );
        cmd_file = g_strdup_printf(
            "%s%c%s",
            (cstrptr != NULL) ? cstrptr : "/",
            DIR_DELIMINATOR,
            EDV_INTERPS_CMD_FILE
        );

	unlink(cmd_file);

	g_free(cmd_file);
}
