/*
                     Endeavour Mime Types Structure
 */

#ifndef EDVMIMETYPES_H
#define EDVMIMETYPES_H

#include <sys/stat.h>
#include <gtk/gtk.h>
#include "edvtypes.h"


/*
 *	MIME Type classes:
 */
#define EDV_MIMETYPE_CLASS_SYSTEM	0	/* System object type. */
#define EDV_MIMETYPE_CLASS_FORMAT	1	/* File format. */
#define EDV_MIMETYPE_CLASS_PROGRAM	2	/* Program object. */
#define EDV_MIMETYPE_CLASS_UNIQUE	3	/* Unique object. */


/* Standard MIME Type strings, these are strings used for member type of
 * the edv_mimetype_struct. MIME Types of these types are created internally
 * at startup by the program (not loaded from file) and always of the
 * category EDV_MIMETYPE_CLASS_SYSTEM.
 */
#define EDV_MIMETYPE_STR_FILE		"file/regular"
#define EDV_MIMETYPE_STR_FILE_EXECUTABLE	"file/executable"
#define EDV_MIMETYPE_STR_DIRECTORY	"directory/regular"
#define EDV_MIMETYPE_STR_LINK		"link/symbolic"
#define EDV_MIMETYPE_STR_DEVICE_BLOCK	"device/block"
#define EDV_MIMETYPE_STR_DEVICE_CHARACTER	"device/character"
#define EDV_MIMETYPE_STR_FIFO		"fifo"
#define EDV_MIMETYPE_STR_SOCKET		"socket"


/*
 *	MIME Type handler codes:
 */
#define EDV_MIMETYPE_HANDLER_COMMAND		0	/* Use defined command. */
#define EDV_MIMETYPE_HANDLER_EDV_ARCHIVER	1
#define EDV_MIMETYPE_HANDLER_EDV_IMAGE_BROWSER	2
#define EDV_MIMETYPE_HANDLER_EDV_RECYCLE_BIN	3


/*
 *	MIME Type structure:
 */
typedef struct {

	gint mt_class;		/* One of EDV_MIMETYPE_CLASS_*. */

	/* String value, its intent is determined by mt_class:
	 *
	 * mt_class			value is a
	 * --------                     ----------
	 * EDV_MIMETYPE_CLASS_SYSTEM	*Ignored*
	 * EDV_MIMETYPE_CLASS_FORMAT	Space separated list of extensions
	 * EDV_MIMETYPE_CLASS_PROGRAM	Path to the program object
	 * EDV_MIMETYPE_CLASS_UNIQUE	Path to the unique object
	 */
	gchar *value;


	/* The MIME Type type, consider this the name of the entry. */
	gchar *type;

	/* A one line, null terminated, verbose description string. */
	gchar *description;


	/* Indicates if this MIME Type structure has been realized,
	 * realized meaning icons and related resources are all loaded.
	 */
	gbool realized;

	/* Indicates that this MIME Type is either created internally or
	 * loaded from a global configuration. Either of which cannot be
	 * modified or removed.
	 */
	gbool read_only;

	/* Each icon comes in sets of 3's. Index 0 is for the normal/closed
	 * version, index 1 is for the opened/selected version, and index 2
	 * is for the highlighted/active version.
	 */
#define EDV_MIMETYPE_ICON_STATE_STANDARD	0
#define EDV_MIMETYPE_ICON_STATE_SELECTED	1	/* Or expanded. */
#define EDV_MIMETYPE_ICON_STATE_EXTENDED	2	/* Reserved. */
#define EDV_MIMETYPE_TOTAL_ICON_STATES		3
	/* Small 20x20 icons. */
	GdkPixmap *small_pixmap[EDV_MIMETYPE_TOTAL_ICON_STATES];
	GdkBitmap *small_mask[EDV_MIMETYPE_TOTAL_ICON_STATES];
	gchar *small_icon_file[EDV_MIMETYPE_TOTAL_ICON_STATES];

	/* Medium 32x32 icons. */
        GdkPixmap *medium_pixmap[EDV_MIMETYPE_TOTAL_ICON_STATES];
        GdkBitmap *medium_mask[EDV_MIMETYPE_TOTAL_ICON_STATES];
	gchar *medium_icon_file[EDV_MIMETYPE_TOTAL_ICON_STATES];

	/* Large 48x48 icons. */
        GdkPixmap *large_pixmap[EDV_MIMETYPE_TOTAL_ICON_STATES];
        GdkBitmap *large_mask[EDV_MIMETYPE_TOTAL_ICON_STATES];
	gchar *large_icon_file[EDV_MIMETYPE_TOTAL_ICON_STATES];


	/* Handler, one of EDV_MIMETYPE_HANDLER_*. This is basically
	 * to determine if a command should be used to open the object
	 * associated with this MIME Type (in the case of
	 * EDV_MIMETYPE_HANDLER_COMMAND) or to use one of Endeavour's
	 * windows to open the object..
	 */
	gint handler;

	/* Command strings and corresponding command name strings, the
	 * first index is usually considered the default.
	 *
	 * Command are only used if the handler is set to
	 * EDV_MIMETYPE_HANDLER_COMMAND.
	 *
	 * The intention of each command is related by the mt_class as
	 * follows:
	 *
         * mt_class			value is a
         * --------			----------
         * EDV_MIMETYPE_CLASS_SYSTEM	*Ignored*
         * EDV_MIMETYPE_CLASS_FORMAT	Program commands or program aliases
         * EDV_MIMETYPE_CLASS_PROGRAM	Program commands (no aliases)
         * EDV_MIMETYPE_CLASS_UNIQUE	Program commands or program aliases
	 */
	gchar **command;
	gchar **command_name;
	gint total_commands;

} edv_mimetype_struct;


extern edv_mimetype_struct *EDVMimeTypeMatchListByType(
        edv_mimetype_struct **list, gint total,
        gint *n,
        const gchar *type, gbool case_sensitive
);

extern edv_mimetype_struct *EDVMimeTypeNew(
	gint mt_class,			/* One of EDV_MIMETYPE_CLASS_*. */
	const gchar *value,
	const gchar *type,		/* MIME Type name. */
	const gchar *description	/* Short verbose description. */
);

extern void EDVMimeTypeLoadSmallIconsData(
        edv_mimetype_struct *m, guint8 ***data
);
extern void EDVMimeTypeLoadMediumIconsData(
        edv_mimetype_struct *m, guint8 ***data
);
extern void EDVMimeTypeLoadLargeIconsData(
        edv_mimetype_struct *m, guint8 ***data
);

extern void EDVMimeTypeRealize(
	edv_mimetype_struct *m, gbool force_rerealize
);

extern void EDVMimeTypeDelete(edv_mimetype_struct *m);


#endif	/* EDVMIMETYPES_H */
