#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "cdialog.h"

#include "edvtypes.h"
#include "edvdevices.h"
#include "edvmount.h"
#include "edvutils.h"
#include "edvutilsgtk.h"


const gchar *EDVMountGetError(void);

gint EDVMountDoMount(edv_device_struct *dev_ptr);
gint EDVMountDoUnmount(edv_device_struct *dev_ptr);
gint EDVMountDoEject(edv_device_struct *dev_ptr);

static const gchar *last_error;


/*
 *	Returns a string describing the last error encountered or NULL
 *	if there was no error. The returned pointer must not be 
 *	deallocated.
 */
const gchar *EDVMountGetError(void)
{
	return(last_error);
}



/*
 *	Procedure to mount the device specified by dev_ptr.
 */
gint EDVMountDoMount(edv_device_struct *dev_ptr)
{
	pid_t pid;
	gchar *cmd = NULL;
	gchar *stdout_path = NULL, *stderr_path = NULL;
        gchar *mount_path = NULL;
	gchar *device_path = NULL;
	const gchar *fs_type_str;
        gint status;


	/* Reset last error string. */
	last_error = NULL;


	if(dev_ptr == NULL)
	{
	    last_error = "Bad input value";
	    return(-1);
	}

#define DO_FREE_LOCALS  \
{ \
 g_free(cmd); \
 cmd = NULL; \
 g_free(stdout_path); \
 stdout_path = NULL; \
 g_free(stderr_path); \
 stderr_path = NULL; \
\
 g_free(mount_path); \
 mount_path = NULL; \
\
 g_free(device_path); \
 device_path = NULL; \
}

        /* Get copy of mount path. */
        mount_path = (dev_ptr->mount_path != NULL) ?
            g_strdup(dev_ptr->mount_path) : NULL;
        if(mount_path == NULL)
        {
            DO_FREE_LOCALS
            last_error = "Unable to get mount path";
            return(-2);
        }

        /* Get copy of device path. */
        device_path = (dev_ptr->device_path != NULL) ?
            g_strdup(dev_ptr->device_path) : NULL;
        if(device_path == NULL)
        {
            DO_FREE_LOCALS
            last_error = "Unable to get device path";
            return(-2);
        }

	/* Get file system type as a string. */
	fs_type_str = EDVDeviceGetFSStringFromNumber(
	    dev_ptr->fs_type
	);
	if(fs_type_str == NULL)
	{
            DO_FREE_LOCALS
            last_error = "Unable to get file system type string";
            return(-2);
        }

	/* Create mount command string. */
	if((dev_ptr->command_mount != NULL) ?
	    (*dev_ptr->command_mount != '\0') : FALSE
	)
	    cmd = g_strdup(dev_ptr->command_mount);
	else
	    cmd = g_strdup_printf(
		"/bin/mount -t %s \"%s\" \"%s\" %s",
		fs_type_str,
		device_path,
		mount_path,
		dev_ptr->read_only ? "-r" : ""
	    );

        /* Generate output file paths. */
        stdout_path = EDVTmpName(NULL);
        stderr_path = EDVTmpName(NULL);

	/* Execute command string. */
	pid = ExecBOE(cmd, stdout_path, stderr_path);
	if(pid == 0)
	{
	    last_error = "Execution of mount command failed";
	    status = -1;
	}
	else
	{
	    status = 0;
	}


	/* Show mount stderr and stdout messages (if any). */
        EDVShowFileInCDialog(
            stderr_path, "Mount Warning", CDIALOG_ICON_WARNING,
            NULL
        );
	EDVShowFileInCDialog(
	    stdout_path, "Mount Message", CDIALOG_ICON_INFO,
	    NULL
	);


        /* Remove output paths. */
        if(stdout_path != NULL)
            unlink(stdout_path);
        if(stderr_path != NULL)
            unlink(stderr_path);

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *	Procedure to unmount the device specified by dev_ptr.
 */
gint EDVMountDoUnmount(edv_device_struct *dev_ptr)
{
        pid_t pid;
        gchar *cmd = NULL;
	gchar *stdout_path = NULL, *stderr_path = NULL;
	gchar *mount_path = NULL;
	gint status;


        /* Reset last error string. */
        last_error = NULL;


        if(dev_ptr == NULL)
        {
            last_error = "Bad input value";
            return(-1);
        }

	/* Device now allowed to be unmounted? */
	if(dev_ptr->no_unmount)
	{
	    last_error = "Device is marked \"no unmount\"";
	    return(-1);
	}

#define DO_FREE_LOCALS  \
{ \
 g_free(cmd); \
 cmd = NULL; \
 g_free(stdout_path); \
 stdout_path = NULL; \
 g_free(stderr_path); \
 stderr_path = NULL; \
\
 g_free(mount_path); \
 mount_path = NULL; \
}

	/* Get copy of mount path. */
	mount_path = (dev_ptr->mount_path != NULL) ?
	    g_strdup(dev_ptr->mount_path) : NULL;
	if(mount_path == NULL)
	{
	    DO_FREE_LOCALS
	    last_error = "Unable to get mount path";
	    return(-2);
	}

        /* Generate output file paths. */
        stdout_path = EDVTmpName(NULL);
        stderr_path = EDVTmpName(NULL);

        /* Create unmount command string. */
        if((dev_ptr->command_unmount != NULL) ?
            (*dev_ptr->command_unmount != '\0') : FALSE
        )
            cmd = g_strdup(dev_ptr->command_unmount);
	else
	    cmd = g_strdup_printf(
		"/bin/umount \"%s\"",
		mount_path
	    );

        /* Execute command string. */
        pid = ExecBOE(cmd, stdout_path, stderr_path);
        if(pid == 0)
        {
            last_error = "Execution of unmount command failed";
            status = -1;
        }
        else
        {
            status = 0;
        }

        /* Show unmount stderr and stdout messages (if any). */
        EDVShowFileInCDialog(
            stderr_path, "Unmount Warning", CDIALOG_ICON_WARNING,
            NULL
        );
        EDVShowFileInCDialog(
            stdout_path, "Unmount Message", CDIALOG_ICON_INFO,
            NULL
        );

	/* Remove output paths. */
	if(stdout_path != NULL)
	    unlink(stdout_path);
	if(stderr_path != NULL)
	    unlink(stderr_path);

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *      Procedure to eject media from the device specified by dev_ptr.
 *
 *	Note that you should call EDVMountDoUnmount as needed to unmount
 *	the device first.
 */
gint EDVMountDoEject(edv_device_struct *dev_ptr)
{
        pid_t pid;
        gchar *cmd = NULL;
        gchar *stdout_path = NULL, *stderr_path = NULL;
        gchar *device_path = NULL;
        gint status;


        /* Reset last error string. */
        last_error = NULL;


        if(dev_ptr == NULL)
        {
            last_error = "Bad input value";
            return(-1);
        }

        /* Media not allowed to be ejected from device? Check if
	 * it is \"no unmount\".
	 */
        if(dev_ptr->no_unmount)
        {
            last_error = "Cannot eject media, device is marked \"no unmount\"";
            return(-1);
        }

#define DO_FREE_LOCALS  \
{ \
 g_free(cmd); \
 cmd = NULL; \
 g_free(stdout_path); \
 stdout_path = NULL; \
 g_free(stderr_path); \
 stderr_path = NULL; \
\
 g_free(device_path); \
 device_path = NULL; \
}

        /* Get copy of device path. */
        device_path = (dev_ptr->device_path != NULL) ?
            g_strdup(dev_ptr->device_path) : NULL;
        if(device_path == NULL)
        {
            DO_FREE_LOCALS
            last_error = "Unable to get device path";
            return(-2);
        }

        /* Generate output file paths. */
        stdout_path = EDVTmpName(NULL);
        stderr_path = EDVTmpName(NULL);

        /* Create eject command string. */
        if((dev_ptr->command_eject != NULL) ?
            (*dev_ptr->command_eject != '\0') : FALSE
        )
            cmd = g_strdup(dev_ptr->command_eject);
        else
            cmd = g_strdup_printf(
                "/usr/bin/eject \"%s\"",
                device_path
            );

        /* Execute command string. */
        pid = ExecBOE(cmd, stdout_path, stderr_path);
        if(pid == 0)
        {
            last_error = "Execution of eject command failed";
            status = -1;
        }
        else
        {
            status = 0;
        }

        /* Show eject error and output message (if any). */
        EDVShowFileInCDialog(
            stderr_path, "Eject Error", CDIALOG_ICON_ERROR,
            NULL
        );
        EDVShowFileInCDialog(
            stdout_path, "Eject Message", CDIALOG_ICON_INFO,
            NULL
        );

        /* Remove output paths. */
        if(stdout_path != NULL)
            unlink(stdout_path);
        if(stderr_path != NULL)
            unlink(stderr_path);

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}
