/*
                     Endeavour Disk Object Data Structure

	Used as GtkCList row data and other places to keep stats on
	disk objects.
 */

#ifndef EDVOBJ_H
#define EDVOBJ_H

#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>
#include "edvtypes.h"


/*
 *	Object type codes:
 */
#define EDV_OBJECT_TYPE_FILE			1
#define EDV_OBJECT_TYPE_DIRECTORY		2
#define EDV_OBJECT_TYPE_LINK			3
#define EDV_OBJECT_TYPE_DEVICE_BLOCK		4
#define EDV_OBJECT_TYPE_DEVICE_CHARACTER	5
#define EDV_OBJECT_TYPE_FIFO			6
#define EDV_OBJECT_TYPE_SOCKET			7


/*
 *	Endeavour disk object structure:
 */
typedef struct {

	gint type;		/* One of EDV_OBJECT_TYPE_*. */

	gchar *name;
	gchar *full_path;

	gchar *linked_to;	/* Link destination, only for type
				 * EDV_OBJECT_TYPE_LINK
				 */

	gbool link_valid;	/* TRUE if link is valid (not dangling),
				 * only for type EDV_OBJECT_TYPE_LINK
                                 */

	guint permissions;	/* Any of EDV_PERMISSION_*. */

	gulong	access_time,	/* In systime seconds. */
		modify_time,
		change_time;

	gint owner_id;
	gint group_id;

	gint hard_links;

	gulong size;		/* In bytes. */

	gulong device;
	gulong inode;

	gint device_type;	/* Device major and minor numbers
				 * (used only if object type is one of
				 * EDV_OBJECT_TYPE_DEVICE_*.
				 * Use EDVGetDeviceNumbers() to parse this
				 * value casted as a dev_t.
				 */

	gulong block_size;
	gulong blocks;

} edv_object_struct;



extern gint EDVObjectGetTypeFromStatMode(mode_t m);
extern mode_t EDVObjectGetTypeFromEDVType(gint type);

extern guint EDVObjectGetPermissionsFromStatMode(mode_t m);
extern mode_t EDVObjectGetPermissionsFromEDVPermissions(guint permissions);


extern edv_object_struct *EDVObjectNew(void);
extern edv_object_struct *EDVObjectCopy(edv_object_struct *obj);
extern void EDVObjectSetPath(
        edv_object_struct *obj, const gchar *path
);
extern void EDVObjectSetPath2(
        edv_object_struct *obj,
        const gchar *parent, const gchar *name
);
extern void EDVObjectSetStat(
	edv_object_struct *obj, const struct stat *lstat_buf
);
extern void EDVObjectValidateLink(edv_object_struct *obj);
extern void EDVObjectDelete(edv_object_struct *obj);


#endif	/* EDVOBJ_H */
