#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"
#include "../include/fio.h"

#include "guiutils.h"
#include "cdialog.h"
#include "clipboard.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvhistory.h"
#include "edvmimetypes.h"
#include "browser.h"
#include "endeavour.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcb.h"
#include "edvcfglist.h"
#include "config.h"


gchar *EDVGetObjectSizeStr(edv_core_struct *core_ptr, gulong i);

GdkCursor *EDVGetCursor(
	edv_core_struct *core_ptr, gint cursor_code
);

GtkCTreeNode *EDVNodeGetParent(GtkCTreeNode *node);
GtkCTreeNode *EDVNodeGetChild(GtkCTreeNode *node);
GtkCTreeNode *EDVNodeGetSibling(GtkCTreeNode *node);
GtkCTreeNode *EDVNodeGetToplevel(GtkCTree *ctree);

gint EDVCListGetSelectedLast(
	GtkCList *clist, GtkCListRow **row_ptr_rtn
);
GtkCTreeNode *EDVCTreeGetSelectedLast(
	GtkCTree *ctree, GtkCTreeRow **row_ptr_rtn
);

gboolean EDVNodeGetIndex(
        GtkCTree *ctree, GtkCTreeNode *node,
        gint *row_rtn, gint *column_rtn
);
GtkCTreeNode *EDVNodeGetByCoordinates(
        GtkCTree *ctree, gint x, gint y
);
void EDVScrollCListToPosition(
	GtkCList *clist, gfloat hpos, gfloat vpos
);

gboolean EDVObjectNameNotationOK(const gchar *name);
gboolean EDVCheckObjectHidden(
	edv_core_struct *core_ptr, edv_object_struct *object
);

gboolean EDVCheckWriteProtect(edv_core_struct *core_ptr, gboolean verbose);

gboolean EDVCheckUIDIsOwner(gint effective_uid, gint owner_id);
gboolean EDVCheckIsOwner(edv_core_struct *core_ptr, gint owner_id);

gboolean EDVCheckObjectAccessable(
	edv_core_struct *core_ptr, edv_object_struct *object
);
gboolean EDVCheckObjectReadable(
        edv_core_struct *core_ptr, edv_object_struct *object
);
gboolean EDVCheckObjectWriteable(
        edv_core_struct *core_ptr, edv_object_struct *object
);
gboolean EDVCheckObjectExecutable(
        edv_core_struct *core_ptr, edv_object_struct *object
);

gboolean EDVCheckImlibImage(
	edv_core_struct *core_ptr, const gchar *path
);
gboolean EDVCheckEDVArchiverArchive(
        edv_core_struct *core_ptr, const gchar *path
);

gint EDVMatchObjectIcon(
	edv_device_struct **device, gint total_devices,
	edv_mimetype_struct **mimetype, gint total_mimetypes,
	gint type,		/* One of EDV_OBJECT_TYPE_*. */
	const gchar *path,	/* Full path or just name. */
	gboolean link_valid, guint permissions,
        gint icon_size,         /* 0 = small, 1 = medium, 2 = large. */
	GdkPixmap **pixmap_closed, GdkBitmap **mask_closed,
        GdkPixmap **pixmap_opened, GdkBitmap **mask_opened,
        GdkPixmap **pixmap_extended, GdkBitmap **mask_extended
);
gint EDVMatchObjectTypeString(
        edv_mimetype_struct **mimetype, gint total_mimetypes,
        gint type,              /* One of EDV_OBJECT_TYPE_*. */
        guint permissions,
        const gchar *path,      /* Full path or just name. */
	gchar **type_str
);

void EDVShowFileInCDialog(
        const gchar *path, const gchar *title, gint cdialog_icon,
	GtkWidget *toplevel
);

void EDVCenterWindowToWindow(GtkWidget *w1, GtkWidget *w2);
void EDVEntrySetDND(edv_core_struct *core_ptr, GtkWidget *w);

void EDVCopyDiskObjectsToDDEPath(
        edv_core_struct *core_ptr,
        edv_object_struct **list, gint total
);
void EDVCopyDiskObjectsToDDEURL(
	edv_core_struct *core_ptr,
	edv_object_struct **list, gint total
);

void EDVAppendHistory(
	edv_core_struct *core_ptr,
	gint type,		/* One of EDV_HISTORY_*. */
	gulong time_start,	/* Time the operation first started. */
	gulong time_end,	/* Time the operation ended. */
	gint status,
	const gchar *src_path, const gchar *tar_path,
	const gchar *comments
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Returns a statically allocated string describing the size
 *	of the object specified by i in units defined by the
 *	configuration on the given core structure.
 *
 *	The returned string will be no longer than 256 bytes and
 *	never NULL.
 */
gchar *EDVGetObjectSizeStr(edv_core_struct *core_ptr, gulong i)
{
	gint comma_countdown, slen;
        gchar ss[256], *ss_ptr;
        static gchar ts[256], *ts_ptr;


        sprintf(ss, "%ld", i);
        slen = strlen(ss);

        /* 3 digits or less? (no commas needed) */
        if(slen <= 3)
        {
            strcpy(ts, ss);
            return(ts);
        }

        ts_ptr = ts;
        ss_ptr = ss;

	/* Initialize comma counter. */
        comma_countdown = slen % 3;
        if(comma_countdown <= 0)
            comma_countdown = 3;

	/* Iterate through size string until end is reached. */
        while(*ss_ptr != '\0')
        {
	    /* Reset comma counter and put in a comma? */
            if(comma_countdown <= 0)
            {
                *ts_ptr++ = ',';
                comma_countdown = 3;
            }

            *ts_ptr++ = *ss_ptr++;
            comma_countdown--;
        }

	/* Null terminate return string. */
        *ts_ptr = '\0';

        return(ts);
}


/*
 *	Returns a GdkCursor specified by cursor_code on the given core
 *	structure.
 */
GdkCursor *EDVGetCursor(
	edv_core_struct *core_ptr, gint cursor_code
)
{
	gint i;
	edv_cursor_struct *cur;


	if(core_ptr == NULL)
	    return(NULL);

	for(i = 0; i < core_ptr->total_cursors; i++)
	{
	    cur = core_ptr->cursor[i];
	    if(cur == NULL)
		continue;

	    if(cur->code == cursor_code)
		return(cur->cursor);
	}

	return(NULL);
}


/*
 *	Returns the parent of the given node or NULL if the given node
 *	does not have a parent.
 */
GtkCTreeNode *EDVNodeGetParent(GtkCTreeNode *node)
{
	GtkCTreeRow *row_ptr = (node != NULL) ? GTK_CTREE_ROW(node) : NULL;
	return((row_ptr != NULL) ? row_ptr->parent : NULL);
}

/*
 *	Returns the first child of the given node or NULL if the node
 *	does not have any children.
 */
GtkCTreeNode *EDVNodeGetChild(GtkCTreeNode *node)
{
	GtkCTreeRow *row_ptr = (node != NULL) ? GTK_CTREE_ROW(node) : NULL;
	return((row_ptr != NULL) ? row_ptr->children : NULL);
}

/*
 *      Returns the next sibling of the given node or NULL if the node
 *	does not have a next sibling.
 */
GtkCTreeNode *EDVNodeGetSibling(GtkCTreeNode *node)
{
	GtkCTreeRow *row_ptr = (node != NULL) ? GTK_CTREE_ROW(node) : NULL;
	return((row_ptr != NULL) ? row_ptr->sibling : NULL);
}

/*
 *	Returns the first toplevel node on the given GtkCTree widget
 *	or NULL if the GtkCTree does not have one.
 */
GtkCTreeNode *EDVNodeGetToplevel(GtkCTree *ctree)
{
	/* Here we just return the first node, since it is always
	 * gauranteed to be the toplevel node.
	 */
	return((ctree != NULL) ? gtk_ctree_node_nth(ctree, 0) : NULL);
}


/*
 *	Returns the index of the last selected row on the given clist.
 *
 *	If row_ptr_rtn is not NULL and there exists a selected row then
 *	*row_ptr_rtn will be updated with the pointer to the row
 *	structure.
 *
 *	Returns -1 on error or no row selected, if a valid row index is
 *	returned it can be considered to be valid and allocated.
 */
gint EDVCListGetSelectedLast(
        GtkCList *clist, GtkCListRow **row_ptr_rtn
)
{
	gint row;
	GList *glist;


	if(row_ptr_rtn != NULL)
	    *row_ptr_rtn = NULL;

	if(clist == NULL)
	    return(-1);

	/* Get last selected row index or -1 if there is none. */
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	/* Check if we need to get a row pointer and that the selected
	 * row index is in bounds.
	 */
	if((row >= 0) && (row < clist->rows) &&
	   (row_ptr_rtn != NULL) && (clist->row_list != NULL)
	)
	{
	    glist = g_list_nth(clist->row_list, row);
	    *row_ptr_rtn = (glist != NULL) ?
		(GtkCListRow *)glist->data : NULL;
	    if(*row_ptr_rtn == NULL)
		row = -1;
	}

	return(row);
}

/*
 *      Returns the pointer of the last selected node on the given ctree.
 *
 *      If row_ptr_rtn is not NULL and there exists a selected node then
 *      *row_ptr_rtn will be updated with the pointer to the node's row
 *	structure.
 *
 *      Returns NULL on error or no node selected, if a valid node is
 *      returned it can be considered to be valid and allocated.
 */
GtkCTreeNode *EDVCTreeGetSelectedLast(
        GtkCTree *ctree, GtkCTreeRow **row_ptr_rtn
)
{
	GList *glist;
	GtkCList *clist;
	GtkCTreeNode *node;


        if(row_ptr_rtn != NULL)
            *row_ptr_rtn = NULL;

        if(ctree == NULL)
            return(NULL);

	clist = GTK_CLIST(ctree);

        /* Get last selected node or -1 if there is none. */
        glist = clist->selection_end;
        node = (glist != NULL) ? (GtkCTreeNode *)glist->data : NULL;

	/* Check if we need to get a row pointer and that there is a
	 * selected node.
	 */
	if((node != NULL) && (row_ptr_rtn != NULL))
	{
	    *row_ptr_rtn = GTK_CTREE_ROW(node);
	    if(*row_ptr_rtn == NULL)
		node = NULL;
	}

	return(node);
}


/*
 *	Returns the column and row indexes of the given GtkCTreeNode node
 *	relative to the given GtkCTree widget.
 *
 *	Returns TRUE if the indexes are found or FALSE on failed match.
 */
gboolean EDVNodeGetIndex(
	GtkCTree *ctree, GtkCTreeNode *node,
	gint *row_rtn, gint *column_rtn
)
{
	gint row;
	GList *glist;
	GtkCList *clist;
	GtkCTreeRow *row_ptr;


	/* Reset returns. */
	if(row_rtn != NULL)
	    *row_rtn = -1;
	if(column_rtn != NULL)
	    *column_rtn = -1;

	if((ctree == NULL) || (node == NULL))
	    return(FALSE);

	clist = GTK_CLIST(ctree);

	/* Get the row pointer the given node. */
	row_ptr = GTK_CTREE_ROW(node);
	if(row_ptr == NULL)
	    return(FALSE);

	/* Count rows until we encounter the given node's row. */
	row = 0;
	glist = clist->row_list;
	while(glist != NULL)
	{
	    if((gpointer)row_ptr == glist->data)
		break;

	    row++;
	    glist = glist->next;
	}
	/* Failed to find row? */
	if(glist == NULL)
	    return(FALSE);

	/* Update returns. */
	if(row_rtn != NULL)
	    *row_rtn = row;
	/* Column is always the ctree column. */
	if(column_rtn != NULL)
	    *column_rtn = ctree->tree_column;

	return(TRUE);
}

/*
 *	Returns the tree node that matches the given coordinates on the
 *	given GtkCTree widget.
 */
GtkCTreeNode *EDVNodeGetByCoordinates(
        GtkCTree *ctree, gint x, gint y
)
{
	gint row, column;
	GtkCList *clist;


	if(ctree == NULL)
	    return(NULL);

	clist = GTK_CLIST(ctree);

	/* Find row and column based on given coordinates. */
	if(!gtk_clist_get_selection_info(
	    clist, x, y, &row, &column
	))
	{
	    row = clist->rows;
	    column = 0;
	}
	if(row < 0)
	    row = 0;

	/* Get tree node matching the calculated row. */
	return(gtk_ctree_node_nth(ctree, row));
}

/*
 *	Scroll the given clist to the given position.
 *
 *	A "value_changed" signal will be emitted to each of the clist's
 *	horizontal and vertical GtkAdjustments.
 */
void EDVScrollCListToPosition(
        GtkCList *clist, gfloat hpos, gfloat vpos
)
{
	GtkAdjustment *adj;

	if(clist == NULL)
	    return;

	adj = clist->hadjustment;
	if(adj != NULL)
	{
	    if(hpos > (adj->upper - adj->page_size))
		hpos = adj->upper - adj->page_size;
	    if(hpos < adj->lower)
		hpos = adj->lower;
	    adj->value = hpos;
	    gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");
	}

        adj = clist->vadjustment;
        if(adj != NULL)
        {
            if(vpos > (adj->upper - adj->page_size))
                vpos = adj->upper - adj->page_size;
            if(vpos < adj->lower)
                vpos = adj->lower;
            adj->value = vpos;
            gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");
        }
}


/*
 *	Returns TRUE if the given disk object name has a valid notation.
 */
gboolean EDVObjectNameNotationOK(const gchar *name)
{
	gchar *strptr;


	if(name == NULL)
	    return(FALSE);

	/* No empty names allowed. */
	if(*name == '\0')
	    return(FALSE);

	/* No special notations allowed. */
        if(!strcmp(name, ".."))
            return(FALSE);
        if(!strcmp(name, "."))
            return(FALSE);
        if(!strcmp(name, "?"))
            return(FALSE);
        if(!strcmp(name, "*"))
            return(FALSE);

	/* No blanks allowed as first char. */
	if(*name == ' ')
            return(FALSE);
        if(*name == '\t')
            return(FALSE);

	/* No deliminators allowed. */
	strptr = strchr(name, DIR_DELIMINATOR);
	if(strptr != NULL)
	    return(FALSE);

	/* All checks passed! */
	return(TRUE);
}

/*
 *      Returns TRUE if the object is `hidden' by checking the given name
 *      of the object and testing if the first character if the name is
 *      a '.' character.
 */
gboolean EDVCheckObjectHidden(
        edv_core_struct *core_ptr, edv_object_struct *object
)
{
	const gchar *name;


        if(object == NULL)
            return(FALSE);

	name = object->name;
	if(name == NULL)
	    return(FALSE);

        /* Check if the first character of the name starts with a '.'
         * character.
         */
        if(*name == '.')
	{
	    /* Special directory notations should not count in this
	     * check.
	     */
	    if(!strcmp(name, "..") || !strcmp(name, "."))
		return(FALSE);
	    else
		return(TRUE);
	}
        else
	{
            return(FALSE);
	}
}

/*
 *	Returns TRUE if the write protect is enabled on the given core
 *	structure's configuration list.
 *
 *	If verbose is set to TRUE and write protect is enabled then a 
 *	standard message will be printed to the user indicating that write
 *	protect is enabled and how to disable it.
 */
gboolean EDVCheckWriteProtect(edv_core_struct *core_ptr, gboolean verbose)
{
	gboolean write_protect;


	if(core_ptr == NULL)
	    return(FALSE);

	/* Get write protect state from the core structure's configuration
	 * list.
	 */
	write_protect = EDVCFGItemListGetValueI(
	    core_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT
	);
	/* Is write protect enabled and verbose specified? */
	if(write_protect && verbose)
	{
            CDialogSetTransientFor(NULL);
            CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Permission Denied",
"The specified operation is not permitted because\n\
write protect is on.\n\
\n\
If you wish to perform this operation then you must turn\n\
off write protect by going to Settings->Write Protect.",
		NULL,
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El Permiso Neg",
"La operacin especificada no se permite porque escribe\n\
protege est activado. Si usted desea realizar la operacin\n\
entonces usted debe apagar escribe protege yendo a\n\
Settings->Write Protect.",
                NULL,
#endif
#ifdef PROG_LANGUAGE_FRENCH
"La Permission A Ni",
"L'opration spcifie n'est pas permise parce que protge\n\
en criture est sur. Si vous souhaitez excuter l'opration\n\
alors vous devez tourner de protge en criture en allant \n\
Settings->Write Protect.",
                NULL,
#endif
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
	}

	return(write_protect);
}


/*
 *	Checks if the given effective user id owns the object who's
 *	owner is the owner_id.
 */
gboolean EDVCheckUIDIsOwner(gint effective_uid, gint owner_id)
{
	return(
	   (effective_uid == 0) ||
	   (effective_uid == owner_id)
	);
}

/*
 *	Checks if the effective user id of this process owns the object
 *	sho's owner is the owner_id.
 */
gboolean EDVCheckIsOwner(edv_core_struct *core_ptr, gint owner_id)
{
	if(core_ptr == NULL)
	    return(FALSE);
	else
	    return(EDVCheckUIDIsOwner(core_ptr->effective_user_id, owner_id));
}

/*
 *	Checks if the user or group id of the process can access the
 *	given directory object.
 *
 *	Where access is defined as the directory object being
 *	executable, since any directory set x means it is
 *	accessable.
 */
gboolean EDVCheckObjectAccessable(
        edv_core_struct *core_ptr, edv_object_struct *object
)
{
	gint euid, eguid;
	guint p;


	if((core_ptr == NULL) || (object == NULL))
	    return(FALSE);

	euid = core_ptr->effective_user_id;
        eguid = core_ptr->effective_group_id;
	p = object->permissions;

	/* Sticky and atleast one catagory has execute permission? */
	if((p & EDV_PERMISSION_STICKY) &&
	   (p & (EDV_PERMISSION_UEXECUTE | EDV_PERMISSION_GEXECUTE |
                 EDV_PERMISSION_AEXECUTE))
	)
	    return(TRUE);	

	/* Anyone can execute? */
	if(p & EDV_PERMISSION_AEXECUTE)
	    return(TRUE);

	/* Group can execute and is member of group? */
	if((p & EDV_PERMISSION_GEXECUTE) &&
	   (eguid == object->group_id)
	)
	    return(TRUE);

	/* Owner can execute, is root, or is the owner? */
        if((p & EDV_PERMISSION_UEXECUTE) &&
           ((euid == 0) || (euid == object->owner_id))
        )
            return(TRUE);

	return(FALSE);
}

/*
 *      Checks if the user or group id of the process can read the
 *      given object.
 */
gboolean EDVCheckObjectReadable(
        edv_core_struct *core_ptr, edv_object_struct *object
)
{
        gint euid, eguid;
        guint p;


        if((core_ptr == NULL) || (object == NULL))
            return(FALSE);

        euid = core_ptr->effective_user_id;
        eguid = core_ptr->effective_group_id;
        p = object->permissions;

        /* Sticky and atleast one catagory has read permission? */
        if((p & EDV_PERMISSION_STICKY) &&
           (p & (EDV_PERMISSION_UREAD | EDV_PERMISSION_GREAD |
                 EDV_PERMISSION_AREAD))
        )
            return(TRUE);

        /* Anyone can read? */
        if(p & EDV_PERMISSION_AREAD)
            return(TRUE);

        /* Group can read and is member of group? */
        if((p & EDV_PERMISSION_GREAD) &&
           (eguid <= object->group_id)
        )
            return(TRUE);

        /* Owner can read and is a owner? */
        if((p & EDV_PERMISSION_UREAD) &&
           ((euid == 0) || (euid == object->owner_id))
        )
            return(TRUE);

        return(FALSE);
}

/*
 *      Checks if the user or group id of the process can write the
 *      given object.
 */
gboolean EDVCheckObjectWriteable(
        edv_core_struct *core_ptr, edv_object_struct *object
)
{
        gint euid, eguid;
        guint p;


        if((core_ptr == NULL) || (object == NULL))
            return(FALSE);

        euid = core_ptr->effective_user_id;
        eguid = core_ptr->effective_group_id;
        p = object->permissions;

        /* Sticky and atleast one catagory has write permission? */
        if((p & EDV_PERMISSION_STICKY) &&
           (p & (EDV_PERMISSION_UWRITE | EDV_PERMISSION_GWRITE |
                 EDV_PERMISSION_AWRITE))
        )
            return(TRUE);

        /* Anyone can write? */
        if(p & EDV_PERMISSION_AWRITE)
            return(TRUE);

        /* Group can write and is member of group? */
        if((p & EDV_PERMISSION_GWRITE) &&
           (eguid <= object->group_id)
        )
            return(TRUE);

        /* Owner can write and is a owner? */
        if((p & EDV_PERMISSION_UWRITE) &&
           ((euid == 0) || (euid == object->owner_id))
        )
            return(TRUE);

        return(FALSE);
}

/*
 *      Checks if the user or group id of the process can execute the
 *      given object.
 */
gboolean EDVCheckObjectExecutable(
        edv_core_struct *core_ptr, edv_object_struct *object
)
{
        gint euid, eguid;
        guint p;


        if((core_ptr == NULL) || (object == NULL))
            return(FALSE);

        euid = core_ptr->effective_user_id;
        eguid = core_ptr->effective_group_id;
        p = object->permissions;

        /* Sticky and atleast one catagory has execute permission? */
        if((p & EDV_PERMISSION_STICKY) &&
           (p & (EDV_PERMISSION_UEXECUTE | EDV_PERMISSION_GEXECUTE |
                 EDV_PERMISSION_AEXECUTE))
        )
            return(TRUE);

        /* Anyone can execute? */
        if(p & EDV_PERMISSION_AEXECUTE)
            return(TRUE);

        /* Group can execute and is member of group? */
        if((p & EDV_PERMISSION_GEXECUTE) &&
           (eguid <= object->group_id)
        )
            return(TRUE);

        /* Owner can execute and is a owner? */
        if((p & EDV_PERMISSION_UEXECUTE) &&
           ((euid == 0) || (euid == object->owner_id))
        )
            return(TRUE);

        return(FALSE);
}

/*
 *	Returns TRUE if the object specified by path is supported by
 *	Imlib.
 */
gboolean EDVCheckImlibImage(
        edv_core_struct *core_ptr, const gchar *path
)
{
        gint i;
        const gchar *ext_list[] = IMLIB_IMAGE_FILE_EXTENSIONS;


	if((core_ptr == NULL) || (path == NULL))
	    return(FALSE);

        /* Check if this is an image file by checking the extension
         * portion of the name.
         */
        for(i = 0; ext_list[i] != NULL; i += 2)
        {
	    if(EDVIsExtension(path, ext_list[i]))
                return(TRUE);
        }

	return(FALSE);
}

/*
 *      Returns TRUE if the object specified by path is an archive
 *      supported by Endeavour's archiver.
 */
gboolean EDVCheckEDVArchiverArchive(
        edv_core_struct *core_ptr, const gchar *path
)
{
        gint i;
        const gchar *ext_list[] = EDV_ARCHIVER_ARCHIVE_FILE_EXTENSIONS;


        if((core_ptr == NULL) || (path == NULL))
            return(FALSE);

        /* Check if this is an image file by checking the extension
         * portion of the name.
         */
        for(i = 0; ext_list[i] != NULL; i += 2)
        {
            if(EDVIsExtension(path, ext_list[i]))
                return(TRUE);
        }

        return(FALSE);
}


/*
 *	Fetches the pixmap and mask pairs that best matches the given
 *	information. Devices and MIME Types will be realized as needed
 *	during this call.
 *
 *	Any of the given information may be NULL or 0.
 *
 *	The returned pixmap and mask pairs do not have any refcounts
 *	added to them.
 *
 *	Returns 0 on success or -1 on failed match.
 */
gint EDVMatchObjectIcon(
        edv_device_struct **device, gint total_devices,
        edv_mimetype_struct **mimetype, gint total_mimetypes,
        gint type,              /* One of EDV_OBJECT_TYPE_*. */
        const gchar *path,      /* Full path or just name. */
	gboolean link_valid, guint permissions,
	gint icon_size,		/* 0 = small, 1 = medium, 2 = large. */
        GdkPixmap **pixmap_closed, GdkBitmap **mask_closed,
        GdkPixmap **pixmap_opened, GdkBitmap **mask_opened,
        GdkPixmap **pixmap_extended, GdkBitmap **mask_extended
)
{
	gboolean got_match = FALSE;
	gboolean is_dir = (type == EDV_OBJECT_TYPE_DIRECTORY) ? TRUE : FALSE;
	gint i;
	edv_device_struct *dev_ptr;
	edv_mimetype_struct *mt_ptr;
	GdkPixmap *pixmap[3];
	GdkBitmap *mask[3];


	/* Reset locals. */
	memset(pixmap, 0x00, 3 * sizeof(GdkPixmap *));
        memset(mask, 0x00, 3 * sizeof(GdkBitmap *));

	/* Reset returns. */
	if(pixmap_closed != NULL)
	    *pixmap_closed = NULL;
        if(mask_closed != NULL)
            *mask_closed = NULL;

        if(pixmap_opened != NULL)
            *pixmap_opened = NULL;
        if(mask_opened != NULL)
            *mask_opened = NULL;

        if(pixmap_extended != NULL)
            *pixmap_extended = NULL;
        if(mask_extended != NULL)
            *mask_extended = NULL;


/* Macro to set the pixmap and mask array values to the MIME Type
 * specified by mt_ptr and realize the MIME Type. If one or more pixmaps
 * were obtained than got_match will be set to TRUE.
 */
#define DO_LOAD_MT_ICONS					\
{								\
 gint n;							\
 gint min = MIN(EDV_MIMETYPE_TOTAL_ICON_STATES, 3);		\
								\
 /* Realize MIME Type first (as needed) to ensure icons		\
  * are loaded.							\
  */								\
 EDVMimeTypeRealize(mt_ptr, FALSE);				\
								\
 /* Get icons by the requested size. */				\
 switch(icon_size)						\
 {								\
  case 0:	/* Small. */					\
   for(n = 0; n < min; n++)					\
   {								\
    pixmap[n] = mt_ptr->small_pixmap[n];			\
    mask[n] = mt_ptr->small_mask[n];				\
   }								\
   break;							\
								\
  case 1:	/* Medium. */					\
   for(n = 0; n < min; n++)					\
   {								\
    pixmap[n] = mt_ptr->medium_pixmap[n];			\
    mask[n] = mt_ptr->medium_mask[n];				\
   }								\
   break;							\
								\
  case 2:	/* Large. */					\
   for(n = 0; n < min; n++)					\
   {								\
    pixmap[n] = mt_ptr->large_pixmap[n];			\
    mask[n] = mt_ptr->large_mask[n];				\
   }								\
   break;							\
 }								\
								\
 /* Check if we got a valid pixmap, if we did then set		\
  * got_match to TRUE.						\
  */								\
 for(n = 0; n < min; n++)					\
 {								\
  if(pixmap[n] != NULL)						\
  {								\
   got_match = TRUE;						\
   break;							\
  }								\
 }								\
}

	/* Begin checking if the object matches a device or MIME Type. */

	/* First check if the object is a link. */
	if(type == EDV_OBJECT_TYPE_LINK)
	{
	    const gchar *mime_type_str = EDV_MIMETYPE_STR_LINK;

	    /* Object is a link, now iterate through MIME Types list and
	     * find the MIME Type of class EDV_MIMETYPE_CLASS_SYSTEM
	     * and use its icons.
	     */
            for(i = 0; i < total_mimetypes; i++)
            {
                mt_ptr = mimetype[i];
                if(mt_ptr == NULL)
                    continue;

                /* Only handle if MIME Type class is a systems object. */
                if((mt_ptr->mt_class == EDV_MIMETYPE_CLASS_SYSTEM) &&
                   (mt_ptr->type != NULL)
                )
                {
                    if(!strcmp(mt_ptr->type, mime_type_str))
                    {
                        DO_LOAD_MT_ICONS
                        got_match = TRUE;
                        break;
                    }
                }
            }
	}

	/* From here on, got_match can be set to TRUE at any time to
	 * indicate that a match was made.
	 */

	/* Check devices if device list is given and path is an
	 * absolute path.
	 */
	if((device != NULL) &&
           ((path != NULL) ? ISPATHABSOLUTE(path) : FALSE) &&
	   !got_match
	)
	{
	    /* Iterate through devices list. */
	    for(i = 0; i < total_devices; i++)
	    {
		dev_ptr = device[i];
		if(dev_ptr == NULL)
		    continue;

		if(dev_ptr->mount_path == NULL)
		    continue;

		if(!strcmp(dev_ptr->mount_path, path))
		{
		    gint n;
		    gint min = MIN(EDV_DEVICE_TOTAL_ICON_STATES, 3);


		    /* Realize this device first to ensure that the icon
		     * images are loaded.
		     */
		    EDVDeviceRealize(dev_ptr, FALSE);

		    /* Got full path match for device's mounted path. */
		    switch(icon_size)
		    {
		      case 0:	/* Small. */
			for(n = 0; n < min; n++)
			{
			    pixmap[n] = dev_ptr->small_pixmap[n];
                            mask[n] = dev_ptr->small_mask[n];
			}
			break;

                      case 1:	/* Medium. */
                        for(n = 0; n < min; n++)
                        {
                            pixmap[n] = dev_ptr->medium_pixmap[n];
                            mask[n] = dev_ptr->medium_mask[n];
                        }
                        break;

                      case 2:	/* Large. */
                        for(n = 0; n < min; n++)
                        {
                            pixmap[n] = dev_ptr->large_pixmap[n];
                            mask[n] = dev_ptr->large_mask[n];
                        }
                        break;
		    }
		    /* Stop itterating through devices once we got a
		     * match. Mark got match only if an actual valid pixmap
		     * was obtained.
		     */
		    for(n = 0; n < min; n++)
		    {
			if(pixmap[n] != NULL)
			{
			    got_match = TRUE;
			    break;
			}
		    }
		    break;
		}
	    }	/* Iterate through devices list. */
	}

        /* Check MIME Types. */
        if((mimetype != NULL) && !got_match)
	{
	    const gchar *value;

	    /* Iterate through MIME Types list. */
            for(i = 0; i < total_mimetypes; i++)
            {
                mt_ptr = mimetype[i];
                if(mt_ptr == NULL)
                    continue;

		value = mt_ptr->value;
		if(value == NULL)
		    continue;

		/* Handle by MIME Type class. */
		switch(mt_ptr->mt_class)
		{
		  case EDV_MIMETYPE_CLASS_UNIQUE:
                  case EDV_MIMETYPE_CLASS_PROGRAM:
		    if((path != NULL) ? ISPATHABSOLUTE(path) : FALSE)
		    {
			if(!strcmp(value, path))
			    DO_LOAD_MT_ICONS
		    }
		    break;

		  case EDV_MIMETYPE_CLASS_FORMAT:
                    if((path != NULL) && !got_match && !is_dir)
                    {
			if(EDVIsExtension(path, value))
                            DO_LOAD_MT_ICONS
                    }
                    break;

		}
		/* Do not stop iterating through MIME Types if got_match
		 * is TRUE, because there might be another MIME type that
		 * has a more specific icon further in the list.
		 */
	    }	/* Iterate through MIME Types list. */
	}



	/* If still did not get match then use basic system MIME types
	 * for the given object.
	 */
	if(!got_match)
	{
	    const gchar *mime_type_str;


	    /* Get MIME Type type string used for matching of system
	     * object type.
	     */
	    switch(type)
	    {
	      case EDV_OBJECT_TYPE_FILE:
		/* Check the file's permissions allow execution, in which
		 * case we use the file/executable MIME Type instead of
		 * file/regular.
		 */
		if(permissions & (EDV_PERMISSION_UEXECUTE |
		    EDV_PERMISSION_GEXECUTE | EDV_PERMISSION_AEXECUTE)
		)
		    mime_type_str = EDV_MIMETYPE_STR_FILE_EXECUTABLE;
		else
		    mime_type_str = EDV_MIMETYPE_STR_FILE;
		break;

	      case EDV_OBJECT_TYPE_DIRECTORY:
		mime_type_str = EDV_MIMETYPE_STR_DIRECTORY;
                break;

              case EDV_OBJECT_TYPE_LINK:
                mime_type_str = EDV_MIMETYPE_STR_LINK;
                break;

              case EDV_OBJECT_TYPE_DEVICE_BLOCK:
                mime_type_str = EDV_MIMETYPE_STR_DEVICE_BLOCK;
                break;

              case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
                mime_type_str = EDV_MIMETYPE_STR_DEVICE_CHARACTER;
                break;

              case EDV_OBJECT_TYPE_FIFO:
                mime_type_str = EDV_MIMETYPE_STR_FIFO;
                break;

              case EDV_OBJECT_TYPE_SOCKET:
                mime_type_str = EDV_MIMETYPE_STR_SOCKET;
                break;

	      default:
		mime_type_str = EDV_MIMETYPE_STR_FILE;
		break;
            }

	    /* Iterate through MIME Types of class system. */
            for(i = 0; i < total_mimetypes; i++)
            {
                mt_ptr = mimetype[i];
                if(mt_ptr == NULL)
                    continue;

                /* Only handle if MIME Type class is a systems object. */
                if((mt_ptr->mt_class == EDV_MIMETYPE_CLASS_SYSTEM) &&
                   (mt_ptr->type != NULL)
		)
                {
		    if(!strcmp(mt_ptr->type, mime_type_str))
		    {
			DO_LOAD_MT_ICONS
			got_match = TRUE;
			break;
		    }
		}
	    }

	    /* If we still didn't get match, then all else return the
	     * file icons.
	     */
/* Work on this later. */
	}
#undef DO_LOAD_MT_ICONS


	/* Update returns. */
        if(pixmap_closed != NULL)
            *pixmap_closed = pixmap[0];
        if(mask_closed != NULL)
            *mask_closed = mask[0];

        if(pixmap_opened != NULL)
            *pixmap_opened = pixmap[1];
        if(mask_opened != NULL)
            *mask_opened = mask[1];

        if(pixmap_extended != NULL)
            *pixmap_extended = pixmap[2];
        if(mask_extended != NULL)
            *mask_extended = mask[2];

	return(0);
}

/*
 *      Fetches the MIME Type type string that best matches the given
 *      information.
 *
 *      Any of the given information may be NULL or 0.
 *
 *      The returned MIME Type string must not be modified or
 *	deallocated.
 *
 *      Returns 0 on success or -1 on failed match.
 */
gint EDVMatchObjectTypeString(
        edv_mimetype_struct **mimetype, gint total_mimetypes,
        gint type,              /* One of EDV_OBJECT_TYPE_*. */
	guint permissions,
        const gchar *path,      /* Full path or just name. */
        gchar **type_str
)
{
        gboolean got_match = FALSE;
        gint i;
	gchar *ltype_str = NULL;
        const gchar *cstrptr, *ext = NULL;
        edv_mimetype_struct *mt_ptr;


	if(type_str != NULL)
	    *type_str = NULL;


        /* Parse extension from the given path if it is not NULL. */
        if(path != NULL)
        {
            /* Seek cstrptr to child portion of path. */
            cstrptr = strrchr(path, DIR_DELIMINATOR);
            if(cstrptr != NULL)
                cstrptr++;
            else
                cstrptr = path;

            /* Child name starts with a '.'? */
            if(*cstrptr == '.')
                cstrptr++;

            /* Seek to extension (if any), include the '.' as the first
             * character if an extension is found.
             */
            ext = strchr(cstrptr, '.');
            if(ext != NULL)
            {
                /* Extension must have atleast one character after it. */
                if(*(ext + 1) == '\0')
                    ext = NULL;
            }
        }


        /* Check MIME Types first. */
        if(mimetype != NULL)
        {
            const gchar *value;


            for(i = 0; i < total_mimetypes; i++)
            {
                mt_ptr = mimetype[i];
                if(mt_ptr == NULL)
                    continue;

                value = mt_ptr->value;
                if(value == NULL)
                    continue;

                /* Handle by MIME Type class. */
                switch(mt_ptr->mt_class)
                {
                  case EDV_MIMETYPE_CLASS_UNIQUE:
                  case EDV_MIMETYPE_CLASS_PROGRAM:
                    if((path != NULL) ? ISPATHABSOLUTE(path) : FALSE)
                    {
                        if(!strcmp(value, path))
                        {
                            ltype_str = mt_ptr->type;
			    if(ltype_str != NULL)
				got_match = TRUE;
                        }
                    }
                    break;

                  case EDV_MIMETYPE_CLASS_FORMAT:
                    if((path != NULL) && !got_match)
                    {
                        if(EDVIsExtension(path, value))
                        {
                            ltype_str = mt_ptr->type;
			    if(ltype_str != NULL)
				got_match = TRUE;
                        }
		    }
		    break;
                }
                /* Do not stop itterating through MIME Types if got_match
                 * is TRUE, because there might be another MIME type that
                 * has a more specific icon further in the list.
                 */
            }
        }

        /* If still did not get match then use basic system MIME types
         * for the given object.
         */
        if(!got_match)
        {
            const gchar *mime_type_str;


            /* Get MIME Type type string used for matching of system
             * object type.
             */
            switch(type)
            {
              case EDV_OBJECT_TYPE_FILE:
                /* Check the file's permissions allow execution, in which
                 * case we use the file/executable MIME Type instead of
                 * file/regular.
                 */
                if(permissions & (EDV_PERMISSION_UEXECUTE |
                    EDV_PERMISSION_GEXECUTE | EDV_PERMISSION_AEXECUTE)
                )
                    mime_type_str = EDV_MIMETYPE_STR_FILE_EXECUTABLE;
		else
		    mime_type_str = EDV_MIMETYPE_STR_FILE;
                break;

              case EDV_OBJECT_TYPE_DIRECTORY:
                mime_type_str = EDV_MIMETYPE_STR_DIRECTORY;
                break;

              case EDV_OBJECT_TYPE_LINK:
                mime_type_str = EDV_MIMETYPE_STR_LINK;
                break;

              case EDV_OBJECT_TYPE_DEVICE_BLOCK:
                mime_type_str = EDV_MIMETYPE_STR_DEVICE_BLOCK;
                break;

              case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
                mime_type_str = EDV_MIMETYPE_STR_DEVICE_CHARACTER;
                break;

              case EDV_OBJECT_TYPE_FIFO:
                mime_type_str = EDV_MIMETYPE_STR_FIFO;
                break;

              case EDV_OBJECT_TYPE_SOCKET:
                mime_type_str = EDV_MIMETYPE_STR_SOCKET;
                break;

              default:
                mime_type_str = EDV_MIMETYPE_STR_FILE;
                break;
            }

	    ltype_str = (gchar *)mime_type_str;
	}


	/* Update returns. */
        if(type_str != NULL)
            *type_str = ltype_str;


	return(0);
}


/*
 *	Opens the file specified by path and displays its contents on
 *	the cdialog.
 */
void EDVShowFileInCDialog(
        const gchar *path, const gchar *title, gint cdialog_icon,
        GtkWidget *toplevel
)
{
	FILE *fp;
	gchar *buf;
	gint buf_len, bytes_read;
	struct stat stat_buf;



	if(path == NULL)
	    return;

	/* Open file for reading. */
	fp = FOpen(path, "rb");
	if(fp == NULL)
	    return;

	/* Get statistics of opened file. */
	if(fstat(fileno(fp), &stat_buf))
	{
	    FClose(fp);
            return;
	}

	/* Get length of buffer and allocate read buffer to match the size
	 * of the file up to 10000 bytes plus one extra byte for the null
	 * terminator byte.
	 */
	buf_len = (gint)((stat_buf.st_size > 10000) ?
	    10000 : stat_buf.st_size
	);
	if(buf_len < 0)
	    buf_len = 0;
	buf = (gchar *)g_malloc((buf_len + 1) * sizeof(gchar));
	if(buf == NULL)
	{
	    FClose(fp);
	    return;
	}

	/* Read file. */
	bytes_read = (gint)fread(buf, sizeof(gchar), buf_len, fp);

	/* Close file. */
	FClose(fp);
	fp = NULL;

	/* Error reading file? */
	if(bytes_read < 0)
	{
	    g_free(buf);
	    return;
	}

	/* Null terminate. */
	if(bytes_read <= buf_len)
	    buf[bytes_read] = '\0';
	else
	    buf[buf_len] = '\0';

	/* File contents empty? */
	if(*buf == '\0')
	{
	    g_free(buf);
	    return;
	}

	/* Display read file contents on cdialog. */
	CDialogSetTransientFor(toplevel);
	CDialogGetResponse(
	    title, buf, NULL,
	    cdialog_icon,
	    CDIALOG_BTNFLAG_OK,
	    CDIALOG_BTNFLAG_OK
	);
	CDialogSetTransientFor(NULL);

	/* Deallocate read file contents buffer. */
	g_free(buf);
	buf = NULL;
}


/*
 *	Moves the window w2 relative to window w1, both w1 and w2
 *	must be GtkWindows.
 */
void EDVCenterWindowToWindow(GtkWidget *w1, GtkWidget *w2)
{
	gint x, y;
	gint width1, height1, width2, height2;
	GdkWindow *window1, *window2;


	if((w1 == NULL) || (w2 == NULL))
	    return;

	if(!GTK_IS_WINDOW(w1) || !GTK_IS_WINDOW(w2))
	    return;

	window1 = w1->window;
	window2 = w2->window;
	if((window1 == NULL) || (window2 == NULL))
	    return;

	gdk_window_get_root_origin(window1, &x, &y);
	gdk_window_get_size(window1, &width1, &height1);
	gdk_window_get_size(window2, &width2, &height2);

	gtk_widget_set_uposition(
	    w2,
	    (width2 > 0) ?
		x + (width1 / 2) - (width2 / 2) : x + 20,
	    (height2 > 0) ?
		y + (height1 / 2) - (height2 / 2) : y + 20
	);
}

/*
 *	Sets up the given widget (which must be a GtkEntry) to allow
 *	it to receive dragged disk object paths.
 *
 *	Only the text on the GtkEntry will be updated when a disk object
 *	path is dragged on to it, no "activate" signal will be emitted.
 */
void EDVEntrySetDND(edv_core_struct *core_ptr, GtkWidget *w)
{
        const GtkTargetEntry dnd_tar_types[] = {
{"text/plain",                          0,      EDV_DND_TYPE_INFO_TEXT_PLAIN},
{"text/uri-list",                       0,      EDV_DND_TYPE_INFO_TEXT_URI_LIST},
{"STRING",                              0,      EDV_DND_TYPE_INFO_STRING}
        };

	if((core_ptr == NULL) || (w == NULL))
	    return;

	if(!GTK_IS_ENTRY(w))
	    return;

        GUIDNDSetTar(
            w,
            dnd_tar_types,
            sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
            GDK_ACTION_COPY,			/* Actions. */
            GDK_ACTION_COPY,			/* Default action if same. */
            GDK_ACTION_COPY,			/* Default action. */
            EDVEntryDragDataReceivedCB,
            core_ptr
        );
        gtk_signal_connect_after(
            GTK_OBJECT(w), "drag_motion",
            GTK_SIGNAL_FUNC(EDVEntryDragMotionCB), core_ptr
        );
}


/*
 *	Coppies the paths of the given list of disk objects to the
 *	DDE as a space separated list of absolute paths.
 */
void EDVCopyDiskObjectsToDDEPath(
        edv_core_struct *core_ptr,
        edv_object_struct **list, gint total
)
{
	gint i;
	gchar *buf = NULL;
	edv_object_struct *obj;


	if((core_ptr == NULL) || (list == NULL) || (total <= 0))
	    return;


	for(i = 0; i < total; i++)
	{
	    obj = list[i];
	    if(obj == NULL)
		continue;

	    if(obj->full_path == NULL)
		continue;

	    buf = strcatalloc(buf, obj->full_path);
	    if((i + 1) < total)
		buf = strcatalloc(buf, " ");
	}

	if(buf != NULL)
	{
	    ClipboardPutBinary((const guint8 *)buf, strlen(buf));
	    g_free(buf);
	}
}

/*
 *      Coppies the paths of the given list of disk objects to the
 *      DDE as a space separated list of urls (without the localhost
 *	address specified).
 */
void EDVCopyDiskObjectsToDDEURL(
        edv_core_struct *core_ptr,
        edv_object_struct **list, gint total
)
{
	gint i;
        gchar *buf = NULL;
        edv_object_struct *obj;


        if((core_ptr == NULL) || (list == NULL) || (total <= 0))
            return;


        for(i = 0; i < total; i++)
        {
            obj = list[i];
            if(obj == NULL)
                continue;

            if(obj->full_path == NULL)
                continue;

	    buf = strcatalloc(buf, "file://");
            buf = strcatalloc(buf, obj->full_path);
            if((i + 1) < total)
                buf = strcatalloc(buf, " ");
        }

        if(buf != NULL)
        {
            ClipboardPutBinary((const guint8 *)buf, strlen(buf));
            g_free(buf);
        }
}


/*
 *	Appends a new operation history event item to the list on the
 *	given core structure.
 */
void EDVAppendHistory(
        edv_core_struct *core_ptr,
        gint type,              /* One of EDV_HISTORY_*. */
        gulong time_start,      /* Time the operation first started. */
        gulong time_end,        /* Time the operation ended. */
        gint status,
        const gchar *src_path, const gchar *tar_path,
        const gchar *comments
)
{
	gint max;
	edv_history_struct *h;


	if(core_ptr == NULL)
	    return;

	/* Get maximum history event items from configuration list on the
	 * core structure.
	 */
	max = EDVCFGItemListGetValueI(
	    core_ptr->cfg_list, EDV_CFG_PARM_HISTORY_EVENTS_MAX
	);

	/* Append a new history event item and discard any old items. */
	h = EDVHistoryListAppendNew(
	    &core_ptr->history_event, &core_ptr->total_history_events,
	    NULL, max
	);
	if(h != NULL)
	{
	    EDVHistoryDiskObjectSet(
		h, type, time_start, time_end, status,
		src_path, tar_path, comments
	    );
	}
}
