#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "../../include/string.h"
#include "../../include/disk.h"
#include "../guiutils.h"
#include "../cdialog.h"
#include "../clipboard.h"
#include "../fb.h"
#include "../pdialog.h"
#include "../lib/endeavour2.h"
#include "hview.h"
#include "config.h"

#include "../images/icon_edit_48x48.xpm"
#include "../images/icon_new_20x20.xpm"
#include "../images/icon_open_20x20.xpm"
#include "../images/icon_save_20x20.xpm"
#include "../images/icon_save_as_20x20.xpm"
#include "../images/icon_exit_20x20.xpm"
#include "../images/icon_cut_20x20.xpm"
#include "../images/icon_copy_20x20.xpm"
#include "../images/icon_paste_20x20.xpm"
#include "../images/icon_add_20x20.xpm"
#include "../images/icon_cancel_20x20.xpm"
#include "../images/icon_search_20x20.xpm"


/*
 *	Hex edit structure:
 */
typedef struct {

	GtkWidget	*toplevel,
			*main_vbox,
			*menu_bar_handle;

	GtkAccelGroup	*accelgrp;

	GtkWidget	*cut_mi,
			*copy_mi,
			*delete_mi,
			*edit_mode;
	hview_struct	*hv;
	gchar		*filename;

	guint8		*last_find_needle;
	gint		last_find_needle_len;

	edv_context_struct *ctx;

} hedit_struct;


static void HViewSignalCB(int s);
static void HEditDNDDragReceivedCB(
        GtkWidget *widget, GdkDragContext *dc, gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);
static void HEditNewCB(GtkWidget *widget, gpointer data);
static void HEditOpenCB(GtkWidget *widget, gpointer data);
static void HEditSaveCB(GtkWidget *widget, gpointer data);
static void HEditSaveAsCB(GtkWidget *widget, gpointer data);
static void HEditExitCB(GtkWidget *widget, gpointer data);
static void HEditCutCB(GtkWidget *widget, gpointer data);
static void HEditCopyCB(GtkWidget *widget, gpointer data);
static void HEditPasteCB(GtkWidget *widget, gpointer data);
static void HEditInsertCB(GtkWidget *widget, gpointer data);
static void HEditDeleteCB(GtkWidget *widget, gpointer data);
static void HEditFindCB(GtkWidget *widget, gpointer data);
static void HEditFindNextCB(GtkWidget *widget, gpointer data);
static void HEditEditModeCB(GtkWidget *widget, gpointer data);
static void HEditCalculateSelectedCB(GtkWidget *widget, gpointer data);

static void HEditBuildMenuBar(hedit_struct *he, GtkWidget *parent);
static void HEditUpdateMenus(hedit_struct *he);


#define DO_CHECK_CHANGES_AND_QUERY_USER \
{ \
        /* Check if current data has modifications and if so, query \
         * user to save first. \
         */ \
        if((hv != NULL) ? hv->modified : FALSE) \
        { \
            gint response; \
 \
            CDialogSetTransientFor(toplevel); \
            response = CDialogGetResponse( \
                "Save Changes?", \
"The data contains changes that have not been\n\
saved, do you want to save those changes?", \
"There are changes made to the data that have\n\
not been saved. If you want to save those changes,\n\
then click on \"Yes\". If you want to discard\n\
those changes then click on \"No\". If you do not\n\
want to continue with this operation then click\n\
on \"Cancel\".", \
                CDIALOG_ICON_WARNING, \
                CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO | \
                CDIALOG_BTNFLAG_CANCEL | CDIALOG_BTNFLAG_HELP, \
                CDIALOG_BTNFLAG_YES \
            ); \
            CDialogSetTransientFor(NULL); \
            switch(response) \
            { \
              case CDIALOG_RESPONSE_YES_TO_ALL: \
              case CDIALOG_RESPONSE_YES: \
                HEditSaveCB(NULL, data); \
                /* Still modified (changes not saved?). */ \
                if(hv->modified) \
                    return; \
                break; \
 \
              case CDIALOG_RESPONSE_NO: \
                break; \
 \
              default: \
                return; \
                break; \
            } \
        } \
}


/*
 *	Signal handler.
 */
static void HViewSignalCB(int s)
{
        switch(s)
        {
          case SIGINT:
          case SIGTERM:
          case SIGSEGV:
            exit(1);
            break;
        }
}

/*
 *	Hex view "delete_event" callback.
 */
static gint HEditCloseCB(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	HEditExitCB(widget, data);
	return(TRUE);
}

/*
 *	Hex view "drag_data_received" callback.
 *
 *	Opens the object specified by the url in the drag data.
 */
static void HEditDNDDragReceivedCB(
        GtkWidget *widget, GdkDragContext *dc, gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
        GtkWidget *w, *toplevel;
	hview_struct *hv;
	hedit_struct *he = (hedit_struct *)data;
        if((widget == NULL) || (dc == NULL) || (he == NULL))
            return;

	toplevel = he->toplevel;
	hv = he->hv;
	if(hv == NULL)
	    return;

	DO_CHECK_CHANGES_AND_QUERY_USER

        HViewSetBusy(hv, TRUE);

        /* Get hv's view drawing area widget. */
        w = hv->view_da;

        /* View widget the same as the callback widget? */
        if(w == widget)
        {
            /* Check if the DND target type is one that we support. */
            if((info == HEDIT_DND_TYPE_INFO_TEXT_PLAIN) ||
               (info == HEDIT_DND_TYPE_INFO_TEXT_URI_LIST) ||
               (info == HEDIT_DND_TYPE_INFO_STRING)
            )
            {
                const gchar *cstrptr;
                const gchar *url = (const gchar *)selection_data->data;
                gchar *path;


                /* The DND data should be a URL string, we need to parse
                 * it and get the absolute path portion of that URL
		 * string.
                 */
                cstrptr = strstr(url, "://");
                if(cstrptr != NULL)
                    cstrptr += strlen("://");
                else
                    cstrptr = url;

                cstrptr = strchr(cstrptr, '/');
                path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;

                /* Got absolute path to object to be loaded? */
                if(path != NULL)
                {
		    g_free(he->filename);
		    he->filename = g_strdup(path);
		    HViewLoadFile(hv, he->filename);

                    g_free(path);
                }
            }
        }

	HEditUpdateMenus(he);
	HViewSetBusy(hv, FALSE);
}

/*
 *	New callback.
 */
static void HEditNewCB(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
        hview_struct *hv;
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        toplevel = he->toplevel;
        hv = he->hv;
        if(hv == NULL)
            return;

	DO_CHECK_CHANGES_AND_QUERY_USER

        HViewSetBusy(hv, TRUE);

	HViewClear(hv);

	hv->modified = FALSE;

	g_free(he->filename);
	he->filename = NULL;

        HEditUpdateMenus(he);
        HViewSetBusy(hv, FALSE);
}

/*
 *	Open callback.
 */
static void HEditOpenCB(GtkWidget *widget, gpointer data)
{
        gboolean status;
        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;
	GtkWidget *toplevel;
	hview_struct *hv;
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

	toplevel = he->toplevel;
	hv = he->hv;
	if(hv == NULL)
	    return;

        if(FileBrowserIsQuery())
            return;

	DO_CHECK_CHANGES_AND_QUERY_USER

	HViewSetBusy(hv, TRUE);

        /* Create file types list. */
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

        /* Query user for file to open. */
        FileBrowserSetTransientFor(toplevel);
        status = FileBrowserGetResponse(
            "Open File",
            "Open", "Cancel",
            NULL,		/* Use last path. */
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

        /* Got user response? */
        if(status)
        {
            const gchar *new_path = (total_path_rtns > 0) ?
                path_rtn[0] : NULL;

            g_free(he->filename);
            he->filename = g_strdup(new_path);
	    HViewLoadFile(hv, he->filename);
	}

        /* Deallocate file types list. */
        FileBrowserDeleteTypeList(ftype, total_ftypes);

	HEditUpdateMenus(he);
	HViewSetBusy(hv, FALSE);
}

/*
 *      Save callback.
 */
static void HEditSaveCB(GtkWidget *widget, gpointer data)
{
        GtkWidget *toplevel;
        hview_struct *hv;
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        toplevel = he->toplevel;
        hv = he->hv;
        if(hv == NULL)
            return;

	/* If there is no filename to save to then call save as callback
	 * so user can be queried for a file name.
	 */
	if(he->filename == NULL)
	{
	    HEditSaveAsCB(widget, data);
	    return;
	}

        HViewSetBusy(hv, TRUE);

	/* Save data to file. */
	if(HViewSaveFile(hv, he->filename))
	{
	    gchar *buf = g_strdup_printf(
"Error occured while saving file:\n\n    %s",
		he->filename
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Save Error",
		buf,
"Please check if you have sufficient permission to\n\
save to the specified file and that the disk capacity\n\
has not been exceeded.",
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	}
	else
	{
	    EDVNotifyQueueObjectModified(
		he->ctx, he->filename, NULL
	    );
	    EDVContextSync(he->ctx);

	    hv->modified = FALSE;
	}

        HEditUpdateMenus(he);
        HViewSetBusy(hv, FALSE);
}

/*
 *	Save as callback.
 */
static void HEditSaveAsCB(GtkWidget *widget, gpointer data)
{
        gboolean status;
        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;
        GtkWidget *toplevel;
        hview_struct *hv;
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        toplevel = he->toplevel;
        hv = he->hv;
        if(hv == NULL)
            return;

        if(FileBrowserIsQuery())
            return;

        HViewSetBusy(hv, TRUE);

        /* Create file types list. */
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

        /* Query user for file to save. */
        FileBrowserSetTransientFor(toplevel);
        status = FileBrowserGetResponse(
            "Save File",
            "Save", "Cancel",
            NULL,               /* Use last path. */
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

        /* Got user response? */
        if(status)
        {
	    gboolean file_exists = FALSE;
            const gchar *new_path = (total_path_rtns > 0) ?
                path_rtn[0] : NULL;

	    if(new_path != NULL)
	    {
		gboolean allow_save = TRUE;

		/* Check if file exists. */
		file_exists = access(new_path, F_OK) ? FALSE : TRUE;
		if(file_exists)
		{
		    gint response;
		    gchar *buf = g_strdup_printf(
"Overwrite existing file:\n\n    %s",
			new_path
		    );
		    CDialogSetTransientFor(toplevel);
		    response = CDialogGetResponse(
			"Confirm Overwrite",
			buf,
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_YES
		    );
		    CDialogSetTransientFor(NULL);
		    g_free(buf);
		    switch(response)
		    {
		      case CDIALOG_RESPONSE_YES_TO_ALL:
		      case CDIALOG_RESPONSE_YES:
			break;

		      default:
			allow_save = FALSE;
			break;
		    }
		}

		/* File not exist or user permits ovewrite? */
		if(allow_save)
		{
		    /* Update filename. */
		    g_free(he->filename);
		    he->filename = g_strdup(new_path);

		    /* Save data to new file name. */
		    if(HViewSaveFile(hv, he->filename))
		    {
			gchar *buf = g_strdup_printf(
"Error occured while saving file:\n\n    %s",
			    he->filename
			);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
			    "Save Error",
			    buf,
"Please check if you have sufficient permission to\n\
save to the specified file and that the disk capacity\n\
has not been exceeded.",
			    CDIALOG_ICON_ERROR,
			    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(buf);
		    }
		    else
		    {
			if(file_exists)
			    EDVNotifyQueueObjectModified(
				he->ctx, he->filename, NULL
			    );
			else
                            EDVNotifyQueueObjectAdded(
                                he->ctx, he->filename
                            );
			EDVContextSync(he->ctx);

			hv->modified = FALSE;
		    }
		}
	    }
        }	/* Got user response? */

        /* Deallocate file types list. */
        FileBrowserDeleteTypeList(ftype, total_ftypes);

        HEditUpdateMenus(he);
        HViewSetBusy(hv, FALSE);
}

/*
 *	Exit callback.
 */
static void HEditExitCB(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	hview_struct *hv;
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        toplevel = he->toplevel;
        hv = he->hv;
        if(hv == NULL)
            return;

	DO_CHECK_CHANGES_AND_QUERY_USER

        gtk_main_quit();
}

/*
 *	Cut callback.
 */
static void HEditCutCB(GtkWidget *widget, gpointer data)
{
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

	HViewCut(he->hv);
}

/*
 *      Copy callback.
 */
static void HEditCopyCB(GtkWidget *widget, gpointer data)
{
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        HViewCopy(he->hv);
}

/*
 *      Paste callback.
 */
static void HEditPasteCB(GtkWidget *widget, gpointer data)
{
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        HViewPaste(he->hv);
}

/*
 *      Insert callback.
 */
static void HEditInsertCB(GtkWidget *widget, gpointer data)
{
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        HViewInsert(he->hv);
}

/*
 *      Delete callback.
 */
static void HEditDeleteCB(GtkWidget *widget, gpointer data)
{
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        HViewDeleteSelected(he->hv);
}

/*
 *	Find callback.
 */
static void HEditFindCB(GtkWidget *widget, gpointer data)
{
	gchar **strv, *last_needle = NULL;
	gint strc;
	GtkWidget *toplevel;
	hview_struct *hv;
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

	toplevel = he->toplevel;
	hv = he->hv;
	if(hv == NULL)
	    return;

	if(PDialogIsQuery())
	    return;

        HViewSetBusy(hv, TRUE);

	PDialogDeleteAllPrompts();
	if((he->last_find_needle != NULL) &&
	   (he->last_find_needle_len > 0)
	)
	{
	    last_needle = (gchar *)g_malloc(
		(he->last_find_needle_len + 1) * sizeof(gchar)
	    );
	    memcpy(
		last_needle, he->last_find_needle, 
		he->last_find_needle_len * sizeof(gchar)
	    );
	    last_needle[he->last_find_needle_len] = '\0';
	}
	PDialogAddPrompt(NULL, "Find:", last_needle);
	g_free(last_needle);
	last_needle = NULL;
	PDialogSetSize(320, -1);
	PDialogSetTransientFor(toplevel);
	strv = PDialogGetResponse(
	    "Find:", NULL, NULL,
	    PDIALOG_ICON_SEARCH,
	    "Find", "Cancel",
	    PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
	    PDIALOG_BTNFLAG_SUBMIT,
	    &strc
	);
	PDialogSetTransientFor(NULL);

	if((strv != NULL) && (strc > 0))
	{
	    gint i;
	    const gchar *cstrptr = strv[0];
	    if((cstrptr != NULL) && (*cstrptr != '\0'))
	    {
		/* Need to convert the user given string from ASCII to
		 * hex and update the last find needle.
		 */
		g_free(he->last_find_needle);
		he->last_find_needle = (guint8 *)g_strdup(cstrptr);
		he->last_find_needle_len = strlen(cstrptr);

		/* Do search. */
		i = HViewFind(
		    hv,
		    he->last_find_needle, he->last_find_needle_len,
		    hv->buf_pos + 1,	/* Start index. */
		    FALSE		/* Not case sensitive. */
		);
		/* Got match? */
		if(i > -1)
		{
		    /* Scroll to position. */
		    HViewSetPosition(hv, i);
		    HViewSelect(hv, i, i + he->last_find_needle_len - 1);
                    if(HViewIsBufferPositionVisible(hv, i) !=
                        GTK_VISIBILITY_FULL
                    )
                        HViewScrollTo(hv, i, 0.5);
		}
		else
		{
		    /* No match, then start search again from the
		     * beginning.
		     */
		    i = HViewFind(
			hv,
			he->last_find_needle, he->last_find_needle_len,
			0,		/* Start index. */
			FALSE		/* Not case sensitive. */
		    );
                    /* Got match the second time? */
                    if(i > -1)
                    {
                        /* Scroll to position. */
                        HViewSetPosition(hv, i);
                        HViewSelect(hv, i, i + he->last_find_needle_len - 1);
                        if(HViewIsBufferPositionVisible(hv, i) !=
                            GTK_VISIBILITY_FULL
                        )
                            HViewScrollTo(hv, i, 0.5);
                    }
		    else
		    {
			/* No match. */
			HViewSetStatusMessage(
			    hv,
			    "String not found",
			    FALSE
			);
		    }	/* Got match the second time? */
		}	/* Got match? */
	    }
	}

        HEditUpdateMenus(he);
        HViewSetBusy(hv, FALSE);
}

/*
 *      Find next callback.
 */
static void HEditFindNextCB(GtkWidget *widget, gpointer data)
{
        GtkWidget *toplevel;
        hview_struct *hv;
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        toplevel = he->toplevel;
        hv = he->hv;
        if(hv == NULL)
            return;

	if(he->last_find_needle == NULL)
	    return;

        HViewSetBusy(hv, TRUE);

	if(1)
	{
	    /* Do search. */
	    gint i = HViewFind(
		hv,
		he->last_find_needle, he->last_find_needle_len,
		hv->buf_pos + 1,	/* Start index. */
		FALSE			/* Not case sensitive. */
	    );
	    /* Got match? */
	    if(i > -1)
	    {
		/* Scroll to position. */
		HViewSetPosition(hv, i);
		HViewSelect(hv, i, i + he->last_find_needle_len - 1);
		if(HViewIsBufferPositionVisible(hv, i) !=
		    GTK_VISIBILITY_FULL
		)
		    HViewScrollTo(hv, i, 0.5);
	    }
	    else
	    {
		/* No match, then start search again from the
		 * beginning.
		 */
		i = HViewFind(
		    hv,
		    he->last_find_needle, he->last_find_needle_len,
		    0,		/* Start index. */
		    FALSE	/* Not case sensitive. */
		);
		/* Got match the second time? */
		if(i > -1)
		{
		    /* Scroll to position. */
		    HViewSetPosition(hv, i);
		    HViewSelect(hv, i, i + he->last_find_needle_len - 1);
		    if(HViewIsBufferPositionVisible(hv, i) !=
			GTK_VISIBILITY_FULL
		    )
			HViewScrollTo(hv, i, 0.5);
		}
		else
		{
		    /* No match. */
		    HViewSetStatusMessage(
			hv,
			"String not found",
			FALSE
		    );
		}	/* Got match the second time? */
	    }		/* Got match? */
	}

        HEditUpdateMenus(he);
        HViewSetBusy(hv, FALSE);
}

/*
 *	Change edit mode callback.
 */
static void HEditEditModeCB(GtkWidget *widget, gpointer data)
{
        GtkWidget *toplevel;
        hview_struct *hv;
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        toplevel = he->toplevel;
        hv = he->hv;
        if(hv == NULL)
            return;

	if(hv->edit_mode == HVIEW_EDIT_MODE_HEX)
	    HViewSetEditMode(hv, HVIEW_EDIT_MODE_ASCII);
	else
	    HViewSetEditMode(hv, HVIEW_EDIT_MODE_HEX);

        HEditUpdateMenus(he);
}

/*
 *	Calculate selected callback.
 */
static void HEditCalculateSelectedCB(GtkWidget *widget, gpointer data)
{
	gint pos;
	guint8 i8 = 0, i8s = 0;
	guint16 i16 = 0, i16s = 0;
	guint32 i24 = 0, i24s = 0;
	guint32 i32 = 0, i32s = 0;
	guint64 i64 = 0, i64s = 0;
	const guint8 *buf;
        GtkWidget *toplevel;
        hview_struct *hv;
        hedit_struct *he = (hedit_struct *)data;
        if(he == NULL)
            return;

        toplevel = he->toplevel;
        hv = he->hv;
        if(hv == NULL)
            return;

	buf = hv->buf;
	if(buf == NULL)
	    return;

	if(CDialogIsQuery())
	    return;

	/* Check if nothing is selected. */
	if(HVIEW_IS_BUF_SELECTED(hv))
	{
	    gint	i = hv->buf_sel_start,
			n = hv->buf_sel_end;
	    pos = i;
	    if((i >= 0) && (i < hv->buf_len) &&
               (n >= 0) && (n < hv->buf_len)
	    )
	    {
		gint len;

		if(i > n)
		{
		    gint t = i;
		    i = n;
		    n = t;
		}

		len = n - i + 1;

		if(len >= sizeof(guint8))
		    i8 = *(guint8 *)(&buf[i]);
                if(len >= sizeof(guint16))
                    i16 = *(guint16 *)(&buf[i]);
                if(len >= sizeof(guint32))
                    i32 = *(guint32 *)(&buf[i]);
                if(len >= sizeof(guint64))
                    i64 = *(guint64 *)(&buf[i]);

                i24 = (i32 & 0x00ffffff);
	    }
	}
	else
	{
	    gint i = hv->buf_pos;
	    pos = i;
            if((i >= 0) && (i < hv->buf_len))
	    {
                if((i + sizeof(guint8)) <= hv->buf_len)
                    i8 = *(guint8 *)(&buf[i]);
                if((i + sizeof(guint16)) <= hv->buf_len)
                    i16 = *(guint16 *)(&buf[i]);
                if((i + sizeof(guint32)) <= hv->buf_len)
                    i32 = *(guint32 *)(&buf[i]);
                if((i + sizeof(guint64)) <= hv->buf_len)
                    i64 = *(guint64 *)(&buf[i]);

                i24 = (i32 & 0x00ffffff);
	    }
	}

	/* Calculate swapped values. */
	if(1)
	{
	    i8s = i8;

	    i16s |= ((i16 & 0x00ff) << 8);
	    i16s |= ((i16 & 0xff00) >> 8);

            i24s |= ((i24 & 0x000000ff) << 16);
            i24s |= ((i24 & 0x0000ff00) >> 0);
            i24s |= ((i24 & 0x00ff0000) >> 16);

            i32s |= ((i32 & 0x000000ff) << 24);
            i32s |= ((i32 & 0x0000ff00) << 8);
            i32s |= ((i32 & 0x00ff0000) >> 8);
            i32s |= ((i32 & 0xff000000) >> 24);

            i64s |= ((i64 & 0x00000000000000ff) << 56);
            i64s |= ((i64 & 0x000000000000ff00) << 40);
            i64s |= ((i64 & 0x0000000000ff0000) << 24);
            i64s |= ((i64 & 0x00000000ff000000) << 8);
            i64s |= ((i64 & 0x000000ff00000000) >> 8);
            i64s |= ((i64 & 0x0000ff0000000000) >> 24);
            i64s |= ((i64 & 0x00ff000000000000) >> 40);
            i64s |= ((i64 & 0xff00000000000000) >> 56);
	}


	if(1)
	{
	    gchar *buf = g_strdup_printf(
"Position: %i\n\
8 Bit Value: %i  Unsigned: %i  0x%.2X\n\
16 Bit Value: %i  Unsigned: %i  0x%.4X\n\
24 Bit Value: %i  Unsigned: %i  0x%.6X\n\
32 Bit Value: %i  Unsigned: %i  0x%.8X\n\
64 Bit Value: %ld  Unsigned: %ld  0x%.16LX\n\
\n\
Swapped Values:\n\
8 Bit Value: %i  Unsigned: %i  0x%.2X\n\
16 Bit Value: %i  Unsigned: %i  0x%.4X\n\
24 Bit Value: %i  Unsigned: %i  0x%.6X\n\
32 Bit Value: %i  Unsigned: %i  0x%.8X\n\
64 Bit Value: %ld  Unsigned: %ld  0x%.16LX",
		pos,
		(gint8)((gint32)i8 - (0xff / 2) - 1), i8, i8,
		(gint16)((gint32)i16 - (0xffff / 2) - 1), i16, i16,
                (gint32)((gint64)i24 - (0xffffff / 2) - 1), i24, i24,
                (gint32)((gint64)i32 - (0xffffffff / 2) - 1), i32, i32,
		(unsigned long)((gint64)i64 - (0xffffffffffffffff / 2) -
1), 
(unsigned long)i64,
(long long unsigned int)i64,
		(gint8)((gint32)i8s - (0xff / 2) - 1), i8s, i8s,
                (gint16)((gint32)i16s - (0xffff / 2) - 1), i16s, i16s,
                (gint32)((gint64)i24s - (0xffffff / 2) - 1), i24s, i24s,
                (gint32)((gint64)i32s - (0xffffffff / 2) - 1), i32s, i32s,
                (unsigned long)((gint64)i64s - (0xffffffffffffffff / 2)
- 1),
(unsigned long)i64s,
(long long unsigned int)i64s
	    );
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(
                "Results",
		buf,
		NULL,
                CDIALOG_ICON_INFO,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
	    g_free(buf);
	}


        HEditUpdateMenus(he);
}



/*
 *	Builds the menu bar.
 */
static void HEditBuildMenuBar(hedit_struct *he, GtkWidget *parent)
{
	u_int8_t **icon;
	const gchar *label;
	gint accel_key;
	guint accel_mods;
	gpointer mclient_data = he;
	void (*func_cb)(GtkWidget *w, gpointer);
	gpointer fw;
        GtkAccelGroup *accelgrp;
        GtkWidget *menu_bar, *menu, *w;


        if((he == NULL) || (parent == NULL))
            return;

        /* Get keyboard accelerator group. */
        accelgrp = he->accelgrp;

        /* Create menu bar. */
        menu_bar = (GtkWidget *)GUIMenuBarCreate(NULL);
	if(GTK_IS_BOX(parent))
	    gtk_box_pack_start(GTK_BOX(parent), menu_bar, FALSE, FALSE, 0);
	else
	    gtk_container_add(GTK_CONTAINER(parent), menu_bar);
        gtk_widget_show(menu_bar);

#define DO_ADD_MENU_ITEM_LABEL  \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp, \
  icon, label, accel_key, accel_mods, (gpointer *)&fw, \
  mclient_data, func_cb \
 ); \
}
#define DO_ADD_MENU_ITEM_CHECK  \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_CHECK, accelgrp, \
  icon, label, accel_key, accel_mods, (gpointer *)&fw, \
  mclient_data, func_cb \
 ); \
}
#define DO_ADD_MENU_SEP         \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

        /* Create file menu. */
        menu = GUIMenuCreateTearOff();
        if(menu != NULL)
        {
            icon = (u_int8_t **)icon_new_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "New";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "New";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "New";
#endif
            accel_key = 'n';
            accel_mods = GDK_CONTROL_MASK;
            func_cb = HEditNewCB;
            DO_ADD_MENU_ITEM_LABEL

            icon = (u_int8_t **)icon_open_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Open...";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Open...";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Open...";
#endif
            accel_key = 'o';
            accel_mods = GDK_CONTROL_MASK;
            func_cb = HEditOpenCB;
            DO_ADD_MENU_ITEM_LABEL

            icon = (u_int8_t **)icon_save_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Save";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Save";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Save";
#endif
            accel_key = 's';
            accel_mods = GDK_CONTROL_MASK;
            func_cb = HEditSaveCB;
            DO_ADD_MENU_ITEM_LABEL

            icon = (u_int8_t **)icon_save_as_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Save As...";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Save As...";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Save As...";
#endif
            accel_key = 'a';
            accel_mods = GDK_CONTROL_MASK;
            func_cb = HEditSaveAsCB;
            DO_ADD_MENU_ITEM_LABEL

	    DO_ADD_MENU_SEP

            icon = (u_int8_t **)icon_exit_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Exit";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Exit";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Exit";
#endif
            accel_key = 'q';
            accel_mods = GDK_CONTROL_MASK;
            func_cb = HEditExitCB;
            DO_ADD_MENU_ITEM_LABEL

	}
        GUIMenuAddToMenuBar(
            menu_bar, menu,
#ifdef PROG_LANGUAGE_ENGLISH
            "File",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Archivo",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Fichier",
#endif
            GUI_MENU_BAR_ALIGN_LEFT
        );

        /* Create edit menu. */
        menu = GUIMenuCreateTearOff();
        if(menu != NULL)
        {
            icon = (u_int8_t **)icon_cut_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Cut";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Cut";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Cut";
#endif
            accel_key = 'x';
            accel_mods = GDK_CONTROL_MASK;
            func_cb = HEditCutCB;
            DO_ADD_MENU_ITEM_LABEL
	    he->cut_mi = w;

            icon = (u_int8_t **)icon_copy_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Copy";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Copy";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Copy";
#endif
            accel_key = 'c';
            accel_mods = GDK_CONTROL_MASK;
            func_cb = HEditCopyCB;
            DO_ADD_MENU_ITEM_LABEL
	    he->copy_mi = w;

            icon = (u_int8_t **)icon_paste_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Paste";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Paste";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Paste";
#endif
            accel_key = 'v';
            accel_mods = GDK_CONTROL_MASK;
            func_cb = HEditPasteCB;
            DO_ADD_MENU_ITEM_LABEL

	    DO_ADD_MENU_SEP

            icon = (u_int8_t **)icon_add_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Insert";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Insert";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Insert";
#endif
            accel_key = GDK_Insert;
            accel_mods = 0;
            func_cb = HEditInsertCB;
            DO_ADD_MENU_ITEM_LABEL

            icon = (u_int8_t **)icon_cancel_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Delete";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Delete";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Delete";
#endif
            accel_key = GDK_Delete;
            accel_mods = 0;
            func_cb = HEditDeleteCB;
            DO_ADD_MENU_ITEM_LABEL
	    he->delete_mi = w;

	    DO_ADD_MENU_SEP

            icon = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Change Edit Mode";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Change Edit Mode";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Change Edit Mode";
#endif
            accel_key = GDK_F4;
            accel_mods = 0;
            func_cb = HEditEditModeCB;
            DO_ADD_MENU_ITEM_LABEL

	    DO_ADD_MENU_SEP

            icon = (u_int8_t **)icon_search_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Find...";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Find...";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Find...";
#endif
            accel_key = GDK_F5;
            accel_mods = 0;
            func_cb = HEditFindCB;
            DO_ADD_MENU_ITEM_LABEL

            icon = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Find Next";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Find Next";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Find Next";
#endif
            accel_key = GDK_F6;
            accel_mods = 0;
            func_cb = HEditFindNextCB;
            DO_ADD_MENU_ITEM_LABEL

	}
        GUIMenuAddToMenuBar(
            menu_bar, menu,
#ifdef PROG_LANGUAGE_ENGLISH
            "Edit",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Redacta",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "diter",
#endif
            GUI_MENU_BAR_ALIGN_LEFT
        );

        /* Create tools menu. */
        menu = GUIMenuCreateTearOff();
        if(menu != NULL)
        {
            icon = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Calculate Selected";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Calculate Selected";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Calculate Selected";
#endif
            accel_key = GDK_F9;
            accel_mods = 0;
            func_cb = HEditCalculateSelectedCB;
            DO_ADD_MENU_ITEM_LABEL

        }
        GUIMenuAddToMenuBar(
            menu_bar, menu,
#ifdef PROG_LANGUAGE_ENGLISH
            "Tools",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Tools",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Tools",
#endif
            GUI_MENU_BAR_ALIGN_LEFT
        );



#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_ITEM_CHECK
#undef DO_ADD_MENU_SEP
}

/*
 *	Updates menus on the hv (but not the hv).
 */
static void HEditUpdateMenus(hedit_struct *he)
{
	gboolean sensitive;
	GtkWidget *w;
        hview_struct *hv;


	if(he == NULL)
	    return;

	hv = he->hv;
	if(hv == NULL)
	    return;

	w = he->toplevel;
	if(w != NULL)
	{
	    const gchar *cstrptr = (he->filename != NULL) ?
		strrchr(he->filename, DIR_DELIMINATOR) : NULL;
	    gchar *title = g_strdup_printf(
		PROG_NAME ": %s",
		(cstrptr != NULL) ? cstrptr + 1 :
		    ((he->filename != NULL) ? he->filename : "Untitled")
	    );
	    gtk_window_set_title(GTK_WINDOW(w), title);
	    g_free(title);
	}

	sensitive = ((hv->buf_sel_start >= 0) && (hv->buf_sel_end >= 0)) ?
	    TRUE : FALSE;
/*
	w = he->cut_mi;
	if(w != NULL)
	    gtk_widget_set_sensitive(w, sensitive);
        w = he->copy_mi;
        if(w != NULL)
            gtk_widget_set_sensitive(w, sensitive);
        w = he->delete_mi;
        if(w != NULL)
            gtk_widget_set_sensitive(w, sensitive);
 */

}

#undef DO_CHECK_CHANGES_AND_QUERY_USER


int main(int argc, char *argv[])
{
	gboolean initialized_gtk = FALSE;
	gint i;
	const gchar *arg_ptr;
	GdkWindow *window;
	GtkWidget *w;
	hedit_struct *he;
	hview_struct *hv;
        edv_context_struct *ctx;


        /* Set up time zone. */
        tzset();

        /* Set up signal callbacks. */
        signal(SIGINT, HViewSignalCB);
        signal(SIGTERM, HViewSignalCB);
        signal(SIGKILL, HViewSignalCB);
        signal(SIGSEGV, HViewSignalCB);
        signal(SIGSTOP, HViewSignalCB);
        signal(SIGCONT, HViewSignalCB);
        signal(SIGPIPE, HViewSignalCB);

	/* Handle for arguments. */
	for(i = 1; i < argc; i++)
	{
	    arg_ptr = argv[i];
	    if(arg_ptr == NULL)
		continue;

	    /* Help? */
	    if(!strcasecmp(arg_ptr, "--help") ||
               !strcasecmp(arg_ptr, "-help") ||
               !strcasecmp(arg_ptr, "--h") ||
               !strcasecmp(arg_ptr, "-h") ||
               !strcasecmp(arg_ptr, "-?")
	    )
	    {
		printf(PROG_HELP_MESG);
		return(0);
	    }
	    /* Version? */
	    else if(!strcasecmp(arg_ptr, "--version") ||
                    !strcasecmp(arg_ptr, "-version")
            )
            {
                printf(
		    "%s %s\n%s",
		    PROG_NAME, PROG_VERSION, PROG_COPYRIGHT
		);
                return(0);
            }

	}

        /* Initialize GTK+ as needed. */
        if(!initialized_gtk)
        {
            if(!gtk_init_check(&argc, &argv))
            {
                fprintf(
                    stderr,
 "This program requires X.\n"
                );
                return(1);
            }
            initialized_gtk = TRUE;
        }

        /* Initialize GDK RGB buffers system. */
        gdk_rgb_init();

        /* Initialize dialogs. */
        CDialogInit();
	ClipboardBrowserInit();
	FileBrowserInit();
        PDialogInit();

        /* Initialize Endeavour context. */
        ctx = EDVContextNew();
        EDVContextLoadConfigurationFile(ctx, NULL);


	/* Begin creating hex edit window. */
	he = (hedit_struct *)g_malloc0(sizeof(hedit_struct));
	he->accelgrp = gtk_accel_group_new();
	he->ctx = ctx;

        /* Toplevel. */
	he->toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
        gtk_widget_set_usize(w, 640, 480);
        gtk_window_set_policy(
            GTK_WINDOW(w),
            TRUE, TRUE, TRUE
        );
        gtk_widget_realize(w);
        window = w->window;
        if(window != NULL)
        {
	    GdkGeometry geometry;

            geometry.min_width = 100;
            geometry.min_height = 70;

            geometry.base_width = 0;
            geometry.base_height = 0;

            geometry.width_inc = 1;
            geometry.height_inc = 1;
/*
            geometry.min_aspect = 1.3;
            geometry.max_aspect = 1.3;
 */
            gdk_window_set_geometry_hints(
                window, &geometry,
                GDK_HINT_MIN_SIZE |
                GDK_HINT_BASE_SIZE |
                /* GDK_HINT_ASPECT | */
                GDK_HINT_RESIZE_INC
            );

            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE | GDK_DECOR_MAXIMIZE
            );
            gdk_window_set_functions(
                window,
		GDK_FUNC_MOVE | GDK_FUNC_RESIZE |
		GDK_FUNC_MINIMIZE | GDK_FUNC_MAXIMIZE |
		GDK_FUNC_CLOSE
            );

            GUISetWMIcon(window, (u_int8_t **)icon_edit_48x48_xpm);
        }
        gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_accel_group_attach(he->accelgrp, GTK_OBJECT(w));

	he->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(he->toplevel), w);
	gtk_widget_show(w);

	he->menu_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(he->main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	HEditBuildMenuBar(he, he->menu_bar_handle);

	he->hv = hv = HViewNew(he->main_vbox);
	HViewMap(hv);
        gtk_signal_connect(
            GTK_OBJECT(he->toplevel), "delete_event",
            GTK_SIGNAL_FUNC(HEditCloseCB), he
        );
        /* Set up hv for DND. */
        w = hv->view_da;
        if(w != NULL)
        {
            const GtkTargetEntry dnd_tar_types[] = {
{"text/plain",          0,              HEDIT_DND_TYPE_INFO_TEXT_PLAIN},
{"text/uri-list",       0,              HEDIT_DND_TYPE_INFO_TEXT_URI_LIST},
{"STRING",              0,              HEDIT_DND_TYPE_INFO_STRING}
            };
            GUIDNDSetTar(
                w,
                dnd_tar_types,
                sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
                GDK_ACTION_DEFAULT |            /* Drag actions. */
                GDK_ACTION_LINK |
                GDK_ACTION_MOVE |
                GDK_ACTION_COPY,
                GDK_ACTION_MOVE,                /* Def act if same. */
                GDK_ACTION_COPY,                /* Def act. */
                HEditDNDDragReceivedCB,
                he
            );
        }
	HEditUpdateMenus(he);
	gtk_widget_show(he->toplevel);

	if(argc > 1)
	{
	    HViewSetBusy(hv, TRUE);
	    g_free(he->filename);
	    he->filename = g_strdup(argv[1]);
	    HViewLoadFile(hv, he->filename);
	    HViewSetBusy(hv, FALSE);
	}


	gtk_main();

	/* Delete hex edit window. */
        HViewClear(hv);
	HViewDelete(hv);
	he->hv = hv = NULL;

	gtk_widget_destroy(he->menu_bar_handle);
	he->menu_bar_handle = NULL;
	gtk_widget_destroy(he->toplevel);
	he->toplevel = NULL;

	gtk_accel_group_unref(he->accelgrp);
	he->accelgrp = NULL;

	g_free(he->filename);
	g_free(he->last_find_needle);
	g_free(he);
	he = NULL;


	/* Delete Endeavour context. */
        EDVContextSync(ctx);
        EDVContextDelete(ctx);
        ctx = NULL;

	/* Shutdown dialogs. */
	CDialogShutdown();
	ClipboardBrowserShutdown();
	FileBrowserShutdown();
	PDialogShutdown();

	return(0);
}
