#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvobj.h"
#include "edvdevices.h"
#include "edvstatusbar.h"
#include "imbr.h"
#include "imbrcb.h"
#include "imbropcb.h"
#include "imbrtlist.h"
#include "imbrimgview.h"
#include "imbrdnd.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvlistseek.h"
#include "edvcfglist.h"
#include "config.h"



void EDVImbrTListItemDestroyCB(gpointer data);

gint EDVImbrDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
void EDVImbrDestroyCB(GtkObject *object, gpointer data);

gint EDVImbrKeyEventCB(
         GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint EDVImbrButtonPressEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
);

void EDVImbrHandleChildAttachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void EDVImbrHandleChildDetachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

void EDVImbrTListSelectCB(
        gpointer tl, GdkEventButton *button, gint thumb_num, gpointer data
);
void EDVImbrTListUnselectCB(
        gpointer tl, GdkEventButton *button, gint thumb_num, gpointer data
);

void EDVImbrComboActivateCB(GtkWidget *widget, gpointer data);

gint EDVImbrLoadingTOCB(gpointer data);

void EDVImbrWriteProtectChangedCB(
        edv_imbr_struct *imbr, gbool state
);
void EDVImbrObjectAddedNotifyCB(
        edv_imbr_struct *imbr, const gchar *path,
        const struct stat *lstat_buf
);
void EDVimbrObjectModifiedNotifyCB(
        edv_imbr_struct *imbr, const gchar *path,
        const gchar *new_path,
        const struct stat *lstat_buf
);
void EDVImbrObjectRemovedNotifyCB(
        edv_imbr_struct *imbr, const gchar *path
);

void EDVImbrMountNotifyCB(
        edv_imbr_struct *imbr,
        gint dev_num, edv_device_struct *dev_ptr,
        gbool is_mounted
);

void EDVImbrRecycledObjectAddedNotifyCB(
        edv_imbr_struct *imbr, guint index
);
void EDVImbrRecycledObjectRemovedNotifyCB(
        edv_imbr_struct *imbr, guint index
);

void EDVImbrReconfiguredNotifyCB(edv_imbr_struct *imbr);

void EDVImbrMimeTypeAddedCB(
        edv_imbr_struct *imbr,
        gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVImbrMimeTypeModifiedCB(
        edv_imbr_struct *imbr,
        gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVImbrMimeTypeRemovedCB(
        edv_imbr_struct *imbr, gint mt_num
);

void EDVImbrDeviceAddedCB(
        edv_imbr_struct *imbr,
        gint dev_num, edv_device_struct *dev_ptr
);
void EDVImbrDeviceModifiedCB(
        edv_imbr_struct *imbr,
        gint dev_num, edv_device_struct *dev_ptr
);
void EDVImbrDeviceRemovedCB(
        edv_imbr_struct *imbr, gint dev_num
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Image browser thumbs list "destroy" signal callback.
 */
void EDVImbrTListItemDestroyCB(gpointer data)
{
        EDVObjectDelete((edv_object_struct *)data);
}


/*
 *	Image browser toplevel GtkWidget "delete_event" signal callback.
 */
gint EDVImbrDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_imbr_struct *imbr = (edv_imbr_struct *)data;
        if(imbr == NULL)
            return(TRUE);

	if(imbr->processing)
	    return(TRUE);

	EDVImbrOPClose(imbr);

	return(TRUE);
}

/*
 *	Image browser toplevel GtkWidget "destroy" signal callback.
 */
void EDVImbrDestroyCB(GtkObject *object, gpointer data)
{
	return;
}


/*
 *	Image browser "key_press_event" or "key_release_event" signal
 *	callback.
 */
gint EDVImbrKeyEventCB(
         GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
        static gbool reenterent = FALSE;
	gint status = FALSE;
	gint etype;
	guint keyval, state;
	gbool press;
	tlist_struct *tlist;
        edv_core_struct *core_ptr;
        edv_imbr_struct *imbr = (edv_imbr_struct *)data;
        if((widget == NULL) || (key == NULL) || (imbr == NULL))
            return(status);

	if(imbr->processing)
	    return(status);

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
            return(status);

        if(reenterent)
            return(status);
        else
            reenterent = TRUE;

	/* Get event type. */
	etype = key->type;

	/* Get other key event values. */
        press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

	tlist = imbr->tlist;
	if(tlist == NULL)
	{
	    reenterent = FALSE;
	    return(status);
	}

/* Macro to emit a signal stop for a key press or release depending
 * on the current event's type.
 */
#define DO_STOP_KEY_SIGNAL_EMIT	\
{ \
 gtk_signal_emit_stop_by_name( \
  GTK_OBJECT(widget), \
  press ? "key_press_event" : "key_release_event" \
 ); \
}

/* Macro to clamp the GtkAdjustment adj and emit a "value_changed"
 * signal.
 */
#define DO_ADJ_CLAMP_EMIT	\
{ \
 if(adj->value > (adj->upper - adj->page_size)) \
  adj->value = adj->upper - adj->page_size; \
\
 if(adj->value < adj->lower) \
  adj->value = adj->lower; \
\
 gtk_signal_emit_by_name( \
  GTK_OBJECT(adj), "value_changed" \
 ); \
}

/* Macro to clamp the GtkCList clist's focus_row. */
#define DO_CLIST_FOCUS_ROW_CLAMP	\
{ \
 if(clist->focus_row >= clist->rows) \
  clist->focus_row = clist->rows - 1; \
 if(clist->focus_row < 0) \
  clist->focus_row = 0; \
}


        if(1)
        {
	    /* Handle by key value. */
	    switch(keyval)
	    {
	      case GDK_Return:
	      case GDK_KP_Enter:
	      case GDK_ISO_Enter:
	      case GDK_3270_Enter:
                /* Handle only if control key modifier is not held, so
                 * that accelerator keys will get handled properly.
                 */
                if(!(state & GDK_CONTROL_MASK))
		{
		    if(press)
			EDVImbrTListDoOpenObject(
			    imbr, imbr->tlist_selected_thumb, state
			);
		    DO_STOP_KEY_SIGNAL_EMIT
		    status = TRUE;
		}
		break;

              case GDK_Delete:
                if(press)
                {
                    EDVImbrOPDelete(imbr);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

	      default:
                /* For all other alphanumeric character keys and while
                 * no modifier keys are held, attempt to seek to the
                 * item who's name starts with the letter of the key
                 * that was pressed.
                 */
                if(isalnum((char)keyval) &&
                   ((state == 0x00000000) || (state == GDK_SHIFT_MASK))
                )
                {
                    TListFreeze(tlist);
                    EDVTListSeekCharacter(
                        tlist, 0,
                        (gchar)((state & GDK_SHIFT_MASK) ?
                            toupper((char)keyval) : keyval
                        )
                    );
                    TListThaw(tlist);

                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

	    }
	}

#undef DO_CLIST_FOCUS_ROW_CLAMP
#undef DO_ADJ_CLAMP_EMIT
#undef DO_STOP_KEY_SIGNAL_EMIT

	reenterent = FALSE;
	return(status);
}

/*
 *	Image browser GtkWidget "button_press_event" signal callback.
 */
gint EDVImbrButtonPressEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	static gbool reenterent = FALSE;
	gint status = FALSE;
	gint etype;
	guint state;
	edv_core_struct *core_ptr;
	tlist_struct *tlist;
        edv_imbr_struct *imbr = (edv_imbr_struct *)data;
        if((widget == NULL) || (button == NULL) || (imbr == NULL))
            return(status);

        if(imbr->processing)
            return(status);

	core_ptr = (edv_core_struct *)imbr->core_ptr;
	if(core_ptr == NULL)
	    return(status);

	tlist = imbr->tlist;
	if(tlist == NULL)
	    return(status);

	if(reenterent)
	    return(status);
	else
	    reenterent = TRUE;

	/* Get event type. */
	etype = button->type;

	state = button->state;

	/* Event occured on the thumbs list? */
        if(widget == tlist->list_da)
        {
	    gint	thumb_num = -1, total_selected_thumbs = 0,
			single_selected_thumb = -1;
	    GList *glist = tlist->selection;


	    /* Get total number of selected thumbs and set
	     * single_selected_thumb to be the only single selected
	     * thumb or -1 of there are none or multiple thumbs selected.
	     */
	    while(glist != NULL)
	    {
		single_selected_thumb = (gint)glist->data;
		total_selected_thumbs++;
		glist = glist->next;
	    }
	    if(total_selected_thumbs != 1)
		single_selected_thumb = -1;

	    /* Get thumb that button event occured over. */
	    TListGetSelection(
		tlist, button->x, button->y, &thumb_num, NULL, NULL
	    );

	    /* Handle by button number. */
	    switch(button->button)
	    {
	      case 3:
		if(etype == GDK_BUTTON_PRESS)
		{
		    GtkMenu *menu;

		    /* Select item before mapping menu? */
		    if(EDVCFGItemListGetValueI(
			core_ptr->cfg_list, EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS
		    ) && (thumb_num > -1))
		    {
			/* Select the thumb that the button was pressed over. */
			TListFreeze(tlist);
			if(!(button->state & GDK_CONTROL_MASK) &&
                           !(button->state & GDK_SHIFT_MASK) &&
			   (thumb_num != single_selected_thumb)
			)
			    TListUnselectAll(tlist);
			tlist->focus_thumb = thumb_num;
			TListSelectThumb(tlist, thumb_num);
			TListThaw(tlist);
		    }

		    /* Update all menus and map right click menu. */
		    EDVImbrUpdateMenus(imbr);
                    menu = (GtkMenu *)imbr->tlist_menu;
		    if(menu != NULL)
			gtk_menu_popup(
			    menu, NULL, NULL,
			    NULL, NULL,
			    button->button, button->time
			);
		}
		status = TRUE;
		break;

	      case 2:
                if(etype == GDK_BUTTON_PRESS)
                {
		    if(thumb_num > -1)
			EDVImbrTListDoFPromptRename(imbr, thumb_num);
		}
                status = TRUE;
		break;

	      case 1:
		/* Double click? */
		if(etype == GDK_2BUTTON_PRESS)
		{
                    if(thumb_num > -1)
		    {
			EDVImbrTListDoOpenObject(imbr, thumb_num, state);
			status = TRUE;
		    }
		}
		break;
	    }
	}
        /* Event occured on the image viewer? */
        else if(widget == (GtkWidget *)ImgViewGetViewWidget(imbr->imgview))
        {
            switch(button->button)
            {
              case 3:
                if(etype == GDK_BUTTON_PRESS)
                {
#if 0
                    GtkMenu *menu;

                    /* Update all menus and map right click menu. */
                    EDVImbrUpdateMenus(imbr);
                    menu = (GtkMenu *)imbr->tlist_menu;
                    if(menu != NULL)
                        gtk_menu_popup(
                            menu, NULL, NULL,
                            NULL, NULL,
                            button->button, button->time
                        );
#endif
                }
                status = TRUE;
                break;

              case 1:
		/* Double click? */
                if(etype == GDK_2BUTTON_PRESS)
                {
		    gint thumb_num = imbr->tlist_selected_thumb;
                    if(thumb_num > -1)
                    {
                        EDVImbrTListDoOpenObject(imbr, thumb_num, state);
                        status = TRUE;
                    }
		}
		break;
	    }
	}

	reenterent = FALSE;
	return(status);
}

/*
 *	GtkHandleBox "child_attached" signal callback.
 */
void EDVImbrHandleChildAttachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
        edv_imbr_struct *imbr = (edv_imbr_struct *)data;
	if((handle_box == NULL) || (imbr == NULL))
	    return;

	gtk_widget_queue_resize(
	    gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}

/*
 *      GtkHandleBox "child_detached" signal callback.
 */
void EDVImbrHandleChildDetachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
        edv_imbr_struct *imbr = (edv_imbr_struct *)data;
        if((handle_box == NULL) || (imbr == NULL))
            return;

        gtk_widget_queue_resize(
            gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
        );
}


/*
 *	Image browser thumbs list select thumb callback.
 */
void EDVImbrTListSelectCB(
        gpointer tl, GdkEventButton *button, gint thumb_num, gpointer data
)
{
        static gbool reenterent = FALSE;
	edv_core_struct *core_ptr;
	tlist_struct *tlist;
        edv_imbr_struct *imbr = (edv_imbr_struct *)data;
        if(imbr == NULL)
            return;

        if(imbr->processing)
            return;

	core_ptr = (edv_core_struct *)imbr->core_ptr;
	if(core_ptr == NULL)
	    return;

	tlist = imbr->tlist;
	if(tlist == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        if(1)
        {
	    /* Change in selection? */
	    if(imbr->tlist_selected_thumb != thumb_num)
	    {
		gint total_selected = 0;
		GList *glist;
		edv_object_struct *obj;


                EDVImbrSetBusy(imbr, TRUE);
		GUIBlockInput(imbr->toplevel, TRUE);
                imbr->processing = TRUE;


		/* Get total number of selected thumbs. */
		glist = tlist->selection;
		while(glist != NULL)
		{
		    total_selected++;
		    glist = glist->next;
		}

                /* Get disk object of selected node. */
                obj = (edv_object_struct *)TListGetThumbData(
                    tlist, thumb_num
                );

		/* Scroll if selected thumb is not visible. */
		if(TListIsThumbVisible(tlist, thumb_num) !=
		    GTK_VISIBILITY_FULL
		)
		    TListMoveTo(tlist, thumb_num, 0.5);

		/* Update current selected thumb on the tlist. */
                imbr->tlist_selected_thumb = thumb_num;

		/* Match device index number from core structure's list of
		 * devices who's mount path matches the selected disk
		 * object's path.
		 */
		if(obj != NULL)
		    EDVeviceMatchListByMountPath(
			core_ptr->device, core_ptr->total_devices,
			&imbr->selected_dev_num,
			obj->full_path
		    );


                /* Update DND icon for thumbs list. */
                EDVImbrTListDNDSetIcon(imbr, thumb_num);

		/* Update title and status bar if selected disk object is
		 * valid.
		 */
		if(obj != NULL)
		{
		    gchar *buf;
		    gchar size_str[80];
		    const gchar *type_str;


                    /* Get object type string and size string. */
                    *size_str = '\0';
                    switch(obj->type)
                    {
                      case EDV_OBJECT_TYPE_FILE:
                        type_str = "File";
                        sprintf(size_str, " (%s bytes)",
                            EDVGetObjectSizeStr(
                                (edv_core_struct *)imbr->core_ptr,
                                obj->size
                            )
			);
                        break;
                      case EDV_OBJECT_TYPE_DIRECTORY:
                        type_str = "Directory";
                        break;
                      case EDV_OBJECT_TYPE_LINK:
                        type_str = "Link";
                        break;
                      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
                        type_str = "Block device";
                        break;
                      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
                        type_str = "Character device";
                        break;
                      case EDV_OBJECT_TYPE_FIFO:
                        type_str = "FIFO Pipe";
                        break;
                      case EDV_OBJECT_TYPE_SOCKET:
                        type_str = "Socket";
                        break;
		      default:	/* Unknown type of object. */
			type_str = "Object";
			break;
                    }

                    /* Format status bar message. */
                    if(total_selected > 1)
                        buf = g_strdup_printf(
                            "%i objects selected",
                            total_selected
                        );
                    else if(!strcmp(obj->name, ".."))
                        buf = g_strdup_printf(
                            "Parent directory selected"
                        );
                    else
                        buf = g_strdup_printf(
                            "%s \"%s\" selected%s",
                            type_str, obj->name, size_str
                        );

		    /* Set status bar message. */
		    EDVStatusBarMessage(
			imbr->status_bar, buf, FALSE
		    );

		    /* Deallocate status bar message. */
		    g_free(buf);


		    /* Check if only one thumb is selected and load image
		     * on the image viewer.
		     */
		    if(total_selected <= 1)
		    {
			EDVImbrImgViewLoad(imbr, obj->full_path);
		    }
		}

		imbr->processing = FALSE;
                GUIBlockInput(imbr->toplevel, FALSE);
                EDVImbrSetBusy(imbr, FALSE);

		EDVImbrUpdateMenus(imbr);
	    }
	}

	reenterent = FALSE;
}

/*
 *      Image browser thumbs list unselect thumb callback.
 */
void EDVImbrTListUnselectCB(
        gpointer tl, GdkEventButton *button, gint thumb_num, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_core_struct *core_ptr;
        tlist_struct *tlist;
        edv_imbr_struct *imbr = (edv_imbr_struct *)data;
        if(imbr == NULL)
            return;

        if(imbr->processing)
            return;

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
            return;

        tlist = imbr->tlist;
        if(tlist == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        if(1)
        {
	    /* Was there a thumb previously selected? */
	    if(imbr->tlist_selected_thumb > -1)
	    {
/*
		EDVImbrSetBusy(imbr, TRUE);
                GUIBlockInput(imbr->toplevel, TRUE);
		imbr->processing = TRUE;
 */

		/* Mark that no thumb is selected. */
		imbr->tlist_selected_thumb = -1;

		/* Mark that no device is selected. */
		imbr->selected_dev_num = -1;

/*
		imbr->processing = FALSE;
                GUIBlockInput(imbr->toplevel, FALSE);
		EDVImbrSetBusy(imbr, FALSE);
 */
		EDVImbrUpdateMenus(imbr);
	    }
        }

        reenterent = FALSE;
}

/*
 *	Location combo activate callback.
 */
void EDVImbrComboActivateCB(GtkWidget *widget, gpointer data)
{
	static gbool reenterent = FALSE;
	GtkCombo *combo = (GtkCombo *)widget;
	edv_imbr_struct *imbr = (edv_imbr_struct *)data;
        if((widget == NULL) || (imbr == NULL))
            return;

	if(imbr->processing)
	    return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;

	/* Check which widget was activated. */
	if(widget == imbr->location_combo)
	{
            gchar *new_path = EDVCopyEvaluateInputPath(
                NULL,           /* No parent path, imply use toplevel. */
                gtk_entry_get_text(GTK_ENTRY(combo->entry))
            );
	    if(new_path != NULL)
	    {
		EDVImbrSetBusy(imbr, TRUE);
                GUIBlockInput(imbr->toplevel, TRUE);

		/* Update location combo, this is really just to record
		 * the new value as history. The combo will be updated
		 * again (without recording history) if the new path is
		 * valid and selected in the callbacks triggered by
		 * EDVImbrSelectPath() farther below.
		 */
		EDVImbrSetLocation(imbr, new_path, TRUE);

		/* Reset loading procedure values and get new directory
		 * listing, then initialize loading procedure values for
		 * the loading of images from the directory specified by
		 * new_path.
		 */
		EDVImbrSelectPath(imbr, new_path);

                GUIBlockInput(imbr->toplevel, FALSE);
                EDVImbrSetBusy(imbr, FALSE);


		EDVImbrUpdateMenus(imbr);


		/* Deallocate copy of the new path. */
		g_free(new_path);
		new_path = NULL;
	    }
	}

	reenterent = FALSE;
}


/*
 *	Loading procedure timeout callback.
 */
gint EDVImbrLoadingTOCB(gpointer data)
{
	gint status;
	edv_core_struct *core_ptr;
	edv_imbr_struct *imbr = (edv_imbr_struct *)data;
	if(imbr == NULL)
	    return(FALSE);

	if(gtk_events_pending() > 0)
	    return(TRUE);

	if(imbr->processing)
	    return(TRUE);

#define DO_FINAL_RETURN		\
{ \
 imbr->loading_tocb = (guint)-1; \
 return(FALSE); \
}

#define DO_RESCHEDUAL_RETURN	\
{ \
 imbr->loading_tocb = gtk_timeout_add( \
  100, \
  (GtkFunction)EDVImbrLoadingTOCB, \
  imbr \
 ); \
 /* Return and indicate to not call this function again since we \
  * already reschedualed a new call to this function. \
  */ \
 return(FALSE); \
}

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
        {
            DO_FINAL_RETURN
        }


	/* Stop loading? */
	if(imbr->stop_count > 0)
	{
	    imbr->stop_count = 0;
	    imbr->loading_tocb = (guint)-1;

	    EDVStatusBarMessage(
		imbr->status_bar, "Loading interrupted", FALSE
	    );
	    EDVStatusBarProgress(imbr->status_bar, 0.0, FALSE);

	    EDVImbrUpdateMenus(imbr);

	    return(FALSE);
	}

	/* Load the next thumb. */
	status = EDVImbrTListLoadIterate(
	    imbr, TRUE,
	    EDVCFGItemListGetValueI(
		core_ptr->cfg_list, EDV_CFG_PARM_IMBR_THUMB_NO_ENLARGE
	    )
	);
	switch(status)
	{
	  case 0:	/* All thumb loaded. */
            imbr->loading_tocb = (guint)-1;
	    EDVImbrUpdateMenus(imbr);
	    return(FALSE);
	    break;

	  case 1:	/* Thumb loaded okay, but still may be more thumbs
			 * that need loading.
			 */
	    DO_RESCHEDUAL_RETURN
	    break;

	  case -3:	/* System error. */
	    DO_FINAL_RETURN
	    break;

	  default:	/* General error. */
	    /* It's okay to try loading again after this. */
	    DO_RESCHEDUAL_RETURN
	    break;
	}
#undef DO_FINAL_RETURN
#undef DO_RESCHEDUAL_RETURN
}



/*
 *	Called whenever the global write protect has changed.
 *
 *	The new state is given as state.
 */
void EDVImbrWriteProtectChangedCB(
        edv_imbr_struct *imbr, gbool state
)
{
        if(imbr == NULL)
            return;

        if(imbr->processing)
            return;

	EDVImbrUpdateMenus(imbr);
}

/*
 *	Called whenever a disk object has been added.
 */
void EDVImbrObjectAddedNotifyCB(
        edv_imbr_struct *imbr, const gchar *path,
        const struct stat *lstat_buf
)
{
	if(imbr == NULL)
	    return;

        if(imbr->processing)
            return;

        EDVImbrTListObjectAddedNotify(
            imbr, path, lstat_buf
        );

	/* Need to update menus because a new loading process may be
	 * queued.
	 */
	EDVImbrUpdateMenus(imbr);
}

/*
 *      Called whenever a disk object has had its properties modified.
 */
void EDVImbrObjectModifiedNotifyCB(
        edv_imbr_struct *imbr, const gchar *path,
	const gchar *new_path,
        const struct stat *lstat_buf
)
{
        if(imbr == NULL)
            return;

	if(imbr->processing)
	    return;

	/* Check current location first, if the old path matches the
	 * current location then the current location needs to be updated
	 * to reflect the new path before notifying the contents clist.
	 */
	if(path != NULL)
	{
	    const gchar *cstrptr;


	    if(new_path == NULL)
		new_path = path;

	    /* Check if the old path matches the current location, if it
	     * does then the current location needs to be updated to the
	     * new path.
	     */
	    cstrptr = EDVImbrCurrentLocation(imbr);
	    if((cstrptr != NULL) ? !strcmp(cstrptr, path) : FALSE)
	    {
		/* Old path matches current location, change values
		 * reflecting the current location to the value of
		 * new_path.
		 */
		EDVImbrSetTitle(imbr, new_path);
		EDVImbrSetLocation(imbr, new_path, FALSE);
                EDVImbrUpdateLocationIcon(imbr, new_path);
	    }
	}

        EDVImbrTListObjectModifiedNotify(
            imbr, path, new_path, lstat_buf
        );

/*	EDVImbrUpdateMenus(imbr); */
}

/*
 *	Called whenever a disk object has been removed.
 */
void EDVImbrObjectRemovedNotifyCB(
        edv_imbr_struct *imbr, const gchar *path
)
{
        if(imbr == NULL)
            return;

        if(imbr->processing)
            return;

        EDVImbrTListObjectRemovedNotify(imbr, path);

	/* Check if the path that was removed is the current location
	 * (after letting contents clist and directory ctree check on it
	 * first), if the path that was removed is the location then
	 * set the current location to be the parent of the path that
	 * was removed.
	 */
        if(path != NULL)
        {
	    const gchar *cstrptr;

            /* Check if the removed object path matches the current
	     * location. If it matches then the current location needs to
	     * be updated to reflect that.
	     */
            cstrptr = EDVImbrCurrentLocation(imbr);
            if((cstrptr != NULL) ? !strcmp(cstrptr, path) : FALSE)
            {
                /* Removed object path matches the current location, so
		 * technically the current location needs to be cleared
		 * however we'll instead just change the current location
		 * value to reflect the value of the parent path of the
		 * removed object path.
                 */
		cstrptr = GetParentDir(path);
		if(cstrptr != NULL)
		{
		    EDVImbrSetTitle(imbr, cstrptr);
		    EDVImbrSetLocation(imbr, cstrptr, FALSE);
                    EDVImbrUpdateLocationIcon(imbr, cstrptr);
		}
            }
        }

/*	EDVImbrUpdateMenus(imbr); */
}


/*
 *      Called whenever a device has been mounted or unmounted.
 */
void EDVImbrMountNotifyCB(
	edv_imbr_struct *imbr,
	gint dev_num, edv_device_struct *dev_ptr,
	gbool is_mounted
)
{
        if(imbr == NULL)
            return;

        if(imbr->processing)
            return;

	EDVImbrTListMountNotify(imbr, dev_ptr, is_mounted);

        EDVImbrUpdateMenus(imbr);
}


/*
 *	Called whenever an object has been added to the recycle bin.
 */
void EDVImbrRecycledObjectAddedNotifyCB(
        edv_imbr_struct *imbr, guint index
)
{
        edv_core_struct *core_ptr;


	if(imbr == NULL)
	    return;

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
            return;


	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects.
	 */
        if(core_ptr->last_recbin_items != imbr->last_recbin_items)
	    EDVImbrUpdateMenus(imbr);
}

/*
 *	Called whenever an object has been removed from the recycle bin.
 */
void EDVImbrRecycledObjectRemovedNotifyCB(
        edv_imbr_struct *imbr, guint index
)
{
	edv_core_struct *core_ptr;


        if(imbr == NULL)
            return;

	core_ptr = (edv_core_struct *)imbr->core_ptr;
	if(core_ptr == NULL)
	    return;


        /* There is not much interest when a recycled object has been
         * added or removed. Only the menus need to be updated if there
         * is a change in the number of recycled objects.
         */
	if(core_ptr->last_recbin_items != imbr->last_recbin_items)
	    EDVImbrUpdateMenus(imbr);
}


/*
 *	Called whenever the global configuration has changed.
 */
void EDVImbrReconfiguredNotifyCB(edv_imbr_struct *imbr)
{
	const gchar *cstrptr;
	gchar *cur_path;
	GtkWidget *w;
	edv_status_bar_struct *status_bar;
	edv_core_struct *core_ptr;
	tlist_struct *tlist;
	imgview_struct *iv;


        if(imbr == NULL)
            return;

        if(imbr->processing)
            return;

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
            return;


        /* Reset last state markers so that resources get explicitly
         * checked due to reconfiguring.
         */
	imbr->last_recbin_items = -1;
	imbr->last_write_protect_state = -1;


	/* Get copy of current location path. */
	cstrptr = EDVImbrCurrentLocation(imbr);
	cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


        /* Update title. */
        EDVImbrSetTitle(imbr, cur_path);

	/* Regenerate tool bar. */
	EDVImbrToolbarRegenerate(imbr);

	/* Show tool bar? */
	w = imbr->tool_bar_handle;
	if(w != NULL)
	{
	    imbr->tool_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_IMBR_SHOW_TOOL_BAR
            );
	    if(imbr->tool_bar_map_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

        /* Show location bar? */
        w = imbr->location_bar_handle;
        if(w != NULL)
        {
            imbr->location_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_IMBR_SHOW_LOCATION_BAR
            );
            if(imbr->location_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show mount bar? */
        w = imbr->mount_bar_handle;
        if(w != NULL)
        {
            imbr->mount_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_IMBR_SHOW_MOUNT_BAR
            );
            if(imbr->mount_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show find bar? */
        w = imbr->find_bar_handle;
        if(w != NULL)
        {
            imbr->find_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_IMBR_SHOW_FIND_BAR
            );
            if(imbr->find_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show status bar? */
        status_bar = imbr->status_bar;
        if(status_bar != NULL)
        {
            imbr->status_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_IMBR_SHOW_STATUS_BAR
            );
            if(imbr->status_bar_map_state)
                EDVStatusBarMap(status_bar);
            else
                EDVStatusBarUnmap(status_bar);
        }


	/* Update thumbs list. */
	tlist = imbr->tlist;
	if(tlist != NULL)
	{
	    gbool lists_double_buffer = (gbool)EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_LISTS_DOUBLE_BUFFER
            );

            tlist->horizontal = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_IMBR_THUMB_LIST_HORIZONTAL
            );

	    tlist->thumb_width = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_IMBR_THUMB_WIDTH
            );
            tlist->thumb_height = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_IMBR_THUMB_HEIGHT
            );
            tlist->thumb_border = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_IMBR_THUMB_BORDER
            );

	    /* Update double buffer setting, this will cause the list
	     * to recreate its internal buffers and redraw.
	     */
	    TListDoubleBuffer(tlist, lists_double_buffer);

	    /* Notify thumbs list of new sizes, this will also cause it
	     * to redraw itself.
	     */
	    TListResize(tlist, -1, -1, FALSE);
	}

        /* Reset displayed thumb values. */
	TListFreeze(tlist);
	EDVImbrTListResetThumbs(imbr);
	TListThaw(tlist);


	/* Begin update image viewer. */
	iv = imbr->imgview;
	if(iv != NULL)
	{
	    /* Image quality. */
	    switch(EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_IMAGE_QUALITY
            ))
	    {
	      case 0:
		iv->quality = 0;
		break;
              case 1:
                iv->quality = 1;
                break;
              case 2:
                iv->quality = 2;
                break;

	      default:
		iv->quality = 1;
		break;
	    }


	    /* Redraw image viewer and update menus. */
	    ImgViewDraw(iv);
	    ImgViewUpdateMenus(iv);
	}


	/* Update menus. */
        EDVImbrUpdateMenus(imbr);


	/* Notify image browser's toplevel widget to resize. */
	w = imbr->toplevel;
	if(w != NULL)
	    gtk_widget_queue_resize(w);


	/* Deallocate copy of current location path. */
	g_free(cur_path);
	cur_path = NULL;
}


/*
 *      Called whenever a MIME Type has been added to the core
 *      structure's list of MIME Types.
 */
void EDVImbrMimeTypeAddedCB(
        edv_imbr_struct *imbr,
        gint mt_num, edv_mimetype_struct *mt_ptr
)
{
        /* Treat a MIME Type added the same as it would be for a MIME Type
         * modified, forward signal to the MIME Type modified callback.
         */
        EDVImbrMimeTypeModifiedCB(imbr, mt_num, mt_ptr);
}

/*
 *      Called whenever a MIME Type has been modified on the core
 *      structure's list of MIME Types.
 */
void EDVImbrMimeTypeModifiedCB(
        edv_imbr_struct *imbr,
        gint mt_num, edv_mimetype_struct *mt_ptr
)
{
        const gchar *cstrptr;
        gchar *cur_path;
        edv_core_struct *core_ptr;
	tlist_struct *tlist;


        if(imbr == NULL)
            return;

        if(imbr->processing)
            return;

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
            return;

	tlist = imbr->tlist;
	if(tlist == NULL)
	    return;


        /* Get copy of current location path. */
        cstrptr = EDVImbrCurrentLocation(imbr);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


        /* Reset displayed thumb values. */
        TListFreeze(tlist);
        EDVImbrTListResetThumbs(imbr);
        TListThaw(tlist);

        /* Deallocate copy of current location path. */
        g_free(cur_path);
        cur_path = NULL;
}

/*
 *      Called whenever a MIME Type has been removed from the core
 *      structure's list of MIME Types.
 */
void EDVImbrMimeTypeRemovedCB(
        edv_imbr_struct *imbr, gint mt_num
)
{
        const gchar *cstrptr;
        gchar *cur_path;
        edv_core_struct *core_ptr;
        tlist_struct *tlist;


        if(imbr == NULL)
            return;

        if(imbr->processing)
            return;

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
            return;

        tlist = imbr->tlist;
        if(tlist == NULL)
            return;


        /* Get copy of current location path. */
        cstrptr = EDVImbrCurrentLocation(imbr);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;



        /* Reset displayed thumb values. */
        TListFreeze(tlist);
        EDVImbrTListResetThumbs(imbr);
        TListThaw(tlist);

        /* Deallocate copy of current location path. */
        g_free(cur_path);
        cur_path = NULL;
}


/*
 *      Called whenever a device has been added to the core
 *      structure's list of devices.
 */
void EDVImbrDeviceAddedCB(
        edv_imbr_struct *imbr,
        gint dev_num, edv_device_struct *dev_ptr
)
{
	/* Treat a device added the same as it would be for a device
	 * modified, forward signal to the device modified callback.
	 */
	EDVImbrDeviceModifiedCB(imbr, dev_num, dev_ptr);
}

/*
 *	Called whenever a device has been modified on the core
 *	structure's list of devices.
 */
void EDVImbrDeviceModifiedCB(
        edv_imbr_struct *imbr,
        gint dev_num, edv_device_struct *dev_ptr
)
{
	const gchar *cstrptr;
	gchar *cur_path;
        edv_core_struct *core_ptr;
        tlist_struct *tlist;


        if(imbr == NULL)
            return;

        if(imbr->processing)
            return;

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
            return;

        tlist = imbr->tlist;
        if(tlist == NULL)
            return;


        /* Get copy of current location path. */
        cstrptr = EDVImbrCurrentLocation(imbr);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;



        /* Reset displayed thumb values. */
        TListFreeze(tlist);
        EDVImbrTListResetThumbs(imbr);
        TListThaw(tlist);

        /* Update menus. */
        EDVImbrUpdateMenus(imbr);

        /* Deallocate copy of current location path. */
        g_free(cur_path);
        cur_path = NULL;
}

/*
 *      Called whenever a device has been removed from the core
 *      structure's list of devices.
 */
void EDVImbrDeviceRemovedCB(
        edv_imbr_struct *imbr, gint dev_num
)
{
        const gchar *cstrptr;
        gchar *cur_path;
	edv_mountbar_struct *mb;
        edv_core_struct *core_ptr;
        tlist_struct *tlist;


        if(imbr == NULL)
            return;

        if(imbr->processing)
            return;

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
            return;

        tlist = imbr->tlist;
        if(tlist == NULL)
            return;


        /* Get copy of current location path. */
        cstrptr = EDVImbrCurrentLocation(imbr);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


	/* Check if mount bar is currently referencing this device, if it
	 * is then its selected device needs to be set to -1.
	 */
	mb = imbr->mountbar;
	if(mb != NULL)
	{
	    if(mb->selected_dev_num == dev_num)
		mb->selected_dev_num = -1;
	    /* Mount bar will be updated further below when menus are
	     * updated.
	     */
	}


        /* Reset displayed thumb values. */
        TListFreeze(tlist);
        EDVImbrTListResetThumbs(imbr);
        TListThaw(tlist);

        /* Update menus. */
        EDVImbrUpdateMenus(imbr);

        /* Deallocate copy of current location path. */
        g_free(cur_path);
        cur_path = NULL;
}
